#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import FdlInventoryInitializer

class TempSensorInventoryDirInitializer( FdlInventoryInitializer.DirInitializer ):
   managedTypeName = "Inventory::SmbusTempSensorDir"

   def getModelName( self, **kwargs ):
      tableVersion = kwargs.get( 'tableVersion' )
      if not tableVersion:
         return self.invTableInfo_[ kwargs[ 'key' ] ][ 4 ]
      else:
         return None

   def initInventoryEntry( self, **kwargs ):
      tableVersion = kwargs.get( 'tableVersion' )
      key = kwargs[ 'key' ]
      getNextSensorId = kwargs.get( 'getNextSensorId' )
      addSmbusDeviceToBus = kwargs.get( 'addSmbusDeviceToBus' )

      if not tableVersion:
         # Required arguments: bus, deviceId, description, position,
         # modelName, api, overheatThreshold, criticalThreshold
         #
         # Optional arguments (can be None): groupName, targetTemperature,
         # ultimateGainFrontToBack,
         # ultimatePeriodFrontToBack,
         # ultimateGainBackToFront,
         # ultimatePeriodBackToFront,
         # offset, hystTemp, ignoreForBoardInit, remoteDiodes

         ( smbus, deviceId, description, position, modelName, api, groupName,
           targetTemperature,
           overheatThreshold,
           criticalThreshold,
           ultimateGainFrontToBack,
           ultimatePeriodFrontToBack,
           ultimateGainBackToFront,
           ultimatePeriodBackToFront,
           offset,
           hystTemp,
           ignoreForBoardInit,
           remoteDiodes ) = self.invTableInfo_[ key ]
         assert modelName in [ "Adt7483", "Max6697", "Max6658", "Max6581", "Lm73" ]

      else:
         # If we ever want to change the structure of the table (add, remove,
         # or rearrange attributes), without having to rewrite old fdls, we
         # can create another tableVersion and put it here.
         return None

      sensorId = key
      if getNextSensorId:
         sensorId = getNextSensorId()

      assert overheatThreshold < criticalThreshold

      sensor = self.invEntity_.newTempSensor( deviceId, sensorId )
      if getNextSensorId is None:
         sensorId += 1
      sensor.description = description
      sensor.position = position
      sensor.modelName = modelName
      sensor.api = api

      if modelName == "Adt7483":
         sensor.readDelay = "delay10ms"
         sensor.writeDelay = "delay10ms"
      elif modelName == "Lm73":
         sensor.writeDelay = "delay50ms"

      if groupName is not None:
         sensor.groupName = groupName
      if targetTemperature is not None:
         assert targetTemperature < overheatThreshold
         sensor.targetTemperature = targetTemperature
      sensor.overheatThreshold = overheatThreshold
      sensor.criticalThreshold = criticalThreshold
      if ultimateGainFrontToBack is not None:
         assert ultimateGainFrontToBack <= 1
         sensor.ultimateGainFrontToBack = ultimateGainFrontToBack
      if ultimatePeriodFrontToBack is not None:
         assert ultimatePeriodFrontToBack >= 0
         sensor.ultimatePeriodFrontToBack = ultimatePeriodFrontToBack

      if ultimateGainBackToFront is not None:
         assert ultimateGainBackToFront <= 1
         sensor.ultimateGainBackToFront = ultimateGainBackToFront
      if ultimatePeriodBackToFront is not None:
         assert ultimatePeriodBackToFront >= 0
         sensor.ultimatePeriodBackToFront = ultimatePeriodBackToFront

      if offset is not None:
         sensor.offset = offset

      if hystTemp is not None:
         sensor.hystTemp = hystTemp

      if ignoreForBoardInit is not None:
         sensor.ignoreForBoardInitialization = ignoreForBoardInit

      if remoteDiodes is not None:
         for rd in remoteDiodes:
            if not tableVersion:
               ( localId, description, position, groupName,
                 targetTemperature, overheatThreshold, criticalThreshold ) = rd[ :7 ]
            else:
               continue

            assert overheatThreshold < criticalThreshold

            if getNextSensorId:
               sensorId = getNextSensorId()
            remote = sensor.newRemoteDiode( sensorId )
            if getNextSensorId is None:
               sensorId += 1

            remote.localId = localId
            remote.description = description
            remote.position = position
            if groupName is not None:
               remote.groupName = groupName
            if targetTemperature is not None:
               assert targetTemperature < overheatThreshold
               remote.targetTemperature = targetTemperature
            remote.overheatThreshold = overheatThreshold
            remote.criticalThreshold = criticalThreshold

            if len( rd[ 7: ] ) > 0:
               if rd[ 7 ] is not None:
                  assert rd[ 7 ] <= 1
                  remote.ultimateGainFrontToBack = rd[ 7 ]
            if len( rd[ 8: ] ) > 0:
               if rd[ 8 ] is not None:
                  assert rd[ 8 ] >= 0
                  remote.ultimatePeriodFrontToBack = rd[ 8 ]
            if len( rd[ 9: ] ) > 0:
               if rd[ 9 ] is not None:
                  assert rd[ 9 ] <= 1
                  remote.ultimateGainBackToFront = rd[ 9 ]
            if len( rd[ 10: ] ) > 0:
               if rd[ 10 ] is not None:
                  assert rd[ 10 ] >= 0
                  remote.ultimatePeriodBackToFront = rd[ 10 ]
            if len( rd[ 11: ] ) > 0:
               if rd[ 11 ] is not None:
                  remote.offset = rd[ 11 ]
            if len( rd[ 12: ] ) > 0:
               if rd[ 12 ] is not None:
                  remote.hystTemp = rd[ 12 ]
            if len( rd[ 13: ] ) > 0:
               if rd[ 13 ] is not None:
                  remote.truThermEnable = rd[ 13 ]
            if len( rd[ 14: ] ) > 0:
               if rd[ 14 ] is not None:
                  remote.alertDisabled = rd[ 14 ]
            if len( rd[ 15: ] ) > 0:
               if rd[ 15 ] is not None:
                  remote.ignoreForBoardInitialization = rd[ 15 ]

      if addSmbusDeviceToBus is not None:
         addSmbusDeviceToBus( sensor )
      else:
         smbus.device.addMember( sensor )
         sensor.upstreamBus = smbus

      return sensor

def Plugin( context ):
   context.registerInitializer( TempSensorInventoryDirInitializer )
