#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re
import tempfile
import threading

import Ark
import Tac
import Tracing

import ExtensionMgr.errors as errors
import ExtensionMgr.rpmutil as rpmutil


__defaultTraceHandle__ = Tracing.Handle( 'ExtensionMgrYum' )
t3 = Tracing.trace3

# Since yum imports rpm (which is blacklisted by Eos/test/CliTests.py),
# we lazy import this module.
yum = Ark.LazyModule( 'yum' )

# This module provides the package manager interface implementation
# for Yum repositories/packages.

YUM_REPO_TEMPLATE = """
[{name}]
name={description}
baseurl={url}
skip_if_unavailable=True
enabled=1
"""

# The downloadonly plugin causes yum to exit with errorcode 1 on
# successful completion. Because... Redhat.
YUM_PLUGIN_EXIT = 1
YUM_DOWNLOADONLY_MSG = 'exiting because --downloadonly specified'

YUM_REPO_DIR = '/etc/yum.repos.d'
RPM_FINAL_DEST = '/mnt/flash/.extensions/'
_name_prefix = 'eos-'
_yumBase = None

RE_YUM_PKG_NOT_AVAIL = re.compile( r'\nNo package (.+?) available\.' )

threadLock = threading.Lock()

def yumbase():
   global _yumBase
   with threadLock:
      if _yumBase is None:
         _yumBase = yum.YumBase()
      return _yumBase

def _repo_fn( repo ):
   return os.path.join( YUM_REPO_DIR, '%s%s.repo' % ( _name_prefix, repo.name ) )

def updateRepository( repo, timeout=60 ):
   t3( 'update repo', repo )
   reponame = repo.name.replace( ' ', '_' )
   contents = YUM_REPO_TEMPLATE.format(
      name=reponame,
      description=repo.description or reponame,
      url=repo.url )
   with tempfile.NamedTemporaryFile() as f:
      f.write( contents )
      f.flush()

      try:
         cmd = [ 'cp', '-f', f.name, _repo_fn( repo ) ]
         Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )
         cmd = [ 'chmod', 'og+r', _repo_fn( repo ) ]
         Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )
      except Tac.SystemCommandError as e:
         t3( 'Failed to update repository filename %s - output: %s'
             % ( _repo_fn( repo ), e.output ) )
         raise errors.Error( 'Failed to update repository %s' % repo.name )

def deleteRepository( repo ):
   t3( 'delete repo', repo )

   fn = _repo_fn( repo )
   try:
      cmd = [ 'rm', '-f', fn ]
      Tac.run( cmd, asRoot=True )
   except Tac.SystemCommandError as e:
      t3( 'failed to delete repo file', fn, 'error', str( e ) )

def _infoFromYumPackages( packages ):
   infos = []
   for p in packages:
      infoKey = Tac.Value( 'Extension::InfoKey', p.name, 'formatRpm', 1 )
      info = Tac.newInstance( 'Extension::Info', infoKey )
      # We can't use readRpmHeaderIntoDict because even the yum library
      # it uses depends upon the physical file existing on the filesystem
      # instead of just reading the information from the matching results
      info.format = 'formatRpm'
      info.presence = 'available'
      info.status = 'notInstalled'
      info.name = p.name
      info.size = p.size
      info.desc = p.description
      info.vendor = p.vendor
      info.epoch = p.epoch
      info.version = p.version
      info.release = p.release
      info.url = p.url
      info.license = p.license
      info.summary = p.summary
      info.primaryPkg = p.name

      infos.append( info )
   return infos

def packageSearch( query, repo ):
   t3( 'package search', query, 'in', repo )
   yb = yumbase()
   matches = yb.searchGenerator( [ 'name' ], [ query ] )
   pkgs = []
   # filter by repo
   for ( p, _ ) in matches:
      if p.ui_from_repo == repo.name:
         pkgs.append( p )

   return _infoFromYumPackages( pkgs )

def packageDownload( name, repo, status ):
   """
   Downloads the package and dependencies to a temporary directory.
   We then go through and create the Info object to return.
   Finally we copy the files to /flash/.extensions.
   """

   # XXX This function could do with some better error checking.
   # I.e. can we screen scrape to see if dependent RPMs weren't
   # downloaded. What should we do in that case?

   t3( 'package download', name, 'from repo', repo )
   tmpDir = tempfile.mkdtemp()
   cmd = [ 'yum', 'install',
           '--downloadonly', '-t', '-y', '--color=never',
           '--downloaddir=%s' % ( tmpDir, ), name ]

   try:
      output = Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
      t3( 'package download output', output )
   except Tac.SystemCommandError as e:
      if YUM_DOWNLOADONLY_MSG in e.output and e.error == YUM_PLUGIN_EXIT:
         # This is a normal case, no error here.
         pass
      elif 'There are no enabled repos.' in e.output:
         # This is a potential issue; repos listed in system config but not
         # written to disk for yum to know of.
         t3( 'Yum has no enabled repos but we acted for repo:', repo )
         raise errors.NoRepoError( repo )
      else:
         match = RE_YUM_PKG_NOT_AVAIL.search( e.output )
         if match:
            pkgname = match.group( 1 )
            t3( 'yum package %r not available' % pkgname )
            raise errors.PackageNotAvailableError( pkgname )
         else:
            t3( 'Package %s failed to install - output: %s' % ( name, e.output ) )
            raise errors.InstallError(
               'Failed to download package %s: %s' % ( name, e ) )

   # Get the filename of the main RPM to create the infoKey
   primaryPkg = None
   primaryInfo = None
   for _, _, files in os.walk( tmpDir ):
      for f in files:
         fPath = os.path.join( tmpDir, f )
         header = rpmutil.readRpmHeaderIntoDict( rpmutil.newTransactionSet(), fPath )
         if header.get( 'name' ).lower() == name.lower():
            primaryPkg = os.path.basename( f ) 
            primaryInfo = _addPkgToStatus( status, fPath, None )

   if primaryPkg is None:
      raise errors.PackageNameDownloadMismatchError( 
         "Package named '%s' was downloaded but could not be read for "
         "installation." % name )

   # call addRpm for each dependency then move the files to the final location
   for _, _, files in os.walk( tmpDir ):
      t3( 'Adding %d RPM to package %s' % ( len( files ), name ) )
      for f in files:
         fPath = os.path.join( tmpDir, f )
         _addPkgToStatus( status, fPath, primaryInfo )
         t3( 'moving', fPath, 'to', RPM_FINAL_DEST )
         cmd = [ 'mv', '-f', fPath, RPM_FINAL_DEST ]
         finalDest = os.path.join( RPM_FINAL_DEST, f )
         try:
            Tac.run( cmd, asRoot=True )
            # The Cli expects to have control over those files. With vfat
            # this was garanteed by mount options (gid=88, umask=007). With
            # ext4, this has to be manually done
            Tac.run( [ 'chown', 'root:eosadmin', finalDest ], asRoot=True,
                     ignoreReturnCode=True )
            Tac.run( [ 'chmod', '660', finalDest ], asRoot=True,
                     ignoreReturnCode=True )
         except Tac.SystemCommandError as e:
            # We didn't successfully move this RPM to its install location.
            raise errors.InstallError( 'Failed to download RPM %s' % ( f, ) )

def _addPkgToStatus( status, fPath, primaryInfo ):
   name = os.path.basename( fPath )
   if primaryInfo is not None and name == primaryInfo.primaryPkg:
      # We have already added this package
      return
   key = Tac.Value( 'Extension::InfoKey', name, 'formatYum', 1 )
   info = status.info.newMember( key )
   info.presence = 'present'
   info.status = 'notInstalled'
   info.primaryPkg = name
   info.filepath = RPM_FINAL_DEST + name 
   header = rpmutil.readRpmHeaderIntoDict( rpmutil.newTransactionSet(), fPath )
   # For dependencies we do not keep track of all the packages downloaded. This
   # will be trakced via the info of the primary package downloaded only.
   rpmutil.addRpm( info, name, header )
   if primaryInfo is not None:
      primaryInfo.package.newMember( name )
   return info


