#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import Agent
import Cell
import os
from socket import (
      socket, AF_INET, SOCK_RAW, IPPROTO_RAW, SOL_IP, AF_INET6, IPPROTO_UDP )
from socket import error as socketError
from Arnet.Device import Tap
from Arnet import PktParserTestLib
from IpLibConsts import DEFAULT_VRF
# pylint: disable-msg=C0412
from Arnet.VxlanPktTestLib import VxlanHdrSize
from Arnet.IpTestLib import (
      IpHdrSize, Ip6HdrSize, UdpHdrSize, EthHdrSize, IPV4, IPV6 )
from Arnet.NsTestLib import nsNameFromVrfName, netNsExists
from Arnet.NsLib import DEFAULT_NS
from eunuchs.if_ether_h import ETH_P_IP, ETH_P_IPV6
import SmashLazyMount
from TypeFuture import TacLazyType

AgentName = 'EvpnrtrEncap'

evpnrtrTraceHandle = Tracing.Handle( AgentName )
t0 = evpnrtrTraceHandle.trace0
t1 = evpnrtrTraceHandle.trace1
t5 = evpnrtrTraceHandle.trace5
t9 = evpnrtrTraceHandle.trace9

# set the mtu of the tun interfaces to max size as we don't want to fragment packets.
MTU = 65535

IP_MTU_DISCOVER = 10
IP_PMTUDISC_DO = 2
IPV6_MTU_DISCOVER = 23
IPV6_PMTUDISC_DO = 2
SOL_IPV6 = 41

TunnelTableIdentifier = TacLazyType( "Tunnel::TunnelTable::TunnelTableIdentifier" )
TunnelTableMounter = TacLazyType( "Tunnel::TunnelTable::TunnelTableMounter" )

vxlanEncapType = Tac.Type( "Vxlan::VxlanEncap" )

def getVxlanDefaultUdpPort():
   vtiCfg = Tac.Type('Vxlan::VtiConfig')
   return vtiCfg.vxlanWellKnownPort

def createTunnelTap( name, vrfName=DEFAULT_VRF ):
   nsName = nsNameFromVrfName( vrfName )
   tun = None
   try:
      # This will issue a system call and it can fail for various reasons.
      # Most typcial failure  observed is, test code/ user script can delete
      # the vrf instance <vrf-name> from the configuration. This results in the
      # deletion of namespace from kernel. When this happens kernel can return
      # with error code as EPERM or ENOENT and sometimes with error code 255.
      tun = Tap( ifname=name, netNs=nsName, mtu=MTU, blocking=False,
            ethernet=False, dadEnabled=False )

   except Tac.SystemCommandError as e:
      # Just log the error returned from syscall and continue to chug along.
      t0( 'createTunnelTap for name:', name, ' vrfName:', vrfName, ' error:',
            e.error )
      # Has someone else created a tun interface with this name?  This agent cleans
      # up its tun interfaces upon death. 
      debug = False
      if debug:
         # Turn on debug mode to get the backtrace of offending system call
         raise e
      return tun
   else:
      return tun


class VrfNetnsReactor( Tac.Notifiee ):
   '''
   The VrfNetnsReactor reacts to changes to the networkNamespace in a VrfStatusLocal.
   
   When the network namespace is set, we create the tun interfaces for it - one tun
   interface per Evpn tunnel entry.
   When network namespace is unset, clean up the tun interfaces and fds.
   '''
   notifierTypeName = 'Ip::VrfStatusLocal'

   def __init__( self, vrfStatus, vrfSet, tunIntfReactors, tunnelTable,
                 vtiStatusDir, bridgingConfig, outSocket, outSocket6 ):
      t5( 'VrfNetnsReactor.__init__' )
      self.vrfStatus_ = vrfStatus
      self.tunIntfReactors_ = tunIntfReactors
      self.vrfSet_ = vrfSet
      self.tunnelTable_ = tunnelTable
      self.vtiStatusDir_ = vtiStatusDir
      self.bridgingConfig_ = bridgingConfig
      self.outSocket_ = outSocket
      self.outSocket6_ = outSocket6
      Tac.Notifiee.__init__( self, vrfStatus )

      if vrfStatus.networkNamespace:
         self.netnsAdded()

   def __del__( self ):
      t9( 'VrfNetnsReactor.__del__' )

   @Tac.handler( 'networkNamespace' )
   def handleNetworkNamespace( self ):
      netNs = self.vrfStatus_.networkNamespace
      t9( 'VrfNetnsReactor.handleNetworkNamespace for netNs:' + netNs )
      if netNs:
         self.netnsAdded()
      else:
         self.netnsRemoved()

   def netnsAdded( self ):
      vrfName = self.vrfStatus_.vrfName
      t5( 'VrfNetnsReactor.netnsAdded for vrfName:' + vrfName )
      self.vrfSet_.add( vrfName )

      # for each tunnel id, create a TunInterfaceReactor and update the dict
      for tunnelKey, vrfToIntfs in self.tunIntfReactors_.iteritems():
         if vrfToIntfs.get( vrfName, None ):
            t0( 'VrfNetnsReactor.netnsAdded error: a TunInterfaceReactor '
                  'already exists for vrfName:' + vrfName )
            raise
         
         tunnelEntry = self.tunnelTable_.entry.get( tunnelKey, None )
         if tunnelEntry is None:
            # The TunnelTableReactor will clean up the tunIntfReactors_ later.
            continue
         tapName = self.tunnelTable_.tunKernelIntfName( tunnelEntry.key )
         tun = createTunnelTap( name=tapName, vrfName=vrfName )
         tun.ip6EnableIs()

         intfReactor = TunInterfaceReactor( tun, tunnelEntry, self.vtiStatusDir_,
               self.bridgingConfig_, self.outSocket_, self.outSocket6_ )
         vrfToIntfs[ vrfName ] = ( intfReactor, tun )

   def netnsRemoved( self ):
      vrfName = self.vrfStatus_.vrfName
      t5( 'VrfNetnsReactor.netnsRemoved for vrfName:' + vrfName )
      self.vrfSet_.remove( vrfName )
      # for each tunnel id, remove the TunInterfaceReactor and update the dict
      for vrfToIntfs in self.tunIntfReactors_.itervalues():
         intfReactor, _ = vrfToIntfs.pop( vrfName, ( None, None ) )
         if not intfReactor:
            t5( 'VrfNetnsReactor.netnsRemoved: no TunInterfaceReactor exists for '
                  'vrfName:' + vrfName + ' but it may have already been removed.' )

class VrfStatusLocalReactor( Tac.Notifiee ):
   '''
   The VrfStatusLocalReactor reacts to changes in the vrf collection.  When a new vrf
   appears, we create a VrfNetnsReactor.  When it disappears, we remove the reactor.
   '''
   notifierTypeName = 'Ip::AllVrfStatusLocal'

   def __init__( self, tunIntfReactors, vrfSet, tunnelTable, vtiStatusDir,
                 bridgingConfig, allVrfStatusLocal, outSocket, outSocket6 ):
      self.tunIntfReactors_ = tunIntfReactors
      self.vrfSet_ = vrfSet
      self.tunnelTable_ = tunnelTable
      self.vtiStatusDir_ = vtiStatusDir
      self.bridgingConfig_ = bridgingConfig
      self.allVrfStatusLocal_ = allVrfStatusLocal
      self.outSocket_ = outSocket
      self.outSocket6_ = outSocket6
      Tac.Notifiee.__init__( self, allVrfStatusLocal )

      self.vrfReactors_ = {}

      for vrfName, vrfStatus in self.allVrfStatusLocal_.vrf.iteritems():
         self.handleVrfAdded( vrfName, vrfStatus )

   def __del__( self ):
      t9( 'VrfStatusLocalReactor.__del__' )

   @Tac.handler( 'vrf' )
   def handleVrf( self, key ):
      vrfStatus = self.allVrfStatusLocal_.vrf.get( key )
      t9( 'VrfStatusLocalReactor.handleVrf for key:' + key +
            ' vrfStatus.networkNamespace:' +
            ( vrfStatus.networkNamespace if vrfStatus else 'None' ) )
      if key in self.vrfReactors_:
         if not vrfStatus:
            # handle vrf deletion and remove vrf netns reactor
            vrfReactor = self.vrfReactors_.pop( key )
            vrfReactor.netnsRemoved()
      else:
         assert vrfStatus
         self.handleVrfAdded( key, vrfStatus )

   def handleVrfAdded( self, vrfName, vrfStatus ):
      # create vrf netns reactor
      self.vrfReactors_[ vrfName ] = VrfNetnsReactor( vrfStatus, self.vrfSet_,
            self.tunIntfReactors_, self.tunnelTable_, self.vtiStatusDir_,
            self.bridgingConfig_, self.outSocket_, self.outSocket6_ )

class TunInterfaceReactor( Tac.Notifiee ):
   '''
   The TunInterfaceReactor reacts to a tun interface's file descriptor when it
   becomes readable.
   It receives the packet and encapsulates it with the correct VXLAN information for
   Evpn. Then, it sends the packet out a raw IP socket to the destination VTEP.
   '''
   notifierTypeName = 'Tac::FileDescriptor'

   def __init__( self, tun, tunnelEntry, vtiStatusDir, bridgingConfig, outSocket,
                 outSocket6 ):
      t5( 'TunInterfaceReactor.__init__ for tun:%s fd:%d'
            % ( tun.name_, tun.fileno() ) )
      self.tun_ = tun
      self.tunnelEntry_ = tunnelEntry
      self.vtiStatusDir_ = vtiStatusDir
      self.bridgingConfig_ = bridgingConfig

      self.tacFd_ = Tac.newInstance( 'Tac::FileDescriptor', tun.name_ )
      self.tacFd_.descriptor = tun.fileno()
      self.tacFd_.nonBlocking = True
      self.tacFd_.notifyOnReadable = True
      self.outSocket_ = outSocket 
      self.outSocket6_ = outSocket6
      self.udpPort_ = getVxlanDefaultUdpPort()

      Tac.Notifiee.__init__( self, self.tacFd_ )
      
   def __del__( self ):
      t9( 'TunInterfaceReactor.__del__' )

   @Tac.handler( 'readableCount' )
   def handleReadableCount( self ):
      t9( 'TunInterfaceReactor.handleReadableCount for fd:' +
            str( self.tun_.fileno() ) )
      try:
         # Receive a raw IP packet from the tun interface
         buf = self.tun_.recv() 
         if not buf:
            return

         t5( 'TunInterfaceReactor.handleReadableCount received ' +
             str( len( buf ) ) + ' bytes.' )
         pkt = self.encapVxlanHdr( buf )

         if pkt is not None:
            self.sendPacket( pkt )
      except OSError as e:
         # Likely a safely ignorable error, eg. in the case of a file descriptor in
         # bad state as a vrf is being torn down.
         t0( 'TunInterfaceReactor.handleReadableCount ignoring error for tun:%s ' \
             'fd:%d message:%s' %
             ( self.tun_.name_, self.tun_.fileno(), e.message ) )
      except Exception:
         t0( 'TunInterfaceReactor.handleReadableCount error for tun:%s fd:%d'
               % ( self.tun_.name_, self.tun_.fileno() ) )
         raise

   # returns a packet encapsulated with info from the tunnel table entry 
   def encapVxlanHdr( self, buf ):
      t9( 'encapVxlanHdr' )
      t9( Tracing.HexDump( buf ) )

      # Read IP header to determine if this is an ipv4 or ipv6 packet
      payload = Tac.newInstance( 'Arnet::Pkt' )
      strLen = len( buf )
      if strLen > payload.maxSharedData:
         buf = buf[ : payload.maxSharedData ]
         strLen = payload.maxSharedData
      payload.newSharedHeadData = strLen
      PktParserTestLib.copyStrToPkt( payload, buf )

      # the packet parser will silently parse a pkt not matching
      # the parser's version, so its important to check the version
      # after parsing to ensure we called the right one.
      headers = None
      headers4, headers6 = [], []
      sAddr, dAddr = None, None
      if PktParserTestLib.ipHdrParser( payload, 0, headers4 ) and \
         headers4[ 0 ][ 1 ].version == 4:
         t9( 'payload is ipv4' )
         headers = headers4

      elif PktParserTestLib.ip6HdrParser( payload, 0, headers6 ) and \
           headers6[ 0 ][ 1 ].version == 6:
         t9( 'payload is ipv6' )
         headers = headers6
      else:
         # not ipv4 or ipv6
         raise AssertionError( "EvpnrtrEncapAgent cannot find ip header" )

      assert( headers )
      t9( 'Parsed headers:', headers )
      _, payloadIpHdr = headers[ 0 ]

      srcVti = self.tunnelEntry_.vti
      if not srcVti:
         srcVti = "Vxlan1"
      vti = self.vtiStatusDir_.vtiStatus.get( srcVti )
      if vti is None:
         # VTI is not up.
         # The IPv6 link local packet won't be encapsulated anyway.
         return None

      # Always use local VTEP IP as source when available
      if vti.vxlanEncap == vxlanEncapType.vxlanEncapIp4:
         ipVersion = IPV4
         srcVtepIp = vti.localVtepAddr
         if srcVtepIp == '0.0.0.0':
            srcVtepIp = vti.mlagVtepAddr
         dstVtepIp = self.tunnelEntry_.vtepAddr.v4Addr
      else:
         ipVersion = IPV6
         srcVtepIp = vti.localVtepAddr6
         if srcVtepIp.isUnspecified:
            srcVtepIp = vti.mlagVtepAddr6
         dstVtepIp = self.tunnelEntry_.vtepAddr.v6Addr

      if payloadIpHdr.version == 4:
         sAddr, dAddr = Tac.Value( "Arnet::IpAddr" ), Tac.Value( "Arnet::IpAddr" )
         sAddr.stringValue, dAddr.stringValue = payloadIpHdr.src, payloadIpHdr.dst
      elif payloadIpHdr.version == 6:
         sAddr, dAddr = payloadIpHdr.src, payloadIpHdr.dst
         if sAddr.isLinkLocalUnicast or dAddr.isLinkLocalUnicast:
            t9( 'Discard the packet on IPv6 link-local unicast address' )
            return None
         # In some rare scenarios, an encapsulated IPv6 Vxlan packets may be
         # looped back and received by the EvpnrtrEncapAgent when the receiver
         # is not ready yet. Resend such packets without further encapsulation.
         if sAddr == srcVtepIp and dAddr == dstVtepIp:
            t0( 'Resend the packet looped back on IPv6 address' )
            return buf
      else:
         raise AssertionError( "EvpnrtrEncapAgent cannot find ip header" )

      innerHash = 0
      # try to get the inner proto hash
      if len( headers ) > 1:
         proto, protoHdr = headers[ 1 ]
         if proto in [ 'TcpHdr', 'UdpHdr' ]:
            innerHash = protoHdr.srcPort ^ protoHdr.dstPort
         t9( ' innerHash after inner proto ' + proto + ' :' + str( innerHash ) )
      else:
         t9( ' no inner proto found' )

      # inner Ip header hash
      sHash, dHash = sAddr.hash, dAddr.hash
      innerHash ^= ( sHash >> 16 ) ^ ( sHash & 0x0000FFFF ) ^ \
                   ( dHash >> 16 ) ^ ( dHash & 0x0000FFFF )
      t9( ' innerHash after inner ip hdr:' + str( innerHash ) )

      udpPort = vti.udpPort
      srcMac = self.bridgingConfig_.bridgeMacAddr
      dstMac = self.tunnelEntry_.mac
      vni = self.tunnelEntry_.vni

      t5( 'TunInterfaceReactor.encapVxlanHdr with srcVtepIp:' +
          str( srcVtepIp ) + ' dstVtepIp:' + str( dstVtepIp ) + ' srcMac:' +
          srcMac + ' dstMac:' + dstMac + ' vni:' + str( vni ) )
      t5( 'Inner packet IP version: ' + str( payloadIpHdr.version ) )
      t5( ' Encap VTI: %s' % vti.intfId )

      # construct Ethernet header
      ethHeader = Tac.newInstance( 'Arnet::Pkt' )
      ethHeader.newSharedHeadData = EthHdrSize
      ethHdr = Tac.newInstance( 'Arnet::EthHdrWrapper', ethHeader, 0 )
      ethHdr.src = srcMac
      ethHdr.dst = dstMac
      if payloadIpHdr.version == 4:
         ethHdr.typeOrLen = ETH_P_IP 
      else:
         ethHdr.typeOrLen = ETH_P_IPV6

      # inner ether hash
      sAddr1, dAddr1 = Tac.Value( "Arnet::EthAddr" ), Tac.Value( "Arnet::EthAddr" )
      sAddr1.stringValue, dAddr1.stringValue = ethHdr.src, ethHdr.dst
      sHash, dHash = sAddr1.hash, dAddr1.hash
      innerHash ^= ( sHash >> 16 ) ^ ( sHash & 0x0000FFFF ) ^ \
                   ( dHash >> 16 ) ^ ( dHash & 0x0000FFFF )
      t9( ' innerHash after inner eth hdr: ' + str( innerHash ) )

      # construct VXLAN header
      vxlanHeader = Tac.newInstance( 'Arnet::Pkt' )
      vxlanHeader.newSharedHeadData = VxlanHdrSize 
      vxlanHdr = Tac.newInstance( 'Arnet::VxlanHdrWrapper', vxlanHeader, 0 )
      vxlanHdr.flagsReserved0 = 0 # must transmit as zeroes
      vxlanHdr.flagsReserved1 = 0 # must transmit as zeroes
      vxlanHdr.vniReserved = 0 # must transmit as zeroes
      vxlanHdr.vni = vni
      vxlanHdr.flagsVniValid = True 

      # construct UDP header
      udpHeader = Tac.newInstance( 'Arnet::Pkt' )
      udpHeader.newSharedHeadData = UdpHdrSize 
      udpHdrType = 'Arnet::UdpHdrWrapper'
      if ipVersion == IPV6:
         udpHdrType += 'Ipv6'
      udpHdr = Tac.newInstance( udpHdrType, udpHeader, 0 )
      udpHdr.srcPort = innerHash
      udpHdr.dstPort = udpPort
      udpHdr.len = UdpHdrSize + VxlanHdrSize + EthHdrSize + payload.bytes
      udpHdr.ipSrc = srcVtepIp
      udpHdr.ipDst = dstVtepIp
      udpHdr.checksum = 0
      if ipVersion == IPV6:
         udpChecksum = udpHdr.computedChecksum
         if udpChecksum == 0:
            udpChecksum = 0xffff
         udpHdr.checksum = udpChecksum

      if ipVersion == IPV4:
         # construct IPv4 outer header
         ipHeader = Tac.newInstance( 'Arnet::Pkt' )
         ipHeader.newSharedHeadData = IpHdrSize
         ipHdr = Tac.newInstance( 'Arnet::IpHdrWrapper', ipHeader, 0 )
         ipHdr.version = 4 # IPv4
         ipHdr.headerLen = 5 # No IP options. in 32-bit words. 5*4=20 bytes
         ipHdr.tos = 0 # default
         ipHdr.totalLen = ( ipHdr.headerLen * 4 ) + udpHdr.len # in bytes
         ipHdr.id = 0x1 # first IP datagram
         ipHdr.fragmentOffset = 0 # not fragmented
         ipHdr.moreFrag = False # not fragmented
         ipHdr.dontFrag = True # don't fragment packets
         ipHdr.reservedFragFlag = False
         ipHdr.ttl = 0x40 # default ttl is 64
         ipHdr.protocolNum = 'ipProtoUdp' # encapsulating a UDP packet
         ipHdr.src = srcVtepIp
         ipHdr.dst = dstVtepIp
         ipHdr.checksum = ipHdr.computedChecksum
      else:
         # construct IPv6 outer header
         ipHeader = Tac.newInstance( 'Arnet::Pkt' )
         ipHeader.newSharedHeadData = Ip6HdrSize
         ipHdr = Tac.newInstance( 'Arnet::Ip6HdrWrapper', ipHeader, 0 )
         ipHdr.version = IPV6
         ipHdr.trafficClass = 0 # default
         ipHdr.flowLabel = 0
         ipHdr.payloadLen = udpHdr.len # in bytes
         ipHdr.nextHeader = IPPROTO_UDP
         ipHdr.hopLimit = 0x40 # default ttl is 64
         ipHdr.src = srcVtepIp
         ipHdr.dst = dstVtepIp

      return ipHeader.stringValue + udpHeader.stringValue + \
            vxlanHeader.stringValue + ethHeader.stringValue + \
            payload.stringValue

   def sendPacket( self, pkt ):
      dstIp = self.tunnelEntry_.vtepAddr
      srcVti = self.tunnelEntry_.vti
      if not srcVti:
         srcVti = "Vxlan1"
      vti = self.vtiStatusDir_.vtiStatus.get( srcVti )
      udpPort = vti.udpPort

      t5( 'TunInterfaceReactor.sendPacket to dstIp:' + dstIp.stringValue + ' port:' +
          str( udpPort ) )
      t9( Tracing.HexDump( str( pkt ) ) )

      try:
         if vti.vxlanEncap == vxlanEncapType.vxlanEncapIp4:
            self.outSocket_.sendto( str( pkt ), ( dstIp.stringValue, udpPort ) )
         else:
            self.outSocket6_.sendto( pkt, ( dstIp.stringValue, 0 ) )

         t9( 'TunInterfaceReactor.sendPacket completed' )

      except socketError, e:
         t0( 'TunInterfaceReactor.sendPacket error:' + str( e ) )

class TunnelTableReactor( Tac.Notifiee ):
   '''
   The TunnelTableReactor reacts to the EvpnVxlanTunnelTable.
   Upon addition of a tunnel table entry, we create a tun interface representing this
   entry in all vrfs.
   Upon removal of a tunnel table entry, we remove the tun interface from all vrfs.
   '''
   notifierTypeName = 'Tunnel::TunnelTable::EvpnVxlanTunnelTable'

   def __init__( self, tunnelTable, vtiStatusDir, bridgingConfig, tunIntfReactors,
                 vrfSet, outSocket, outSocket6 ):
      t5( 'TunnelTableReactor.__init__' )
      self.tunnelTable_ = tunnelTable
      self.vtiStatusDir_ = vtiStatusDir
      self.bridgingConfig_ = bridgingConfig
      self.vrfSet_ = vrfSet
      self.outSocket_ = outSocket
      self.outSocket6_ = outSocket6
      self.tunIntfReactors_ = tunIntfReactors
      Tac.Notifiee.__init__( self, self.tunnelTable_ )
      # counts tunnel table entry updates for testing purposes
      self.entryUpdateCount_ = 0

      # iterate tunnel table entries to create the tun interface reactors
      for key in self.tunnelTable_.entry.iterkeys():
         self.handleEntry( key )

   def __del__( self ):
      t9( 'TunnelTableReactor.__del__' )

   @Tac.handler( 'entry' )
   def handleEntry( self, key ):
      t9( 'TunnelTableReactor.handleEntry for key:' + str( key ) )
      tunnelEntry = self.tunnelTable_.entry.get( key )
      if not tunnelEntry:
         self.handleRemoveEntry( key )
         return

      vrfToIntfs = self.tunIntfReactors_.get( key )
      if vrfToIntfs:
         t5( 'TunnelTableReactor.handleEntry update key:' + str( key ) )
         # The only thing that can change is the vlan id, which we don't use for
         # encapsulation.  Assert that the rest of the information is the same, for
         # one of the TunInterfaceReactors
         intfReactor, _ = vrfToIntfs.itervalues().next()
         assert intfReactor 
         if intfReactor.tunnelEntry_.vtepAddr != tunnelEntry.vtepAddr:
            raise RuntimeError( 'tunnel entry vtepAddr has changed' )
         if intfReactor.tunnelEntry_.mac != tunnelEntry.mac:
            raise RuntimeError( 'tunnel entry mac has changed' )
         if intfReactor.tunnelEntry_.vni != tunnelEntry.vni:
            raise RuntimeError( 'tunnel entry vni has changed' )

         # update our entry update count; this is done merely for test purposes
         self.entryUpdateCount_ += 1

      else:
         t5( 'TunnelTableReactor.handleEntry add key:' + str( key ) )
         tapName = self.tunnelTable_.tunKernelIntfName( tunnelEntry.key )
         vrfToIntfs = dict()

         for vrf in self.vrfSet_:
            nsName = nsNameFromVrfName( vrf )
            t5( 'nsName: %s' % nsName )
            if nsName == DEFAULT_NS or netNsExists( nsName ):
               # Either default NS or user defined NS exist
               t5( 'tapName: %s vrf: %s' % ( tapName, vrf ) )
               tun = createTunnelTap( name=tapName, vrfName=vrf )
               if tun:
                  tun.ip6EnableIs()
                  intfReactor = TunInterfaceReactor( tun, tunnelEntry,
                        self.vtiStatusDir_, self.bridgingConfig_, self.outSocket_,
                        self.outSocket6_ )
                  vrfToIntfs[ vrf ] = ( intfReactor, tun )

         self.tunIntfReactors_[ key ] = vrfToIntfs 

   def handleRemoveEntry( self, key ):
      t5( 'TunnelTableReactor.handleRemoveEntry remove key:' + str( key ) )
      vrfToIntfs = self.tunIntfReactors_.pop( key, None )
      if not vrfToIntfs:
         t5( 'TunnelTableReactor.handleRemoveEntry: no TunInterfaceReactor to '
               'remove for key:' + str( key ) +
               ' but it may have already been removed.' )
      vrfToIntfs.clear()

class EvpnrtrEncapAgent( Agent.Agent ):
   '''
   EvpnrtrEncapAgent performs software forwarding of Evpn Vxlan packets to the remote
   VTEP.
   It creates a tun interface per EvpnVxlanTunnelTable entry.  When a packet is
   received on any of these tun interfaces, this agent performs the necessary
   encapsulation of the packet and sends it back to the kernel.
   '''
   def __init__( self, entityManager ):
      t1( 'EvpnrtrEncapAgent.__init__ starting' )
      self.vtiStatusDir_ = None
      self.tunnelTable_ = None
      self.bridgingConfig_ = None
      self.allVrfStatusLocal_ = None
      self.tunnelTableReactor_ = None
      self.vrfStatusReactor_ = None

      # Raw IP socket on which we will send out vxlan encapsulated packets 
      self.outSocket_ = socket( AF_INET, SOCK_RAW, IPPROTO_RAW )
      self.outSocket_.setsockopt( SOL_IP, IP_MTU_DISCOVER, IP_PMTUDISC_DO )
      # Raw socket for IPv6 underlay
      sock = socket( AF_INET6, SOCK_RAW, IPPROTO_RAW )
      sock.setsockopt( SOL_IPV6, IPV6_MTU_DISCOVER, IPV6_PMTUDISC_DO )
      self.outSocket6_ = sock

      # a dictionary of dictionaries where the outer key is the tunnelId and the
      # inner key is the vrfName
      #
      #    { [ tunnelId1, { [ vrfName1, ( tunInterfaceReactor1, tun1 ) ],
      #                     [ vrfName2, ( tunInterfaceReactor2, tun2 ) ],
      #                   } ],
      #      [ tunnelId2, { [ vrfName1, ( tunInterfaceReactor3, tun3 ) ],
      #                     [ vrfName2, ( tunInterfaceReactor4, tun4 ) ],
      #                   } ],
      #    }
      self.tunIntfReactors_ = {}
      Tac.activityManager.useEpoll = True

      # a set of vrfs that have their network namespaces created
      # default vrf is added since it always exists
      self.vrfSet_ = { 'default' }

      Agent.Agent.__init__( self, entityManager, agentName=AgentName )
   
   def __del__( self ):
      t9( 'EvpnrtrEncapAgent.__del__' )

   def doInit( self, entityManager ):
      t5( 'doInit mounting entities' )
      mg = entityManager.mountGroup()
      self.vtiStatusDir_ = mg.mount( 'interface/status/eth/vxlan',
                                     'Vxlan::VtiStatusDir', 'r' )
      tableInfo = TunnelTableMounter.getMountInfo(
         TunnelTableIdentifier.evpnVxlanTunnelTable ).tableInfo
      self.tunnelTable_ = SmashLazyMount.mount(
         self.entityManager, tableInfo.mountPath, tableInfo.tableType,
         SmashLazyMount.mountInfo( "keyshadow" ) )

      self.bridgingConfig_ = mg.mount( 'bridging/config', 'Bridging::Config', 'r' )
      self.allVrfStatusLocal_ = mg.mount( Cell.path( 'ip/vrf/status/local' ),
                                          'Ip::AllVrfStatusLocal', 'r' )

      def _finish():
         t5( 'done mounting' )
         self.tunnelTableReactor_ = TunnelTableReactor( self.tunnelTable_,
               self.vtiStatusDir_, self.bridgingConfig_, self.tunIntfReactors_,
               self.vrfSet_, self.outSocket_, self.outSocket6_ )

         self.vrfStatusReactor_ = VrfStatusLocalReactor( self.tunIntfReactors_,
               self.vrfSet_, self.tunnelTable_, self.vtiStatusDir_,
               self.bridgingConfig_, self.allVrfStatusLocal_, self.outSocket_,
               self.outSocket6_ )

      mg.close( _finish )
      t5( 'doInit completed' )

   # helper function for tests that get the number of tunnel entry updates
   def getTunnelEntryUpdates( self ):
      return self.tunnelTableReactor_.entryUpdateCount_

def main():
   tfm = Tac.singleton( 'Tac::TraceFacilityMan' )
   traceSetting = AgentName + '/01'
   if 'TRACE' in os.environ.keys():
      traceSetting = os.environ[ 'TRACE' ] + ',' + traceSetting
   os.environ[ 'TRACE' ] = traceSetting
   tfm.doReadEnvironment()

   container = Agent.AgentContainer( [ EvpnrtrEncapAgent ],
                                     agentTitle=AgentName,
                                     passiveMount=True )
   # Process name and syslog name equals agent class name. This makes it
   # inconsistent with another agent name used in mount profile, log filename,
   # etc. DO NOT COPY THIS. See AID/8010 Appendix G for more info.
   container.useLegacyName_DO_NOT_COPY_THIS()
   container.runAgents()

if __name__ == '__main__':
   main()
