#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
# pylint: disable-msg=W0702
# pylint: disable-msg=E0602
# pkgdeps : library EventMon
import Tac
import Agent
import os
import sqlite3
import Tracing
import re
import Logging
import Plugins
import glob

Logging.logD( id="EVENTMON_DB_WRITE_FAILED",
              severity=Logging.logError,
              format="A sqlite %s exception occurred "
              "when writing to the EventMon database",
              explanation="An error occurred when trying to write to "
              "EventMon Database",
              recommendedAction=Logging.CONTACT_SUPPORT )

__defaultTraceHandle__ = Tracing.Handle( "EventMon" )
th = Tracing.defaultTraceHandle()
t0 = th.trace0
t8 = th.trace8

QT_PARSE_RE = re.compile( "\(.*\)" )

def removeFile( filePath ):
   try:
      if os.path.isdir( filePath ):
         raise OSError( 'EventMonAgent attempting to delete a directory' )
      else:
         os.remove( filePath )
   except OSError:
      t0( 'could not delete', filePath )

def QTFileIterator( qtFileName ):
   ''' returns msg lines for qt file '''
   try:
      os.stat( qtFileName )
   except OSError:
      t0( 'could not find qt file', qtFileName )
      return

   # typical output line looks as below:
   # 2012-05-18 09:13:18 0 00051.325776031, +965174 "1 %
   # (1.1.1.0/32,,,,removed,11)
   #
   # space split output looks as follows:
   # ['2012-05-18', '09:13:18', '0', '00051.325776031,',
   # '+965174', '"1', '%', '(1.1.1.0/32,,,,removed,11)', '"']
   #
   # first,second fields are date,time
   # seventh field is msg - use QT_PARSE_RE to strip the braces
   #   and quotes to get the user msg
   try:
      DATE_FIELD = 0
      TIME_FIELD = 1

      output = Tac.run( [ '/usr/bin/qtcat', qtFileName ], stdout=Tac.CAPTURE )
      if not output:
         return

      traceLines = output.split( '\n' )
      for traceLine in traceLines:
         if not traceLine or "first msg" in traceLine:
            continue

         traceAttrArr = traceLine.split( " " )
         msgMatch = QT_PARSE_RE.search( traceLine )
         if msgMatch:
            msgLine = msgMatch.group( 0 )
            timestamp = traceAttrArr[ DATE_FIELD ] \
                  + " " + traceAttrArr[ TIME_FIELD ]
            msg = ''.join( c for c in msgLine if c not in '()\'\ ' )
            yield ( timestamp, msg )
   except:
      t0( "qtcat failed", qtFileName )

def foreverLogEnabled( config, tableId=None ):
   if not tableId:
      return config.foreverLogEnabled
   table = config.table[ tableId ]
   if ( table.foreverLogOverride and table.foreverLogEnabled ) or \
      ( not table.foreverLogOverride and config.foreverLogEnabled ):
      return True
   return False

def foreverLogPath( config, tableId ):
   table = config.table[ tableId ]
   if table.foreverLogOverride:
      foreverPath = "%s/%s" % ( table.foreverLogDirectory,
                                 tableId + table.foreverLogFileName )
   else:
      foreverPath = "%s/%s" % ( config.foreverLogDirectory,
                                 tableId + config.foreverLogFileName )
   return foreverPath

class EventMonDbMgr( object ):
   def __init__( self, dbLocation, tableSchema ):
      self.initialized = False
      self.dbLocation = dbLocation
      self.tableSchema = tableSchema
      self.flushDb()

   def flushDb( self, ):
      self.deleteDb()
      self.createDb()

      if not self.initialized:
         t0( "db creation attempt failed, deleting empty eventMon.db" )
         self.deleteDb()

   def flushTable( self, tableName ):
      t0( "flushTable", tableName )
      if not self.initialized:
         t0( "eventMon.db not initialized" )
         return

      db = sqlite3.connect( self.dbLocation )
      try:
         with db:
            columnNames = ",".join( [ ( "%s %s" % ( attr, attrType ) )
               for attr, attrType, _ in self.tableSchema[ tableName ] ] )
            db.execute( "drop table if exists %s" % tableName )
            db.execute( "create table %s (%s)" % ( tableName, columnNames ) )
            for attr, attrType, indexed in self.tableSchema[ tableName ]:
               if indexed:
                  db.execute( "create index idx_%s_%s on %s (%s)" %
                        ( tableName, attr, tableName, attr ) )

      except sqlite3.Error as e:
         t0( "flushTable: exception", e )
         Logging.log( EVENTMON_DB_WRITE_FAILED, e )
      finally:
         db.close()

   def createDb( self, ):
      t0( "createDb at", self.dbLocation )
      db = sqlite3.connect( self.dbLocation )
      try:
         with db:
            for tableName in self.tableSchema:
               columnNames = ",".join( [ ( "%s %s" % ( attr, attrType ) )
                  for attr, attrType, _ in self.tableSchema[ tableName ] ] )
               db.execute( "create table if not exists %s (%s)" %
                     ( tableName, columnNames ) )
               # Index needs to be created after the table is,
               # meaning we have to go though again
               for attr, attrType, indexed in self.tableSchema[ tableName ]:
                  if indexed:
                     db.execute( "create index idx_%s_%s on %s (%s)" %
                           ( tableName, attr, tableName, attr ) )
            self.initialized = True
      except sqlite3.Error as e:
         t0( "createDb: exception", e )
         self.initialized = False
         Logging.log( EVENTMON_DB_WRITE_FAILED, e )
      finally:
         db.close()

   def deleteDb( self, ):
      t0( "deleteDb at", self.dbLocation )
      removeFile( self.dbLocation )

   def syncQTToDb( self, qtFileName, tableName ):
      t8( "syncQTToDb", qtFileName, "to", tableName )
      if not self.initialized:
         t0( "eventMon.db not initialized..attempting flush" )
         # attempt creating db
         self.flushDb()
         if not self.initialized:
            t0( "eventMon.db recreation failed. not syncing" )
            return False
         else:
            t0( "eventMon.db recreate successful..attempting sync" )

      if ( not tableName ) or ( tableName not in self.tableSchema ):
         t0( "table", tableName, "not found" )
         return True

      db = sqlite3.connect( self.dbLocation )
      try:
         with db:
            fileIter = QTFileIterator( qtFileName )
            if not fileIter:
               t0( "no qt records found in ", qtFileName )
               return True
            for timeStamp, evtMsg in fileIter:
               attrTuple = ( timeStamp, ) + tuple( evtMsg.split( "," ) )
               numFields = len( self.tableSchema[ tableName ] )
               if len( attrTuple ) != numFields:
                  t0( "len(attrTuple) is", len( attrTuple ), "expected", numFields )
                  continue
               sql = ( "insert into %s values ( %s )" %
                     ( tableName, ",".join( "?" * numFields ) ) )
               try:
                  db.execute( sql, attrTuple )
                  t8( sql, attrTuple )
               except sqlite3.IntegrityError as e:
                  # duplicate val exception - ignore
                  pass
      except sqlite3.Error as e:
         t0( "syncQtToDb: exception ", e )
         Logging.log( EVENTMON_DB_WRITE_FAILED, e )
         return False
      finally:
         db.close()
      return True

   def removeFromDb( self, table, fileName ):
      if not self.initialized:
         t0( "eventMon.db not initialized" )
         return

      db = sqlite3.connect( self.dbLocation )
      fileIter = QTFileIterator( fileName )
      if not fileIter:
         t0( "no qt records found in ", fileName )
         return

      msgs = [ msg for _, msg in fileIter ]
      if not msgs:
         return
      splitLastMsg = msgs[ -1 ].split( ',' )
      if not splitLastMsg:
         return
      counter = splitLastMsg[ -1 ]
      sql = ( "delete from %s where counter <= %s" % ( table, counter ) )
      try:
         with db:
            db.execute( sql )
      except sqlite3.Error as e:
         t0( "removeFromDb exception ", e )
         Logging.log( EVENTMON_DB_WRITE_FAILED, e )
      finally:
         db.close()

# This class manages the the sync of the qt files to the sqlite db and
# buffer flush. Both the events are triggered from Cli
class EventMonEpochReactor( Tac.Notifiee ):
   notifierTypeName = "EventMon::EpochMarker"
   MAX_SYNCS_PER_ROUND = 5

   def __init__( self, epochMarker, agent ):
      self.agent_ = agent
      Tac.Notifiee.__init__( self, epochMarker )
      self.timer_ = Tac.ClockNotifiee( self.syncFiles, timeMin=Tac.endOfTime )
      self.tableChangeList = []
      for table in self.agent_.config.table:
         self.agent_.syncedFile[ table ] = {}

      # create qt directory if it doesn't exist
      qtDir = self.agent_.config.qtDir
      if not os.path.exists( qtDir ):
         os.makedirs( qtDir )
      self.handleSyncEpoch()

   def foreverDirQueue( self, eventMonTableId ):
      ''' queue the files in the forever dir for later synchronization '''
      t8( "foreverDirQueue", eventMonTableId )
      if not foreverLogEnabled( self.agent_.config, eventMonTableId ):
         return

      foreverFullPath = foreverLogPath( self.agent_.config, eventMonTableId )
      for fn in glob.iglob( "%s.[0-9]*" % foreverFullPath ):
         if not re.search( r'\.\d+$', fn ):
            continue

         if fn not in self.agent_.syncedFile[ eventMonTableId ]:
            self.agent_.syncedFile[ eventMonTableId ][ fn ] = False

   def eventMonTableChangeList( self, ):
      t8( "eventMonTableChangeList" )
      changeList = []
      for tableName in self.agent_.config.table:
         if( tableName in self.agent_.statusMount ) and (
               self.agent_.status.tableSyncCount[ tableName ] !=
               self.agent_.statusMount[ tableName ].count() ):
            t8( tableName, " eventCount : ",
                  self.agent_.statusMount[ tableName ].count(), " syncCount : ",
                  self.agent_.status.tableSyncCount[ tableName ] )
            changeList.append( tableName )
      return changeList

   @Tac.handler( 'bufferFlushEpoch' )
   def handleBufferFlush( self, name=None ):
      ''' remove eventmon current and forever logs '''
      self.agent_.eventMonDbMgr.flushDb()
      pathname = "%s/*%s*" % ( self.agent_.config.qtDir,
               self.agent_.config.qtFileName )
      for fn in glob.iglob( pathname ):
         removeFile( fn )

      if self.agent_.config.foreverLogEnabled:
         pathname = "%s/*%s*" % ( self.agent_.config.foreverLogDirectory,
               self.agent_.config.foreverLogFileName )
         for fn in glob.iglob( pathname ):
            if re.search( r'\.\d+$', fn ):
               removeFile( fn )

      for tableConfig in self.agent_.config.table.itervalues():
         if tableConfig.foreverLogOverride and tableConfig.foreverLogEnabled:
            pathname = "%s/*%s*" % ( self.agent_.config.foreverLogDirectory,
                self.agent_.config.foreverLogFileName )
            for fn in glob.iglob( pathname ):
               if re.search( r'\.\d+$', fn ):
                  removeFile( fn )
         self.agent_.syncedFile[ tableConfig.name ] = {}

      # We cleanup Forever directories here, the inotify watcher must watch new
      # directories. Hence we reinstantiate the config reactor
      self.agent_.configReactor = EventMonConfigReactor( self.agent_.config,
            self.agent_ )
      self.agent_.status.newBufferFlushEpoch = self.notifier().bufferFlushEpoch

   def syncFiles( self ):
      t8( 'syncFiles', self.agent_.syncedFile )

      syncSuccess = True
      t0( "syncing foreverLog files" )
      for tableName in self.tableChangeList:
         # files in the collection that have not yet been synced
         # take 1 at a time
         unSyncedFiles = sorted( [ f for ( f, synced ) in self.agent_.syncedFile[
            tableName ].iteritems() if not synced ],
            key=lambda x: int( x.split( '.' )[ -1 ] ) )[ :
                  EventMonEpochReactor.MAX_SYNCS_PER_ROUND ]
         t8( unSyncedFiles )
         if unSyncedFiles:
            for fn in unSyncedFiles:
               syncSuccess = self.agent_.eventMonDbMgr.syncQTToDb( fn, tableName )
               if syncSuccess:
                  self.agent_.syncedFile[ tableName ][ fn ] = True
               else:
                  t0( "Failed syncing %s %s " % ( fn, tableName ) )
                  break

            if not syncSuccess:
               break
            else:
               t0( "Scheduling timer for next round" )
               self.timer_.timeMin = Tac.now()
               return
         else:
            t0( "no unsynced files..disabling timer" )
            self.timer_.timeMin = Tac.endOfTime

      # sync qt files
      t0( "syncing qt files" )
      if syncSuccess:
         qtDir = self.agent_.config.qtDir
         qtFileName = self.agent_.config.qtFileName
         for tableName in self.tableChangeList:
            t8( "event count for ", tableName, " ",
                  self.agent_.statusMount[ tableName ].count() )
            qtPathName = qtDir + tableName + qtFileName
            oldQtFileList = sorted( [ ( int( fn.split( "." )[ -1 ] ), fn )
                  for fn in glob.iglob( qtPathName + ".[0-9]" ) ] )
            for _, fn in oldQtFileList:
               syncSuccess = self.agent_.eventMonDbMgr.syncQTToDb( fn, tableName )
               if not syncSuccess:
                  t0( "Failed syncing %s %s " % ( fn, tableName ) )
                  break

            if syncSuccess:
               syncSuccess = self.agent_.eventMonDbMgr.syncQTToDb( qtPathName,
                     tableName )
               if not syncSuccess:
                  t0( "Failed syncing %s %s " % ( qtPathName, tableName ) )

      # update status counts on successful sync
      if syncSuccess:
         for tableName in self.tableChangeList:
            if tableName in self.agent_.statusMount:
               self.agent_.status.tableSyncCount[ tableName ] = \
                   self.agent_.statusMount[ tableName ].count()

      # sync epoch is updated anyways to signify the end of this sync round
      # note that the syncSuccess doesn't play a part in finishing a sync
      # epoch round, so if the error condition causing sync failure is resolved
      # user can issue another sync request to get the latest state
      self.agent_.status.newSyncEpoch = self.notifier().syncEpoch
      t8( 'done syncing' )

   @Tac.handler( 'syncEpoch' )
   def handleSyncEpoch( self, ):
      t8( "handleSyncEpoch" )
      # turn off existing timer
      t0( "turning off existing timer" )
      self.timer_.timeMin = Tac.endOfTime
      self.tableChangeList = self.eventMonTableChangeList()

      for tableName in self.tableChangeList:
         self.agent_.syncedFile[ tableName ] = {}
         self.agent_.eventMonDbMgr.flushTable( tableName )
         if foreverLogEnabled( self.agent_.config, tableName ):
            self.foreverDirQueue( tableName )
      self.syncFiles()

class ForeverCleanupReactor( Tac.Notifiee ):
   notifierTypeName = "EventMon::ForeverFileTableCounter"
   MAX_DELETES_PER_ROUND = 5

   def __init__( self, config, foreverFileTableCounter, agent ):
      Tac.Notifiee.__init__( self, foreverFileTableCounter )
      self.timer_ = Tac.ClockNotifiee( self.handleForeverCleanup,
            timeMin=Tac.endOfTime )
      self.agent_ = agent
      self.config = config
      self.deletedFileCountPerRound = 0
      self.handleForeverCleanup()

   def foreverCleanupExtraLogs( self, tableName ):
      t8( "foreverCleanupExtraLogs for ", tableName )
      pathname = "%s/%s*" % ( self.config.foreverLogDirectory,
            tableName + self.config.foreverLogFileName )
      foreverLogFileList = [ ( int( fn.split( "." )[ -1 ] ), fn )
            for fn in glob.iglob( pathname ) if re.search( r'\.\d+$', fn ) ]
      excess = len( foreverLogFileList ) - self.config.foreverLogMaxSize.val
      if excess > 0:
         t8( "excess files for ", tableName, " is ", excess )
         for ( _, foreverLogName ) in sorted( foreverLogFileList )[ : excess ]:
            if( self.deletedFileCountPerRound >
                  ForeverCleanupReactor.MAX_DELETES_PER_ROUND ):
               t8( "deferring deletion ", tableName )
               self.deletedFileCountPerRound = 0
               self.timer_.timeMin = Tac.now()
               return False
            else:
               self.timer_.timeMin = Tac.endOfTime

            self.agent_.eventMonDbMgr.removeFromDb( tableName, foreverLogName )
            t8( "removing forever log from synced file list ", foreverLogName )
            if foreverLogName in self.agent_.syncedFile[ tableName ]:
               del self.agent_.syncedFile[ tableName ][ foreverLogName ]

            removeFile( foreverLogName )
            self.deletedFileCountPerRound += 1
      return True

   @Tac.handler( 'fileCount' )
   def handleForeverCleanup( self, tableName=None ):
      if type( self.config ) == Tac.Type( "EventMon::Config" ):
         if tableName is None:
            for tableName in self.config.table:
               if not self.foreverCleanupExtraLogs( tableName ):
                  return
         else:
            if tableName not in self.config.table:
               t8( "unknown table name %s ignoring" % tableName )
               return
            self.foreverCleanupExtraLogs( tableName )
      else:
         if tableName and tableName != self.config.name:
            t8( "unknown table name %s ignoring" % tableName )
            return
         self.foreverCleanupExtraLogs( self.config.name )

class EventMonTableConfigReactor( Tac.Notifiee ):
   notifierTypeName = "EventMon::TableConfig"

   def __init__( self, config, agent ):
      Tac.Notifiee.__init__( self, config )
      self.config = config
      self.agent_ = agent
      self.foreverDirWatcher = None
      self.foreverCleanupReactor = None
      self.setForeverDirWatcher()

   @Tac.handler( 'foreverLogEnabled' )
   def handleForeverLogEnabled( self, ):
      t8( "EventMonTableConfigReactor:handleForeverLogEnabled %s %s " % (
         self.config.fullName, self.config.foreverLogEnabled ) )
      self.setForeverDirWatcher()

   @Tac.handler( 'foreverLogFileName' )
   def handleForeverLogFileName( self, ):
      t8( "EventMonTableConfigReactor:handleForeverLogFileName %s %s" % (
         self.config.fullName, self.config.foreverLogFileName ) )
      self.setForeverDirWatcher()

   @Tac.handler( 'foreverLogDirectory' )
   def handleForeverLogDir( self, ):
      t8( "EventMonTableConfigReactor:handleForeverLogDir %s %s" % (
         self.config.fullName, self.config.foreverLogDirectory ) )
      self.setForeverDirWatcher()

   @Tac.handler( 'foreverLogOverride' )
   def handleForeverLogOverride( self, ):
      t8( "EventMonTableConfigReactor:handleForeverLogOverride %s %s" % (
         self.config.fullName, self.config.foreverLogOverride ) )
      self.setForeverDirWatcher()

   @Tac.handler( 'foreverLogMaxSize' )
   def handleForeverLogMaxSize( self, ):
      t8( "EventMonTableConfigReactor:handleForeverLogMaxSize %s %s " % (
         self.config.fullName, self.config.foreverLogMaxSize ) )
      if self.foreverCleanupReactor:
         self.foreverCleanupReactor.handleForeverCleanup()

   def setForeverDirWatcher( self, ):
      t0( "EventMonTableConfigReactor: setForeverDirWatcher" )
      if self.config.foreverLogEnabled and self.config.foreverLogOverride:
         try:
            os.makedirs( self.config.foreverLogDirectory )
         except OSError:
            pass
         self.foreverDirWatcher = Tac.newInstance( "EventMon::ForeverDirWatcher",
            self.config.foreverLogDirectory, self.config.foreverLogFileName )
         self.foreverCleanupReactor = ForeverCleanupReactor( self.config,
              self.foreverDirWatcher.foreverFileTableCounter, self.agent_ )
      else:
         self.foreverCleanupReactor = None
         self.foreverDirWatcher = None

class EventMonConfigReactor( Tac.Notifiee ):
   notifierTypeName = "EventMon::Config"

   def __init__( self, config, agent ):
      Tac.Notifiee.__init__( self, config )
      self.config = config
      self.agent_ = agent
      self.tableConfigReactor = {}
      self.foreverCleanupReactor = None
      self.foreverDirWatcher = None
      self.handleTableConfig()
      self.setForeverDirWatcher()

   def handleTableConfig( self, ):
      for table, tableConfig in self.config.table.iteritems():
         self.tableConfigReactor[ table ] = EventMonTableConfigReactor( tableConfig,
               self.agent_ )

   @Tac.handler( 'foreverLogEnabled' )
   def handleForeverLogEnabled( self, ):
      t8( "EventMonConfigReactor: handleForeverLogEnabled %s %s " % (
         self.config.fullName, self.config.foreverLogEnabled ) )
      self.setForeverDirWatcher()

   @Tac.handler( 'foreverLogFileName' )
   def handleForeverLogFileName( self, ):
      t8( "EventMonConfigReactor: handleForeverLogFileName %s %s" % (
         self.config.fullName, self.config.foreverLogFileName ) )
      self.setForeverDirWatcher()

   @Tac.handler( 'foreverLogDirectory' )
   def handleForeverLogDir( self, ):
      t8( "EventMonConfigReactor: handleForeverLogDir %s %s" % (
         self.config.fullName, self.config.foreverLogDirectory ) )
      self.setForeverDirWatcher()

   @Tac.handler( 'foreverLogMaxSize' )
   def handleForeverLogMaxSize( self, ):
      t8( "EventMonConfigReactor: handleForeverLogMaxSize %s %s " % (
         self.config.fullName, self.config.foreverLogMaxSize ) )
      if self.foreverCleanupReactor:
         self.foreverCleanupReactor.handleForeverCleanup()

   def setForeverDirWatcher( self, ):
      if self.config.foreverLogEnabled:
         try:
            os.makedirs( self.config.foreverLogDirectory )
         except OSError:
            pass

         self.foreverDirWatcher = Tac.newInstance( "EventMon::ForeverDirWatcher",
            self.config.foreverLogDirectory, self.config.foreverLogFileName )
         self.foreverCleanupReactor = ForeverCleanupReactor( self.config,
              self.foreverDirWatcher.foreverFileTableCounter, self.agent_ )
      else:
         self.foreverCleanupReactor = None
         self.foreverDirWatcher = None

class EventMon( Agent.Agent ):
   def __init__( self, entityManager, blocking=False ):
      Agent.Agent.__init__( self, entityManager )
      self.tableSchema = {}
      self.statusMount = {}
      self.statusReactor = {}
      self.epochReactor = None
      self.configReactor = None
      self.syncedFile = {}
      self.eventMonDbMgr = None

      class Context:
         def __init__( self, agent ):
            self.registerDb = agent.registerDb

      ctx = Context( self )
      Plugins.loadPlugins( 'EventMonPlugin', ctx, None, None )

      def _finish():
         t0( "mounts finished" )
         self.eventMonDbMgr = EventMonDbMgr( self.config.dbLocation,
               self.tableSchema )

         # Register status mount reactors
         for tableName in self.tableSchema:
            # add new table status
            self.status.tableSyncCount[ tableName ] = 0
         self.epochReactor = EventMonEpochReactor( self.epochMarker, self )
         self.configReactor = EventMonConfigReactor( self.config, self )

      # do the initial mounts
      mg = entityManager.mountGroup()
      for tableName in self.tableSchema:
         self.statusMount[ tableName ] = mg.mount( "eventMon/%s/status" % tableName,
               "EventMon::TableStatus", "r" )
      self.config = mg.mount( "eventMon/config", "EventMon::Config", "r" )
      self.status = mg.mount( "eventMon/status", "EventMon::Status", "w" )
      self.epochMarker = mg.mount( "eventMon/epochMarker", "EventMon::EpochMarker",
            "r" )
      if blocking:
         mg.close( blocking=True )
         _finish()
      else:
         mg.close( _finish )

   def registerDb( self, tableSchema ):
      name = tableSchema.name()
      t0( 'Register event monitor for', name )
      schema = tableSchema.schema()
      self.tableSchema[ name ] = schema

def main():
   container = Agent.AgentContainer( [ EventMon ], passiveMount=True )
   container.runAgents()
