# Copyright (c) 2008, 2009, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re, struct, socket, random
from eunuchs.if_ether_h import *
from eunuchs.in_h import *
IP_HDR_LEN = 20
IP6_HDR_LEN = 40
IP6_FRAG_EXTENSION_LEN = 8
UDP_HDR_LEN = 8
TCP_HDR_LEN = 20
ICMP_HDR_LEN = 8
VXLAN_HDR_LEN = 8
UDP_SPORT_DEFAULT = 1999
UDP_DPORT_DEFAULT = 0xAAAA
TCP_SPORT_DEFAULT = 1999
TCP_DPORT_DEFAULT = 0xAAAA
UDP_DPORT_VXLAN = 0x12B5
E_TAG_ETHER_TYPE = 0x893f
VN_TAG_ETHER_TYPE = 0x8926

# NDP related defines
ND_OPTION_SRC_LL_ADDR = 1
ND_OPTION_DST_LL_ADDR = 2
ND_TYPE_RTR_SOLICIT = 133
ND_TYPE_NBR_SOLICIT = 135
ND_TYPE_NBR_ADVT = 136
IP6_EXT_HDR_ICMP6 = 0x3A

def macAddrToBytes( addr ):
   pair = '([0-9a-fA-F]{1,2})'
   quad = '([0-9a-fA-F]{1,4})'
   macAddrColnPattern = '%s:%s:%s:%s:%s:%s$' % ( pair, pair, pair, pair, pair, pair )
   macAddrDotPattern = r'%s\.%s\.%s$' % ( quad, quad, quad )
   colonMatch = re.match( macAddrColnPattern, addr )
   dotMatch = re.match( macAddrDotPattern, addr )
   if colonMatch:
      bytes = ''.join( [ chr( int( s, 16 ) ) for s in colonMatch.groups() ] )
   elif dotMatch:
      # zero pad on left of each group of numbers until length of group is 4 
      # this way an addr of 1.2.3 becomes 0001.0002.0003 and will be valid
      zeroPaddedMatch = ''.join( [ s.zfill( 4 ) for s in dotMatch.groups() ] )
      bytes = ''
      for i in range( 0, 12, 2 ):
         byte = zeroPaddedMatch[ i:i+2 ]
         bytes += chr( int( byte, 16 ) )
   else:
      raise ValueError( "invalid MAC address: %s" % addr )
   return bytes

def bytesToMacaddr( s ):
   return "%x:%x:%x:%x:%x:%x" % tuple( [ord(i) for i in s[:6] ] )

def strContains( string, c ):
   # Return True if string contains c.
   if c in string:
      return True
   else:
      return False

def buildVlan( vlan, vpri, tpid=ETH_P_8021Q ):
   if ( vlan is not None ) or ( vpri is not None ):
      tci = ( ( vpri or 0 ) << 13 ) | ( vlan or 0 )
      vlanTagStr = struct.pack( '>HH', tpid, tci )
   else:
      vlanTagStr = ""
   return vlanTagStr

def buildSvTag( signature, channelId ):
   if signature is None:
      signature = 0xFF
   if channelId is None:
      svTagStr = ""
   else:
      svTag = ( signature << 24 ) | channelId
      svTagStr = struct.pack( '>I', svTag )
   return svTagStr

def buildBrTag( br=None ):
   # 802.1br tag header is 8B long and consists of:
   # ethType = 0x893f( 16bits )
   # e-pcp : priority bits ( 3bits )
   # DEI : drop eligible indicator ( 1bit ) 
   # ingress ECID : ingress extended port ID ( 12bits )
   #                [ source Vif in vn-tag ]
   # reserve ( 2bits )
   # ECID : extended port ID ( 14bits )
   #        [ destination Vif in vn-tag ]
   # reserve ( 16bits )
   # the content is simplified in this funciton for testing: 
   #     only set ethType and set 0 to all rest bits
   eTagStr = ""
   if br is not None:
      ethType = E_TAG_ETHER_TYPE 
      eTagHeader = 0
      reserve = 0
      eTagStr = struct.pack('>HIH', ethType, eTagHeader, reserve)
   return eTagStr

def buildVNTag( sVif=None, dVif=None ):
   # vn tag header is 6B( 48bits ) long and consits of :
   # ethType = 0x8926 ( 16bits )
   # destination bit ( 1bit ) : 0 from IV to bridge, 1: from the bridge to IV
   # pointer bit ( 1bit ): 1: a vif_list_id is in the tag, 0: dVif_id in the tage
   # destination Vif ( 12bits ), or vif_list_id( 14bits )
   #      dVif_id/vif_list_id is reserved if d is 0
   # looped ( 1bit ): 1 indicates that this is a multicast fram out the bridge
   # Reserved ( 3bit )
   # source Vif ( 12bits ): vif_id of the port that add the vntag,
   #                        reserved if d=1 and l=0 
   # version ( 2bits ): is 0
   # unicast d = 1, multicast p = 1
   vnTag = ""
   if( sVif is not None ) or ( dVif is not None ):
      ethType = VN_TAG_ETHER_TYPE
      if sVif is not None:
         sVif = sVif << 2
         dVif = 0 << 18
         vnTag = dVif |  sVif
         vnTagStr = struct.pack('>HI', ethType, vnTag )
         return vnTagStr
      if dVif is not None:
         # non multicast case:
         dstBit = 1 << 31
         pointer = 0 << 30
         dVif = dVif << 18
         sVif = 0 << 2
         vnTag = dstBit | pointer | dVif | sVif
         vnTagStr = struct.pack('>HI', ethType, vnTag )
         return vnTagStr
   return vnTag
         
def buildMplsHeader( mpls_label=None, mpls_ttl=None, mpls_exp=None, mpls_bos=True ):
   if mpls_label >= 0:
      # MPLS header is 32 bits long and consits of (in order):
      # - 20 bit label
      # - 3 bit QoS and ECN
      # - 1 bit bottom of stack flag (if set means the current label is the last)
      # - 8 bit TTL field
      # We set the bottom of stack flag because we are sending one label
      if mpls_exp is None:
         mpls_exp = 0
      bos = 1 if mpls_bos else 0
      value = 0
      value |= mpls_label << 12
      value |= mpls_exp << 9
      value |= bos << 8
      value |= mpls_ttl

      mplsStr = struct.pack('>I', value )
   else:
      mplsStr = ""
   return mplsStr

def buildEthernetHeader( dmac, smac, ethertype, 
                         vlan=None, vpri=None, innerVlan=None, innerVpri=None,
                         vlanTpid=ETH_P_8021Q, innerVlanTpid=ETH_P_8021Q,
                         sVif=None, dVif=None,
                         br=None, svTagSignature=None, svTagChannelId=None ):
   dmacStr = macAddrToBytes( dmac )
   smacStr = macAddrToBytes( smac )
   svTagStr  = buildSvTag( svTagSignature, svTagChannelId )
   vlanTagStr = buildVlan( vlan, vpri, vlanTpid )
   innerVlanTagStr = buildVlan( innerVlan, innerVpri, innerVlanTpid )
   ethertypeStr = struct.pack( '>H', ethertype )

   if br is not None:
      tagStr = str( buildBrTag( br ) )
   elif sVif is not None or dVif is not None: 
      tagStr = str( buildVNTag( sVif, dVif ) ) 
   else:
      tagStr = ""

   return dmacStr + smacStr + svTagStr + tagStr + vlanTagStr + innerVlanTagStr + \
          ethertypeStr

def buildEthernetPacket( size, initialData='', 
                         dataType='incrementing', dataValue=0, extraHeaderSize=0 ):
   minSize = ETH_HLEN + len( initialData ) + ETH_FCS_LEN + extraHeaderSize
   if size < minSize:
      raise ValueError( "size too small: must be at least %d" % minSize )

   dataLen = size - minSize

   if dataType == 'constant':
      dataValue = int( dataValue, base=0 )
      if dataValue < 0 or dataValue > 255:
         raise ValueError( "data value out of range: must be between 0 and 255" )
      bytes = [ dataValue for i in range( dataLen ) ]
   elif dataType == 'incrementing':
      bytes = [ i % 256 for i in range( dataLen ) ]
   elif dataType == 'random':
      bytes = [ random.randrange( 256 ) for i in range( dataLen ) ]
   elif dataType == 'raw':
      bytes = [ int( dataValue[ i:i+2 ], base=16 )
                for i in range( 0, len( dataValue ), 2 ) ]
      padding = [ 0 ] * max( ( dataLen - len( bytes ) ), 0 )
      bytes += padding
   elif dataType == 'binary':
      bytes = dataValue
      padding = [ 0 ] * max( ( dataLen - len( bytes ) ), 0 )
      bytes += padding
   else:
      assert False

   return initialData + ''.join( [ chr( b ) for b in bytes ] )

def buildArpPacket( srcMacAddr, targetIpAddr, dstMacAddr='ff:ff:ff:ff:ff:ff', 
                    srcIpAddr='1.1.1.1', requestOrReply='request' ):
   hardwareAddrSpace = 1
   protocolAddrSpace = ETH_P_IP
   hardwareAddrLen = 6
   protocolAddrLen = 4
   if requestOrReply == 'request':
      opcode = 1
   elif requestOrReply == 'reply':
      opcode = 2
   else:
      opcode = 0
   dmacStr = macAddrToBytes( dstMacAddr )
   smacStr = macAddrToBytes( srcMacAddr )
   arpHeader = struct.pack( '>HHBBH',
                            hardwareAddrSpace,
                            protocolAddrSpace,
                            hardwareAddrLen,
                            protocolAddrLen,
                            opcode )
   ipSrcAddr = struct.pack( '>I', struct.unpack( '>L', 
                            socket.inet_aton( srcIpAddr ) )[ 0 ] )
   ipDstAddr = struct.pack( '>I', struct.unpack( '>L', 
                            socket.inet_aton( targetIpAddr ) )[ 0 ] )

   arpPacket = arpHeader + smacStr + ipSrcAddr + dmacStr + ipDstAddr
   return arpPacket

# OR packed Ipv6 addressses 
def ip6AddrOR( addr1, addr2 ):
   addr1 = struct.unpack( '4I', addr1 )
   addr2 = struct.unpack( '4I', addr2 )
   ret = map( lambda x, y: x | y, addr1, addr2 )
   t = ''.join( map( lambda x: struct.pack( 'I', x ), ret ) )
   return t

# AND packed Ipv6 addressses 
def ip6AddrAND( addr1, addr2 ):
   addr1 = struct.unpack( '4I', addr1 )
   addr2 = struct.unpack( '4I', addr2 )
   ret = map( lambda x, y: x & y, addr1, addr2 )
   t = ''.join( map( lambda x: struct.pack( 'I', x ), ret ) )
   return t

# returns NS multicast IP address (packed bin)
def getNsMcastAddr( addr ):
   # Multicast addr format ff02::1:ff00:0/104
   r = ip6AddrAND( addr, socket.inet_pton( socket.AF_INET6, '::ff:ffff' ) )
   r = ip6AddrOR( socket.inet_pton( socket.AF_INET6, 'ff02::1:ff00:0' ), r )
   return r

# returns NS multicast MAC address (string)
def getNsMcastMac( addr ):
   addr = struct.unpack( '16B', addr )[ -4: ]
   mac = '33:33:'       # Multicast MAC
   mac += ':'.join( map( lambda x: '%.2x' %x, addr ) )
   return mac

# ICMPv6 pseudo Hdr for checksum calculation
def buildIcmp6PseudoHdr( targetIpAddr, srcIpAddr, upperLayerLen, extHdr ):
   extHdr = struct.pack( ">I", extHdr )
   upperLayerLen = struct.pack( ">I", upperLayerLen )
   return ( srcIpAddr + targetIpAddr + upperLayerLen + extHdr )


def buildNdpPacket( ndpType, srcMacAddr, dstMacAddr, targetIpAddr, srcIpAddr ):

   ipSrcAddr = socket.inet_pton( socket.AF_INET6, srcIpAddr )
   ipDstAddr = socket.inet_pton( socket.AF_INET6, targetIpAddr )

   if ndpType == "nbr-solicit" or ndpType == "rtr-solicit":
      if ndpType == "nbr-solicit":
         typeData = ND_TYPE_NBR_SOLICIT # Type code for Nbr Solicit
         # The address of the target neighbor
         reqIpDstAddr = ipDstAddr
      else:
         typeData = ND_TYPE_RTR_SOLICIT # Type code for Rtr Solicit
         reqIpDstAddr = ""

      code = 0
      chkSum = 0        # Initially setting to 0
      reserved = 0
      hdr = struct.pack( ">BBHI", typeData, code, chkSum, reserved )

      # options Field in the solicitation packet
      optType = ND_OPTION_SRC_LL_ADDR
      optLen = 1        # 8 bytes : size of TLD
      optData = macAddrToBytes( srcMacAddr )
      optHdr = struct.pack( ">BB", optType, optLen ) 
      data = hdr + reqIpDstAddr + optHdr + optData
      upperLayerLen = len( data )

      # If dmac is broadcast, get Multicast Addr from the dest-ip, otherwise use
      # the unmodified target IP. Update the target IP of the pkt accordingly.
      dstIp = getNsMcastAddr( ipDstAddr ) if dstMacAddr == 'ff:ff:ff:ff:ff:ff' \
              else ipDstAddr
      targetIpAddr = socket.inet_ntop( socket.AF_INET6, dstIp )

      # Needed for ICMP6 checksum
      icmp6PseudoHdr = buildIcmp6PseudoHdr( dstIp, ipSrcAddr, upperLayerLen,
                                             IP6_EXT_HDR_ICMP6 )
      chkSum = ipChecksum( icmp6PseudoHdr + data )

      # Repack the data stream with calculated checksum
      hdr = struct.pack( ">BBHI", typeData, code, chkSum, reserved )
      data = hdr + reqIpDstAddr + optHdr + optData

   elif ndpType == "nbr-advt":
      typeData = ND_TYPE_NBR_ADVT # Type code for Nbr Advt
      code = 0
      chkSum = 0
      flag = 0xE000     # 1-Router, 1-Solicited reply, 1-Override
      reserved = 0
      hdr = struct.pack( ">BBHHH", typeData, code, chkSum, flag, reserved )

      # options Field in the Advt packet
      optType = ND_OPTION_DST_LL_ADDR 
      optLen = 1
      optHdr = struct.pack( ">BB", optType, optLen )
      optData = macAddrToBytes( srcMacAddr )
      data = hdr + ipSrcAddr + optHdr + optData
      upperLayerLen = len( data )

      # Needed for ICMP6 checksum
      icmp6PseudoHdr = buildIcmp6PseudoHdr( ipDstAddr, ipSrcAddr, upperLayerLen,
                                             IP6_EXT_HDR_ICMP6 )
      chkSum = ipChecksum( icmp6PseudoHdr + data )

      # Repack the data stream with calculated checksum
      hdr = struct.pack( ">BBHHH", typeData, code, chkSum, flag, reserved )
      data = hdr + ipSrcAddr + optHdr + optData

   else:
      # Should not reach here
      raise ValueError( "invalid NDP rquest type: %s" % ndpType )
  
   ip6Hdr = buildIp6Header( srcIpAddr, targetIpAddr, IP6_EXT_HDR_ICMP6,
                           upperLayerLen + IP6_HDR_LEN,
                           hopLimit=255, tos=0x00 )

   return ( ip6Hdr + data )

def ipChecksum( packet ):
   """Compute checksum for the packet. This algorithm is used by
   both TCP (header+pkt) and IP (header only)."""
   total = 0
   while len( packet ) > 1:
      total += struct.unpack( ">H", packet[ :2 ] )[ 0 ]
      packet = packet[ 2: ]
   if packet:
      total += ord( packet[ 0 ] )
   total = ( total & 0xFFFF ) + ( total >> 16 )
   total = ( total & 0xFFFF ) + ( total >> 16 )
   total &= 0xFFFF
   return ( 0xFFFF & ~total )

def buildIpHeader( ipSrc, ipDst, ipProtocol, ttl, totalLength, tos,
                   ipOptionData='', dontFragFlag=False, moreFragsFlag=False,
                   fragOffset=0, ipId=0, ipCkSum=0, ipVer=0, ipHl=0 ):
   ipSrcAddr = struct.unpack( '>L', socket.inet_aton( ipSrc ) )[ 0 ]
   ipDstAddr = struct.unpack( '>L', socket.inet_aton( ipDst ) )[ 0 ]
   assert len( ipOptionData ) % 4 == 0
   if ipVer == 0 and ipHl == 0:
      versionAndIhl = 0x40 | ( ( 20 + len( ipOptionData ) ) / 4 )
   else:
      versionAndIhl = ipVer | ( ipHl + ( len( ipOptionData ) ) / 4 )
   flags = 0
   if dontFragFlag:
      flags |= 0x2
   if moreFragsFlag:
      flags |= 0x1
   flagsAndFragOffset = ( flags << 13 ) | ( fragOffset & 0x1fff )
   ipHeader = struct.pack( '>BBHHHBBHII',
                           versionAndIhl,
                           tos,
                           totalLength,
                           ipId, # Id (should be unique, but 0 should be OK)
                           flagsAndFragOffset, # Flags and fragment offset
                           ttl,
                           ipProtocol,
                           0, # Header checksum
                           ipSrcAddr,
                           ipDstAddr ) + ipOptionData
   if ipCkSum == 0:
      checksum = ipChecksum( ipHeader )
   else:
      checksum = ipCkSum

   ipHeader = ipHeader[ :10 ] + struct.pack( '>H', checksum ) + ipHeader[ 12: ]

   return ipHeader


def buildIp6FragExtension( ipProtocol, moreFragsFlag, fragOffset, ipId ):
   fragBits = ( fragOffset << 3 ) | moreFragsFlag
   fragHdr = struct.pack( ">BBHI", ipProtocol, 0, fragBits, ipId )
   return fragHdr

def buildIp6Header( ipSrc, ipDst, ipProtocol, payloadLen, hopLimit, tos=0,
                    moreFragsFlag=False, fragOffset=0, ipId=0, flowLabel=0 ):
   fragHdr = None
   if ( fragOffset != 0 ) or moreFragsFlag:
      fragHdr = buildIp6FragExtension( ipProtocol, moreFragsFlag,
                                       fragOffset, ipId )
      ipProtocol = 44
   srcAddr = socket.inet_pton( socket.AF_INET6, ipSrc )
   dstAddr = socket.inet_pton( socket.AF_INET6, ipDst )
   hi = ( tos & 0xF0 ) >> 4
   lo = ( tos & 0x0F ) << 4
   flowLabelHi = ( flowLabel & 0xF0000 ) >> 16
   flowLabelLo = flowLabel & 0x0FFFF
   ipHdr = struct.pack( ">BBHHBB", ( 0x60 | ( hi & 0x0F ) ) ,
                        ( ( lo & 0xF0 ) | ( flowLabelHi & 0x0F ) ), flowLabelLo,
                        payloadLen - IP6_HDR_LEN,
                        ipProtocol, hopLimit ) + srcAddr + dstAddr

   if fragHdr:
      ipHdr += fragHdr
   return ipHdr

# The extraHeaderSize parameter is to account for any extra headers such as MPLS or
# GRE that need to be taken into account in the minimum packet size calculation
def buildIpPacket( size, ipSrc, ipDst,
                   ipProtocol=63, # 63 = "any local network"
                   ttl=64, tos=0, ipOptionData='',
                   dataType='incrementing', dataValue=0, initialData='', version=4,
                   dontFragFlag=False,
                   moreFragsFlag=False,
                   fragOffset=0,
                   ipId=0,
                   extraHeaderSize=0,
                   ipCkSum=0,
                   ipVer=0,
                   ipHl=0,
                   flowLabel=0 ):
   dataLen = size  - ( ETH_HLEN + ETH_FCS_LEN ) - extraHeaderSize
   ipHeader = None
   if version == 4:
      ipHeader = buildIpHeader( ipSrc,
                                ipDst,
                                ipProtocol,
                                ttl,
                                dataLen,
                                tos,
                                ipOptionData,
                                dontFragFlag=dontFragFlag,
                                moreFragsFlag=moreFragsFlag,
                                fragOffset=fragOffset,
                                ipId=ipId,
                                ipCkSum=ipCkSum,
                                ipVer=ipVer,
                                ipHl=ipHl )
   else:
      ipHeader = buildIp6Header( ipSrc,
                                 ipDst,
                                 ipProtocol,
                                 dataLen,
                                 ttl,
                                 tos=tos,
                                 moreFragsFlag=moreFragsFlag,
                                 fragOffset=fragOffset,
                                 ipId=ipId,
                                 flowLabel=flowLabel )

   initialData = ipHeader + initialData
   return buildEthernetPacket( size, initialData=initialData, dataType=dataType,
                               dataValue=dataValue,
                               extraHeaderSize=extraHeaderSize )

def buildIpInIpPacket( size, outerIpSrc, outerIpDst, ipSrc, ipDst,
                       ipProtocol=63, # 63 = "any local network"
                       ttl=64, tos=0, ipOptionData='',
                       dataType='incrementing', dataValue=0,
                       outerDontFragFlag=False,
                       outerMoreFragsFlag=False,
                       outerFragOffset=0,
                       outerIpId=0 ):
   outerDataLen = size - ( ETH_HLEN + ETH_FCS_LEN )
   outerIpHeader = buildIpHeader( outerIpSrc,
                                  outerIpDst,
                                  IPPROTO_IPIP,
                                  ttl,
                                  outerDataLen,
                                  tos,
                                  ipOptionData,
                                  dontFragFlag=outerDontFragFlag,
                                  moreFragsFlag=outerMoreFragsFlag,
                                  fragOffset=outerFragOffset,
                                  ipId=outerIpId )

   dataLen = outerDataLen - len( outerIpHeader )
   ipHeader = buildIpHeader( ipSrc,
                             ipDst,
                             ipProtocol,
                             ttl,
                             dataLen,
                             tos )

   return buildEthernetPacket( 
      size, initialData=outerIpHeader + ipHeader,
      dataType=dataType, dataValue=dataValue )

def buildUdpHeader( size, udpSport, udpDport, version=4,
                    ipOptionData='', ipFrag=False ):
   if version == 4:
      ipHdrLen = IP_HDR_LEN
   else:
      ipHdrLen = IP6_HDR_LEN
      if ipFrag:
         ipHdrLen += IP6_FRAG_EXTENSION_LEN

   udpLength = size - ( ETH_HLEN + ipHdrLen + len( ipOptionData ) + ETH_FCS_LEN )
   return struct.pack( '>HHHH', udpSport, udpDport, udpLength, 0 )

def buildUdpPacket( size, udpSport, udpDport, ipSrc, ipDst, ipProtocol=IPPROTO_UDP,
                    ttl=64, tos=0, ipOptionData='', initialData='',
                    dataType='incrementing', dataValue=0, version=4, 
                    dontFragFlag=False,
                    moreFragsFlag=False,
                    fragOffset=0,
                    ipId=0,
                    flowLabel=0 ):
   ipFrag = ( ( fragOffset != 0 ) or moreFragsFlag )
   udpHeader = buildUdpHeader( size, udpSport, udpDport, version,
                               ipOptionData, ipFrag )
   initialData = udpHeader + initialData
   return buildIpPacket( size, ipSrc, ipDst, ipProtocol, ttl, tos, ipOptionData,
                         dataType=dataType, dataValue=dataValue,
                         initialData=initialData, version=version, 
                         dontFragFlag=dontFragFlag,
                         moreFragsFlag=moreFragsFlag,
                         fragOffset=fragOffset,
                         ipId=ipId,
                         flowLabel=flowLabel )

def buildTcpHeader( sPort=TCP_SPORT_DEFAULT, dPort=TCP_DPORT_DEFAULT, sequenceNo=1,
                    ackNum=0, tcpFlags='', windowSize=1024, urgPointer=0 ):
   checksum = 0
   tcpHdrLen = TCP_HDR_LEN
   dataOffset = ( tcpHdrLen / 4 ) << 4

   flagUrg = strContains( tcpFlags, "U" )
   flagAck = strContains( tcpFlags, "A" )
   flagPsh = strContains( tcpFlags, "P" )
   flagRst = strContains( tcpFlags, "R" )
   flagSyn = strContains( tcpFlags, "S" )
   flagFin = strContains( tcpFlags, "F" )

   if flagAck:
      ackNo = ackNum
   else:
      ackNo = 0
   if flagUrg:
      urgPtr = urgPointer
   else:
      urgPtr = 0
   flags = ( int( flagUrg ) << 5 ) | \
           ( int( flagAck ) << 4 ) | \
           ( int( flagPsh ) << 3 ) | \
           ( int( flagRst ) << 2 ) | \
           ( int( flagSyn ) << 1 ) | \
           ( int( flagFin ) )

   header = struct.pack( '>HHIIBBHHH', sPort, dPort, sequenceNo, ackNo,
                         dataOffset, flags, windowSize, checksum, urgPtr )
   return header

def buildTcpPacket( size, ipSrc, ipDst, ipVersion=4, tos=0, ipId=0,
                    dontFragFlag=False, moreFragsFlag=False, fragOffset=0, ttl=64,
                    ipOptionData='', tcpSPort=TCP_SPORT_DEFAULT,
                    tcpDPort=TCP_DPORT_DEFAULT, ackNo=1, tcpFlags='',
                    initialData='' ):

   tcpHeader = buildTcpHeader( sPort=tcpSPort, dPort=tcpDPort, ackNum=ackNo,
                               tcpFlags=tcpFlags )
   if (ipVersion == 4):
      ipSrcAddr = struct.unpack( '>L',
                                 socket.inet_pton( socket.AF_INET,  ipSrc ) )[ 0 ]
      ipDstAddr = struct.unpack( '>L',
                                 socket.inet_pton( socket.AF_INET,  ipDst ) )[ 0 ]
      pseudoHeader = struct.pack( '>IIBBH', ipSrcAddr, ipDstAddr, 0,
                                  IPPROTO_TCP, TCP_HDR_LEN )
   elif (ipVersion == 6):
      ipSrcAddr = struct.unpack( '>QQ',
                                 socket.inet_pton( socket.AF_INET6,  ipSrc ) )[ 0 ]
      ipDstAddr = struct.unpack( '>QQ',
                                 socket.inet_pton( socket.AF_INET6,  ipDst ) )[ 0 ]
      pseudoHeader = struct.pack( '>QQBBH', ipSrcAddr, ipDstAddr, 0,
                                  IPPROTO_TCP, TCP_HDR_LEN )
   else:
      assert (False), "ipVersion = %r. IP version must be 4 or 6\n"\
            %ipVersion

   headerForChecksum = pseudoHeader + tcpHeader
   tcpChecksum = ipChecksum ( headerForChecksum )

   tcpHeader = tcpHeader[ :16 ] + struct.pack( '>H', tcpChecksum ) + tcpHeader[ 18: ]
   initialData = tcpHeader + initialData
   return buildIpPacket( size, ipSrc, ipDst, ipProtocol=IPPROTO_TCP, ttl=ttl,
                         tos=tos, ipOptionData=ipOptionData,
                         dataType='incrementing', dataValue=0,
                         initialData=initialData,
                         version=ipVersion, dontFragFlag=dontFragFlag,
                         moreFragsFlag=moreFragsFlag, fragOffset=fragOffset,
                         ipId=ipId )

def buildIcmpHeader( icmpType, icmpCode ):
   icmpHeader = struct.pack( '>BBHL', icmpType, icmpCode, 0, 0 )
   checksum = ipChecksum( icmpHeader )
   icmpHeader = struct.pack( '>BBHL', icmpType, icmpCode, checksum, 0 )
   return icmpHeader

def buildIcmpPacket( ipSrc, ipDst, ipProtocol=IPPROTO_ICMP,
                     icmpType=8, icmpCode=0, ttl=64, tos=0, ipOptionData='',
                     version=4, dontFragFlag=False, moreFragsFlag=False,
                     fragOffset=0, ipId=0, flowLabel=0 ):
   # We don't support data payload here. Because ICMP checksum
   # includes the entire packet, not just the header. The payload is
   # not even generated yet when we build the ICMP header. The payload
   # generation code will need to be restructured to support this.
   size = ETH_HLEN + IP_HDR_LEN + len( ipOptionData ) + ICMP_HDR_LEN + \
          ETH_FCS_LEN
   icmpHeader = buildIcmpHeader( icmpType, icmpCode )
   return buildIpPacket( size, ipSrc, ipDst, ipProtocol, ttl, tos,
                         ipOptionData, dataType='incrementing', dataValue=0,
                         initialData=icmpHeader, version=version,
                         dontFragFlag=False, moreFragsFlag=False,
                         fragOffset=fragOffset, ipId=ipId, flowLabel=flowLabel )

def buildVxlanHeader( flags=0x08, vni=None ):
   hdrPart1 = struct.pack( '>BBH' , flags, 0, 0 )
   # 5 bytes - 4 of VNI and 1 of Reserved
   temp1 = struct.pack( '>IB' , vni, 0 ) 
   # Truncate first byte and discard it
   ( _discard, temp2 ) = struct.unpack( '>BI', temp1 ) 
   hdrPart2 = struct.pack( '>I', temp2 ) 
   return hdrPart1 + hdrPart2

def buildGreHeader( greVersion, greProtocol,
                    greChecksum=None, greKey=None, greSequence=None ):
   flags = ( 1 if greChecksum else 0 ) << 7
   flags += ( 1 if greKey else 0 ) << 5
   flags += ( 1 if greSequence else 0 ) << 4
   version = greVersion & 0x0FFF
   protocol = greProtocol & 0xFFFF
   header = struct.pack( '>BBH', flags, version, protocol )
   if greChecksum:
      header += struct.pack( '>HH', greChecksum, 0 )
   if greKey:
      header += struct.pack( '>I', greKey )
   if greSequence:
      header += struct.pack( '>I', greSequence )
   
   return header

def buildControlWord( data, raw=False ):
   # if in raw format, data is 4-byte hex string,
   # else, it is ( pwType, flags, frg, seqno ) tuple
   if raw:
      assert len( data ) == 8
      assert isinstance( data, str )
      bytes = [ int( data[ i:i+2 ], base=16 ) for i in range( 0, len( data ), 2 ) ]
      return ''.join( [ chr( b ) for b in bytes ] )
   else:
      assert len( data ) == 4
      assert isinstance( data, tuple )
      ( pwType, flags, frg, seqno ) = data
      pwTypeFlags = ( ( pwType & 0xF ) << 4 ) + ( flags & 0xF )
      frgLength = ( frg & 0x3 ) << 6
      return struct.pack( '>BBH', pwTypeFlags, frgLength, seqno )

def insertIsl( orig, f1=None, f2=None, sglort=None, dglort=None ):
   ''' Pack the f1, f2, sglort, and dglort into four 16-bit big-endin 
   integers and insert them after the ethernet MAC addresses '''
   args = [ f1, f2, sglort, dglort ]
   for idx in range( len( args ) ):
      if args[ idx ] is None:
         args[ idx ] = '0'
      args[ idx ] = int( args[ idx ] )

   isl = struct.pack( '>HHHH', args[ 0 ], args[ 1 ], args[ 2 ], args[ 3 ] )
   new = orig[:ETH_ALEN*2] + isl + orig[ETH_ALEN*2:]
   return new

def insertPtchItmh( orig, txrawFap=None, sysDestPort=None ):
   ''' Prepend a txrawHeader followed by a 2 byte PTCH and a 4 byte ITMH.
   The txrawHeader contains a 64 byte string identifying the source FAP, then
   a 2 byte field for the CPU channel (which is hardcoded).
   Hardcoded PTCH specifies a pp_ssp of 128 and leaves the Parse_Program_Control
   (bit 15) to zero which indicates that ITMH header follows the PTCH.
   For the ITMH, hardcoded dropPrecedence=1, traffic class=7,
   and only permitting a dest_sys_port for the FWD_DEST_INFO field'''
   assert isinstance( txrawFap, str ), "txraw must be a string"
   assert isinstance( sysDestPort, int ), "sysDestPort must be an integer"
   dropPrecedence = 1
   trafficClass = 7
   fillLen = 64 - len( txrawFap )
   txrawHeaderDeviceName = txrawFap + '\x00' * fillLen
   txrawHeaderCpuChannel = struct.pack( '<H', 1 << 9 )
   ptch = struct.pack( '>H', 128 ) #pp_ssp
   itmh = struct.pack( '>L',
                       ( sysDestPort | 0xc0000 ) |
                       ( dropPrecedence << 20 ) |
                       ( trafficClass << 22 ) )
   txrawHeader = txrawHeaderDeviceName + txrawHeaderCpuChannel + ptch + itmh
   return txrawHeader + orig
