# Copyright (c) 2006-2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module contains utilities for working with Ethernet addresses.
#-------------------------------------------------------------------------------
import re

#-------------------------------------------------------------------------------
# A regular-expression for matching MAC addresses.
#-------------------------------------------------------------------------------
pair = '([0-9a-fA-F]{1,2})'
quad = '([0-9a-fA-F]{1,4})'
colonPattern = '%s:%s:%s:%s:%s:%s$' % ( pair, pair, pair, pair, pair, pair )
dashPattern = '%s-%s-%s-%s-%s-%s$' % ( pair, pair, pair, pair, pair, pair )
dotPattern = '%s\.%s\.%s$' % ( quad, quad, quad )
macAddrPattern = '(%s|%s|%s)' % ( colonPattern, dashPattern, dotPattern )

def convertMacAddrCanonicalToDisplay( colonAddr ):
   '''Function to convert from the canonical string representation of
   a MAC address, namely the format in which we receive them from C++,
   to the format in which MAC addresses are displayed in the output of
   the 'show interfaces' command.'''
   m = re.match( colonPattern, colonAddr )
   if not m:
      raise ValueError( "Invalid canonical MAC address" )
   # This gives the correct result because the canonical representation pads
   # each of the bytes to two digits.
   return ( "%s%s.%s%s.%s%s" % ( m.group( 1 ), m.group( 2 ), m.group( 3 ),
                                 m.group( 4 ), m.group( 5 ), m.group( 6 ) ) )

def convertMacAddrToCanonical( macAddr ):
   '''Converts a MAC address from any string representation to the
   canonical string representation.'''
   m = re.match( colonPattern, macAddr )

   if m is None:
      m = re.match( dashPattern, macAddr )

   if m is None:
      m = re.match( dotPattern, macAddr )
      if m is None:
         raise ValueError( "Invalid MAC address" )

      paddedDotAddr = '%04x.%04x.%04x' % tuple(
                            [ int( m.group( i ), 16 ) for i in range( 1, 4 ) ] )

      paddedDotPat = '%s%s\.%s%s\.%s%s' % ( pair, pair, pair, pair, pair, pair )
      m = re.match( paddedDotPat, paddedDotAddr )

   assert m is not None

   colonAddr = '%02x:%02x:%02x:%02x:%02x:%02x' % tuple(
                            [ int( m.group( i ), 16 ) for i in range( 1, 7 ) ] )
   return colonAddr

def canonicalMacAddrRelative( baseAddress, increment ):
   '''Given a base address in canonical form, and an increment value, return the
   addr which is "increment" higher relative to the base address.'''
   addr = [int(i, 16) for i in baseAddress.split(":")]
   addr[5] += increment
   for i in range(5, 0, -1):
      while addr[i] > 255:
         addr[i] -= 256
         addr[i-1] += 1
   return "%02x:%02x:%02x:%02x:%02x:%02x" % tuple(addr)

def convertMacAddrToPackedString( macAddr ):
   '''Converts a MAC address from any string representation to a packed string'''
   colonAddr = convertMacAddrToCanonical( macAddr )
   m = re.match( colonPattern, colonAddr )
   assert m is not None
   return ''.join( [ chr( int( m.group( i ), 16 ) ) for i in range( 1, 7 ) ] )
   
def convertMacAddrToDisplay( macAddr ):
   '''Converts a MAC address from any string representation to the
   format in which MAC addresses are displayed in the output of 'show'
   commands.'''
   canonicalAddr = convertMacAddrToCanonical( macAddr )
   return convertMacAddrCanonicalToDisplay( canonicalAddr )

def isUnicast( macAddr ):
   """Returns True if a MAC address (in any string representation) is a unicast
   address."""
   colonAddr = convertMacAddrToCanonical( macAddr )
   m = re.match( colonPattern, colonAddr )
   return int( m.group( 1 ), 16 ) % 2 == 0

def isMulticast( macAddr ):
   """Returns True if a MAC address (in any string representation) is a multicast
   address (not a unicast or the broadcast address)."""
   return not ( isUnicast( macAddr ) or isBroadcast( macAddr ) )
  
def isIPv4Multicast( macAddr, allowIanaReserved ):
   """Returns True if a MAC address (in any string representation) is an IP v4
   multicast address (starts with 01:00:5e). If allowIanaReserved is True,
   considers this an IP v4 multicast address even if the 24th bit is 1."""
   colonAddr = convertMacAddrToCanonical( macAddr )
   m = colonAddr.split(":")
   return colonAddr.startswith( "01:00:5e" ) and \
      ( allowIanaReserved or not ( int( m.group( 4 ), 16 ) >> 7 ) )

def isIPv6Multicast( macAddr ):
   """Returns True if a MAC address (in any string representation) is an IP v6
   multicast address (starts with 33:33)."""
   colonAddr = convertMacAddrToCanonical( macAddr )
   return colonAddr.startswith( "33:33:" )

def isIPMulticast( macAddr, allowIanaReserved=True ):
   """Returns True if a MAC address (in any string representation) is an IP 
   v4 or v6 multicast address. If allowIanaReserved is True,
   considers this an IP v4 multicast address even if the 24th bit is 1
   and the address otherwise satisfies the IP v4 conditions."""
   return isIPv4Multicast( macAddr, allowIanaReserved ) or isIPv6Multicast( macAddr )

def isBroadcast( macAddr ):
   """Returns True if a MAC address (in any string representation) is the broadcast
   address (ff:ff:ff:ff:ff:ff)."""
   colonAddr = convertMacAddrToCanonical( macAddr )
   return colonAddr == 'ff:ff:ff:ff:ff:ff'

# MAC protocols
macProtoByName = {
   "aarp": ( 0x80f3, "Appletalk Address Resolution Protocol" ),
   "arp" : ( 0x806, "Address Resolution Protocol" ),
   "appletalk": ( 0x809b, "Appletalk" ),
   "ip": ( 0x800, "Internet Protocol Version 4" ),
   "ipv6": ( 0x86dd, "Internet Protocol Version 6" ),
   "ipx": ( 0x8137, "Internet Packet Exchange" ),
   "novell": ( 0x8138, "Novell" ),
   "lldp": ( 0x88cc, "LLDP" ),
   "rarp": ( 0x8035, "Reverse Address Resolution Protocol" ),
   }

def convertMacProtoToString( proto ):
   '''From MAC protocol value return a string. If it is a well-known
   protocol, we return the protocol name. Otherwise, we return the
   numeric value as a hex string.'''

   for k, v in macProtoByName.iteritems( ):
      if v[ 0 ] == proto:
         return k
   return "0x%x" % proto

def convertMacProtoToInt( proto ):
   '''Convert a MAC protocol in string form into an integer. The input
   can be a recognised protocol name or an integer in string
   form. Also, an integer argument will be returned unchanged without
   error.
   '''

   for k, v in macProtoByName.iteritems( ):
      if k == proto:
         return v[ 0 ]
   return int( proto )
