#!/usr/bin/env python
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
# 
# Module contains utilities for accessing EthIntf information from sysdb.

peerIntfPrefix = "Peer"
ethernetIntfPrefix = "Ethernet"
mgmtIntfPrefix = "Management"
vxlanIntfPrefix = "Vxlan"

def isPeerIntf( intfName ):
   return intfName.startswith( peerIntfPrefix )

def isEthernetIntf( intfName ):
   return intfName.startswith( ethernetIntfPrefix )

def isMgmtIntf( intfName ):
   return intfName.startswith( mgmtIntfPrefix )

def isVxlanIntf( intfName ):
   return intfName.startswith( vxlanIntfPrefix )

def isSubIntf( intfName ):
   return '.' in intfName

#-----------------------------------------------------------------------------
# Returns the list of all physical Ethernet interface names
#-----------------------------------------------------------------------------
def allPhyEthernetInterfaceNames( phyIntfConfig ):
   return [ intfName for intfName in phyIntfConfig if isEthernetIntf( intfName ) ]

#-----------------------------------------------------------------------------
# Returns the list of all physical Management interface names
#-----------------------------------------------------------------------------
def allPhyMgmtInterfaceNames( phyIntfConfig ):
   return [ intfName for intfName in phyIntfConfig if isMgmtIntf( intfName ) ]

#----------------------------------------------------------------------------
# Returns the list of all switchport interface names.
# If 'includeEligible' is set to True then the names for all interfaces capable
# of becoming switchport are returned. Else only currently configured switchport
# names are returned.
# By default subintfs are not included (since they are not really switchports) unless
# the caller explicitly wants them.
# 'sysdbRoot' points to the root of sysdb.
#---------------------------------------------------------------------------
def allSwitchportNames( bridgingConfig, includeEligible=False, 
                        includeSubIntf=False ):
   switchIntfConfig = bridgingConfig.switchIntfConfig
   switchportNames = []
   
   for intfName, intfConfig in switchIntfConfig.iteritems():
      if isPeerIntf( intfName ) or isVxlanIntf( intfName ):
         continue
      if isSubIntf( intfName ) and not includeSubIntf:
         continue
      # Skip routed ports, if includeEligible is not set.
      if not includeEligible and intfConfig.switchportMode == 'routed':
         continue
      switchportNames.append( intfName )
   
   return switchportNames

def allPhySwitchportNames( bridgingConfig ):
   switchIntfConfig = bridgingConfig.switchIntfConfig
   switchportNames = []
   
   for intfName, _ in switchIntfConfig.iteritems():
      if not isEthernetIntf( intfName ):
         continue
      switchportNames.append( intfName )
   
   return switchportNames

def allNonPhySwitchportNames( bridgingConfigCli ):
   switchIntfConfig = bridgingConfigCli.switchIntfConfig
   switchportNames = []
   
   for intfName, _ in switchIntfConfig.iteritems():
      if isPeerIntf( intfName ) or isVxlanIntf( intfName ) or \
                                   isEthernetIntf( intfName ):
         continue
      switchportNames.append( intfName )
   
   return switchportNames
