#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import itertools
import re
import collections
import Cell
import HumanReadable
import Tac
import Tracing
from collections import namedtuple

traceHandle = Tracing.Handle( 'SysdbEthIntf' )
t0 = traceHandle.trace0

sliceNameCache = {}

phyIntfRe = re.compile( r'(Ethernet|Management)(\d+)' )

def sliceName( intfName ):
   if intfName in sliceNameCache:
      return sliceNameCache[ intfName ]
   if Cell.cellType() == 'supervisor':
      m = phyIntfRe.match( intfName )
      if m:
         modId = m.group( 2 )
         if modId == '0' and m.group( 1 ) == 'Management':
            # Special case for Management 0
            # We will stash this under the active cell slice dir
            sliceId = str( Cell.cellId() )
         elif m.group( 1 ) == 'Ethernet':
            # Ethernet ports will be on Linecards
            sliceId = 'Linecard%s' % modId
         else:
            sliceId = str( modId )
      else:
         # Interface not handled
         return None
   else:
      sliceId = str( Cell.cellId() )

   sliceNameCache[ intfName ] = sliceId
   return sliceId

def lookupEthPhyIntfConfigDir( intfName,
                               ethPhyIntfConfigSliceDir ):
   if ( intfName.startswith( 'Ethernet' ) or
        intfName.startswith( 'Management' ) or
        intfName.startswith( 'UnconnectedEthernet' ) ):
      return ethPhyIntfConfigSliceDir.get( sliceName( intfName ) )
   return None

def ethPhyIntfConfigIter( ethPhyIntfConfigSliceDir ):
   allIntfConfigs = [ iter( ethPhyDir.intfConfig ) \
                      for ethPhyDir in ethPhyIntfConfigSliceDir.values() ]
   return itertools.chain.from_iterable( allIntfConfigs )

def defaultSwitchedIntfConfigName( linkMode ):
   return { 'linkModeForced10GbpsFull' : 'DefaultEthSwitchedPort',
            'linkModeForced25GbpsFull' : 'DefaultEthSwitchedPort2',
            'linkModeForced40GbpsFull' : 'DefaultEthSwitchedPort3',
            'linkModeForced50GbpsFull' : 'DefaultEthSwitchedPort4',
            'linkModeForced100GbpsFull' : 'DefaultEthSwitchedPort5' }[ linkMode ]

# Tokens for error-correction encodings.
tokenToFecEncodingAttrs = {
   'reed-solomon' : [ 'fecEncodingReedSolomon', 'fecEncodingReedSolomon544' ],
   'fire-code' : [ 'fecEncodingFireCode' ]
   }

# returns the type of encoding
def encodingType( encoding=None ):
   if encoding.startswith( 'coherent' ):
      return "coherent"
   elif encoding in tokenToFecEncodingAttrs:
      return "non-coherent"
   else:
      return "invalid"

# This table depends on the fact that the enum values of
# EthFecBpassMode and the attributes of EthFecBypassCapabilitySet
# are the same. We use it to convert the token to the config value
# or to lookup support for bypass.
tokenToBypassAttr = {
   'correction' : 'fecBypassCorrection',
   'indication' : 'fecBypassIndication'
   }

linkModeToSpeedLanesDuplex = collections.OrderedDict( [
   ( 'linkModeUnknown', ( 'speedUnknown',
                          'laneCountUnknown',
                          'duplexUnknown' ) ),
   ( 'linkModeAutoneg', ( 'speedUnknown',
                          'laneCountUnknown',
                          'duplexUnknown' ) ),
   ( 'linkModeAuto40GbpsFull', ( 'speedUnknown',
                                 'laneCountUnknown',
                                 'duplexUnknown' ) ),
   ( 'linkMode10MbpsFull', ( 'speed10Mbps',
                             'laneCount1',
                             'duplexFull' ) ),
   ( 'linkMode100MbpsFull', ( 'speed100Mbps',
                              'laneCount1',
                              'duplexFull' ) ),
   ( 'linkModeForced10MbpsHalf', ( 'speed10Mbps',
                                   'laneCount1',
                                   'duplexHalf' ) ),
   ( 'linkModeForced10MbpsFull', ( 'speed10Mbps',
                                   'laneCount1',
                                   'duplexFull' ) ),
   ( 'linkModeForced100MbpsHalf', ( 'speed100Mbps',
                                    'laneCount1',
                                    'duplexHalf' ) ),
   ( 'linkModeForced100MbpsFull', ( 'speed100Mbps',
                                    'laneCount1',
                                    'duplexFull' ) ),
   ( 'linkModeForced1GbpsHalf', ( 'speed1Gbps',
                                  'laneCount1',
                                  'duplexHalf' ) ),
   ( 'linkModeForced1GbpsFull', ( 'speed1Gbps',
                                  'laneCount1',
                                  'duplexFull' ) ),
   ( 'linkModeForced10GbpsFull', ( 'speed10Gbps',
                                   'laneCount1',
                                   'duplexFull' ) ),
   ( 'linkModeForced25GbpsFull', ( 'speed25Gbps',
                                   'laneCount1',
                                   'duplexFull' ) ),
   ( 'linkModeForced40GbpsFull', ( 'speed40Gbps',
                                   'laneCount4',
                                   'duplexFull' ) ),
   ( 'linkModeForced50GbpsFull', ( 'speed50Gbps',
                                   'laneCount2',
                                   'duplexFull' ) ),
   ( 'linkModeForced50GbpsFull1Lane', ( 'speed50Gbps',
                                        'laneCount1',
                                        'duplexFull' ) ),
   ( 'linkModeForced100GbpsFull', ( 'speed100Gbps',
                                    'laneCount4',
                                    'duplexFull' ) ),
   ( 'linkModeForced100GbpsFull1Lane', ( 'speed100Gbps',
                                         'laneCount1',
                                         'duplexFull' ) ),
   ( 'linkModeForced100GbpsFull2Lane', ( 'speed100Gbps',
                                         'laneCount2',
                                         'duplexFull' ) ),
   ( 'linkModeForced200GbpsFull8Lane', ( 'speed200Gbps',
                                         'laneCount8',
                                         'duplexFull' ) ),
   ( 'linkModeForced200GbpsFull4Lane', ( 'speed200Gbps',
                                         'laneCount4',
                                         'duplexFull' ) ),
   ( 'linkModeForced200GbpsFull2Lane', ( 'speed200Gbps',
                                         'laneCount2',
                                         'duplexFull' ) ),
   ( 'linkModeForced400GbpsFull4Lane', ( 'speed400Gbps',
                                         'laneCount4',
                                         'duplexFull' ) ),
   ( 'linkModeForced400GbpsFull8Lane', ( 'speed400Gbps',
                                         'laneCount8',
                                         'duplexFull' ) ),
   ( 'linkModeForced800GbpsFull8Lane', ( 'speed800Gbps',
                                         'laneCount8',
                                         'duplexFull' ) ),
   ] )

nonExistentLinkStrToSpeedLanesDuplex = collections.OrderedDict( [
   ( '2p5G/full', ( 'speed2p5Gbps',
                    'laneCount1',
                    'duplexFull' ) ),
   ( '5G/full', ( 'speed5Gbps',
                  'laneCount1',
                  'duplexFull' ) )
   ] )

# Construct global map that maps speed string notation (e.g. 100g) to EthSpeed enums
_ethSpeedEnum = Tac.Type( "Interface::EthSpeed" )
_ethSpeedRe = re.compile( r"speed(?P<speed>\d+(p\d+)?[GM])bps" )
_speedStrToEthSpeed = {}
for attr in _ethSpeedEnum.attributes:
   if attr == "speedUnknown":
      continue
   _speedStrToEthSpeed[ _ethSpeedRe.match( attr ).group( "speed" ).lower() ] = attr

# Construct global map that maps integer value to EthLaneCount enums
_laneCountEnum = Tac.Type( "Interface::EthLaneCount" )
_laneCountRe = re.compile( r"laneCount(?P<numLanes>\d+)" )
_lanesStrToEthLanes = {}
for attr in _laneCountEnum.attributes:
   if attr == "laneCountUnknown":
      continue
   _lanesStrToEthLanes[ int( _laneCountRe.match( attr ).group( "numLanes" ) ) ] = \
      attr

# Helper to expand Interface::EthLinkModeSet to list of Interface::EthSpeed
# and Interface::EthLaneCount tuples
def linkModeSetToSpeedLaneList( elms ):
   supportedSpeeds = []
   modeRe = re.compile( r"mode((\S+[GM])bps\S+)" )
   for _attr in elms.attributes:
      if not _attr.startswith( "mode" ):
         continue
      mode = getattr( elms, _attr )
      if mode:
         match = modeRe.match( _attr )
         assert match
         speedLaneDuplex, speed = match.groups()
         if speed in ( "2p5G", "5G" ):
            s, l, _ = nonExistentLinkStrToSpeedLanesDuplex[ "%s/full" % speed ]
         else:
            linkMode = "linkModeForced%s" % speedLaneDuplex
            s, l, _ = linkModeToSpeedLanesDuplex[ linkMode ]
         supportedSpeeds.append( ( s, l ) )
   return supportedSpeeds

# Mapping of FEC string to EthFecEncoding
_fecStrToEthFecEncoding = {
   "" : "fecEncodingUnknown",
   "NOFEC" : "fecEncodingDisabled",
   "RS544" : "fecEncodingReedSolomon544",
   "RS528" : "fecEncodingReedSolomon",
   "FCFEC" : "fecEncodingFireCode",
}

def modeStrToEthSpeedAndLaneCount( mode ):
   """
   Returns Interface::EthSpeed and Interface::EthLaneCount given a string
   representation of a speed-lane-fec link mode.

   Parameters
   ----------
   mode : str
      A speed-lane-fec link mode (e.g. 100g-4, 50g-1, 100m-1, 50g-1 RS544)

   Returns
   -------
   ethSpeed : Interface::EthSpeed
   ethLaneCount : Interface::EthLaneCount
   """
   speed = laneCount = None
   mode = mode.split()[ 0 ]
   if "-" in mode:
      speedStr, laneCountStr = mode.split( "-" )
      speed, laneCount = ( _speedStrToEthSpeed[ speedStr.lower() ],
                           _lanesStrToEthLanes[ int( laneCountStr ) ] )
   else:
      speed = _speedStrToEthSpeed[ mode.lower() ]
      laneCount = Tac.Type( "Interface::EthLaneCount" ).laneCountUnknown
   return speed, laneCount

def modeStrToFec( mode ):
   """
   Returns Interface::EthFecEncoding given a string representation of a
   speed-lane-fec link mode.

   Parameters
   ----------
   mode : str
      A speed-lane-fec link mode (e.g. 100g-4, 50g-1, 100m-1, 50g-1 RS544)

   Returns
   -------
   fec : Interface::EthFecEncoding
   """
   modeSplit = mode.split()
   fecStr = modeSplit[ 1 ] if len( modeSplit ) == 2 else ""
   return _fecStrToEthFecEncoding[ fecStr ]

def modeStrToEthIntfMode( intfId, mode ):
   """
   Returns an EthIntfMode given a string representation of a speed-lane-fec
   link mode and an interface.

   Parameters
   ----------
   intfId : Arnet::IntfId
   mode : str
      A speed-lane-fec link mode (e.g. 100g-4, 50g-1, 100m-1, 100g-4 RS528).
      Note, fec is optional: NOFEC, FCFEC, RS528, RS544.

   Returns
   -------
   eim : Interface::EthIntfMode
   """
   ethSpeed, ethLaneCount = modeStrToEthSpeedAndLaneCount( mode )
   ethFec = modeStrToFec( mode )
   eim = Tac.Value( "Interface::EthIntfMode", intfId )
   eim.speed = ethSpeed
   eim.laneCount = ethLaneCount
   # Assume full-duplex for all speeds
   eim.duplex = Tac.Type( "Interface::EthDuplex" ).duplexFull
   eim.fec = ethFec
   return eim


#-------------------------------------------------------------------------------
# Utility functions that help format EthIntf-related CLI commands
#-------------------------------------------------------------------------------
# Formatting mappings for EthIntf Enums. Using OrderedDict to help format some
# CLI commands (e.g. `show interfaces interactions`).
speedLanestoSpeedLanesStr = collections.OrderedDict(
      [ ( ( 'speed400Gbps', 'laneCount8' ), '400G-8' ),
        ( ( 'speed200Gbps', 'laneCount4' ), '200G-4' ),
        ( ( 'speed200Gbps', 'laneCount8' ), '200G-8' ),
        ( ( 'speed100Gbps', 'laneCount2' ), '100G-2' ),
        ( ( 'speed100Gbps', 'laneCount4' ), '100G-4' ),
        ( ( 'speed50Gbps', 'laneCount1' ), '50G-1' ),
        ( ( 'speed50Gbps', 'laneCount2' ), '50G-2' ),
        ( ( 'speed40Gbps', 'laneCount4' ), '40G' ),
        ( ( 'speed25Gbps', 'laneCount1' ), '25G' ),
        ( ( 'speed10Gbps', 'laneCount1' ), '10G' ),
        ( ( 'speed1Gbps', 'laneCount1' ), '1G' ),
        ( ( 'speed100Mbps', 'laneCount1' ), '100M' ),
        ( ( 'speed10Mbps', 'laneCount1' ), '10M' ),
      ] )

autonegToforcedMapping = collections.OrderedDict(
                   [ ( 'mode400GbpsFull8Lane' , 'linkModeForced400GbpsFull8Lane' ),
                     ( 'mode200GbpsFull4Lane' , 'linkModeForced200GbpsFull4Lane' ),
                     ( 'mode200GbpsFull8Lane' , 'linkModeForced200GbpsFull8Lane' ),
                     ( 'mode100GbpsFull2Lane' , 'linkModeForced100GbpsFull2Lane' ),
                     ( 'mode100GbpsFull' , 'linkModeForced100GbpsFull' ),
                     ( 'mode50GbpsFull1Lane' , 'linkModeForced50GbpsFull1Lane' ),
                     ( 'mode50GbpsFull' , 'linkModeForced50GbpsFull' ),
                     ( 'mode25GbpsFull' , 'linkModeForced25GbpsFull' ),
                   ] )

duplexStringsFormat1 = { 'duplexHalf' : 'Half-duplex',
                         'duplexFull' : 'Full-duplex',
                         'duplexUnknown' : 'Unconfigured' }

duplexStringsFormat2 = { 'duplexHalf' : 'half',
                         'duplexFull' : 'full',
                         'duplexUnknown' : 'unconfigured' }

duplexStringsFormat3 = { 'duplexHalf' : 'a-half',
                         'duplexFull' : 'a-full',
                         'duplexUnknown' : 'unconfigured' }

duplexStrings = { 'Format1' : duplexStringsFormat1,
                  'Format2' : duplexStringsFormat2,
                  'Format3' : duplexStringsFormat3 }

autonegDuplexStrings = { 'Format1' : 'Auto-duplex',
                         'Format2' : 'auto',
                         'Format3' : 'auto' }

anegStateIntfStrMap = { 'unknown' : 'off',
                        'off' : 'off',
                        'negotiating' : 'on',
                        'success' : 'on',
                        'failed' : 'fail' }


def _duplexStr( value, stringFormat ):
   """
   Returns formatted string for EthDuplex.

   Parameters
   ----------
   value : string representation of Interface::EthDuplex
   stringFormat : string describing desired format (one of "Format[123]")
   """
   d = duplexStrings.get( stringFormat )
   return d[ value ] if d else None


CapList = namedtuple( 'CapList', [ 'speed', 'lanes', 'showLanes', 'duplex',
                                   'default', 'pllIncompatible' ] )

FecCapList = namedtuple( 'FecCapList', [ 'encoding', 'speed', 'lanes',
                                         'showLanes', 'default' ] )

# input : Interface::EthLinkModeSet
# return : List of CapList
def linkModeCapabilitiesToList( caps, autonegCapable=False, defaultMode=None,
                                showLanes=False, imcompatElms=None ):
   if showLanes:
      mode100g4Lane = '100G-4/full'
      mode50g2Lane = '50G-2/full'
   else:
      mode100g4Lane = '100G/full'
      mode50g2Lane = '50G/full'

   if imcompatElms is None:
      imcompatElms = Tac.Value( 'Interface::EthLinkModeSet' )

   capLists = []
   for capable, baseStr, pllIncompatible, mode in (
      ( caps.mode10MbpsHalf, '10M/half', False, 'linkModeForced10MbpsHalf' ),
      ( caps.mode10MbpsFull, '10M/full', False, 'linkModeForced10MbpsFull' ),
      ( caps.mode100MbpsHalf, '100M/half', False, 'linkModeForced100MbpsHalf' ),
      ( caps.mode100MbpsFull, '100M/full', False, 'linkModeForced100MbpsFull' ),
      ( caps.mode1GbpsHalf, '1G/half', False, 'linkModeForced1GbpsHalf' ),
      ( caps.mode1GbpsFull, '1G/full', imcompatElms.mode1GbpsFull,
        'linkModeForced1GbpsFull' ),

      # 2.5G/5G doesn't have corresponding linkmodes. They are only used in autoneg.
      ( caps.mode2p5GbpsFull, '2p5G/full', False, 'NON_EXISTENT_LINK_MODE' ),
      ( caps.mode5GbpsFull, '5G/full', False, 'NON_EXISTENT_LINK_MODE' ),

      ( caps.mode10GbpsFull, '10G/full', imcompatElms.mode10GbpsFull,
        'linkModeForced10GbpsFull' ),
      ( caps.mode25GbpsFull, '25G/full', imcompatElms.mode25GbpsFull,
        'linkModeForced25GbpsFull' ),
      ( caps.mode40GbpsFull, '40G/full', imcompatElms.mode40GbpsFull,
        'linkModeForced40GbpsFull' ),
      ( caps.mode50GbpsFull, mode50g2Lane, imcompatElms.mode50GbpsFull,
        'linkModeForced50GbpsFull' ),
      ( caps.mode50GbpsFull1Lane, '50G-1/full', imcompatElms.mode50GbpsFull1Lane,
        'linkModeForced50GbpsFull1Lane' ),
      ( caps.mode100GbpsFull, mode100g4Lane, imcompatElms.mode100GbpsFull,
        'linkModeForced100GbpsFull' ),
      ( caps.mode100GbpsFull2Lane, '100G-2/full', imcompatElms.mode100GbpsFull2Lane,
        'linkModeForced100GbpsFull2Lane' ),
      ( caps.mode200GbpsFull8Lane, '200G-8/full', imcompatElms.mode200GbpsFull8Lane,
        'linkModeForced200GbpsFull8Lane' ),
      ( caps.mode200GbpsFull4Lane, '200G-4/full', imcompatElms.mode200GbpsFull4Lane,
        'linkModeForced200GbpsFull4Lane' ),
      ( caps.mode400GbpsFull8Lane, '400G-8/full', imcompatElms.mode400GbpsFull8Lane,
        'linkModeForced400GbpsFull8Lane' ),
      ( autonegCapable, 'auto', False, 'auto' ) ):
      if capable:
         default = bool( mode == defaultMode )
         if mode == 'auto':
            speed = 'auto'
            lanes = 'laneCountUnknown'
            duplex = 'full'
         elif mode == 'NON_EXISTENT_LINK_MODE':
            speed, lanes, duplex = nonExistentLinkStrToSpeedLanesDuplex[ baseStr ]
            if not showLanes:
               lanes = 'laneCountUnknown'
         else:
            if '/' in baseStr:
               modeStr, duplex = baseStr.split( '/' )
            speed, lanes = modeStrToEthSpeedAndLaneCount( modeStr )

         if duplex.startswith( 'duplex' ):
            duplex = duplex[ len( 'duplex' ): ].lower()
         capList = CapList( speed, lanes, showLanes, duplex, default,
                            pllIncompatible )
         capLists.append( capList )

   return capLists

# linkModeCapabilitiesToStr() is a legacy non-CAPI function, to be removed once
# its callers are CAPI-fied
def linkModeCapabilitiesToStr( caps, autonegCapable=False, defaultMode=None,
                               showLanes=False, imcompatElms=None ):
   if showLanes:
      mode100g4Lane = '100G-4/full'
      mode50g2Lane = '50G-2/full'
   else:
      mode100g4Lane = '100G/full'
      mode50g2Lane = '50G/full'
   if imcompatElms is None:
      imcompatElms = Tac.Value( 'Interface::EthLinkModeSet' )

   capList = []
   for capable, baseStr, incompatible, mode in (
      ( caps.mode10MbpsHalf, '10M/half', False, 'linkModeForced10MbpsHalf' ),
      ( caps.mode10MbpsFull, '10M/full', False, 'linkModeForced10MbpsFull' ),
      ( caps.mode100MbpsHalf, '100M/half', False, 'linkModeForced100MbpsHalf' ),
      ( caps.mode100MbpsFull, '100M/full', False, 'linkModeForced100MbpsFull' ),
      ( caps.mode1GbpsHalf, '1G/half', False, 'linkModeForced1GbpsHalf' ),
      ( caps.mode1GbpsFull, '1G/full', imcompatElms.mode1GbpsFull,
        'linkModeForced1GbpsFull' ),

      # 2.5G/5G doesn't have corresponding linkmodes. They are only used in autoneg.
      ( caps.mode2p5GbpsFull, '2.5G/full', False, 'NON_EXISTENT_LINK_MODE' ),
      ( caps.mode5GbpsFull, '5G/full', False, 'NON_EXISTENT_LINK_MODE' ),

      ( caps.mode10GbpsFull, '10G/full', imcompatElms.mode10GbpsFull,
        'linkModeForced10GbpsFull' ),
      ( caps.mode25GbpsFull, '25G/full', imcompatElms.mode25GbpsFull,
        'linkModeForced25GbpsFull' ),
      ( caps.mode40GbpsFull, '40G/full', imcompatElms.mode40GbpsFull,
        'linkModeForced40GbpsFull' ),
      ( caps.mode50GbpsFull, mode50g2Lane, imcompatElms.mode50GbpsFull,
        'linkModeForced50GbpsFull' ),
      ( caps.mode50GbpsFull1Lane, '50G-1/full', imcompatElms.mode50GbpsFull1Lane,
        'linkModeForced50GbpsFull1Lane' ),
      ( caps.mode100GbpsFull, mode100g4Lane, imcompatElms.mode100GbpsFull,
        'linkModeForced100GbpsFull' ),
      ( caps.mode100GbpsFull2Lane, '100G-2/full', imcompatElms.mode100GbpsFull2Lane,
        'linkModeForced100GbpsFull2Lane' ),
      ( caps.mode200GbpsFull8Lane, '200G-8/full', imcompatElms.mode200GbpsFull8Lane,
        'linkModeForced200GbpsFull8Lane' ),
      ( caps.mode200GbpsFull4Lane, '200G-4/full', imcompatElms.mode200GbpsFull4Lane,
        'linkModeForced200GbpsFull4Lane' ),
      ( caps.mode400GbpsFull8Lane, '400G-8/full', imcompatElms.mode400GbpsFull8Lane,
        'linkModeForced400GbpsFull8Lane' ),
      ( autonegCapable, 'auto', False, 'auto' ) ):
      if not capable:
         continue
      if incompatible:
         capStr = baseStr.replace( '/', '*/' )
      else:
         capStr = baseStr
      if mode == defaultMode:
         capStr += '(default)'
      capList.append( capStr )

   return ','.join( capList )

def generateOperDuplex( stringFormat, autonegActive, duplex ):
   """
   Returns formatted string for operational duplex

   Parameters
   ----------
   stringFormat : string describing desired format (one of "Format[123]")
   autonegActive : boolean describing whether autoneg is active or not
   duplex : string representation of Interface::EthDuplex
   """
   # If auto-negotiation is active and the duplex has not yet
   # been determined, just report auto-duplex
   if autonegActive and duplex == 'duplexUnknown':
      return autonegDuplexStrings[ stringFormat ]
   return _duplexStr( duplex, stringFormat )

def generateAutoNegotiation( autonegState ):
   """
   Helper function that returns the string representation of the interface's
   autonegotiation status

   Parameters
   ----------
   autonegState : string representation of autoneg state
   """
   return anegStateIntfStrMap[ autonegState ]

def getSpeedString( speed, autonegActive, stringFormat ):
   """
   Given a speed in bit/sec and if auto-negotiate this will produce a human
   readable string of the speed of the interface.

   Parameters
   ----------
   speed : floating point value representing speed in bit/sec
   autonegActive : boolean describing whether autoneg is active
   stringFormat : string corresponding to output formats. Example if given a
                  bit/s of 10000000
                  Format1 - 10Mb/s
                  Format2 - 10M
                  Format3 - a-10M
   """
   if autonegActive and speed == 0:
      return "Auto-speed" if stringFormat == 'Format1' else "auto"
   elif speed == 0:
      return "Unconfigured" if stringFormat == 'Format1' else "unconfigured"

   ( scaledBps, units ) = HumanReadable.scaleValueSi( speed )
   # We should not be doing a floating point comparison here
   if scaledBps % 1.0 == 0.0:
      fmt = "%d%s"
   else:
      fmt = "%.1f%s"

   if stringFormat == 'Format1':
      fmt = "%sb/s" % fmt
   elif stringFormat == 'Format2':
      fmt = "%s" % fmt
   else:
      fmt = "a-%s" % fmt

   return fmt % ( scaledBps, units )

def getLaneCountString( laneCount ):
   """
   Returns description of how many electrical lanes are used for a configuration

   Parameters
   ----------
   laneCount : integer representing number of electrical lanes
   """
   if laneCount == 0:
      return ""
   elif laneCount == 1:
      return " over 1 lane"
   else:
      return " over %d lanes" % laneCount

def showIntStatusSpeedStr( autonegActive, bandwidth ):
   """
   Returns formatted string for speed. Utility function used by
   "showInterfacesStatus()"

   Parameters
   ----------
   autonegActive : boolean describing whether autoneg is active
   bandwidth : floating point value representing bps
   """
   fmt = 'Format3' if autonegActive else 'Format2'
   return getSpeedString( bandwidth, autonegActive, fmt )

def showIntStatusDuplexStr( autonegActive, duplex ):
   """
   Returns formatted string for duplex. Utility function used by
   "showInterfacesStatus()"

   Parameters
   ----------
   autonegActive : boolean describing whether autoneg is active
   duplex : string representation of Interface::EthDuplex
   """
   fmt = 'Format3' if autonegActive else 'Format2'
   return generateOperDuplex( fmt, autonegActive, duplex )

def createEthIntfSysdbState( entMan ):
   """
   Sets up some reactors to be run by Sysdb related to 
   reacting to  EthPhyIntfConfig and EthPhyIntfStatus, and
   also sets up the reactors to populate state in the AllEthReactor.

   Parameters
   ----------
   entMan : The EntityManager object in Sysdb.
   """
   t0( "SysdbEthIntf.py plugin loading..." )
   # Create the Interface::AllEthPhyIntf{Config,Status}Dir's
   eicd = entMan.lookup( "interface/config/eth/phy*/all" )
   eisd = entMan.lookup( "interface/status/eth/phy*/all" )

   # Create the Interface::EthIntf{Config,Status}Dir's
   aecd = entMan.lookup( "interface/config/eth/intf" )
   aesd = entMan.lookup( "interface/status/eth/intf" )
   gicd = entMan.lookup( "interface/config/global" )

   eicsd = entMan.lookup( "interface/config/eth/phy*/slice" )
   eissd = entMan.lookup( "interface/status/eth/phy*/slice" )

   # these are the cell-specific Interface::EthIntfStatusLocal entities,
   # which track the per-supervisor state such as the namespace.
   eisld = entMan.lookup( Cell.path( "interface/status/eth/local" ) )
   episld = entMan.lookup( Cell.path( "interface/status/eth/phy/local" ) )

   portIdDir = entMan.lookup( "interface/eth/portid" )

   # The dependency from the SysdbEthIntf plugin to the SysdbIntf plugin has
   # been removed. So it is possible that the plugin loader is loading the 
   # SysdbEthIntf plugin first before it loads the SysdbIntf plugin. 
   # The SysdbIntf plugin used to create the allIntfDir, which was then accessed 
   # by this plugin. Since there is no explicit dependency now, the allIntfDir 
   # object might not have yet been created. Create this object if it hasn't been
   # created.
   localRoot = entMan.root().parent
   intfAgent = localRoot.newEntity( "SysdbIntf::Agent", "SysdbIntf" )
   entMan.allIntfDir = intfAgent.allIntfDir

   allIntfDir = entMan.allIntfDir
   assert allIntfDir

   # Create locally-mounted reactors in sysdbEthIntfSm
   localRoot = entMan.root().parent
   agent = localRoot.newEntity( "SysdbEthIntf::Agent", "SysdbEthIntf" )

   allIntfConfig = allIntfDir.config
   allIntfStatus = allIntfDir.status
   allEthIntfConfig = agent.allEthDir.config
   allEthIntfStatus = agent.allEthDir.status

   # Stash away the allEthDir
   entMan.allEthDir = agent.allEthDir

   aisld = Cell.root( entMan.root() )[ 'interface' ][ 'status' ][ 'local' ]

   # Initialize reactors to copy EthPhyIntfConfigs and EthPhyIntfStatuses
   # to entMan.allIntfDir and entMan.allEthDir
   # For the EthPhyIntfConfigs, we first need a reactor to handle
   # the config requests.
   # This reactor populates interface/config/eth/phy/all
   # The second reactor, ethConfigReactor, in turn populates the allIntf and
   # allEthInt dirs
   agent.ethConfigReactor = ( eicd, allEthIntfConfig, allIntfConfig )

   # Setup a reactor that handles slice dirs added (by Fru) under 
   # interface/status/eth/phy/slice
   agent.ethPhyInfStatusSliceDirReactor = ( eissd, eicsd, eisd, eicd, \
                                            episld, \
                                            allEthIntfStatus, allIntfStatus, \
                                            eisld, aisld, portIdDir, gicd )

   def _handleActive( active ):
      if active:
         t0( "Creating locally-mounted reactors" )
         # Create locally-mounted reactors under /ar/SysdbEthIntf

         # Instantiate an AllEthReactor to sync out our local collection of all 
         # EthIntfs.
         if not ( hasattr( agent, 'allEthReactor' ) and 
                  agent.allEthReactor ):
            agent.allEthReactor = ( aecd, aesd )

         # Setup the reactor for processing the intfConfig requests (from Sysdb
         # and Cli)
         agent.ethPhyInfConfigSliceDirReactor = ( eicsd, eicd )

         # EthPhyIntfStatusDirReactor should no longer be running in standby mode.
         agent.ethPhyInfStatusSliceDirReactor.standbyMode = False
      else:
         # Note: this cleanup code is always a no-op except in testing.
         agent.ethPhyInfStatusSliceDirReactor.standbyMode = True
         t0( "Destroying locally-mounted reactors" )
         agent.ethPhyInfConfigSliceDirReactor = None
         agent.allEthReactor = None

   entMan.registerActiveCallback( _handleActive )
