# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# This library encapsulates some things about finding out the version
# of hardware and software.

import Tac
import os
import errno, re

SwiVersion = Tac.Type( "EosUtils::SwiVersion" )

# For breadth testing, control this by setting os.environ[ "VERSION_ETC_DIR" ].
def etcDir():
   return SwiVersion.etcDir()

# Constants for flavor strings
swiFlavorDefault = "DEFAULT"
swiFlavor2GB = "2GB"
swiFlavorCloud = "cloud"
swiFlavorDPE = "DPE"
swiFlavorPdp = "PDP"

# Constants of swim variant/flavor strings
swimUSDefault = "us"
swimInternationalDefault = "int"
swimUSDPE = "dpe"
swimUS2GB = "2gb"
swimInternational2GB = "2gbInt"

swimVariantFlavorMap = {
   swimUSDefault : [ "US", swiFlavorDefault ],
   swimInternationalDefault : [ "International", swiFlavorDefault ],
   swimUSDPE : [ "US", swiFlavorDPE ],
   swimUS2GB : [ "US", swiFlavor2GB ],
   swimInternational2GB : [ "International", swiFlavor2GB ],
}

def getFanDirection( entityMibRoot ):
   fanDirectionChar = None
   fanTraySlots = entityMibRoot.fanTraySlot
   for fanTraySlot in fanTraySlots.values():
      if not fanTraySlot.fanTray: # no fan in slot
         continue
      match = re.search('-([FR])$', fanTraySlot.fanTray.modelName)
      if match:
         currentFanChar = match.groups()[0]
      else: # unrecognized fan type (or modular chassis)
         continue
      if not fanDirectionChar:
         fanDirectionChar = currentFanChar
      elif fanDirectionChar != currentFanChar: # conflicting fan directions
         return 'Unknown'

   if fanDirectionChar is None: # no fans present
      return 'Unknown'
   elif fanDirectionChar == 'F':
      return 'Forward'
   elif fanDirectionChar == 'R':
      return 'Reverse'
   else: # shouldn't happen
      return 'Unknown'

def getModelNameExtended( modelName, fanDirection ):
   if modelName:
      if fanDirection == 'Unknown':
         return modelName
      else:
         return "%s-%s" % ( modelName, fanDirection[ 0 ] )
   else:
      return "(unknown model name)"

def swiVersion( filename ):
   '''Returns a VersionInfo for the .swi file given in argument.  The file is
   assumed to be a well-formed ZIP archive.'''
   import zipfile
   with zipfile.ZipFile( filename ) as archive:
      # Previously this code was just doing archive.read()
      # but this is unsafe to do because of Python bug 13133
      # which causes a file descriptor leak if the file does
      # not exist.
      try:
         if archive.getinfo( 'version' ):  # If the file exists.
            version = archive.read( 'version' )
      except KeyError:  # File doesn't exist.
         version = ''   # Just pretend it's empty.
   return VersionInfo( None, versionFile=version, arch="" )

class VersionInfo:
   '''This class fetches and aggregates version information for EOS,
   both hardware and software.  Construct this object and pass a
   sysdbRoot (you can pass None, but I may not be able to get all
   fields if you do).  I will then have the following fields:

      modelName():        Model name, such as 'DCS-7148S', or None if not known.

      hardwareRev():      Hardware revision, such as '02.02', or None if not known.

      deviations():       List of deviations as strings, such as ['D000782'],
                          or [] if no deviations.

      serialNum():        Manufacturing serial number, such as 'JFL08290034', or
                          None if not known.

      version():          EOS software version, such as "4.0.0" if blessed, or
                          "4.0.0-121957.2009.intfRange (engineering build)" if not.
                          None if not known.

      internalVersion():  EOS full version, such as "4.0.0-121957.EOS-4.0.0", or
                          None if not known.

      architecture():     CPU architecture, such as 'i386', or None if not known.

      internalBuildId():  Fingerprint like '633eed1e-8f7e-4cec-8af2-16bf267f64aa'.
                          None if not known.

      fanDirection():     Returns either 'Forward', 'Reverse' or 'Unknown'
                          (Unknown could signify a mix of forward and reverse fans,
                          no/unrecognized fans, or a modular chassis)

      modelNameExtended():   Returns the model name with the
                             fan direction, if applicable

   Other optional arguments:

      versionFile:        String: If given, will be used as the contents of the
                          version file instead of reading /etc/swi-version.

      arch:               String: If given, will be used as the name of the
                          architecture instead of reading /etc/arch.
   '''

   def __init__( self, sysdbRoot, versionFile=None, arch=None ):
      mfgName = None
      modelName = None
      hardwareRev = None
      serialNum = None
      deviations = []
      fanDirection = None
      modelNameExtended = None
      try:
         entityMibRoot = sysdbRoot and sysdbRoot[ 'hardware' ][ 'entmib' ].root
      except KeyError:
         pass
      else:
         if getattr( entityMibRoot, 'initStatus', None ) == 'ok':
            mfgName = entityMibRoot.mfgName
            modelName = entityMibRoot.modelName
            hardwareRev = entityMibRoot.hardwareRev
            deviations = entityMibRoot.deviation.keys()
            serialNum = entityMibRoot.serialNum
            fanDirection = getFanDirection( entityMibRoot )
            modelNameExtended = getModelNameExtended( modelName, fanDirection )
      self.mfgName_ = mfgName
      self.modelName_ = modelName
      self.hardwareRev_ = hardwareRev
      self.deviations_ = deviations
      self.serialNum_ = serialNum
      self.fanDirection_ = fanDirection
      self.modelNameExtended_ = modelNameExtended

      self.swiVersion_ = SwiVersion.parse( versionFile or "" )
      try:
         # Note that /etc/arch and /etc/swi-version are generally not present
         # in workspaces, because they are set up by the EosImage makefile
         # (and the "swi installrootfs" tool in EosUtils).
         arch = arch or self.swiVersion_.swiArch or \
                file( os.path.join( etcDir(), 'arch' ) ).read().strip()
      except IOError, e:
         if e.errno != errno.ENOENT: raise

      self.architecture_ = arch
      self.swiMaxHwEpoch_ = int( self.swiVersion_.swiMaxHwEpoch )
      self.isCEos_ = False
      self.cEosToolsVersion_ = None
      self.readCEosToolsVersion()

   def readCEosToolsVersion( self ):
      # The name of the ceos version file and key
      # should probably come from somewhere?
      cEOSFile = os.path.join( etcDir(), "cEOS-release" )
      if os.path.exists( cEOSFile ):
         self.isCEos_ = True
         with open( cEOSFile, "r" ) as f:
            cEosFileContents = f.read()
            for line in cEosFileContents.split('\n'):
               name, var = line.partition( "=" )[ ::2 ]
               if name.strip() == "CEOSUTILS_VERSION":
                  self.cEosToolsVersion_ = var

   def mfgName( self ):           return self.mfgName_
   def modelName( self ):         return self.modelName_
   def hardwareRev( self ):       return self.hardwareRev_
   def deviations( self ):        return self.deviations_
   def serialNum( self ):         return self.serialNum_
   def version( self ):           return self.swiVersion_.version or None
   def internalVersion( self ):   return self.swiVersion_.internalVersion or None
   def architecture( self ):      return self.architecture_
   def internalBuildId( self ):   return self.swiVersion_.internalBuildId or None
   def isCEos( self ):            return self.isCEos_
   def cEosToolsVersion( self ):  return self.cEosToolsVersion_ or None
   def fanDirection( self ):      return self.fanDirection_
   def modelNameExtended( self ): return self.modelNameExtended_
   def swiArch( self ):           return self.swiVersion_.swiArch or None
   def swiVariant( self ):        return self.swiVersion_.swiVariant or None
   def swiFlavor( self ):         return self.swiVersion_.swiFlavor or None
   def swiMaxHwEpoch( self ):     return self.swiMaxHwEpoch_
   def isIntlVersion( self ):     return self.swiVersion_.isIntlVersion
   def is2GBVersion( self ):      return self.swiVersion_.is2GBVersion
   def isCloudVersion( self ):      return self.swiVersion_.isCloudVersion
   def isDPEVersion( self ):      return self.swiVersion_.isDPEVersion
   def isPdpVersion( self ):      return self.swiVersion_.isPdpVersion
   def isBlessed( self ):         return self.swiVersion_.isBlessed
