# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re
import shutil
import time

import Logging
import Tac # pylint: disable-msg=unused-import

Logging.logD( id="SYSDB_STARTUP_CONFIG_PARSE_ERROR",
              severity=Logging.logError,
              format="Errors encountered in parsing the startup-config",
              explanation="One or more errors were encountered while parsing the" \
                          " startup-config due to invalid, incomplete and/or " \
                          "unavailable commands.",
              recommendedAction="Please run 'show startup-config errors' to look " \
                                "at the exact error(s) and fix them in the " \
                                "startup-config." )

startupConfigOutputPath = '/var/log/startup-config-output'

def getPrefdl():
   try:
      with open( '/etc/prefdl' ) as f:
         return f.read()
   except IOError:
      return ""

# Get the name of the path of the boot image from boot-config
# boot-config contains SWI=flash:/EOS.swi, what this returns is /mnt/flash/EOS.swi
def getBootImagePathname( ):

   # Find the path for the SWI filename from boot-config filename,
   # we need to copy .boot-image.swi to it
   bootConfig = '/mnt/flash/boot-config'
   if os.path.isfile( bootConfig ):
      with open( bootConfig ) as f:
         config = f.read()
         swiNameRe = re.compile( '^SWI=(.*)$', flags=re.IGNORECASE|re.MULTILINE )
         reMatch = re.search( swiNameRe, config )
         if reMatch:
            swiName = reMatch.group( 1 )

            # Build the bash pathname, '/mnt/flash/EOS.swi'
            (dev, fname) = swiName.split( ':' )
            swiPath = "/mnt/" + dev + "/" + fname
            return swiPath
   return None

# If this is a dut with small flash device, we need to copy
# the /mnt/flash/.boot-image.swi to /mnt/flash/EOS.swi
def restoreASUBootImage( ):

   # Check if this EOS.swi image needs to be copied back, if not, just return
   cmdLineFilename = '/proc/cmdline'
   cmdFile = file( cmdLineFilename )
   m = re.search( r"arista\.doDelayedSwiCopyBack", cmdFile.read() )
   if not m:
      return

   # Get the bash version of the pathname
   swiPath = getBootImagePathname()
   if not swiPath:
      return

   # Initiate a copy at low prio in the background. Do it in a separate process so
   # that it doesn't affect the ASU timelines.
   forkResult = os.fork()
   if not forkResult:
      # Start the background copy after 5 minutes. All the ASU stages should have
      # completed by then. Record the pid so that we can stop this if a fatal error
      # occurs later on.
      f = open( '/var/imageCopyPidFile', 'w' )
      f.write( str( os.getpid() ) )
      f.close()
      time.sleep( 300.0 )
      shutil.copyfile( '/mnt/flash/.boot-image.swi', swiPath )
      os.remove( '/var/imageCopyPidFile' )
      os._exit( 0 ) # pylint: disable-msg=W0212

