#!/usr/bin/python
# Copyright (c) 2006-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import PyWrappers.Mkfs as mkfs
import os, re, shutil, stat, tempfile, time, pexpect

bytesPerSector = 512
MB = 1024 * 1024
FDISK_COMMAND_LINE = r'Command \(m for help\): '

# Filesystem identifier, filesystem name(mkfs alias), options to pass to fsck,
# size of the crash partition, extra commands after creating the filesystem
filesystems = {
   'vfat' : [ 'c', 'vfat', [ '-F', '32' ], 0, [] ],
   'ext3' : [ '83', 'ext3', [], 0, [] ],
   'ext4' : [ '83', 'ext4', [ '-m', '0' ], 64 * MB,
              [ [ 'tune2fs', '-c', '1' ] ] ],
}

mkfsLabelOption = {
    'vfat': '-n',
    'ext4': '-L',
    'ext3': '-L',
}

# NOTE: fdisk specifies a preferred partition alignment/ granularity of 1 MiB.
# This granularity and therefore minimum partition size is valid unless either
# of two conditions occur:
#       1. The disk block size is over 1 MiB.
#       2. The disk is extremely small (see update_sector_offset() in the
#          util-linux version of fdisk.c)
# These cases are not handled for now because most modern disk block sizes are
# not over 1 MiB, and the disks in our systems are not so small that case 2
# applies.
minPartitionSize = 1024 * 1024

def diskSizeBytes( disk ):
   if stat.S_ISBLK( os.stat( disk ).st_mode ):
      fd = os.open( disk, os.O_RDONLY )
      try:
         res = os.lseek( fd, 0, os.SEEK_END )
      finally:
         os.close( fd )
      return res
   else:
      return os.stat( disk ).st_size

def setupLoop( disk, opts=[] ):
   cmd = ["/sbin/losetup"] + opts + ["--show", "-vf", disk]
   x = Tac.run( cmd, stdout=Tac.CAPTURE,
                             asRoot=True )
   m = re.match( r"(/dev/\w+)", x.strip() )
   assert m, x
   return m.group( 1 )

def deleteLoop( dev ):
   # There doesn't seem to be any reliable way to sync a loop device
   # before attempting to delete it (and on F12, even the trivial
   # 'losetup /dev/loop1 /dev/sdc; losetup -d /dev/loop1' fails
   # intermittently) so we try a bunch of times before giving up
   for _ in range( 10 ):
      x = Tac.run( ["/sbin/losetup", "-d", dev],
                   stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
                   asRoot=True, ignoreReturnCode=True )
      if "Device or resource busy" not in x:
         break
      time.sleep( 1 )

   # Ensure the loopback device buffers get flushed to the real block device
   os.system("sync")
   time.sleep(2)

def mountLoop( disk, loopOpts=[], mountOpts=[], dir=None ):
   if not dir:
      dir = tempfile.mkdtemp()
   try:
      dev = setupLoop( disk, loopOpts )
      open( "%s/loopdev" % dir, "w" ).write( dev )
      Tac.run( ["/bin/mount"] + mountOpts + [dev, dir],
               stdout=Tac.CAPTURE, asRoot=True )
      return dir
   except:
      unmountLoop( dir, removeDir=False )
      raise

def unmountLoop( dir, removeDir=True ):
   Tac.run( ["/bin/umount", dir], asRoot=True, ignoreReturnCode=True )
   try:
      dev = open( "%s/loopdev" % dir ).read()
   except IOError:
      pass
   else:
      deleteLoop( dev )
   if removeDir:
      shutil.rmtree( dir, ignore_errors=True )

def fixupDirPerms( fsDir ):
   # extX filesystem supports POSIX permissions properly. This means that
   # we need to be carefull because in EOS we want everything on the flash to
   # be owned by root:eosadmin (gid=88).
   # We use 88 here instead of eosadmin because Arora doesn't know
   # about the eosadmin group.
   # These commands won't have any effects on vfat, which garantees correct
   # permissions via mount options.
   Tac.run( [ "chown", "-R", "root:88", fsDir ], stdout=Tac.DISCARD,
            stderr=Tac.DISCARD, asRoot=True, ignoreReturnCode=True )
   Tac.run( [ "chmod", "-R", "ug=rwX,o=", fsDir ], stdout=Tac.DISCARD,
            stderr=Tac.DISCARD, asRoot=True, ignoreReturnCode=True )

# Offset defaults to sector 2048, which is the usual start of partition 1
def mountHdFs( disk, offset=2048*bytesPerSector, writable=True, dir=None ):
   return mountLoop( disk, [ "-o", str( offset ) ], [ writable and "-w" or "-r" ],
                     dir )

def unmountHdFs( dir, removeDir=True ):
   unmountLoop( dir, removeDir )

def emptyHd( disk ):
   return file( disk ).read( 2048 ) == chr( 0 ) * 2048

def mountedHd( disk ):
   return re.findall( "^%s[0-9]* " % re.escape( disk ),
                      file( "/proc/mounts" ).read(), re.MULTILINE )

def createFileSystem( disk, fsType, fsPartType, fsOpts, tuneCmds, parts,
                      label=None ):
   loopDev = setupLoop( disk, [ "-o", str( parts[ 0 ] )] )

   if label != None:
      fsOpts += [ mkfsLabelOption[ fsType ], label ]

   try:
      Tac.run( [ mkfs.name() + '.' + fsType ] + fsOpts
               + [ loopDev, str( parts[ 1 ] / 1024 ) ],
               stdout=Tac.CAPTURE, stderr=Tac.CAPTURE, asRoot=True )
      if tuneCmds is None:
         tuneCmds = []
      for tuneCmd in tuneCmds:
         Tac.run( tuneCmd + [ loopDev ],
                  stdout=Tac.CAPTURE, stderr=Tac.CAPTURE, asRoot=True )

      if fsType.startswith( 'ext' ):
         fsDir = mountHdFs( disk, offset=parts[0] )
         try:
            fixupDirPerms( fsDir )
         finally:
            unmountHdFs( fsDir )
   finally:
      deleteLoop( loopDev )

def createNewPartition( fdisk, fsPartType, sectorSize=0, sectorLeft=0 ):
   fdisk.expect( FDISK_COMMAND_LINE )
   fdisk.sendline( "n" )
   fdisk.expect( r"Select \(default p\):" )
   fdisk.sendline( "p" )
   fdisk.expect( r"Partition number \(\d+[-,]\d+, default (\d+)\):" )
   partitionNumber = int( fdisk.match.group( 1 ) )
   fdisk.sendline( "" )
   fdisk.expect( r"First sector \((\d+)-(\d+), default.*:" )

   # We get the first & last sectors from fdisk
   firstSector, lastSector = fdisk.match.groups()
   firstSector, lastSector = int( firstSector ), int( lastSector )

   # We start on the first sector
   fdisk.sendline( str( firstSector ) )
   fdisk.expect( r"Last.*:" )

   # And we end with enough room for sectorLeft sectors
   fsEndSector = lastSector - sectorLeft
   if sectorSize > 0:
      # - 1 because we can use the last sector
      fsEndSector = firstSector + sectorSize - 1
      if fsEndSector > lastSector:
         fsEndSector = lastSector

   fdisk.sendline( str( fsEndSector ) )
   fdisk.expect( FDISK_COMMAND_LINE )
   fdisk.sendline( "t" )
   fdisk.expect( r'Selected partition \d+|Partition number .*:' )
   if str( fdisk.match.group( 0 ) ).startswith( 'Partition number' ):
      fdisk.sendline( str( partitionNumber ) )

   fdisk.expect( r"Hex code \(type L to list (all )?codes\):" )
   fdisk.sendline( fsPartType )

   return firstSector, fsEndSector

def bytesToSector( size, align=0 ):
   if size == 0:
      return 0

   if align:
      size = ( size + ( align - 1 ) ) & ~( align - 1 )
   return ( size - 1 ) / bytesPerSector + 1

def computeStartAndLength( startInSectors, lastInSectors ):
   """Compute in bytes the offset at which the partition starts and its length
   from the start/end sectors.
   """
   start = startInSectors * bytesPerSector
   # We add 1 here because the last sector is actually usable
   length = ( lastInSectors + 1 ) * bytesPerSector - start
   return start, length

def formatHd( disk, fsPartType, fsType, fsOpts=[], crashPartSize=0, tuneCmds=None,
              swi=None, bootConfig=None, startupConfig=None, zeroTouchConfig=None,
              kickStartConfig=None, recoverySwi=None, addToRecovery=None,
              flashSize=0 ):
   """ Set up the partition table.
   Makes two partitions, for the FS and the recovery image.
   Returns [[offset, len], [offset, len]] for the two partitions, in bytes,
   which you can use to loopback mount them.
   """

   file( disk, "r+" ).write( chr( 0 ) * 2048 )

   recovSize = makeRecovFile( "/dev/null", swi, bootConfig, startupConfig,
                              zeroTouchConfig, kickStartConfig, recoverySwi,
                              addToRecovery )

   flashSizeSectors = bytesToSector( flashSize * MB, MB )
   crashSizeSectors = bytesToSector( crashPartSize, MB )
   recovSizeSectors = bytesToSector( recovSize, MB )

   fdisk = pexpect.spawn( "/sbin/fdisk", [ disk ] )

   # Get the size of the disk to be sure that the requested size for the flash
   # will fit. If it doesn't fit, we set flashSize to 0 to use as much space
   # as possible.
   if flashSize:
      fdisk.expect( FDISK_COMMAND_LINE )
      fdisk.sendline( "p" )
      fdisk.expect( r"Disk [^\s]+: \d+(\.\d+)? (MB|GB), (\d+) bytes" )
      # pylint: disable-msg=maybe-no-member
      diskSizeSectors = bytesToSector( int( fdisk.match.group( 3 ) ) )
      totalEstimatedSizeSectors = flashSizeSectors + crashSizeSectors + \
                                  recovSizeSectors
      if totalEstimatedSizeSectors > diskSizeSectors:
         print 'WARNING: Requested flash size is bigger than the available size'
         print 'Going to use all the space available'
         flashSizeSectors = 0

   fdisk.expect( FDISK_COMMAND_LINE )
   fdisk.sendline( "o" )

   # If we want the flash to have a fixed size then do it. Otherwise, compute
   # the size to be left for the crash and recovery partition, if they need
   # to be created.
   if flashSizeSectors > 0:
      flashFirstSector, flashLastSector = \
         createNewPartition( fdisk, fsPartType, sectorSize=flashSizeSectors )
   else:
      flashFirstSector, flashLastSector = \
         createNewPartition( fdisk, fsPartType,
                             sectorLeft=crashSizeSectors + recovSizeSectors )

   fdisk.expect( FDISK_COMMAND_LINE )
   fdisk.sendline( "a" )
   # CentOS 7 fdisk automatically selects the only partition, FC18 version asks
   prompt = fdisk.expect( [ r"Partition number \(1-4\):", r"Selected partition 1" ] )
   if prompt == 0:
      fdisk.sendline( "1" )

   # Create the crash partition if needed
   if crashPartSize:
      # If we are using a fixed size flash, then create the crash partition
      # with the size given in parameter. Otherwise, only leave space for
      # the recovery, because enough room was reserved when we created the flash
      if flashSizeSectors > 0:
         crashFirstSector, crashLastSector = \
            createNewPartition( fdisk, fsPartType, sectorSize=crashSizeSectors )
      else:
         crashFirstSector, crashLastSector = \
            createNewPartition( fdisk, fsPartType, sectorLeft=recovSizeSectors )

   # Create the recovery partition
   recovFirstSector, recovLastSector = \
      createNewPartition( fdisk, '12', sectorSize=recovSizeSectors )

   fdisk.expect( FDISK_COMMAND_LINE )
   fdisk.sendline( "w" )
   fdisk.expect( r"The partition table has been altered!" )

   fsStart, fsLength = computeStartAndLength( flashFirstSector, flashLastSector )
   recovStart, recovLength = computeStartAndLength( recovFirstSector,
                                                    recovLastSector )
   if crashPartSize:
      crashStart, crashLength = computeStartAndLength( crashFirstSector,
                                                       crashLastSector )

   parts = [ [ fsStart, fsLength ], [ recovStart, recovLength ] ]

   # Create our fileystem for the flash
   createFileSystem( disk, fsType, fsPartType, fsOpts, tuneCmds, parts[ 0 ],
                     label='eos_flash' )

   # Create our fileystem for the crash partition if it exists
   if crashPartSize:
      crashOpts = fsOpts + [ '-J', 'size=1' ] if fsType == 'ext4' else []
      createFileSystem( disk, fsType, fsPartType, crashOpts, tuneCmds,
                        [ crashStart, crashLength ], label='eos_crash' )

   return parts

def writeRecovFile( disk, parts, swi, bootConfig=None, startupConfig=None,
                    copyTreeFrom=None, zeroTouchConfig=None, kickStartConfig=None,
                    recoverySwi=None, addToRecovery=None ):
   """Write recovery file, populate filesystem"""
   recovSize = makeRecovFile( "/dev/null", swi, bootConfig, startupConfig,
                              zeroTouchConfig, kickStartConfig, recoverySwi,
                              addToRecovery )
   assert recovSize <= parts[1][1]
   loopDev = setupLoop( disk, ["-o", str(parts[1][0])] )
   try:
      makeRecovFile( loopDev, swi, bootConfig, startupConfig, zeroTouchConfig,
                     kickStartConfig, recoverySwi, addToRecovery )
      fsDir = mountHdFs( disk, offset=parts[0][0] )
      try:
         Tac.run( ["sh", "-c",
                   "(cd %s; cpio -i 2>/dev/null) <%s" % (fsDir, loopDev)],
                  stdout=Tac.CAPTURE,
                  asRoot=True )
         
         if recoverySwi:
            # If recoverySwi is provided, the cpio cmd above copies it to flash
            # from recovery partition, remove that swi from flash
            Tac.run( [ "rm", "-rf", ( os.path.join( fsDir,
                     os.path.basename( recoverySwi ) ) ) ],
                     stdout=Tac.CAPTURE, asRoot=True )

         if swi:
            # Copy the swi to flash and update boot-config if required
            Tac.run( [ "cp", swi, fsDir ], stdout=Tac.CAPTURE, asRoot=True )
            if recoverySwi and not bootConfig:
               file( os.path.join( fsDir, "boot-config" ), "w" ).write(
                     "SWI=flash:/%s\n" % os.path.basename( swi ) )

         if copyTreeFrom:
            Tac.run( [ "sh", "-c",
                       "cp -r %s/. %s" % ( copyTreeFrom, fsDir ) ],
                     stdout=Tac.CAPTURE,
                     asRoot=True )
         fixupDirPerms( fsDir )
      finally:
         unmountHdFs( fsDir )
   finally:
      deleteLoop( loopDev )

   # Make sure all data makes it to hardware before returning. This prevents
   # the unfortunate situation where a user decides to power cycle the box
   # immediately after formatting the hd, and ends up with a completely
   # corrupted hd, without even a recovery partition to use.
   fd = os.open( disk, os.O_RDONLY )
   os.fsync( fd )
   os.close( fd )
   Tac.run( [ "/bin/sync" ] )

def makeRecovFile( recovFile, swi, bootConfig=None, startupConfig=None,
                   zeroTouchConfig=None, kickStartConfig=None,
                   recoverySwi=None, addToRecovery=None ):
   """Create a cpio archive comprising these files:
         - a software image (swi), if specified
         - a "startup-config" file.
         - a "boot-config" file.
         - a "zerotouch-config" see http://aid/723
         - a "kickstart-config" file see: http://cl/2455454
      If bootConfig is not specified, then the "boot-config" file points
   to the software image.
      Note that startupConfig must contain the complete contents of the
   startup-config file, including the trailing newline."""

   fsDir = tempfile.mkdtemp()
   try:
      files = []
      if recoverySwi:
         os.symlink( os.path.abspath( recoverySwi ),
                     os.path.join( fsDir, os.path.basename( recoverySwi ) ) )
         files.append( os.path.basename( recoverySwi ) )
      if recoverySwi or swi or bootConfig:
         file( os.path.join( fsDir, "boot-config" ), "w" ).write(
               bootConfig or "SWI=flash:/%s\n" % os.path.basename(
               recoverySwi or swi ) )
         files.append( "boot-config" )
      file( os.path.join( fsDir, "startup-config" ), "w" )
      if startupConfig:
         file( os.path.join( fsDir, "startup-config" ), "w" ).write(
                                                             startupConfig )
      files.append( "startup-config" )
      if kickStartConfig:
         file( os.path.join( fsDir, "kickstart-config" ), "w" ).write(
               kickStartConfig )
         files.append( "kickstart-config" )
      if zeroTouchConfig:
         file( os.path.join( fsDir, "zerotouch-config" ), "w" ).write(
               zeroTouchConfig )
         files.append( "zerotouch-config" )

      if addToRecovery:
         # add file recursively to recovFile
         for ( dirPath, _, fileNames ) in os.walk( addToRecovery ):
            for fileName in fileNames:
               fullFilePath = os.path.join( dirPath, fileName )
               os.symlink( os.path.abspath( fullFilePath ),
                           os.path.join( fsDir, fileName ) )
               files.append( fileName )

      fixupDirPerms( fsDir )
      recovSize = Tac.run( [ "sh", "-c",
                             "(cd %s; cpio -ocL 2>/dev/null) | tee %s | wc -c" %
                             ( fsDir, recovFile ) ],
                           input="\n".join( files ), stdout=Tac.CAPTURE,
                           asRoot=True )

      return max( minPartitionSize, int( recovSize.strip() ) )

   finally:
      shutil.rmtree( fsDir, ignore_errors=True )

def blockDeviceNode( name ):
   pat = dict( [ reversed( x.split() ) for x in file( "/etc/blockdev" ) ] )[ name ]
   for dev in os.listdir( "/sys/block" ):
      if re.match( "mmcblk.*(boot.*|rpmb)", dev ):
         # eMMC device creates a mmcblk*boot0/1 partitions by default.
         # and Linux allows access to the RPMB partition since 3.8 (090d25f),
         # skip these, as we really want to manage the actual
         # device instead of these partitions
         continue
      devid = os.path.realpath( os.path.join( "/sys/block", dev, "device" ) )
      if devid.startswith( "/sys/devices/" ) and re.match( pat, devid[13:] ):
         return "/dev/%s" % dev
