#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Arnet
import socket
import json
import Tracing
import time
import calendar
from IpLibConsts import DEFAULT_VRF
import QuickTrace
import errno
import os
import Toggles.DhcpServerToggleLib
import Tac

__defaultTraceHandle__ = Tracing.Handle( "DhcpServer" )
t8 = Tracing.t8

qt8 = QuickTrace.trace8

overrideLockfileDir = "/var/run/kea"
tmpKeaConfFile = "/tmp/kea-dhcp{version}-{vrf}.conf.tmp"
tmpToCheckKeaConfFile = "/tmp/kea-dhcp{version}-{vrf}-ToCheck.conf.tmp"
keaLeasePath = "/var/lib/kea/dhcp{version}-{vrf}.leases"
keaConfigPath = "/etc/kea/kea-dhcp{version}-{vrf}.conf"
keaControlConfigPath = "/etc/kea/kea-ctrl{version}-{vrf}.conf"
keaPidPath = "/var/lib/kea/kea-dhcp{version}-{vrf}.kea-dhcp{version}.pid"
keaServiceName = "keactrl{version}-{vrf}"
keaControlSock = "/tmp/kea-dhcp{version}-{vrf}-ctrl.sock"

# These are the default lease times for kea
leaseTimeDurationKeaDefault = ( 0, 2, 0 )
leaseTimeSecondsKeaDefault = 7200

vendorSubOptionType = Tac.Type( 'DhcpServer::VendorSubOptionType' )
defaultVendorId = 'default'

# Messages for interface status
class DhcpIntfStatusMessages( object ):
   NO_INTF_STATUS_LOCAL_MSG = "Could not determine VRF"
   NOT_IN_DEFAULT_VRF_MSG = "Not in default VRF"
   NO_KNI_MSG = "Kernel interface not created"
   NOT_UP_MSG = "Not up"
   NO_IP_ADDRESS_MSG = "No IP address"
   DHCP_RELAY_CFG_MSG = "DHCP relay is configured for this interface"
   DHCP_RELAY_ALWAYS_MSG = "DHCP relay is always on"

# Messages for subnet direct status
class Dhcp6DirectRelayMessages( object ):
   MULTIPLE_INTERFACES_MATCH_SUBNET = "Multiple interfaces match this subnet: {}"
   MULTIPLE_SUBNETS_MATCH_INTERFACE = "This and other subnets match interface {}"
   NO_IP6_ADDRESS_MATCH = "No IPv6 addresses on interfaces match this subnet"

unknownSubnets = { 4: Arnet.Prefix( "0.0.0.0/0" ),
                   6: Arnet.Ip6Prefix( '::0/0' ) }

def dhcpServerActive( status ):
   return bool( status.ipv4ServerDisabled == "" or
                status.ipv6ServerDisabled == "" )

def convertLeaseSeconds( seconds ):
   seconds = int( seconds )
   val = seconds // 60
   minutes = val % 60
   val = val // 60
   days = val % 24
   hours = val // 24

   return hours, days, minutes

def convertDaysHoursMinutes( days, hours, minutes ):
   return ( ( days * 24 * 60 * 60 ) +
            ( hours * 60 * 60 ) +
            ( minutes * 60 ) )

def featureOption43Status():
   return Toggles.DhcpServerToggleLib.\
                  toggleFeatureOption43Enabled()

def featureStaticIPv4AddressStatus():
   return Toggles.DhcpServerToggleLib.\
                  toggleFeatureStaticIPv4AddressEnabled()

def subOptionCmd( optionCode, optionType=None, optionData=None, optionIsArray=None,
                  disable=False ):
   # optionData: can be a double quoted string or a list
   if disable:
      cmd = 'no sub-option {}'.format( optionCode )
      return cmd

   cmd = 'sub-option {} type'.format( optionCode )
   if optionIsArray:
      cmd += ' array'

   # set proper type and data
   if optionType == vendorSubOptionType.string:
      eosCliType = 'string'

   elif optionType == vendorSubOptionType.ipAddress:
      eosCliType = 'ipv4-address'
      optionData = ' '.join( optionData )

   else:
      assert False, "Unknown sub-option type"
      return None

   cmd += ' {} data {}'.format( eosCliType, optionData )
   return cmd

def getSubOptionData( subOption, raw=False ):
   # raw=False means get optionData "as" is written in the kea config
   # raw=True means get optionData "as" entered by the user.
   # string: double quoted string
   # ipAddress: array of ip addresses
   if subOption.type == vendorSubOptionType.string:
      # add quotes
      dataRaw = '"{}"'.format( subOption.dataString )
      # allow "," in the string
      data = dataRaw if raw else subOption.dataString.replace( ",", r"\," )
      return data

   else:
      dataRaw = subOption.dataIpAddress.values()
      data = dataRaw if raw else ','.join( dataRaw )
      return data

def tftpServerOptions( option66, option150 ):
   # E.g. one option is configured
   # TFTP Server: 1.1.1.1 (option 66)
   # E.g. both options are configured (print each option on a new line)
   # TFTP Server:
   # 1.1.1.1 (option 66)
   # 2.2.2.2 3.3.3.3 (option 150)
   serverOptions = []
   serverOptionsStr = ''
   if option66:
      serverOptions.append( "{} (Option 66)".format( option66 ) )
   if option150:
      serverOptions.append( "{} (Option 150)".format(
         ' '.join( option150 ) ) )
   if serverOptions:
      fmt = '\n' if len( serverOptions ) == 2 else ' '
      serverOptionsStr = 'TFTP server:{}{}'.format(
         fmt, '{}'.format( fmt.join( serverOptions ) ) )
   return serverOptionsStr

class Lease( object ):
   def __init__( self, leaseData, endTime, cltt ):
      self.leaseData = leaseData
      self.ipAddress = None
      self.end = endTime
      self.lastTransaction = cltt

   def cltt( self ):
      return self.lastTransaction

   def endTime( self ):
      return self.end

   def ip( self ):
      return self.ipAddress

   def mac( self ):
      return self.leaseData[ 'hw-address' ]

   def __str__( self ):
      return "<Lease {}: E:{} L:{} M:{}>".format(
         self.ip(),
         self.endTime(),
         self.cltt(),
         self.mac() )

   def __repr__( self ):
      return str( self )

class Ipv4Lease( Lease ):
   def __init__( self, leaseData, endTime, cltt ):
      Lease.__init__( self, leaseData, endTime, cltt )
      self.ipAddress = Arnet.IpAddr( leaseData[ 'ip-address' ] )

class Ipv6Lease( Lease ):
   def __init__( self, leaseData, endTime, cltt ):
      Lease.__init__( self, leaseData, endTime, cltt )
      self.ipAddress = Arnet.Ip6Addr( leaseData[ 'ip-address' ] )

class Subnet( object ):
   def __init__( self, subnet, ipVersion ):
      self.subnet = subnet
      self.leases = {}
      self.ipVersion = ipVersion

   def addLease( self, leaseData, end, cltt ):
      lease = Ipv4Lease( leaseData, end, cltt ) if self.ipVersion == 4 else \
              Ipv6Lease( leaseData, end, cltt )
      # Overrwrite if it exist, since the later one in the file is newer
      self.leases[ lease.ip() ] = lease

   def numActiveLeases( self ):
      return len( self.leases )

   def lease( self, leaseIp ):
      return self.leases[ leaseIp ]

class KeaDhcpLeaseData( object ):
   def __init__( self, subnets, ipVersion=4, vrf=DEFAULT_VRF, keaLimit=1024,
                       filterFunction=None ):
      self.subnetData = {}
      self.subnets = subnets
      self.ipVersion = ipVersion
      self.vrf = vrf
      self.keaLimit = keaLimit
      try:
         self._parseData( filterFunction=filterFunction )
      except socket.error as e:
         # ENOENT when control socket doesn't exist
         # EPIPE when server closes on us
         if e.errno not in [ errno.ENOENT, errno.EPIPE ]:
            qt8( "DhcpServer: Socket had unexpected error: %s, %s" %
                 ( e.errno, os.strerror( e.errno ) ) )

   def _parseData( self, filterFunction=None ):
      # Example lease output
      # { u'text': u'1 IPv4 lease(s) found.',
      #   u'arguments': { u'count': 1,
      #                   u'leases': [ { u'fqdn-rev': False,
      #                                  u'state': 0,
      #                                  u'ip-address':
      #                                  u'192.0\.2.1',
      #                                  u'cltt': 1554224535,
      #                                  u'valid-lft': 3600,
      #                                  u'hw-address':
      #                                  u'ba:25:91:1e:3f:1e',
      #                                  u'hostname': u'',
      #                                  u'fqdn-fwd': False,
      #                                  u'subnet-i\d': 1 } ]
      #                 },
      #   u'result': 0 }
      now = time.gmtime()
      t8( "GM now", calendar.timegm( now ) )
      for leases in leaseDataIter( self.ipVersion, self.vrf, self.keaLimit,
                                   filterFunction=filterFunction ):
         for lease in leases[ "arguments" ][ "leases" ]:
            # Only care about active leases
            # const uint32_t Lease::STATE_DEFAULT = 0x0;
            # const uint32_t Lease::STATE_DECLINED = 0x1;
            # const uint32_t Lease::STATE_EXPIRED_RECLAIMED = 0x2;
            if lease[ "state" ] != 0:
               continue
            cltt = float( lease[ "cltt" ] )
            end = cltt + float( lease[ "valid-lft" ] )
            # Skip if expired
            if end > now:
               continue
            subnet = self.subnets.get( lease[ "subnet-id" ],
                                       unknownSubnets[ self.ipVersion ] )
            try:
               subnetObj = self.subnetData[ subnet ]
            except KeyError:
               subnetObj = Subnet( subnet, self.ipVersion )
               self.subnetData[ subnet ] = subnetObj
            subnetObj.addLease( lease, end, cltt )

   def subnet( self, subnet ):
      return self.subnetData.get( subnet )

   def totalActiveLeases( self ):
      return sum( [ data.numActiveLeases() for data in self.subnetData.values() ] )

_serverPathMapping = {
   "dhcp4": keaControlSock.format( version="4", vrf=DEFAULT_VRF ),
}

class KeaDhcpSocket( object ):
   def __init__( self, ipVersion, vrf, bufSize=4096 ):
      self.bufSize = bufSize
      path = keaControlSock.format( version=ipVersion, vrf=vrf )
      self.sock = socket.socket( socket.AF_UNIX, socket.SOCK_STREAM )
      self.sock.connect( path )

   def runCmd( self, cmdData ):
      try:
         return self._runCmd( cmdData )
      finally:
         # The dhcp server closes the connection after every command, so we should
         # too
         self.sock.close()

   def _runCmd( self, cmdData ):
      self.sock.sendall( json.dumps( cmdData ) )
      resultData = []

      # The server will close the connection, so then recv won't block anymore
      data = self.sock.recv( self.bufSize )
      while data:
         resultData.append( data )
         data = self.sock.recv( self.bufSize )

      return json.loads( "".join( resultData ) )

class KeaDhcpCommandError( Exception ):
   pass

def runKeaDhcpCmd( ipVersion, vrf, cmdData ):
   sock = KeaDhcpSocket( ipVersion, vrf )
   result = sock.runCmd( cmdData )
   # Four result codes possible:
   # 0 No error, results found
   # 1 General error
   # 2 Command not supported
   # 3 No error, no results found
   if result[ "result" ] in [ 1, 2 ]:
      t8( 'KeaDhcpCommandError result:', result[ 'result' ] )
      raise KeaDhcpCommandError( "Command failed with {}".format(
         result[ "result" ] ) )
   return result

def configReloadCmdData():
   return {
      "command": "config-reload"
   }

def _leaseDataIterData( startFrom, limit, ipVersion='4' ):
   return {
      "command": "lease{}-get-page".format( ipVersion ),
      "arguments": {
         "from": startFrom,
         "limit": limit
      }
   }

def leaseDataIter( ipVersion, vrf, limit=1024, filterFunction=None ):
   cmdData = _leaseDataIterData( "start", limit, ipVersion )
   result = runKeaDhcpCmd( ipVersion, vrf, cmdData )
   while True:
      leaseCount = result[ "arguments" ][ "count" ]
      leases = result[ "arguments" ][ "leases" ]

      # filter leases and update the result
      if filterFunction:
         fLeases = [ l for l in leases if filterFunction( l ) ]
         result[ "arguments" ][ "count" ] = len( fLeases )
         result[ "arguments" ][ "leases" ] = fLeases

      t8( result )
      yield result
      if leaseCount < limit:
         # No more leases
         break
      # Start off from where we left off
      lastIp = result[ "arguments" ][ "leases" ][ -1 ][ "ip-address" ]
      cmdData = _leaseDataIterData( lastIp, limit, ipVersion )
      result = runKeaDhcpCmd( ipVersion, vrf, cmdData )
