#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
from subprocess import Popen, PIPE
import sys
import re
import shutil
import Tac
import SecretCli
import json
from Fru.FruBaseVeosDriver import parseVeosConfig
import VeosHypervisor
from ArPyUtils import arch

metadataIp = '169.254.169.254'

# EOS paths
userData = "/mnt/flash/.userdata"
kickstartConfig = "/mnt/flash/kickstart-config"
defaultStartupConfig = "/mnt/flash/.default_startup-config"
startupConfig = "/mnt/flash/startup-config"

## Process downloaded user data by find start markers and write contents into
## corresponding config files
def startMark( name ):
   return '%%%s-START%%' % name

def endMark( name ):
   return '%%%s-END%%' % name

class configFile:
   def __init__( self, name, filename, append=False ):
      self.name = name
      self.endMarker = endMark( name )
      self.filename = filename
      self.append = append

forceUserDataString = "%FORCE-USER-DATA%"
eosStartupConfig = 'EOS-STARTUP-CONFIG'
startMarkStartupConfig = startMark( eosStartupConfig )
endMarkStartupConfig = endMark( eosStartupConfig )

emptyUserData = "%s\n%s\n" % ( startMarkStartupConfig, endMarkStartupConfig )

## user data needs to have start and end markers to denote the specific config
## eg for EOS-STARTUP-CONFIG the markers need to be
## %EOS-STARTUP-CONFIG-START%
## %EOS-STARTUP-CONFIG-END%
defaultConfigs = {
   startMark( 'EOS-STARTUP-CONFIG' ) : configFile( "EOS-STARTUP-CONFIG",
                                                   startupConfig,
                                                   append=True ),
   startMark( 'CLOUDHA-CONFIG' ) : configFile( "CLOUDHA-CONFIG",
                                               "/mnt/flash/cloud_ha_config.json" ),
   startMark( 'LICENSE-IPSEC' ) : configFile(
      "LICENSE-IPSEC", "/persist/secure/license/store/ipsec_license.json" ),
   startMark( 'LICENSE-BANDWIDTH' ) : configFile(
      "LICENSE-BANDWIDTH", "/persist/secure/license/store/bandwidth_license.json" ),
   startMark( 'VEOS-CONFIG' ) : configFile( "VEOS-CONFIG",
                                            "/mnt/flash/veos-config" ),
}

def runCmd( cmd, dump=True, background=False ):
   p = Popen( cmd, stdout=PIPE, stderr=PIPE )
   if background:
      return
   ( output, _ ) = p.communicate()
   rc = p.returncode
   if dump:
      print "Execute " + " ".join( cmd )
      print "Output: %s" % output
   return output, rc

def cleanupExit( errStr, rc=0 ):
   print errStr
   # cleanup dhclients if any
   runCmd( [ 'pkill', 'dhclient' ] )
   sys.exit( rc )

def readFile( fileName ):
   f = open( fileName, "r" )
   contents = f.readlines()
   f.close()
   return contents

def getNetDevName():
   ## get network device name
   devName = [ 'eth0', 'ma1' ]
   devDir = '/sys/class/net'
   netDevs = [ dev for dev in os.listdir( devDir ) if dev in devName ]
   if netDevs:
      return netDevs[ 0 ]
   return None

def getIP( devname ):
   p = Popen( [ 'ip', 'addr', 'show', 'dev', devname ],
              stdout=PIPE, stderr=PIPE )
   out, _ = p.communicate()
   pat = re.compile( r"inet\s(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}/\d{1,3})" )
   m = pat.search( out )
   return m.groups()[ 0 ] if m else None

def writeFile( fileName, contents ):
   f = open( fileName, "w" )
   contents = "".join( contents )
   f.write( contents )
   f.close()

def getDefaultGw( devname='' ):
   output, _ = runCmd( [ "ip", "route" ], dump=False )
   pat = re.compile( "default via (.*) dev %s" % devname )
   m = pat.search( output )
   if m is not None:
      print "GW is %s" % m.group( 1 )
      return m.group( 1 )
   else:
      return None

def getDhcpLeaseContents():
   leasePaths = [
      # in Azure we noticed sometimes that the default lease
      # file was missing but dhclient.leases was present
      # so including both for redundancy
      '/var/lib/dhclient/dhclient.leases',
      '/var/lib/dhclient/dhclient-default.leases'
   ]
   def leaseFile():
      for leasePath in leasePaths:
         if os.path.isfile( leasePath ):
            return leasePath
      return None

   Tac.waitFor( leaseFile, timeout=10.0 )

   leaseFilePath = leaseFile()
   if leaseFilePath:
      leaseContents = ''.join( readFile( leaseFilePath ) )
   else:
      cleanupExit( "Could not acquire dhcp lease", rc=1 )

   return leaseContents

def getDnsIp():
   leaseContents = getDhcpLeaseContents()
   m = re.search( 'domain-name-servers ([0-9]+.[0-9]+.[0-9]+.[0-9]+)',
                  leaseContents )
   if m:
      return m.groups()[ 0 ]
   else:
      cleanupExit( "DNS server IP not found in dhcp lease", rc=1 )

def getAllIntfs():
   devDir = '/sys/class/net'
   pattern = r'(eth[0-9]+)'
   netDevs = [ dev for dev in os.listdir( devDir )
                if re.match( pattern, dev ) is not None ]
   return netDevs

def setupConnection( ping=True ):
   ## Bring up the interface
   ## Aboot renames the network interface to ma1 if corresponding network
   ## drivers are enabled in Aboot kernel.
   netDev = getNetDevName()
   if netDev == None:
      cleanupExit( "Unable find eth0/ma1..network Device", rc=1 )

   runCmd( ["ip", "link"] )
   _, rc = runCmd( ["ip", "link", "set", "dev", netDev, "up"] )
   if rc != 0:
      cleanupExit( "Unable to bringup" + netDev + \
                   " no network access. RC %d" % rc, rc )
   runCmd( ["ip", "link"] )

   ## Get IP address using DHCP
   _, rc = runCmd( [ "dhclient", netDev ] )
   if rc != 0:
      cleanupExit( "dhclient failed return code %d" % rc, rc )

   runCmd( ["ip", "addr"] )

   ## get default GW
   gw = getDefaultGw()
   if not gw:
      cleanupExit( "Unable to find default GW..exiting", rc=1 )

   ## allow established connections so we can retrieve provisioning information
   runCmd( [ "iptables-save", "-c" ] )
   _, rc = runCmd( [ "iptables", "-I", "INPUT", "-m", "conntrack", "--ctstate",
                     "established,related", "-j", "ACCEPT" ] )
   if rc != 0:
      print "Could not add iptables rule to accept outgoing connections; continuing"
   else:
      runCmd( [ "iptables-save", "-c" ] )

   ## wait until we can ping the default GW
   if ping:
      for _ in range( 0, 5 ):
         _, rc = runCmd( ["ping", gw, "-w", "5", "-c", "1"] )
         if rc == 0:
            break
      if rc != 0:
         cleanupExit( "Unable to ping GW. RC %d" % rc, rc )

   return gw

def findMarkIndex( mark, contents ):
   markIndices = [ i for i, s in enumerate( contents ) if mark in s ]
   if markIndices:
      return markIndices[ 0 ]
   else:
      return -1

def generateUserConfig( userName, userPassword, keyName='key.pub' ):
   userConfig = []
   if userPassword:
      hashAlg = 'sha512'
      hashPassword = SecretCli.generalEncryption( userPassword, hashAlg )
      # add username with hashed password
      userConfig.append(
         'username %s secret %s %s\n' % ( userName, hashAlg, hashPassword ) )
   else:
      # add username with no password
      userConfig.append( 'username %s secret *\n' % userName )
   # add public key file location for username
   userConfig.append(
      'username %s ssh-key file flash:%s\n' % ( userName, keyName ) )
   return userConfig

def firstTimeBoot():
   return not os.path.exists( startupConfig )

def isForcedUserData():
   if not os.path.exists( userData ):
      return False
   with open( userData ) as f:
      if f.readline().strip() == forceUserDataString:
         return True
   return False

def copyDefaultStartupConfig( extraConfig=None ):
   shutil.copyfile( defaultStartupConfig, startupConfig )
   if extraConfig:
      contents = readFile( startupConfig )
      contents.extend( extraConfig )
      writeFile( startupConfig, contents )

def processUserData( configs, extraConfig=None, copyDefaultConfig=True ):
   configs.update( defaultConfigs )

   config = None
   fo = None

   if not os.path.exists( userData ):
      print "User Data file does not exist"
      return

   with open( userData ) as f:
      ## %FORCE-USER-DATA% in the first line of the user data will make
      ## EOS treat the boot as a first time boot and honor the user data over
      ## existing configuration
      cline = f.readline().strip()
      if isForcedUserData():
         print "Forcing first time boot"
         if os.path.exists( startupConfig ):
            shutil.copyfile( startupConfig, "/mnt/flash/startup-config-backup" )
      elif firstTimeBoot():
         f.seek( 0, 0 )
      else:
         return
      if copyDefaultConfig:
         copyDefaultStartupConfig( extraConfig=extraConfig )
      ## loop through user data splitting into configuration files
      for line in f:
         cline = line.strip()
         ## Process the various configs
         if cline in configs:
            ## found start marker
            if fo != None:
               ## Previous config file is still open
               print "End marker not found for %s" % config.name
               fo.close()
            config = configs[ line.strip() ]
            print "found start marker for %s" % config.name
            try:
               if config.append is True:
                  fo = open( config.filename, "a" )
               else:
                  fo = open( config.filename, "w" )
            # pylint: disable=W0703
            except Exception, error:
               print "Unable to open file to write for %s error %r" % ( config.name,
                                                                        error )
         elif fo != None and cline == config.endMarker :
            print "found end marker for %s" % config.name
            fo.close()
            fo = None
         else:
            if fo == None:
               ## ignore and continue in case we haven't found start marker
               continue
            fo.write( line )
      if fo != None:
         print "End marker not found for %s" % config.name
         fo.close()

def isModeSfe( modeFile = '/mnt/flash/veos-config' ):
   return parseVeosConfig( modeFile )[ 'MODE' ] == 'sfe'

def systemMemoryGB():
   memGB = 0
   with open('/proc/meminfo') as f:
      meminfo = f.readlines()[ 0 ]
      matched = re.search(r'^MemTotal:\s+(\d+)', meminfo)
      if matched: 
         memKB = int(matched.groups()[0])
         memGB = (memKB / ( 1024 * 1024 ))

   return memGB

# To determine number of threads running the same CPU core.
## cat /sys/devices/system/cpu/cpu0/topology/thread_siblings_list
## 0; if Hyper-Threading is disabled.
## 0,2; if Hyper-Threading is enabled.
# The output above indicates that Logical CPU 0 and Logical CPU 2 are
# threads on the same physical core CPU 0.
# We need to parse this and find number of siblings on physical core 0.
def siblingCount():
   output = None
   try:
      with open( "/sys/devices/system/cpu/cpu0/topology/thread_siblings_list" ) as f:
         output = f.read() 
   except IOError:
      return 1
   # Some kernels use comma and some hypen in the list file
   # so account for both
   siblings = re.split( '[,-]', output )
   return len( siblings )

# Simultaneous Multithreading (SMT) a.k.a Hyper-Threading (HT) allows
# multiple execution threads to be executed on a single physical CPU core.
# To find if the VM has hyper-threading enabled, identify the number of
# threading executing on the same physical core using siblingCount(). If
# hyper-threading is enabled, then disable it, if the platform is not Azure
# or total number of cpus are more than or equal to 4 cpus by appending
# "nr_cpus=" to half the number of cpus /mnt/flash/kernel-params.
def disableHT():
   ncpu = os.sysconf( "SC_NPROCESSORS_ONLN" )
   platform = VeosHypervisor.getPlatform()
   count = siblingCount()

   # If hyper-threading is already disabled, set nr_cpus to ncpu.
   if platform != 'Azure' and count == 1 and ncpu >= 2:
      return " nr_cpus=" + str( ncpu )

   # Don't disable HyperThreading on Azure cloud platform
   if ncpu < 4 or platform == 'Azure':
      return ""

   if count <= 1:
      return ""
   # We expect to see a bunch of CPUs online at boot-up
   totalCpus = os.sysconf("SC_NPROCESSORS_CONF")
   if totalCpus == (ncpu * count):
      return ""
   # To suppress kernel warning during initialization on GCP/AWS
   # platform set nr_cpus to half the number of total cpus.
   val = int( totalCpus ) / int( count )
   return " nr_cpus=" + str( val )

def readBessdMemConfigFile( cfgFile ):
   with open( cfgFile ) as f:
      config = json.loads( f.read() )
   return ( config[ 'hugepages' ].encode( 'ascii' ),
            config[ 'memoryForBessdInMB' ].encode( 'ascii' ),
            config[ 'buffersForBessd' ].encode( 'ascii' ) )

# Look at CPU and Memory requirements to create a kernel command line
def sfeKernelParams():

   # Allow for configuration of bessd memory paramters through a config file if
   # such a file exists. This config file is for arista use only.
   bessdMemConfigFile = '/mnt/flash/bessd-mem-config.json'
   if os.path.exists( bessdMemConfigFile ):
      try:
         retStr, memoryForBessdInMB, buffersForBessd = readBessdMemConfigFile(
            bessdMemConfigFile )
      except: # pylint: disable-msg=W0702
         pass
      else:
         retStr += disableHT()
         return ( retStr, memoryForBessdInMB, buffersForBessd )

   retStr = ""
   # We need to write how much memory should bessd start with into a file on
   # mnt/flash so that systemd service can use it when launching bessd
   memoryForBessdInMB = "1536"
   buffersForBessd = "65536"

   cpus = os.sysconf( "SC_NPROCESSORS_ONLN" )
   if cpus > 4:
      retStr = "isolcpus=2,3,4,5,6,7,8,9,10,11,12,13,14,15 "
   else:
      retStr = "isolcpus=1,2,3,4,5,6,7,8,9,10,11,12,13,14,15 "

   memGB = systemMemoryGB()
   # If less than 4 GB RAM on system, allocate only 2M hugepages as the system
   # would not allow 1G hugepages under 4G RAM (who knew!)
   # On Sfe platforms dpdk 19.11 memics older dpdk by running in legacy-mode
   # and iova-mode as a physical address mode (RTE_IOVA_PA). The total memory
   # available to dpdk is specified by option --socket-mem/-m (for a 32-bit
   # application this is 1.5G). This memory is pre-allocated by dpdk during
   # initialization. In 32-bit architecture, this total memory allocated is
   # sum of the total number of contiguous memory segments found in this dpdk
   # memory (specified by option --socket-mem) and sum of page separator
   # between those contiguous memory segments. Maximum allowed contiguous
   # segments for dpdk per numa node is limited to 64 (RTE_MAX_MEMSEG_LISTS).
   # Hence the maximum hugepages that can be pre-allocated by dpdk cannot exceed
   # the sum of RTE_MAX_MEMSEG_LISTS (64) and socket-mem (1.5G) / hugepagesize (2M)
   # = 832. Request 832 hugepages of 2M size to the kernel via kernel-params.
   # These separator hugepages allocated by dpdk are freed after initialization,
   # making them available for other processes to use.

   if memGB <= 4:
      retStr += "hugepagesz=2M hugepages=832"
   # > 4 GB RAM
   else:
      if arch() == 64: # 2x1G Pages + 256x2M pages, so 2.5GB of RAM in HugePages
         retStr += "default_hugepagesz=1G hugepagesz=1G"
         retStr += " hugepages=2 hugepagesz=2M hugepages=256"
         memoryForBessdInMB = "2560"
      else: # 32-bit - 832x2M pages, so 1.625GB RAM HugePages
         retStr += "default_hugepagesz=2M hugepagesz=2M hugepages=832"
         memoryForBessdInMB = "1536"

   # Disable HyperThreading if needed
   retStr += disableHT()
   return ( retStr, memoryForBessdInMB, buffersForBessd )

# Create or append to a kernel params file on /mnt/flash/kernel-params
# It should look something like 
# default_hugepagesz=1G hugepagesz=1G hugepages=2 hugepagesz=2M hugepages=256
# isolcpus=2,3,4,5,6,7,8,9,10,11,12,13,14,15
# If the file did not have relevant config already, we need to trigger a reboot
# after the write. The values of the memory allocation depends on how much actual
# memory is present in the system.
# Also sets up memory and buffer parameters for bessd that are consumed by the
# bessd.service.in file.
# Returns true if reboot is required
def setupSfeKernelParams( kernelParamsFile='/mnt/flash/kernel-params',
                          bessdMemFile='/mnt/flash/bessd.mem',
                          bessdBuffersFile='/mnt/flash/bessd.buffers' ):
   neededStr, memForBessd, buffersForBessd = sfeKernelParams()
   otherParams = ""

   with open( bessdMemFile, "w+" ) as f:
      f.write(memForBessd)

   with open( bessdBuffersFile, "w+" ) as f:
      f.write( buffersForBessd )

   if not os.path.exists( kernelParamsFile ):
      with open(kernelParamsFile, "w") as f:
         f.write(neededStr)
         return True

   with open(kernelParamsFile, "r") as f:
      lines = f.readlines()
      # Something in the file ?
      if len(lines) != 0:
         content = lines[ 0 ]
         # no need to do anything if params are in order
         if neededStr in content:
            return False
      else:
         content = ""

      for tag in content.split():
         if re.search('^default_hugepagesz=', tag):
            continue
         if re.search('^hugepagesz=', tag):
            continue
         if re.search('^hugepages=', tag):
            continue
         if re.search('^isolcpus=', tag):
            continue
         if re.search('^nosmt=', tag):
            continue
         if re.search('^nr_cpus=', tag):
            continue

         # This contains stuff in the file we should not over-write
         otherParams = otherParams + ' ' + tag

   with open(kernelParamsFile, "w") as f:
      # Write all this down into the file
      f.write(otherParams + ' ' + neededStr)

   # We need to reboot after this
   return True

