#!/usr/bin/env python
# Copyright (c) 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import smtplib, Tac
from email.utils import make_msgid, formatdate, parseaddr, getaddresses
import time
from IpLibConsts import DEFAULT_VRF
from Arnet.NsLib import DEFAULT_NS

def format_smtperrors( errors ):
   return [ "%s: %d %s" % ( who, code, text ) for
                                    ( who, ( code, text ) ) in errors.items() ]

class EmailFailure( Exception ):
   def __init__( self, message, recipients=None, originalException=None ):
      if recipients:
         message += ": " + ", ".join( format_smtperrors( recipients ) )
      Exception.__init__( self, message )
      self.recipients = recipients
      self.originalException = originalException

def add_headers( settings, netSettings, msg ):
   if not(msg.has_key('Message-ID')):
      msg['Message-ID'] = make_msgid('email')
   if not(msg.has_key('Date')):
      msg['Date'] = formatdate(time.time(), True)
   if not(msg.has_key('From')):
      if settings.fromUser:
         msg['From'] = settings.fromUser
      elif netSettings.domainName:
         msg['From'] = 'unconfigured-switch@' + netSettings.domainName
      else:
         raise EmailFailure( 'you must configure an email from address' )
   return msg

def netNsList():
   cmd = [ "ip", "netns", "list" ]
   nsList = Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE )
   return nsList.split()

def setProcessNsByName( ns ):
   setns = Tac.newInstance( 'Arnet::Netns', 'dummy' )
   setns.ns = ns

def send_smtp( settings, netSettings, allVrfStatusLocal,
               msg, bcc=None, debug=False ):
   if settings.host == '':
      raise EmailFailure( 'you must configure an outgoing SMTP server' )
   vrf = settings.vrfName
   if vrf and vrf != DEFAULT_VRF and vrf not in allVrfStatusLocal.vrf:
      raise EmailFailure( 'VRF %s does not exist' % vrf )
   if vrf in allVrfStatusLocal.vrf and \
         allVrfStatusLocal.vrf[ vrf ].state != 'active':
      raise EmailFailure( 'the state of VRF %s is not active' % vrf )
   add_headers( settings, netSettings, msg )
   (fname, frm) = parseaddr(msg.get('From'))
   addrlist = msg.get_all('To') + msg.get_all('Cc', [])
   if bcc:
      addrlist += [bcc]
   to = [addr for name, addr in getaddresses(addrlist)]
   server = None
   connected = False
   try:
      ns = DEFAULT_NS
      if vrf in allVrfStatusLocal.vrf and \
            allVrfStatusLocal.vrf[ vrf ].state == 'active':
         ns = allVrfStatusLocal.vrf[ vrf ].networkNamespace
      if ns in netNsList():
         setProcessNsByName( ns )
      server = smtplib.SMTP( timeout=60 )
      if debug:
         server.set_debuglevel(1)
      server.connect(settings.host, settings.port)
      connected = True
      if settings.useTls:
         server.ehlo()
         if 'starttls' not in server.esmtp_features:
            raise EmailFailure('TLS configured but starttls not supported')
         (retval, retmsg) = server.starttls()
         if retval != 220:
            raise EmailFailure('TLS configured but tls failed: %d %s' % (
                                                                retval, retmsg ))
         # Send a new EHLO, since without TLS the server might not
         # advertise the AUTH capability.
         server.ehlo()
      if settings.user and settings.password:
         server.login(settings.user, settings.password)
      ret = server.sendmail(frm, to, msg.as_string())
      if ret != {}:
         raise EmailFailure( 'The following addresses(s) were not accepted', ret )
   except Exception, e:
      if type( e ) is smtplib.SMTPRecipientsRefused:
         raise EmailFailure( 'None of the addresses were valid', e.recipients, e )
      elif type( e ) is EmailFailure:
         raise
      else:
         raise EmailFailure( 'Failed to send email: %s' % e, None, e )
   finally:
      if connected:
         server.quit()
