# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from collections import OrderedDict
import string

class DecoderException( Exception ):
   """ Exceptions raised by an EepromDecoder class """
   pass

class EepromDecoder( object ):
   def __init__( self ):
      # preFdl defaults
      self.preFdl = OrderedDict( [
         ( "DataFormat", "0002" ),
         ( "PCA", "PCA999990000" ),
         ( "SerialNumber", "" ),
         ( "KVN", "111" ),
         ( "SKU", "" ),
         ( "HwRev", "01.00" ),
         ( "MfgTime", "" ),
         ( "ASY", "ASY999999999" ),
         ( "HwApi", "01.00" ),
         ( "SID", "" ),
         ( "HwEpoch", "02.00" ),
         ( "cellType", "fixed" ),
         ( "slotId", "1" ),
      ] )

   def read( self ):
      self._read()
      return self

   def _read( self ):
      # Read the raw encoded data from an eeprom device and parses it into the
      # Arista prefdl format
      raise NotImplementedError

   def get( self, key ):
      # After reading Eeprom, returns value associated with 'key'
      return self.preFdl.get( key )

   def set( self, key, value ):
      # This function allows values to be directly defined or overriden
      self.preFdl[ key ] = value

   def format( self ):
      # Generate a preFdl string from the data stored in self.preFdl dictionary
      if self.preFdl.get( "SKU" ):
         self.preFdl[ "SKU" ] = self.preFdl[ "SKU" ].replace( " ", "-" )

      # BUG176377: If the prefdl has a DiagVersion and it has any unprintable
      # characters, then replace the DiagVersion with a dummy value.
      diagVers = self.preFdl.get( "DiagVersion" )
      if diagVers:
         if any( x not in string.printable for x in diagVers ):
            self.preFdl[ "DiagVersion" ] = "0.0"

      lines = [ "{}: {}\n".format( key, self.preFdl[ key ] ) for key in self.preFdl ]
      return "".join( lines )
