# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import errno, os, weakref
import Tac, Tracing, Ethernet, SharedMem, Smash
from collections import namedtuple
from Intf import SyncMtu
from EbraTestBridgePort import EbraTestPortBase, EbraTestPort
from eunuchs.if_ether_h import ETH_P_PAUSE, ETH_P_8021Q, ETH_P_ALL
from EbraTestBridgeLib import (
   HANDLER_PRIO_HIGHEST,
   HANDLER_PRIO_NORMAL,
   HANDLER_PRIO_LOWEST,
   PKTEVENT_ACTION_NONE,
   PKTEVENT_ACTION_REMOVE,
   PKTEVENT_ACTION_ADD,
   PKTEVENT_ACTION_REPLACE,
   applyVlanChanges,
   macAddrsAsStrings,
)
from TypeFuture import TacLazyType
import EbraTestBridgeLib
import importlib
import CEosHelper
from Toggles import EtbaDutToggleLib

try:
   # We're not using "import Arnet.PacketFormat" to prevent a
   # dependency on scapy to be generated.
   PacketFormat = importlib.import_module('Arnet.PacketFormat').PacketFormat
except ImportError:
   # PacketFormat uses scapy, which is GPL'd. In order to avoid distributing it
   # to customers (in vEOS specifically), use Tracing.HexDump if PacketFormat
   # is not available
   PacketFormat = Tracing.HexDump

__defaultTraceHandle__ = Tracing.Handle( "EbraTestBridge" )
t0 = Tracing.trace0
t4 = Tracing.trace4    # misc important events
t5 = Tracing.trace5    # Packet tx/rx, error level
t6 = Tracing.trace6    # Packet tx/rx, alert level
t7 = Tracing.trace7    # Packet tx/rx, normal level
t8 = Tracing.trace8    # Packet tx/rx, debug level
t8b = Tracing.trace3   # Packet tx/rx, super-debug level
t9 = Tracing.trace9    # MAC address table aging/flushing

mountHelperTac = TacLazyType( "Interface::EthIntfCounterMountHelper" )
EtbaIntfCounterHelper = TacLazyType( 'Arfa::ArfaIntfCounterHelper' )
HostKey = TacLazyType( "Bridging::HostKey" )
SmashFdbStatus = TacLazyType( "Smash::Bridging::SmashFdbStatus" )
RoutingHandlerCounter = TacLazyType( 'Bridging::Etba::RoutingHandlerCounter' )

# the maximum number of fdbStatus entries that an etba dut will handle. Note that
# setting this higher will require more memory for each etba dut regardless of
# whether the entries are used
ETBA_SMASH_SIZE = 16 * 1024

from EbraTestBridgeConstants import * # pylint: disable-msg=wildcard-import

# Return type for EbraTestBridge.bridgeFrame
BridgeFrameInfo = namedtuple( 'BridgeFrameInfo', [
   'vlanInfo',
   'data',
   'egressIntfs',
   ] )

# EbraTestBridge routing handler counters
ebraRoutingHandlerCounter = {}
ebraRoutingHandlerCounter[ "overall" ] = RoutingHandlerCounter()

# FwdIntfDevice routing handler counters
fwdIntfRoutingHandlerCounter = {}

# Interface plugins (e.g., Lag) register themselves in this list with an
# intfStatus type name and an EbraTestPort subclass.
interfaceHandlers = []

# Callback function that can be registered from a plugin to determine the action
# that needs to be performed on the packet right when it enters on the front panel
# interface. Return values could be 'permit', 'deny' or 'trap'. If 'deny' is
# returned, the packet is dropped and subsequent stages are not invoked (including
# bridging). 'trap' would result in the packet getting trapped to cpu (subsequent
# stages are skipped) and 'permit'/any other return value any other return value
# would cause the packet to proceed to the subsequent stages in the pipeline
preBridgingHandler = None

# Callback function that can be registered from a plugin that decrypts the packets
# on links that have macsec enabled.
macsecDecryptHandler = None

# Callback function that can be registered from a plugin that encrypts the packets
# before transmit on links that have macsec enabled.
macsecEncryptHandler = None

# Callback function that can be registered from a plugin to determine the action
# that needs to be performed on the packet right when it enters on the front panel
# interface and before bridging the frame. Return values could be (False, False)
# (chgData, chgSrcPort) if expected Tunnel header was found to be stripped.
# In any case packet to proceed to the subsequent stages in the pipeline.
preTunnelHandler = []
preTunnelHdlrPrioMap = { k: [] for k in range( HANDLER_PRIO_HIGHEST,
                                               HANDLER_PRIO_LOWEST ) }

# Destination lookup (e.g., IgmpSnooping) plugins register themselves in this
# list with a function to perform the lookup.  All registered functions
# are called, in arbitrary [Plugin registration] order, and return
# a tuple of ( matched, intfList ) -- if a plugin returns True in
# matched, then no further plugins are consulted.  If no plugin returns
# True, then the fdb for the VLAN is consulted.
destLookupHandlers = []

# Floodset-Includes-Cpu plugins register themselves with a function
# that is called to determine whether a given packet should be flooded
# to the CPU.  If the Cpu interface is in the floodset (meaning there is
# an SVI for that VLAN), then if the packet is not broadcast, the
# Floodset-Includes-Cpu plugins are called in arbitrary order, and
# return False (don't send this packet to the Cpu), True (do), or
# None (I don't have an opinion about sending this packet to the Cpu).
floodsetIncludesCpuHandlers = []

# Rewrite-Packet-On-The-Way-To-The-Cpu plugins register themselves with a
# function that is called just before a packet is sent to the CPU. These
# plugins get the chance to modify the packet, for instance to rewrite the
# destination MAC address, so that the CPU handles the packet properly.
# The plugin returns a copy of the packet so that destinations beside the
# CPU do not see the modifications.
# The plugins are called in arbitrary order, and return ( data, dstMac )
# if they have made a modification to either of those values (or want to
# prevent any later plugins from making a modification).
# The "metadata" destination MAC address is passed as a separate argument
# and so can be modified separately from the address written into the packet.
rewritePacketOnTheWayToTheCpuHandlers = []

# Tunnel termination handlers. If this switch terminate e.g. an GRE tunnel,
# the handler will decap the packet and form a new packet to go through the rest of
# processing. 
tunnelTerminationHandlers = []

# Packet replication plugins register themselves with a function that is
# called to modify the outgoing list of interfaces. The function is
# given the source port and a list of output interfaces.  Examples include
# port mirroring and Mlag constrained flooding.
packetReplicationHandlers = []

# Trap lookup (e.g., IgmpSnooping) plugins register themselves in this
# list with a function to perform the lookup.  This hook is for things
# that redirect packets to the cpu or force a copy of the packet to be
# sent to the cpu.  All registered functions are called, in arbitrary
# [Plugin registration] order, and return a tuple of ( matched, trap,
# copyToCpu ) -- only 1 plugin is expected to return true in matched,
# If no plugin returns True, then the packet is forwarded normally.
# Typically, the value of variable trap would be True for trapping or
# False for not trapping, but in some special case, like Ptp, it could
# also be a constant TRAP_ANYWAY, which is used when stpState is
# stpDiscarding, but we want to trap it anyway. This is because, though
# most protocol agents listen on the port-channel interfaces for their
# respective protocols on lags for packets, a subset like PTP agent
# listens on the individual member interfaces instead of the port-channel
# interface itself, but for Etba the packets don't make it to the member
# interfaces as the stpState is marked as stpDiscarding on the member
# interfaces, and stpForwarding on the port-channel itself.
trapLookupHandlers = []

# Sflow plugin registers in this list
# with a function to preform sampling if necessary.
# We currently are interested in the following packets:
#  - Arrived on an interface that is not the Fabric.
#  - Is not being trapped to the CPU.
#  - Has no reason to be dropped by hardware.
sflowHandlers = []

# Post bridging handlers will  register here .Plugin is called just before sending
# the frame out of the interface ( pkts not destined to CPU ) and can modify
# contents of Ethernet frame, including smac and/or dmac.
# This plugin is currrently used by FHRP to modify outgoing ARP packets with smac
# as varp mac if outgoing interface belongs to an unnumbered SVI/Vlan
postBridgingHandlers = []

# Routing plugin ( e.g., Mroute ) register themselves in this list
# with a function to perform the routing. All registered functions are
# called succesively in no particular order, the packets are then bridged
ebraRoutingHandlers = []
preTunnelHandlerBpfFilter = {}

# Plugins may need to initialize themselves at bridge-init or agent-init
# time.
bridgeInitHandlers = []
agentInitHandlers = []
# subset of bridgeInitHandlers. These plugins will be initialized during switchover.
switchoverBridgeInitHandlers = []

# using for bridgeInitHandlers and switchoverBridgeInitHandlers for now, because
# agent may try initialize these plugins at more than one place. This list will be
# used to guard against such multiple initializations
pluginsInitialized = []

# Learning handlers are invoked at learnAddr() time.  For example, if there
# is an active peerLink (in an Mlag configuration)
learningHandlers = []
postAgingHandlers = []

# During a switchover, Etba agent will take care of platform and hardware
# agents that don't run on a namespace dut so that all switchover stages
# can be marked as complete
agentsAndStagesDict = {}
agentsAndStagesDict[ 'PlxPcie-system' ] = [ 'PCIEAcquired',
                                            'PcieConfigurable' ]
agentsAndStagesDict[ 'ForwardingAgent' ] = [ 'DmaReady',
                                             'HwSyncWaitNormal',
                                             'HwSyncWaitForSlowCards' ]
agentsAndStagesDict[ 'ElectionMgr' ] = [ 'SwitchoverReady' ]
agentsAndStagesDict[ 'Fru' ] = [ 'Fru-Plugins' ]
agentsAndStagesDict[ 'ModularSystem' ] = [ 'ModularSystem-switchover' ]
agentsAndStagesDict[ 'NorCalCard' ] = [ 'CardPowerAgents' ]
agentsAndStagesDict[ 'Pca9555' ] = [ 'CardPowerAgents' ]
agentsAndStagesDict[ 'Ucd9012-system' ] = [ 'CardPowerAgents' ]
agentsAndStagesDict[ 'MactAgent' ] = [ 'InitMacTable' ]
agentsAndStagesDict[ 'SandL3Unicast' ] = [ 'PlatformLfibSync' ]

class PluginContext( object ):
   TRAP_PRIORITIES = TRAP_PRIORITIES

   def registerBridgeInitHandler( self, func ):
      bridgeInitHandlers.append( func )

   def registerAgentInitHandler( self, func ):
      agentInitHandlers.append( func )

   def registerSwitchoverBridgeInitHandler( self, func ):
      global switchoverBridgeInitHandlers
      switchoverBridgeInitHandlers.append( func )

   def registerInterfaceHandler( self, typeName, klass ):
      interfaceHandlers.append( ( typeName, klass ) )

   def registerPreBridgingHandler( self, func ):
      global preBridgingHandler
      preBridgingHandler = func

   def registerPreTunnelHandler( self, func, hdlrPrio=HANDLER_PRIO_NORMAL,
                                 bpfFilterInfo=None ):
      if ( hdlrPrio == HANDLER_PRIO_HIGHEST and
         preTunnelHdlrPrioMap[ HANDLER_PRIO_HIGHEST ] ):
         assert False, "Multiple handlers are not allowed at HANDLER_PRIO_HIGHEST"
      preTunnelHdlrPrioMap[ hdlrPrio ].append( func )
      global preTunnelHandler
      hdlrlist = []
      for prio in preTunnelHdlrPrioMap:
         hdlrlist.extend( preTunnelHdlrPrioMap[ prio ] )

      if not EtbaDutToggleLib.toggleFastEtbaEnabled():
         assert bpfFilterInfo is None
      else:
         assert bpfFilterInfo is not None
         bpfFilterKey, bpfFilterString = bpfFilterInfo
         assert bpfFilterKey not in preTunnelHandlerBpfFilter
         preTunnelHandlerBpfFilter[ bpfFilterKey ] = ( bpfFilterString, func )
      # As the EbraTestBridge file imports this one, we can't create a new list
      del preTunnelHandler[ : ]
      preTunnelHandler.extend( hdlrlist )

   def registerDestLookupHandler( self, func ):
      destLookupHandlers.append( func )

   def registerFloodsetIncludesCpuHandler( self, func ):
      floodsetIncludesCpuHandlers.append( func )

   def registerRewritePacketOnTheWayToTheCpuHandler( self, func ):
      rewritePacketOnTheWayToTheCpuHandlers.append( func )

   def registerTunnelTerminationHandler( self, func ):
      tunnelTerminationHandlers.append( func )

   def registerPacketReplicationHandler( self, func ):
      packetReplicationHandlers.append( func )

   def registerPostBridgingHandler( self, func ):
      postBridgingHandlers.append( func )

   def registerTrapLookupHandler( self, func,
                                  priority=TRAP_PRIORITIES[ "DEFAULT" ] ):
      # Higher priority number means higher priority
      trapLookupHandlers.append( ( priority, func ) )

   def registerLearningHandler( self, func ):
      learningHandlers.append( func )

   def registerPostAgingHandler( self, func ):
      postAgingHandlers.append( func )

   def registerSflowHandler( self, func ):
      sflowHandlers.append( func )

   def registerRoutingHandler( self, func ):
      ebraRoutingHandlers.append( func )
      ebraRoutingHandlerCounter[ func.__name__ ] = RoutingHandlerCounter()

   def registerMacsecDecryptHandler( self, func ):
      global macsecDecryptHandler
      macsecDecryptHandler = func

   def registerMacsecEncryptHandler( self, func ):
      global macsecEncryptHandler
      macsecEncryptHandler = func

# learning handlers are invoked in learnAddr(), for example to
# disable learning for peer links in an Mlag configuration.
# A learning handler should be invoked with ( vlanId, macAddr, portName )
# and should return either a string or None.

def invokeLearningHandlers( vlanId, macAddr, portName, entryType ):
   result = []
   for h in learningHandlers:
      r = h( vlanId, macAddr, portName, entryType )
      if( r is not None ):
         result.append( r )
   return result

def invokePostAgingHandlers( ctx, vlanId, macAddr ):
   for h in postAgingHandlers:
      h( ctx, vlanId, macAddr )

def isIEEEReserved( addr ):
   return addr[ :15 ] == '01:80:c2:00:00:'

def isIEEELinkConstrained( addr ):
   return ((addr[ :16 ] == '01:80:c2:00:00:0') or (addr == PVST_BPDU_ADDR))

# Starting in Linux 4.5, if the interface of a tun/tap device is down,
# the kernel returns EIO on writes instead of silently dropping the
# packet.

def writeToTunTapDevice( fd, buf ):
   try:
      os.write( fd, buf )
   except OSError as e:
      if e.errno != errno.EIO:
         raise

class VlanStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Bridging::VlanStatusDir'
   def __init__( self, bridgingVlanStatusDir, bridge, entityManager ):
      assert( bridge )
      self.bridge_ = bridge
      self.entityManager_ = entityManager
      Tac.Notifiee.__init__( self, bridgingVlanStatusDir )

   def getFid( self, vlanId ):
      if 0 < vlanId < 4095:
         fid = self.bridge_.brConfig_.vidToFidMap.get( vlanId )
      else:
         fid = None
      if not fid:
         fid = vlanId
      return fid

   @Tac.handler( 'vlanStatus' )
   def handleVlanStatus( self, vlanId ):
      fid = self.getFid( vlanId )
      if vlanId in self.notifier_.vlanStatus:
         # vlan being added, add existing configured hosts, if any
         t0( 'handleVlanStatus Up, vlanId %d ' % vlanId )
         fdbConfig = self.bridge_.brConfig_.fdbConfig.get( fid )
         if fdbConfig:
            for key in fdbConfig.configuredHost:
               hostEntry = fdbConfig.configuredHost.get( key )
               if hostEntry:
                  t0( "handleVlanStatus: adding %d/%s to %s" %
                     ( vlanId, key, hostEntry.intf ) )
                  self.bridge_.learnAddr( vlanId, key,
                     hostEntry.intf, hostEntry.entryType )
         return

      isActive = self.entityManager_.redundancyStatus().mode == "active"

      if isActive:
         # Vlan is down, flush all learnedHosts
         for lh in self.bridge_.smashBrStatus_.smashFdbStatus.itervalues():
            if lh.fid == fid:
               t9( "handleVlanStatus flushing macAddr %s/%s" %
                     ( str( lh.address ), lh.entryType ) )
               self.bridge_.deleteMacAddressEntry( vlanId, str( lh.address ),
                                                   lh.entryType )

class FdbConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Bridging::FdbConfig'

   def __init__( self, fdbConfig, bridge ):
      Tac.Notifiee.__init__( self, fdbConfig )
      self.bridge_ = bridge
      self.vlanId = fdbConfig.fid
      for key in fdbConfig.configuredHost:
         self.handleConfiguredHost( key )

   @Tac.handler( 'configuredHost' )
   def handleConfiguredHost( self, key ):
      hostEntry = self.notifier_.configuredHost.get( key )
      if hostEntry:
         t0( "handleConfiguredHost: adding %d/%s to %s" %
               ( self.vlanId, key, hostEntry.intf ) )
         self.bridge_.learnAddr( self.vlanId, key,
                                 hostEntry.intf, hostEntry.entryType )
      else:
         t0( "handleConfiguredHost: removing %d/%ss" %
               ( self.vlanId, key ) )
         self.bridge_.deleteMacAddressEntry( self.vlanId, key,
                                             'configuredStaticMac' )

def isPhyControl( srcMac, dstMac, etherType, data ):
   return dstMac == '01:1c:73:ff:ff:ff' and etherType == 0xd28b \
         and data[ 0:4 ] == '\xe1\xba\x00\x00'

class HostTableFlushReactor( Tac.Notifiee ):
   notifierTypeName = 'Bridging::HostTableFlushRequestDir'

   def __init__( self, brConfig, bridge, flushReplyDir ):
      Tac.Notifiee.__init__( self, brConfig )
      self.bridge_ = bridge
      self.flushReplyDir_ = flushReplyDir
      self.dequeueActivity_ = Tac.ClockNotifiee(
         handler=self._processHostTableFlushRequestQueue )
      self.dequeueActivity_.timeMin = Tac.endOfTime

   @Tac.handler( 'hostTableFlushRequest' )
   def handleHostTableFlushRequest( self, ignored ):
      self.dequeueActivity_.timeMin = Tac.beginningOfTime
      # Defer processing of the queue to a different activity, because otherwise
      # when we remove an event from the queue it triggers a recursive call to
      # handleHostTableFlushRequest.

   def _processHostTableFlushRequestQueue( self ):
      for key in self.notifier_.hostTableFlushRequest:
         req = self.notifier_.hostTableFlushRequest[ key ]
         self.flushReplyDir_.hostTableFlushReply[ key ] = True

         t9( 'Handling MAT flush request for vlanId', req.vlanId or '<all>',
             'interface', req.intf or '<all>' )

         learnedMacs = [ 'learnedDynamicMac' , 'learnedRemoteMac' ]
         for learnedHost in self.bridge_.smashBrStatus_.smashFdbStatus.itervalues():
            if learnedHost.entryType in learnedMacs:
               if ( ( req.vlanId == 0 or req.vlanId == learnedHost.fid ) and
                    ( req.intf == "" or req.intf == learnedHost.intf ) ):
                  self.bridge_.deleteMacAddressEntry( learnedHost.fid,
                                                      learnedHost.address )

class EbraTestPhyPortIntfStatusReactor( Tac.Notifiee ):
   '''Synchronizes the configured MTU with the interface if
   it becomes a routed port.'''
   notifierTypeName = 'Interface::EthIntfStatus'
   def __init__( self, intfStatus, intfStatusLocal, intfConfig ):
      Tac.Notifiee.__init__( self, intfStatus )
      self.intfConfig_ = intfConfig
      self.intfStatusLocal_ = intfStatusLocal

   @Tac.handler( 'forwardingModel' )
   def handleForwardingModel( self ):
      # Synchronize the MTU with the kernel and the status if the forwarding
      # model becomes routed.
      SyncMtu.syncMtu( self.notifier_, self.intfStatusLocal_, self.intfConfig_ )

class EbraTestPhyPortIntfConfigReactor( Tac.Notifiee ):
   '''Controls IntfStatus::linkState based on IntfConfig::enabled,
   and synchronizes the configured MTU with the interface when it
   changes.'''
   notifierTypeName = 'Interface::EthIntfConfig'

   def __init__( self, intfConfig, intfStatus, intfStatusLocal, ebraTestPhyPort ):
      Tac.Notifiee.__init__( self, intfConfig )
      self.intfStatus_ = intfStatus
      self.intfStatusLocal_ = intfStatusLocal
      self.ebraTestPhyPort = ebraTestPhyPort
      self.handleEnabledStateLocal()
      self.handleMtu()

   @Tac.handler( 'enabledStateLocal' )
   def handleEnabledStateLocal( self ):
      # Note that we react to 'enabledStateLocal' because neither 'enabled' nor
      # 'enabledState' are notifying.
      self.ebraTestPhyPort.updateLinkStatus()

   @Tac.handler( 'mtu' )
   def handleMtu( self ):
      # Synchronize the MTU for L3 interfaces.
      SyncMtu.syncMtu( self.intfStatus_, self.intfStatusLocal_, self.notifier_ )

class PktReader( Tac.Notifiee ):
   notifierTypeName = "Arnet::EthDevPam"

   def __init__( self, pam, handler ):
      self.pam = pam
      Tac.Notifiee.__init__( self, pam )
      self.handler = handler
      self.counter = 0

   @Tac.handler( "readableCount" )
   def readHandler( self ):
      t0( 'readHandler called %s ' % self.pam.readableCount )
      pkt = self.pam.rxPkt()
      if pkt:
         self.counter += 1
         self.handler( pkt.stringValueMaybeWithVlan )

class EbraTestPhyPort( EbraTestPort ):
   """Simulates a physical port."""

   def __init__( self, bridge, tapDevice, trapDevice, intfConfig,
                 intfStatus, intfXcvrStatus,
                 intfStatusLocal, intfCounters, entityManager ):
      """Initialize the port to its default state."""
      EbraTestPort.__init__( self, bridge, intfConfig, intfStatus )
      self.noTrapDevice = False
      self.pktReader = None
      if CEosHelper.isCeos():
         t0( 'Skip creating the trapDevices in CEos' )
         self.noTrapDevice = True

      # pylint is confused by the value returned by Tac.nonConst.
      # pylint: disable=maybe-no-member, E1103
      if tapDevice.fileno == 0:
         # Bind to the interface ourselves (this is used in the vEOS
         # environment where we don't already have a socket set up
         # for us)
         EthDevPam = Tac.Type( 'Arnet::EthDevPam' )
         pam =  EthDevPam( tapDevice.name )
         pam.ethProtocol = ETH_P_ALL
         pam.maxRxPktData = 10000
         pam.separateDot1QHdr = True
         pam.enabled = True

         self.pktReader = PktReader( pam, self._tapReadHandler )
         tapDevice = Tac.nonConst( tapDevice )
         tapDevice.fileno = pam.sd
         if self.noTrapDevice:
            trapDevice = tapDevice
         self.tapFile_ = None

      self.tapDevice_ = tapDevice
      self.trapDevice_ = trapDevice

      self.processFrame_ = None

      self.phyIsUp = True
      self.prevIntfConfigEnabled = False

      self.intfCounters_ = intfCounters

      self.bridge_ = bridge
      self.intfConfig_ = intfConfig
      self.intfStatus_ = intfStatus
      self.intfXcvrStatus_ = intfXcvrStatus
      self.intfStatusLocal_ = intfStatusLocal

   def onActive( self ):
      t0( "onActive EbraTestPhyPort" )
      if not self.pktReader:
         self.tapFile_ = Tac.File( self.tapDevice_.fileno, self._tapReadHandler,
                                   self.tapDevice_.name, readBufferSize=16000 )
      if not self.noTrapDevice:
         self.trapFile_ = Tac.File( self.trapDevice_, self._trapDeviceReadHandler,
                                    self.trapDevice_.name, readBufferSize=16000 )


      self.intfPhyConfigReactor_ = EbraTestPhyPortIntfConfigReactor(
         self.intfConfig_, self.intfStatus_, self.intfStatusLocal_, self )
      self.intfPhyStatusReactor_ = EbraTestPhyPortIntfStatusReactor(
         self.intfStatus_, self.intfStatusLocal_, self.intfConfig_ )
      self.operStatusReactor = Tac.newInstance( "EthIntf::OperStatusReactor",
                                                self.intfStatus_.intfId,
                                                self.intfStatus_,
                                                self.intfXcvrStatus_)

      self.intfXcvrStatus_.xcvrPresence = 'xcvrPresent'
      self.intfXcvrStatus_.xcvrType = 'EbraTestPhyPort'
      self.intfStatus_.duplex = 'duplexFull'
      self.intfStatus_.addr = self.intfStatus_.burnedInAddr
      if CEosHelper.isCeos():
         # In cEOS, the environment gives us the
         # interface MAC address.
         self.intfStatus_.routedAddr = self.intfStatus_.addr
      else:
         self.intfStatus_.routedAddr = self.bridge_.bridgeMac()


   def sendPhyControl( self ):
      raw = EbraTestBridgeLib.rawPhyControlPacket( self.intfConfig_.enabled,
               src=Ethernet.convertMacAddrToPackedString( self.intfStatus_.addr ) )
      writeToTunTapDevice( self.tapDevice_.fileno, raw )

   def updateLinkStatus( self ):
      # The link state is dependent on admin state and on the link
      # status obtained from the phy control protocol.
      linkStatus = 'linkUp' if (self.intfConfig_.enabled and self.phyIsUp) \
                   else 'linkDown'
      t4( 'Set link state to', linkStatus, 'for', self.intfName_ )
      if self.prevIntfConfigEnabled != self.intfConfig_.enabled:
         # Send out a PhyControl frame to let the other end know about
         # the admin status of this port
         self.sendPhyControl()
      self.prevIntfConfigEnabled = self.intfConfig_.enabled
      self.intfStatus_.linkStatus = linkStatus

   def setProcessFrame( self, func ):
      """Hook frame input for this port"""
      self.processFrame_ = func

   def _handlePhyControl( self, data ):
      """Handle a phy control frame.  This can be used to bring up or
      down the link by sending in a packet.  See AID6280"""
      self.phyIsUp = (ord(data[18]) != 0)
      self.updateLinkStatus()

   def _countIn( self, intf, octets, dstMacAddr ):
      if Ethernet.isBroadcast( dstMacAddr ):
         self.intfCounters_.broadcastIn( intf, octets )
      elif Ethernet.isMulticast( dstMacAddr ):
         self.intfCounters_.multicastIn( intf, octets )
      else:
         self.intfCounters_.unicastIn( intf, octets )

   def _countOut( self, intf, octets, dstMacAddr ):
      if Ethernet.isBroadcast( dstMacAddr ):
         self.intfCounters_.broadcastOut( intf, octets )
      elif Ethernet.isMulticast( dstMacAddr ):
         self.intfCounters_.multicastOut( intf, octets )
      else:
         self.intfCounters_.unicastOut( intf, octets )

   def _tapReadHandler( self, data ):
      ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( data )
      etherType = ( ord( data[ 12 ] ) << 8 ) | ord( data[ 13 ] )
      if isPhyControl( srcMacAddr, dstMacAddr, etherType, data[ 14: ] ):
         t8( "got phy control frame!", srcMacAddr, dstMacAddr, etherType )
         self._handlePhyControl( data )
         return

      if self.intfConfig_.enabled:
         t8( "Received frame on %s" % self.intfName_, PacketFormat( data ) )
         bad = False
         size = len( data ) + 4 # Add 4 bytes for CRC
         # We emulate Tahoe for now and round the mtu up to the nearest
         # multiple of 4.
         # intfStatus_.mtu is the L3 MTU, so we add lower layer headers
         # to come up with the L2 MTU.
         mtu = self.intfStatus_.l2Mtu if self.intfStatus_.l2Mtu else 10000
         if (mtu % 4) != 0:
            mtu += (4 - (mtu % 4))
         if size > mtu:
            bad = True

         self._countIn( self.intfName_, len( data ) + 4, dstMacAddr )

         if etherType == ETH_P_PAUSE:
            t5( 'Discarding MAC control frame on %s' % self.intfName_ )
         elif bad:
            t5( 'Discarding bad frame (size %d) on %s' % ( size, self.intfName_ ) )
         else:
            processFrame = self.processFrame_ or self.bridge_.processFrame
            processFrame( data, srcMacAddr, dstMacAddr, self, False )
      else:
         t8( "Discarding frame received on disabled port %s" % self.intfName_ )

   def sendFrame( self, data, srcMacAddr, dstMacAddr, srcPortName,
                  priority, vlanId, vlanAction ):
      if self.intfConfig_.enabled:
         data = applyVlanChanges( data, priority, vlanId, vlanAction )

         if macsecEncryptHandler is not None:
            chgData = macsecEncryptHandler( self, data )
            if chgData:
               data = chgData

         t8b( "Sending frame out %s" % self.intfName_, PacketFormat( data ) )
         writeToTunTapDevice( self.tapDevice_.fileno, data )

         if not dstMacAddr and not srcMacAddr:
            ( _, dstMacAddr ) = macAddrsAsStrings( data )
         self._countOut( self.intfName_, len( data ) + 4, dstMacAddr )
      else:
         t8( "Refusing to transmit on disabled port %s" % self.intfName_ )

   def _trapDeviceReadHandler( self, data ):
      if not self.noTrapDevice:
         finalIntfs = { self.intfName_: None }
         t8( "trapping frame output on %s" % self.intfName_, PacketFormat( data ) )
         for func in packetReplicationHandlers:
            func( self.bridge_, finalIntfs=finalIntfs )
         for intfName in finalIntfs:
            self.bridge_.port[ intfName ].sendFrame(
               data, None, None, None, None, None, PKTEVENT_ACTION_NONE )

   def trapFrame( self, data ):
      if not self.noTrapDevice:
         t8( "trapping frame input on %s" % self.intfName_, PacketFormat( data ) )
         os.write( self.trapDevice_.fileno(), data )

class EbraTestCpuPort( EbraTestPortBase ):
   """Simulates the interface from the switch chips to the CPU/fabric"""

   def __init__( self, intfName, bridge, tapDevice, trapDevice ):
      super( EbraTestCpuPort, self ).__init__( intfName, bridge )
      self.tapDevice_ = tapDevice
      self.trapDevice_ = trapDevice

   def onActive( self ):
      t0( "onActive EbraTestCpuPort" )
      self.tapFile_ = Tac.File( self.tapDevice_, self._tapReadHandler,
                                self.tapDevice_.name, readBufferSize=16000 )
      t8( "trap", self.trapDevice_.name, self.trapDevice_.fileno() )
      self.trapFile_ = Tac.File( self.trapDevice_, self._trapDeviceReadHandler,
                                 self.trapDevice_.name, readBufferSize=16000 )

   def _tapReadHandler( self, data ):
      ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( data )
      self.bridge_.processFrame( data, srcMacAddr, dstMacAddr, self, False )

   def _trapDeviceReadHandler( self, data ):
      t8( "trapping frame 'output' on Cpu", PacketFormat( data ) )
      writeToTunTapDevice( self.tapDevice_.fileno(), data )

   def sendFrame( self, data, srcMacAddr, dstMacAddr, srcPortName,
                  priority, vlanId, vlanAction ):
      t8( "Sending frame out %s" % self.intfName_, PacketFormat( data ) )
      data = applyVlanChanges( data, priority, vlanId, vlanAction )
      writeToTunTapDevice( self.tapDevice_.fileno(), data )

   def trapFrame( self, data ):
      t8( "trapping frame 'input' on Cpu", PacketFormat( data ) )
      os.write( self.trapDevice_.fileno(), data )

class EbraTestBridge( object ):

   def __init__( self, name, entityManager, inNamespace, ethIntfConfigDir,
                 ethIntfStatusDir, ethIntfStatusLocalDir,
                 allEthPhyIntfStatusDir, ebraCounterConfig, ebraCounterStatus ):
      self.name_ = name
      self.entityManager_ = entityManager
      self.shmemEntityManager_ = SharedMem.entityManager( entityManager.sysname(),
                                                          False )
      self.inNamespace_ = inNamespace
      self.ethIntfConfigDir_ = ethIntfConfigDir
      self.ethIntfStatusDir_ = ethIntfStatusDir
      self.ethIntfStatusLocalDir_ = ethIntfStatusLocalDir
      self.allEthPhyIntfStatusDir_ = allEthPhyIntfStatusDir
      self.ebraCounterConfig_ = ebraCounterConfig
      self.ebraCounterStatus_ = ebraCounterStatus
      self.brConfig_ = entityManager.entity( 'bridging/config' )
      self.bridgeMac_ = entityManager.entity( 'bridging/config' ).bridgeMacAddr
      self.brStatus_ = entityManager.entity( 'bridging/status' )
      self.brVlanStatus_ = entityManager.entity( 'bridging/vlan/status' )
      self.brHwCapabilities_ = entityManager.entity( 'bridging/hwcapabilities' )
      self.flush_ = entityManager.entity( 'bridging/flush/request/cli' )
      self.flushReplyDir_ = entityManager.entity( 'bridging/flush/reply/all' )
      self.topoConfig = entityManager.entity( 'bridging/topology/config' )
      self.topoInstStatus = entityManager.entity( 'bridging/topology/inst/etba' )
      self.etbaConfig = entityManager.entity( 'bridging/etba/config' )
      self.port = {}
      self.agingInterval_ = 15 # Check for aged hosts every 15 seconds

      # create the Smash::Bridging::Status entity and initialize it for writing
      writerInfo = Smash.mountInfo(
         'writer',
         collectionInfo=[ ( 'smashFdbStatus', ETBA_SMASH_SIZE ) ],
         cleanup=True
      )

      self.smashBrStatus_ = self.shmemEntityManager_.doMount(
         'bridging/status',
         'Smash::Bridging::Status',
         writerInfo
      )

      # start a shim to populate Bridging::Status from changes in the Smash table
      # for the benefit of plugins that look at brStatus_ directly.  Once these
      # plugins are rewritten to use smash, we can eliminate this reactor and
      # the Bridging::Status altogether.
      self.shim_ = Tac.Type( 'Bridging::ReaderFdbStatusSm' )( self.brStatus_,
                                                              self.smashBrStatus_ )

      self.agingTable = {}

      # Vxlan datapath learning
      self.datapathLearningMode_ = False

      # Per-packet context, may be used by plugins to transmit information to a
      # later stage in the pipeline.
      self.packetContext = {}

      self.hostAgingActivity_ = Tac.ClockNotifiee( self.hostAging )
      self.hostAgingActivity_.timeMin = Tac.now() + self.agingInterval_
      self.bridge_ = weakref.proxy( self )

      self.fdbConfigReactor_ = Tac.collectionChangeReactor(
         self.brConfig_.fdbConfig, FdbConfigReactor, reactorArgs=( self.bridge_, ) )

      self.vlanStatusReactor_ = VlanStatusReactor( self.brVlanStatus_, self.bridge_,
                                                   self.entityManager_ )

      self.brHwCapabilities_.dot1qTunnelSupported = True
      self.brHwCapabilities_.taggedNativeVlanSupported = True
      self.brHwCapabilities_.pvstSupported = True
      self.brHwCapabilities_.backupInterfacesSupported = True
      self.brHwCapabilities_.privateVlansSupported = True
      self.brHwCapabilities_.vxlanSupported = True
      self.brHwCapabilities_.vxlanRoutingSupported = True
      self.brHwCapabilities_.overlayEcmpSupported = True
      self.brHwCapabilities_.igmpSnoopingSupported = True
      # Needed to unblock the CLI configuration guard in Etba for
      # Vxlan interface.
      self.brHwCapabilities_.vxlanUnderlayMcastSupported = True
      self.brHwCapabilities_.ieeeReservedMacForwardAddrSupported = True
      # XXX Should be mounted read, and we need another mount point to
      # specify the hardware capabilities. Bug 15122 opened to track this.
      self.topoInstStatus.maximumTopologiesSupported = 63
      self.topoInstStatus.hardwareConfigured = True
      self.brHwCapabilities_.staticMcastSupported = True
      self.brHwCapabilities_.ndProxySupported = True
      self.brHwCapabilities_.tunnelIntfBumCountersSupported = True
      self.brHwCapabilities_.tunnelIntfUmCountersSupported = False
      self.brHwCapabilities_.tunnelIntfErrorCountersSupported = True
      # Needed to unblock the CLI configuration guard in Etba for the
      # 'no mac address learning' command for VLANs
      self.brHwCapabilities_.vlanMacLearningSupported = True
      # Needed to unblock the CLI configuration guard in Etba for
      # 'no switchport mac address learning' command
      self.brHwCapabilities_.noMacLearningSupported = True

   # Functions to call on active

   def createHostReactor( self ):
      self.hostTableFlushReactor_ = HostTableFlushReactor( self.flush_,
                                                           self.bridge_,
                                                           self.flushReplyDir_ )

   def addReflector( self ):
      self.reflector_ = Tac.newInstance( 'Bridging::StateReflector',
                                         self.topoConfig,
                                         self.topoInstStatus,
                                         self.brConfig_, self.smashBrStatus_,
                                         Tac.activityManager.clock )

   def startRoot( self ):
      # create libState and topologyStatusLibSm to create topoStatus and
      # topoPortStatus
      libState = Tac.root.newEntity( 'Topology::Library::LibState', 'topoLibState' )
      root = Tac.root.newEntity( 'Topology::Library::Root', 'topoLibRoot' )
      root.doStartLibSm( "topoLibSm", self.topoConfig, None,
                         self.topoInstStatus, self.brConfig_, libState,
                         "createStatus", False, False, "LocalInterface" )
      root.topologyStatusLibSm = ( self.topoInstStatus, libState,
                                   self.topoConfig, self.ethIntfStatusDir_ )

   def createCounterConfigReactor( self ):
      t0( "creating EtbaCounterConfigReactor" )
      self.etbaCounterConfigReactor_ = (
         EbraTestBridgeLib.EtbaCounterConfigReactor( self.ebraCounterConfig_,
                                                     self.ebraCounterStatus_,
                                                     ebraRoutingHandlerCounter ) )

   def runPlugins( self ):
      t0( "runPlugins" )
      for f in bridgeInitHandlers:
         if f not in pluginsInitialized:
            t0( "Initializing", f.__name__ )
            f( self.bridge_ )
            pluginsInitialized.append( f )

   def runSwitchoverBridgePlugins( self ):
      t0( "runSwitchoverBridgePlugins" )
      global switchoverBridgeInitHandlers
      for f in switchoverBridgeInitHandlers:
         if f not in pluginsInitialized:
            t0( "Initializing", f.__name__ )
            f( self.bridge_ )
            pluginsInitialized.append( f )

   def name( self ):
      return self.name_

   def em( self ):
      return self.entityManager_

   def sEm( self ):
      return self.shmemEntityManager_

   def inNamespace( self ):
      return self.inNamespace_

   def bridgeMac( self ):
      return self.bridgeMac_
   
   def addPort( self, port ):
      assert port.name() not in self.port
      self.port[ port.name() ] = port

   def delPort( self, intf ):
      if intf in self.port:
         self.port[ intf ].close()
         del self.port[ intf ]

   def forwardIeeeReservedMac( self, mac ):
      forwardAll = self.brConfig_.ieeeReservedForwarding.get(
                        self.brHwCapabilities_.ieeeReservedMacForwardAddrAll )
      return forwardAll or self.brConfig_.ieeeReservedForwarding.get( mac )

   def vlanTagState( self, srcPortName, srcIsFabric, etherType, data, dstMacAddr ):
      """Returns a tuple:
      ( vlanId, priority, vlanTagNeedsUpdating, tagged, routed )
      """
      dot1qIgnoreTag = False
      routed = False

      if not srcIsFabric:
         switchIntfConfig = self.brConfig_.switchIntfConfig.get( srcPortName )

         if switchIntfConfig is None:
            return ( None, None, None, None, None )

         untaggedVlan = switchIntfConfig.nativeVlan

         # If packet comes from dot1qTunnel, ignore 802.1q tag
         if switchIntfConfig.switchportMode == 'dot1qTunnel':
            dot1qIgnoreTag = True
         elif switchIntfConfig.switchportMode == 'routed':
            routed = True
            dot1qIgnoreTag = True

      # frames from the fabric should always be tagged.

      # Calculate the VLAN ID of the frame.
      vlanTagNeedsUpdating = False
      if etherType == ETH_P_8021Q and not dot1qIgnoreTag:
         vlanTag = ( ( ord( data[ 14 ] ) << 8 ) | ord( data[ 15 ] ) )
         vlanId = vlanTag & 0x0fff
         priority = ( vlanTag & 0xe000 ) >> 13
         if vlanId == 0:
            t7( '802.1q priority tag present, with priority', priority )
            assert not srcIsFabric
            vlanId = untaggedVlan
            tagged = False
            vlanTagNeedsUpdating = True
         else:
            t7( '802.1q tag present, with VLAN ID', vlanId, ', priority', priority )
            tagged = True
      else:
         vlanId = None if srcIsFabric else untaggedVlan
         t7( 'No 802.1q tag present, initial vlan id is', vlanId )
         # Drop untagged frames on the fabric interface by setting the
         # vlanId to None.  We don't really expect these, but some
         # linux process sends untagged IPv6 packets to this
         # interface.
         priority = 0
         tagged = False

      return( vlanId, priority, vlanTagNeedsUpdating, tagged, routed )

   def vlanAction( self, srcIntf, dstIntf, vlanId, destIsFabric,
                   etherType, vlanTagNeedsUpdating ):
      vlanAction = PKTEVENT_ACTION_NONE
      dot1qIgnoreTag = False

      if destIsFabric:
         sendTagged = True
      else:
         if srcIntf != 'Cpu':
            srcSwitchIntfConfig = self.brConfig_.switchIntfConfig[ srcIntf ]
            if( srcSwitchIntfConfig.switchportMode == 'dot1qTunnel' ):
               dot1qIgnoreTag = True
         dstSwitchIntfConfig = self.brConfig_.switchIntfConfig[ dstIntf ]

         sendTagged = (dstSwitchIntfConfig.switchportMode == 'trunk') and \
                      (vlanId != dstSwitchIntfConfig.nativeVlan)

      if etherType == ETH_P_8021Q and not dot1qIgnoreTag:
         if not sendTagged:
            vlanAction = PKTEVENT_ACTION_REMOVE
         elif vlanTagNeedsUpdating:
            vlanAction = PKTEVENT_ACTION_REPLACE
      elif sendTagged:
         vlanAction = PKTEVENT_ACTION_ADD
      return vlanAction

   def processFrame( self, data, srcMacAddr, dstMacAddr, srcPort, tracePkt ):
      """Processes a frame received on one of the simulated ports, performing zero
      or more of the following actions:

      -  Queueing the frame for transmission to the CPU. XXX We don't do this yet.
      -  Sending the frame out one or more simulated ports.
      -  Learning the location of the MAC address that sent the frame.
      -  Re-writing the MAC address and/or VLAN tag of the frame.

      If tracePkt is True, this will process the frame as usual, but instead of
      sending the packets out the determined egress interfaces, it will simply return
      that list of egress interfaces without sending anything.
      """

      self.packetContext = {}
      srcPortName = srcPort.name()
      t8( 'Processing frame with srcMacAddr: %s, dstMacAddr: %s, srcPort: %s, '
          'length: %d' % ( srcMacAddr, dstMacAddr, srcPortName, len( data ) ) )

      if preBridgingHandler is not None:
         action = preBridgingHandler( self, data, srcMacAddr, dstMacAddr, srcPort )
         if action == 'trap':
            t7( 'Trapping frame with source: %s, destination: %s, srcPort: %s '
                % ( srcMacAddr, dstMacAddr, srcPortName ) )
            srcPort.trapFrame( data )
            return None
         elif action == 'deny':
            t7( 'Dropping frame with source: %s, destination: %s, srcPort: %s '
                % ( srcMacAddr, dstMacAddr, srcPortName ) )
            return None

      if macsecDecryptHandler is not None:
         chgData = macsecDecryptHandler( self, data, srcPort )
         if chgData:
            data = chgData

      # Invoke Tunnel handler, which may change packet data and/or source port
      t8( "Invoke Tunnel handlers" )
      for preTunnelHdlr in preTunnelHandler:
         ( chgData, chgSrcPort, drop ) = preTunnelHdlr( self, dstMacAddr, data, 
                                                        srcPort )
         if drop:
            t7( '%s consumed the frame' % preTunnelHdlr.__name__ )
            return None
         if chgData:
            t7( '%s changed data' % preTunnelHdlr.__name__ )
            data = chgData
            # Get the src and dst Mac from the modified packet
            ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( data )
         if chgSrcPort:
            t7( '%s changed srcPort to %s' % ( preTunnelHdlr.__name__, 
                                               chgSrcPort.name() ) )
            srcPort = chgSrcPort
      t8( "Tunnel handlers complete, srcPort: %s, length: %d" % ( srcPort.name(), 
                                                                  len( data ) ) )

      # Do normal bridging of packet in ingress vlan
      vlanInfo, data, _ = self.bridgeFrame( data, srcMacAddr, dstMacAddr, srcPort,
                                            False )
      t8( 'Bridging in ingress VLAN complete' )

      # Get the priority and tagging action since every copy of the packet
      # will need to be retagged
      ( vlanId, priority, _vlanTagNeedsUpdating, tagged, _routed ) = vlanInfo

      # This is a list of packets and the vlans they need to go out on
      routedPackets = []
      # Route the packet in these vlans
      for routingHandler in ebraRoutingHandlers:
         modifiedPktVlanTupleList =  routingHandler( self, vlanId, dstMacAddr, data )
         for ( modifiedPkt, vlans ) in modifiedPktVlanTupleList: 
            if modifiedPkt:
               t8( "Routed packet appended by %s in vlans", 
                     ( routingHandler.__name__, vlans ) )
               routedPackets.append( ( modifiedPkt, vlans ) )
               ebraRoutingHandlerCounter[ routingHandler.__name__ ].routed += 1
            else:
               ebraRoutingHandlerCounter[ routingHandler.__name__ ].ignored += 1
      if routedPackets:
         ebraRoutingHandlerCounter[ "overall" ].routed += 1
      else:
         ebraRoutingHandlerCounter[ "overall" ].ignored += 1
      vlanAction = PKTEVENT_ACTION_REPLACE if tagged else PKTEVENT_ACTION_ADD
      t8( 'Routing handlers complete, routed packets: %s' % routedPackets )

      bridgeFrameInfoList = []
      for ( modifiedPkt, vlans ) in routedPackets:
         # Get the src and dst Mac from the modified packet
         ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( modifiedPkt )
         for vlan in vlans:
            # For each packet rewrite the vlan id because it has to be tagged
            # since it comes from the cpu port
            newdata = applyVlanChanges( modifiedPkt, priority, vlan, 
                                        vlanAction )
            # Bridge the packet, and if tracePkt=True, get the egress intfs
            t8( 'Bridging post-routed frame with srcMac: %s, dstMac: %s in VLAN %d' \
                  % ( srcMacAddr, dstMacAddr, vlan ) )
            bridgeFrameInfo = self.bridgeFrame( newdata, srcMacAddr, dstMacAddr,
                                                self.port[ 'Cpu' ],
                                                tracePkt )
            bridgeFrameInfoList.append( bridgeFrameInfo )
      t8( 'processFrame return' )
      return bridgeFrameInfoList

   def bridgeFrame( self, data, srcMacAddr, dstMacAddr, srcPort, tracePkt ):
      dropReasonList = []
      srcPortName = srcPort.name()
      srcIsFabric = srcPortName == 'Cpu'

      etherType = ( ord( data[ 12 ] ) << 8 ) | ord( data[ 13 ] )
      t7( 'Bridging frame with source: %s, destination: %s, EtherType: 0x%04x, '
          'length: %d srcPort: %s' % ( srcMacAddr, dstMacAddr, etherType, 
             len( data ) + 4, srcPortName ) )

      if not Ethernet.isUnicast( srcMacAddr ):
         t5( 'Warning: frame has multicast source address: %s' % srcMacAddr )

      vlanInfo = self.vlanTagState( srcPortName, srcIsFabric, etherType,
                                    data, dstMacAddr )
      ( vlanId, priority, vlanTagNeedsUpdating, tagged, routed ) = vlanInfo

      t8( 'Using VLAN ID', vlanId, ', priority', priority, ' tagged', tagged )
      if vlanId is None:
         return BridgeFrameInfo( vlanInfo, data, None )

      # In pvlans, multiple (secondary) vlans share same (primary vlan's)
      # fid/fdbConfig. So use fid instead of vlanId to get fdbConfig.
      if 0 < vlanId < 4095:
         fid = self.brConfig_.vidToFidMap.get( vlanId )
      else:
         fid = None

      if not fid:
         fid = vlanId

      if 0 < fid < 4095:
         fdbConfig = self.brConfig_.fdbConfig.get( fid )
      else:
         fdbConfig = None

      if 0 < vlanId < 4095:
         vlanConfig = self.brConfig_.vlanConfig.get( vlanId )
      else:
         vlanConfig = None

      if not isIEEELinkConstrained( dstMacAddr ):
         if fdbConfig:
            dropHostSrc = fdbConfig.configuredHost.get( srcMacAddr )

            if dropHostSrc and dropHostSrc.intf == '':
               t0( "dropping due to drop src mac" )
               dropReasonList.append( "drop src mac entry" )

            dropHostDst = fdbConfig.configuredHost.get( dstMacAddr )
            if dropHostDst and dropHostDst.intf == '':
               t0( "dropping due to drop dst mac" )
               dropReasonList.append( "drop dst mac entry" )
               return BridgeFrameInfo( vlanInfo, data, None )

         if not vlanConfig:
            t4( 'Dropping packet on bad vlan %s' % vlanId )
            dropReasonList.append( 'badVlan' )
         elif srcPortName != 'Cpu':
            if srcPortName not in vlanConfig.intf:
               t4( 'bad ingress port %s on vlan %d' % ( srcPort, vlanId ) )
               dropReasonList.append( 'ingress vlan violation' )
               return BridgeFrameInfo( vlanInfo, data, None )
            elif vlanConfig.intf[ srcPortName ].allowedTrafficDirection == 'egress':
               t4( 'ingress traffic not allowed on port %s on vlan %d' % \
                      ( srcPort, vlanId ) )
               dropReasonList.append( 'allowedTrafficDirection vlan violation' )
               return BridgeFrameInfo( vlanInfo, data, None )

      if not srcIsFabric:
         stpState = self.lookupStpState( vlanId, srcPortName )

         forwardingModel = 'unknown'
         if srcPortName in self.ethIntfStatusDir_:
            forwardingModel = self.ethIntfStatusDir_[ srcPortName ].forwardingModel
         if forwardingModel == "intfForwardingModelUnauthorized" and \
            dstMacAddr != IEEE_8021X_ADDR:
            t5( 'Got a non 802.1x packet when the intf forwarding '
                'model is unauthorized. Dropping packet. ' )
            dropReasonList.append( 'stpdiscarding' )
         if( (stpState == 'discarding') and \
             not isIEEELinkConstrained( dstMacAddr ) ):
            # packet will be discarded for vlanId == 0 signifying
            # an untagged non-control packet received on a trunk port with
            # native vlan tagging enabled
            t5( 'Dropping packet, input forwarding state discarding for ',
                'address %s on port %s' % ( srcMacAddr, srcPortName ) )
            dropReasonList.append( 'stpdiscarding' )

         if( ( stpState == 'learning' or stpState == 'forwarding' ) and
               not isIEEELinkConstrained( dstMacAddr ) ):
            if( ( not self.datapathLearningMode_ ) and
                srcPortName.startswith( 'Vxlan' ) ):
               t9( 'Not learning on Vxlan : vlan %d, mac %s port %s' %
                   ( vlanId, srcMacAddr, srcPortName ) )
            elif( srcPortName.startswith( 'MplsTrunk' ) ):
               t9( 'Not learning on MplsTrunk1 : vlan %d, mac %s port %s' %
                   ( vlanId, srcMacAddr, srcPortName ) )
            else:
               # Learn the location of the source MAC address.
               entryType = 'learnedRemoteMac' if srcPortName.startswith( 'Vxlan' ) \
                     else 'learnedDynamicMac'
               self.learnAddr( vlanId, srcMacAddr, srcPortName, entryType=entryType )

            if( stpState == 'learning' ):
               t5( 'Dropping packet, input forwarding state learning for ',
                   'address %s on port %s' % ( srcMacAddr, srcPortName ) )
               dropReasonList.append( 'stplearning' )
      else:
         if vlanId is None:
            t7( 'Source is fabric/cpu' )
            # drop untagged packets
            return BridgeFrameInfo( vlanInfo, data, None )

      # Check for packets that are to be trapped to the CPU.
      trap = False
      copyToCpu = False

      bestMatch = None
      maxTrapPriority = -1
      # Allow for extensions to force trap or copy to cpu actions here.
      for trapPriority, h in trapLookupHandlers:
         ( match, chgTrap, chgCopyToCpu ) = h( self, routed, vlanId, dstMacAddr,
                                               data, srcPortName )
         if match:
            if trapPriority > maxTrapPriority:
               # Use the trap handler override with the highest priority.
               maxTrapPriority = trapPriority
               bestMatch = match
               trap = chgTrap
               copyToCpu = chgCopyToCpu
            elif trapPriority == maxTrapPriority:
               dupMatches = [ bestMatch, match ]
               assert False, "Multiple trap handlers with the same priority " \
                     "not allowed: %s" % dupMatches

      if not bestMatch:
         if routed:
            trap = True
         elif isIEEELinkConstrained( dstMacAddr ):
            if dstMacAddr == IEEE_PAUSE_ADDR:
               etherType = ( ord( data[ 12 ] ) << 8 ) | ord( data[ 13 ] )
               t7( 'dst mac is pause addr %x ' % etherType )
               if etherType == ETH_P_PAUSE or \
                     not self.forwardIeeeReservedMac( dstMacAddr ):
                  t5( 'Dropping frame sent to PAUSE address from address %s '
                      'on port %s' % ( srcMacAddr, srcPortName ) )
                  dropReasonList.append( 'destaddrpause' )
            elif self.forwardIeeeReservedMac( dstMacAddr ):
               trap = False
            elif dstMacAddr == IEEE_BPDU_ADDR:
               if self.topoConfig.trapStpBpdus:
                  t7( 'dst mac is ieee bpdu addr' )
                  trap = True
            elif dstMacAddr == PVST_BPDU_ADDR:
               if self.topoConfig.trapPvstBpdus:
                  t7( 'dst mac is pvst bpdu addr' )
                  trap = True
            else:
               trap = True
         else:
            # Fall through, and use default trap and copyToCpu values
            pass

      finalIntfs = {}
      destIsFabric = dstMacAddr == self.bridgeMac_

      # Handle for packet that needs to be terminated
      if destIsFabric:
         for func in tunnelTerminationHandlers:
            ( chgData, chgDstMac ) = func( self, routed, vlanId, dstMacAddr, data )
            if chgData:
               data = chgData
               dstMacAddr = chgDstMac

      if trap == TRAP_ANYWAY or ( not dropReasonList and trap ):
         t8( 'Trapping frame to CPU' )
         dataToSend = data
         for func in rewritePacketOnTheWayToTheCpuHandlers:
            ( chgData, chgDstMac ) = func( self, routed, vlanId,
                                           dstMacAddr, data, srcPortName )
            if chgData:
               t7( 'data changed by %s' % func.__name__ )
               dataToSend = chgData
               break
         t8( 'Rewrite packet on the way to the CPU handlers complete' )
         srcPort.trapFrame( dataToSend )
         dropReasonList.append( 'trap' )

      if not dropReasonList:
         if copyToCpu:
            t7( 'Copying frame to CPU' )
            srcPort.trapFrame( data )

         outputIntfs = []
         if vlanId:
            vlanConfig = self.brConfig_.vlanConfig.get( vlanId )
         else:
            vlanConfig = None
         # Look up the location of the destination MAC address.
         if destIsFabric:
            # Make sure a vlan interface was created.
            if not vlanConfig:
               t4( 'Dropping packet on bad vlan %s' % vlanId )
               dropReasonList.append( 'badVlan' )
            else:
               if 'Cpu' in vlanConfig.intf:
                  # Great, send the frame to the kernel.
                  outputIntfs = [ 'Cpu', ]
               else:
                  t4( 'No vlan interface created' )
         else:
            destMatch = False
            for h in destLookupHandlers:
               (destMatch, outputIntfs) = h( self, vlanId, dstMacAddr, data )
               if destMatch:
                  t8( 'destLookupHandler', h, 'returned outputIntfs', outputIntfs )
                  break
            if not destMatch and vlanId:
               (destMatch, outputIntfs) = self.destLookup( vlanId, dstMacAddr )
               t8( 'destLookup returned', destMatch, outputIntfs )
            if not destMatch:
               t8( 'Frame has unknown destination address' )
               if not vlanConfig:
                  t4( 'Dropping packet on bad vlan %s' % vlanId )
                  dropReasonList.append( 'badVlan' )
                  outputIntfs = []
               else:
                  # In pvlans, the ports in isolated vlans are not egress allowed in
                  # their own vlans, so exclude those ports
                  outputIntfs = \
                    [i for i in vlanConfig.intf if ((i in self.port) and \
                    ( vlanConfig.intf[ i ].allowedTrafficDirection != 'ingress' )) ]
                  t8( 'outputIntfs is', outputIntfs )

            # We only want to include the Cpu in the floodset for
            # 1) broadcasts, 2) local multicasts, 3) other packets identified by
            # floodsetIncludesCpuHandlers or 4) there is a match for 'Cpu' in
            # outputIntfs[]
            if srcIsFabric:
               # If the srcPort is Cpu, then there is no need to figure out if
               # Cpu needs to be included in the floodset.
               cpuBelongs = False
            elif Ethernet.isIPv6Multicast( dstMacAddr ):
               cpuBelongs = True
               t8( "IPv6 Multicast Frame." )
            elif Ethernet.isBroadcast( dstMacAddr ) or (
               dstMacAddr >= IP_MCAST_BASE and
               dstMacAddr <= IP_MCAST_LOCAL_END ) or (
               destMatch and 'Cpu' in outputIntfs):
               cpuBelongs = True
            else:
               cpuBelongs = None
               if vlanId != 0:
                  for h in floodsetIncludesCpuHandlers:
                     cpuBelongs = h( self, vlanId, dstMacAddr, data )
                     if cpuBelongs is not None:
                        break
            # If it's there and doesn't belong, remove it.  If it's
            # not there (e.g., IgmpSnooping's destLookup handler
            # provided a constrained set of output interfaces),
            # add it.
            if 'Cpu' in outputIntfs:
               if not cpuBelongs:
                  outputIntfs.remove( 'Cpu' )
                  t7( 'Removing cpu from vlan', vlanId, 'dst', dstMacAddr )
            elif cpuBelongs:
               if vlanConfig is not None and 'Cpu' in vlanConfig.intf:
                  outputIntfs.append( 'Cpu' )
                  t7( 'Adding cpu for vlan', vlanId, 'dst', dstMacAddr )

         # Filter output ports on spanning tree state and remove the source port
         # if it is there. Also, do not send frame out non-local interfaces.
         for intf in outputIntfs:
            if intf not in self.port:
               continue
            if intf == 'Cpu':
               if intf != srcPortName:
                  finalIntfs.setdefault( intf, None )
            else:
               state = self.lookupStpState( vlanId, intf )
               if( intf in self.ethIntfStatusDir_ ):
                  forwardingModel = self.ethIntfStatusDir_[ intf ].forwardingModel
                  t8( "interface", intf, "has stp state", state,
                      "and forwarding model", forwardingModel )
                  # Vxlan allows tunnel to tunnel forwarding via Vxlan interface when
                  # vtep-to-vtep bridging is enabled
                  if( ( (state == 'forwarding') and
                      ( forwardingModel in ( 'intfForwardingModelBridged',
                                            'intfForwardingModelRouted' ) ) ) and 
                      ( ( intf != srcPortName ) or
                      ( ( intf == srcPortName ) and ( 'Vxlan' in srcPortName ) and
                      self.port[ intf ].checkVtepToVtepBridging() ) ) ):
                     t8( "intf ", intf, "srcPortName ", srcPortName )
                     finalIntfs.setdefault( intf, None )

      if not trap and not srcIsFabric and not dropReasonList:
         for sflowHandler in sflowHandlers:
            sflowHandler( self, data, srcPort, finalIntfs )

      for func in packetReplicationHandlers:
         func( self, finalIntfs, srcPort, dropReasonList, vlanId, dstMacAddr,
               data=data )

      if tracePkt:
         t4( "PacktTracer: Return egress intfs and don't send the frames" )

      if dropReasonList:
         if tracePkt:
            t4( "PacketTracer: No egress intfs, dropReasonList:", dropReasonList )
         return BridgeFrameInfo( vlanInfo, data, None )

      t8( 'Host table or VID table indicates ports ', finalIntfs )
      # List of the interfaces that we actually tried to send the frame out of. For
      # the packet tracer, we want to get to most accurate representation of this
      # list of egress interfaces.
      egressIntfs = []
      for intf in finalIntfs.keys():
         sendToMacAddr = dstMacAddr
         dataToSend = data
         sendSrcMacAddr = srcMacAddr
         skipThisIntf = False
         if finalIntfs[ intf ] != None:
            vlanAction = finalIntfs[ intf ]
         else:
            if vlanId != fid:
               if intf == 'Cpu':
                  vlanId = fid
                  vlanTagNeedsUpdating = True
            vlanAction = self.vlanAction( srcPortName, intf, vlanId, intf == 'Cpu',
                                          etherType, vlanTagNeedsUpdating )
         if intf == 'Cpu':
            for func in rewritePacketOnTheWayToTheCpuHandlers:
               ( chgData, chgDstMac ) = func( self, True, vlanId, dstMacAddr, data,
                                              srcPortName )
               if chgData or chgDstMac:
                  sendToMacAddr = chgDstMac
                  dataToSend = chgData
                  break
         else:
            for func in postBridgingHandlers:
               ( action, chgData, chgSrcMac, chgDstMac ) = func( self, srcMacAddr,
                                                                 dstMacAddr, vlanId,
                                                                 data, srcPortName,
                                                                 intf, finalIntfs )
               if action == 'deny':
                  skipThisIntf = True
                  t4( 'postBridgingHandler denied bridging the frame '
                      'on egress interface ', intf )
                  break # from inner for
               if chgData or chgDstMac or chgSrcMac :
                  dataToSend = chgData
                  sendToMacAddr = chgDstMac
                  sendSrcMacAddr = chgSrcMac
                  break

         if skipThisIntf:
            continue
         # If we're tracing the packet, we don't want to actually send the frame
         if tracePkt:
            egressIntfs.append( intf )
         else:
            self.port[ intf ].sendFrame( dataToSend, sendSrcMacAddr, sendToMacAddr,
                                         srcPortName, priority, vlanId, vlanAction )
      return BridgeFrameInfo( vlanInfo, data, egressIntfs )

   def lookupStpState( self, vlanId, portName ):
      '''Lookup the (vlanId, portName) pair in the Topology structures to find
      the forwarding state to use for packets on that port in that vlan.'''

      if not ( 0 < vlanId < 4095 ):
         t8( 'invalid vlanId' )
         return 'discarding'

      topo = self.topoConfig.vidToTopoMap.get( vlanId, None )
      if not topo:
         t8( 'no topo for vlan', vlanId )
         return 'discarding'

      topoPort = topo.topoPortConfig.get( portName, None )
      if not topoPort:
         t8( 'no topoPort for port', portName, 'in', topo.name )
         return 'discarding'

      return topoPort.state

   def learnAddr( self, vlanId, macAddr, portName, entryType='learnedDynamicMac',
                  moves=None ):
      t9( 'Maybe learn vlan %d, mac %s, port %s entryType %s moves %s' %
          ( vlanId, macAddr, portName, entryType, moves ) )

      # Do not learn MAC if learning is disabled in this VLAN or on this port
      vlanConfig = self.brConfig_.vlanConfig.get( vlanId )
      try:
         switchIntfConfig = self.brConfig_.switchIntfConfig.get( portName )
      except IndexError:
         switchIntfConfig = None
      if ( vlanConfig and not vlanConfig.macLearning ) or ( switchIntfConfig and
           not switchIntfConfig.macLearningEnabled ):
         return

      learnedMacs = [ 'learnedDynamicMac' , 'learnedRemoteMac' ]
      configuredRemoteMacs = [ 'configuredRemoteMac' , 'evpnConfiguredRemoteMac' ]
      r = invokeLearningHandlers( vlanId, macAddr, portName, entryType )
      if( 'dontLearn' in r ):
         return

      # Create these on demand rather than via reactors.
      fid = self.brConfig_.vidToFidMap.get( vlanId )
      if not fid:
         fid = vlanId

      if self.brConfig_.fdbConfig.has_key( fid ):
         fdbConfig = self.brConfig_.fdbConfig[ fid ]
         staticTable = fdbConfig.configuredHost
         staticHost = staticTable.has_key( macAddr )
         if staticHost:
            if entryType == 'configuredStaticMac':
               t9( "Adding static entry" )
               staticHostEntry = staticTable[ macAddr ]
               if moves is None:
                  moves = 1
               host = SmashFdbStatus( HostKey( fid, macAddr ) )
               host.intf = staticHostEntry.intf
               host.moves = moves
               host.lastMoveTime = Tac.now()
               host.entryType = "configuredStaticMac"
               self.smashBrStatus_.smashFdbStatus.addMember( host )
               t9( "Entry added in %s" % fid )
            else:
               t9( "Static entry exists, ignoring learn event" )
            return

      if not Ethernet.isUnicast( macAddr ):
         # We don't learn multicast addresses
         t4( 'not learning multicast host vlan %d mac %s' % (vlanId, macAddr) )
         return
      if macAddr == '00:00:00:00:00:00':
         # We don't learn the zero address either
         t4( 'not learning zero host vlan %d mac %s' % (vlanId, macAddr) )
         return

      if( entryType is None ):
         entryType = "learnedDynamicMac"
      key = HostKey( fid, macAddr )
      if self.smashBrStatus_.smashFdbStatus.has_key( key ):
         host = self.smashBrStatus_.smashFdbStatus[ key ]
         if host.intf == portName and host.entryType.startswith( "peer" ):
            t4( 'Keeping existing learnedHost with type', host.entryType )
            return
         if host.entryType == 'configuredRemoteMac' or \
            host.entryType == 'configuredStaticMac':
            t4( 'Keeping existing learnedHost with type', host.entryType )
            return
         if entryType in learnedMacs and \
            ( host.entryType == 'evpnIntfStaticMac' or \
              host.entryType == 'evpnConfiguredRemoteMac' ):
            t4( 'Keeping existing learnedHost with type', host.entryType )
            return
         # configured MACs have high priority
         if ( ( entryType not in configuredRemoteMacs ) and ( host.intf == portName )
              and ( host.entryType == 'receivedRemoteMac' or
                host.entryType == 'evpnDynamicRemoteMac' ) ):
            t4( 'Keeping existing learnedHost with type', host.entryType )
            return
         if host.intf == portName and entryType in learnedMacs and \
            host.entryType in learnedMacs:
            self.agingTable[ (fid, macAddr) ] = ( Tac.now(), portName )
            return
         t4( 'host entryType %s/%s move to vlanId %d, mac %s, port %s' %
             ( host.entryType, entryType, vlanId, macAddr, portName ) )
         newHost = SmashFdbStatus( key )
         newHost.intf = portName
         if moves is None:
            newHost.moves = ( host.moves + 1 ) & 0xffffffff
         else:
            newHost.moves = moves
         newHost.lastMoveTime = Tac.now()
         newHost.entryType = entryType
         self.smashBrStatus_.smashFdbStatus.addMember( newHost )
      else:
         t4( 'host learn vlanId %d, mac %s, port %s' %
             ( vlanId, macAddr, portName ) )
         if moves is None:
            moves = 1
         host = SmashFdbStatus( HostKey( fid, macAddr ) )
         host.intf = portName
         host.moves = moves
         host.lastMoveTime = Tac.now()
         host.entryType = entryType
         self.smashBrStatus_.smashFdbStatus.addMember( host )

      # Only age learned Mac entries
      if entryType in learnedMacs:
         self.agingTable[ (fid, macAddr) ] = ( Tac.now(), portName )

   def destLookup( self, vlanId, macAddr ):
      if macAddr == '00:00:00:00:00:00':
         # Don't forward packets to address zero.
         return (True, [])
      vlanConfig = self.brConfig_.vlanConfig.get( vlanId, None )
      if not vlanConfig:
         return (False, [])

      fid = self.brConfig_.vidToFidMap.get( vlanId )
      if not fid:
         fid = vlanId

      key = HostKey( fid, macAddr )
      if self.smashBrStatus_.smashFdbStatus.has_key( key ):
         host = self.smashBrStatus_.smashFdbStatus[ key ]
         t4( 'destLookup found dynamic host vlan %d mac %s' % (vlanId, macAddr) )
         intfs = [ host.intf ]
         return (True, intfs)

      return (False, [])

   def hostAging( self ):
      now = Tac.now()
      self.hostAgingActivity_.timeMin = now + self.agingInterval_

      if self.brConfig_.hostAgingTime == 0:
         # Aging time set to 0 means to disable aging.
         return

      c = 0
      for ((vlanId, macAddr), (lastSeenTime, _portName)) \
              in self.agingTable.items():
         if( (now - self.brConfig_.hostAgingTime) > lastSeenTime ):
            t4( 'now', now, 'hostAgingTime', self.brConfig_.hostAgingTime,
                'lastSeenTime', lastSeenTime )
            c += 1
            self.deleteMacAddressEntry( vlanId, macAddr )

      if c > 0:
         t4( 'Aged out ', c, '  MAC address table entries' )

   def macTableFlush( self ):
      for ((vlanId, macAddr), (_lastSeenTime, portName)) \
              in self.agingTable.items():
         t9( 'Flushing MAT entries on port', portName )
         self.deleteMacAddressEntry( vlanId, macAddr )

   def macTableFlushPort( self, intfName ):
      t9( 'Flushing MAT entries on port', intfName )
      for ((vlanId, macAddr), (_lastSeenTime, portName)) \
              in self.agingTable.items():
         if portName == intfName:
            self.deleteMacAddressEntry( vlanId, macAddr )

   def deleteMacAddressEntry( self, vlanId, macAddr, entryType=None ):
      t9( 'Deleting MAC entry at vlan ', vlanId, ' mac ', macAddr )
      fid = self.brConfig_.vidToFidMap.get( vlanId )
      if not fid:
         fid = vlanId

      key = HostKey( fid, macAddr )
      if self.smashBrStatus_.smashFdbStatus.has_key( key ):
         # It is possible that the address was already deleted by a flushing
         # operation.
         hostEntry = self.smashBrStatus_.smashFdbStatus[ key ]
         if hostEntry.entryType == 'configuredStaticMac' and \
            entryType != 'configuredStaticMac':
            t4( "Do not Flush static entry" )
         else:
            del self.smashBrStatus_.smashFdbStatus[ key ]
      if self.agingTable.has_key( (fid, macAddr) ):
         del self.agingTable[ (fid, macAddr) ]
      invokePostAgingHandlers( self, vlanId, macAddr )

