#!/usr/bin/env python
# Copyright (c) 2008-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Tracing, Ethernet, struct, re
from EbraTestBridgePort import EbraTestPort
import Arnet.Device
from Arnet.PktParserTestLib import parsePktStr, findHeader
from Arnet.VxlanPktTestLib import VxlanHdrSize
from BridgingHostEntryType import isEntryTypeController
from EbraTestBridgeLib import (
   makeTap,
   macAddrsAsStrings,
   applyVlanChanges,
   PKTEVENT_ACTION_ADD,
)
from if_ether_arista import ETH_P_ARISTA_VXLAN_ENCAP, ETH_P_ARISTA_VXLAN_ARP, \
      VXLAN_DOT1QTUNNEL_FLAG, VXLAN_CPUPORT_FLAG, VXLAN_MLAG_SRCPORT_FLAG, \
      VXLAN_SRC_VTEP_EXT_FLAG
import VxlanLib
import SharedMem
import Smash
import MlagMountHelper
from IpUtils import IpAddress
from Toggles import EtbaDutToggleLib
import Toggles.VxlanToggleLib
import socket, sys

handle = Tracing.Handle( 'EbraVxlan' )
t0 = handle.trace0    #
t2 = handle.trace2    #
t4 = handle.trace4    #
t5 = handle.trace5    #
t6 = handle.trace6    # less frequent tracing/exception cases
t7 = handle.trace7    # per-packet exception cases
t8 = handle.trace8    # per-packet tracing

# Global handle on Vxlan agent devices
vxlanAgentDevs = None

# Global config directory
vxlanConfigDir = None

# Global H/W status directory
vxlanHwStatusDir = None

# Global Bridging and Vxlan H/W capabilities
bridgingHwCapabilities = None

# Global varMac status directory
vrMacStatus = None

# Global Vxlan Learned Hosts (in HwStatusDir)
vxlanLearnedHost = None
multiVtepLearnedHost = None

# Global view of floodset published by L2Rib
l2RibOutput = None
l2RibLoadBalance = None
l2RibDest = None
l2RibFloodSet = None
vxlanStaticSource = 0x00000004

# Global mlagStatus
mlagStatusMount = None

# Global EVPN status for ethernet segment handling
evpnStatus = None

# global flag
mountCompleted = False

BridgingEntryType = Tac.Type( "Bridging::EntryType" )
VxlanEntryType = VxlanLib.EntryType()
BumReplicationMode = Tac.Type( "Vxlan::BumReplicationMode" )
VtepType = Tac.Type( "L2Rib::VtepType" )
VxlanTunnelType = Tac.Type( "Vxlan::VxlanTunnelType" )
destIdNotInSmash = Tac.Value( 'L2Rib::Constants' ).objIdNotInSmash

# global DMA driver seq No
encapSeqNo = 0

# Global L2Rib host input of type "vxlan-dynamic"
vxlanDynamicL2RibHostInput = None

# Global remote VTEP config
remoteVtepConfigDir = None

# Global vtep HW Status
vtepHwStatus = None

#Global decap config
decapConfig = None
defaultVrfId  = Tac.newInstance( 'Vrf::VrfIdMap::VrfId' ).defaultVrf
UnderlayAddrType =  Tac.Type( 'Routing::Multicast::UnderlayAddr' )
UnderlayRouteType = Tac.Type( "Routing::Multicast::UnderlayRoute" )
uSrcV4 = UnderlayAddrType( "" )
uSrcV4.af = "ipv4"

class VxlanAgentDevs( object ):
   """ All interactions with Vxlan agent defined here """
   def __init__( self, bridge ):
      # XXX: the kernel won't be able to find these devices if not properly named
      self.vxlanTapFile_ = None
      self.txrawTapFile_ = None
      dev = 'vxlan' if bridge.inNamespace() else '%s-vxlan' % bridge.name()
      self.vxlanTapDevice_ = makeTap( dev, hw=bridge.bridgeMac() )
      dev = 'txraw' if bridge.inNamespace() else '%s-txraw' % bridge.name()
      self.txrawTapDevice_ = makeTap( dev, hw=bridge.bridgeMac(), mtu=10000 )

      t0( 'created Vxlan Tap Devices %s %s ' % ( self.vxlanTapDevice_.name,
               self.txrawTapDevice_.name ) )
      self.bridge_ = bridge

   def onActive( self ):
      if not self.vxlanTapFile_:
         self.vxlanTapFile_ = Tac.File( self.vxlanTapDevice_.fileno(),
                                        self._vxlanTapReadHandler,
                                        self.vxlanTapDevice_.name,
                                        readBufferSize=16000 )
      if not self.txrawTapFile_:
         self.txrawTapFile_ = Tac.File( self.txrawTapDevice_.fileno(),
                                        self._txrawTapReadHandler,
                                        self.txrawTapDevice_.name,
                                        readBufferSize=16000 )

   def __del__( self ):
      self.vxlanTapFile_ = None
      self.txrawTapFile_ = None
      del self.vxlanTapDevice_
      del self.txrawTapDevice_

   def _vxlanTapReadHandler( self, data ):
      """ Process frames from Vxlan Agent """
      # No packets expected on this tap, so if we receive any we drop
      # We still need to consume the packet to prevent memory leak
      t0( 'received unexpected frame from vxlan netdevice' ) 

   def _txrawTapReadHandler( self, data ):
      """ Process decap frames from Vxlan Agent """
      if self.bridge_ is None:
         t0( 'txraw dropping frame because there is no bridge configured' )
         return

      if self.bridge_.vxlanPort_ is None:
         t2( 'txraw dropping frame because vxlan port does not exist' )
         return
      ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( data )
      t0( 'txraw received frame for smac(%s)/dmac(%s)' % \
            ( srcMacAddr, dstMacAddr ) )
      t7( 'Raw frame is', data ) 
      self.bridge_.processFrame( data, srcMacAddr, dstMacAddr,
                                 self.bridge_.vxlanPort_, False )

class VxlanStatusFloodListReactor( Tac.Notifiee ):
   notifierTypeName = 'Vxlan::VxlanStatus'
   def __init__( self, vxlanStatus, vtiStatusReactor, bridge ):
      t2( 'VxlanStatusFloodListReactor::__init__()' )
      self.vxlanStatus_ = vxlanStatus
      self.vtiStatusReactor_ = vtiStatusReactor
      self.bridge_ = bridge
      self.vlanToLearnRestrictReactor_ = {}
      Tac.Notifiee.__init__( self, vxlanStatus )
      self.handleVlanToLearnRestrict( vlanId=None )

   @Tac.handler( 'vlanToLearnRestrict' )
   def handleVlanToLearnRestrict( self, vlanId ):
      t2( 'handleVlanToLearnRestrict %s' % ( vlanId ) )

      if vlanId is None:
         for vlan in self.vxlanStatus_.vlanToLearnRestrict:

            t2( 'handleVlanToLearnRestrict create a new reactor for vlan=%d' % \
               ( vlan ) )
            Tac.handleCollectionChange( VlanToLearnRestrictReactor, vlan,
               self.vlanToLearnRestrictReactor_,
               self.notifier_.vlanToLearnRestrict,
               reactorArgs=( vxlanHwStatusDir.learnStatus, self.vxlanStatus_,
                             self.bridge_ ),
               reactorTakesKeyArg=True,
               reactorFilter=None )
         return

      # If the key doesn't exist, handleCollectionChange will delete the reactor
      Tac.handleCollectionChange( VlanToLearnRestrictReactor, vlanId,
                self.vlanToLearnRestrictReactor_,
                self.notifier_.vlanToLearnRestrict,
                reactorArgs=( vxlanHwStatusDir.learnStatus, self.vxlanStatus_,
                              self.bridge_ ),
                reactorTakesKeyArg=True,
                reactorFilter=None )

      t4( 'active vlist reactors exist for vlans:', \
            self.vlanToLearnRestrictReactor_ )

   def __del__( self ):
      t2( 'VxlanStatusFloodListReactor cleanup' )

class VxlanStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Vxlan::VxlanStatus'
   def __init__( self, vxlanStatus, vtiStatusReactor ):
      t2( 'VxlanStatusReactor::__init__()' )
      self.vxlanStatus_ = vxlanStatus
      self.vtiStatusReactor_ = vtiStatusReactor
      self.bumStatus_ = vxlanHwStatusDir.bumStatus
      self.vxlanStatusFloodListReactor_ = None 
      Tac.Notifiee.__init__( self, vxlanStatus )

   @Tac.handler( 'vlanToLearnRestrict' )
   def handleVlanToLearnRestrict( self, vlanId ):
      t2( 'VxlanStatusReactor::handleVlanToLearnRestrict vlan=', vlanId )
      if ( hasattr( self.vtiStatusReactor_, 'vxlanStatusDirReactor_' ) and \
           self.vtiStatusReactor_.vxlanStatusDirReactor_ and
           'Vxlan1' in self.vtiStatusReactor_.vxlanStatusDirReactor_.
           vxlanStatusReactor_ ): 
         vxlanStatusReactor = self.vtiStatusReactor_.vxlanStatusDirReactor_. \
             vxlanStatusReactor_['Vxlan1']
      else:
         vxlanStatusReactor = None

      # Make sure we always track the flood and learnRestrict lists
      if vxlanStatusReactor and not vxlanStatusReactor.vxlanStatusFloodListReactor_:
         t2( "Launch vxlanStatusFloodListReactor")
         vxlanStatusReactor.vxlanStatusFloodListReactor_ = \
             VxlanStatusFloodListReactor( self.vxlanStatus_, self.vtiStatusReactor_,
                                          self.vtiStatusReactor_.bridge_ )

   @Tac.handler( 'vlanToClearCounterRequestTime' )
   def vlanToClearCounterRequestTime( self, vlanId ):
      t2( 'vlanToClearCounterRequestTime %s' % ( vlanId ) )

      learnList = vxlanHwStatusDir.learnStatus.learnStatusList
      if vlanId is None:
         for vlan in self.vxlanStatus_.vlanToClearCounterRequestTime:
            t4( 'calling vlanToClearCounterRequestTime %d' % ( vlan ) )
            self.vlanToClearCounterRequestTime( vlan )
         return

      if self.vxlanStatus_.vlanToClearCounterRequestTime.has_key( vlanId ):
         time = self.vxlanStatus_.vlanToClearCounterRequestTime[ vlanId ]
         # Did we get a new clearCounterRequest?
         if ( learnList.has_key( vlanId ) and
              time != learnList[ vlanId ].lastClearTime ):
            t4( 'handleVlanToClearCounterRequestTime: clear counters vlan %d' % 
                ( vlanId ) )
            for p in learnList[ vlanId ].numMatches:
               learnList[ vlanId ].numMatches[ p ] = 0
            learnList[ vlanId ].numMatchAny = 0
            learnList[ vlanId ].numMatchList = 0
            learnList[ vlanId ].numMatchFloodList = 0
            learnList[ vlanId ].numRejectList = 0
            learnList[ vlanId ].numRejectFloodList = 0
            learnList[ vlanId ].lastClearTime = time

class VxlanStatusDirReactor( Tac.Notifiee ):
   notifierTypeName = 'Vxlan::VxlanStatusDir'
   def __init__( self, bridge, vtiStatus, vxlanStatusDir, vtiStatusReactor ):
      self.bridge_ = bridge
      self.vtiStatus_ = vtiStatus
      self.vxlanStatusDir_ = vxlanStatusDir
      self.vtiStatusReactor_ = vtiStatusReactor
      self.vxlanStatus_ = {}
      self.vxlanStatusReactor_ = {}
      Tac.Notifiee.__init__( self, vxlanStatusDir )

   @Tac.handler( 'vxlanStatus' )
   def handleVxlanStatus( self, intfName ):
      t2( 'VxlanStatusDirReactor::handleVxlanStatus::%s' % intfName )
      vxlanStatusDir = self.bridge_.em().entity( 'vxlan/status' )
      if intfName in vxlanStatusDir.vxlanStatus:
         self.vxlanStatus_[ intfName ] = vxlanStatusDir.vxlanStatus[ intfName ]
         
         self.vxlanStatusReactor_[ intfName ] = \
               VxlanStatusReactor( self.vxlanStatus_[ intfName ],
                                   self.vtiStatusReactor_ )
         self.vxlanStatusReactor_[ intfName ].handleVlanToLearnRestrict(
             vlanId=None )

   def __del__( self ):
      t2( 'VxlanStatusDirReactor::__del__()' )
      for intf in self.vxlanStatus_:
         t2( 'VxlanStatusDirReactor::__del__() %s' % intf )
         self.vxlanStatus_[ intf ] = None

class VtiConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Vxlan::VtiConfig'

   def __init__( self, bridge, *args, **kwargs ):
      Tac.Notifiee.__init__( self, *args, **kwargs )
      self.bridge_ = bridge

   @Tac.handler( "udpPort" )
   def handlePortChange( self ):
      updateBpfFilterString( self.bridge_ )

class VtiStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Vxlan::VtiStatus'

   def __init__( self, vtiStatus, bridge, vxlanStatusDir ):
      t2( 'VtiStatusReactor::__init__()' )
      assert bridge
      self.vtiStatus_ = vtiStatus
      self.bridge_ = bridge
      self.bridge_.controllerClientMode_ = False
      self.bridge_.datapathLearningMode_ = False
      self.vxlanStatusDir_ = vxlanStatusDir

      # Instantiate reactor classes
      self.vxlanStatusDirReactor_ = \
            VxlanStatusDirReactor( bridge, vtiStatus, vxlanStatusDir, self )
      self.vxlanStatusDirReactor_.handleVxlanStatus( vtiStatus.intfId )

      if self.vtiStatus_:
         self.bridge_.controllerClientMode_ = self.vtiStatus_.controllerClientMode
         self.bridge_.datapathLearningMode_ = self.vtiStatus_.datapathLearning
         self.bridge_.controllerControlPlane_ = \
               self.vtiStatus_.controllerControlPlane
         self.computeReplicationMode()
         for vlanId in self.vtiStatus_.vlanToVniMap:
            self.handleVlanToVniMap( vlanId )
            self.handleVlanToGroup( vlanId )
      Tac.Notifiee.__init__( self, vtiStatus )
      # process the current state
      self.handleLinkStatus()
      self.handleVlanToGroup()
      self.handleVlanToVniMap()

   def __del__( self ):
      t2( 'VtiStatusReactor::__del__()' )
      self.vxlanStatusDirReactor_ = None

   @Tac.handler( 'controllerClientMode' )
   def handleControllerClientMode( self ):
      t2( 'controllerClientMode %s' % self.notifier_.controllerClientMode )
      self.bridge_.controllerClientMode_ = self.notifier_.controllerClientMode
      self.computeReplicationMode()

   @Tac.handler( 'controllerControlPlane' )
   def handleControllerControlPlane( self ):
      t2( 'controllerControlPlane %s' % self.notifier_.controllerControlPlane )
      self.bridge_.controllerControlPlane_ = self.notifier_.controllerControlPlane
      self.computeReplicationMode()

   @Tac.handler( 'datapathLearning' )
   def handleDatapathLearningMode( self ):
      t2( 'datapathLearningMode %s' % self.notifier_.datapathLearning )
      self.bridge_.datapathLearningMode_ = self.notifier_.datapathLearning
      self.computeReplicationMode()
      if self.bridge_.datapathLearningMode_:
         cleanupReceivedRemoteMacs( self.bridge_ )
      else:
         deleteLearnedRemoteMacs( self.bridge_ )

   @Tac.handler( 'linkStatus' )
   def handleLinkStatus( self ):
      t2( 'linkStatus %s' % self.notifier_.linkStatus )
      if self.notifier_.linkStatus == 'linkUp':
         if not self.bridge_.vxlanFdbConfigReactor_ is None:
            self.bridge_.vxlanFdbConfigReactor_.handleConfiguredHosts()
         if self.bridge_.mlagHostTableReactor_ is not None:
            self.bridge_.mlagHostTableReactor_.handleMlagHostEntry()
      self.computeReplicationMode()

   @Tac.handler( 'vlanToGroup' )
   def handleVlanToGroup( self, vlanId=None ):
      if vxlanHwStatusDir.bumStatus.bumReplicationMode != \
            BumReplicationMode.multicast:
         # recompute the replication mode and return
         t2( 'handleVlanToGroup: recompute replication mode' )
         self.computeReplicationMode()
         return
      elif vlanId is None:
         for vlan in self.vtiStatus_.vlanToGroup:
            self.handleVlanToGroup( vlan )
         return

      group = self.vtiStatus_.vlanToGroup.get( vlanId )
      if group:
         t2( "vlanToGroup add vlan %d group %s" % ( vlanId, group ) )
         self.modifyBumMcastGrp( 'add', group, vlanId )
      else:
         t2( "vlanToGroup del vlan %d group %s" % ( vlanId, group ) )
         self.modifyBumMcastGrp( 'del', group, vlanId )

   @Tac.handler( 'vlanToVniMap' )
   def handleVlanToVniMap( self, vlanId=None ):
      if vlanId is None:
         for vlan in self.vtiStatus_.vlanToVniMap:
            self.handleVlanToVniMap( vlan )
         return

      t2( 'handleVlanToVniMap %s/%d' % ( self.vtiStatus_.intfId, vlanId ) )

      vniSource = self.vtiStatus_.vlanToVniMap.get( vlanId )
      vni = vniSource.vni if vniSource and vniSource.vni != \
            Tac.Value( "Vxlan::VniOrNone" ) else None
      if self.bridge_.vxlanPort_:
         vtiIntf = self.bridge_.vxlanPort_.intfName_
         if ( self.vxlanStatusDirReactor_ and
           self.vxlanStatusDirReactor_.vxlanStatus_.has_key( vtiIntf ) ):
            vxlanStatus = self.vxlanStatusDirReactor_.vxlanStatus_[ vtiIntf ]
         else:
            vxlanStatus = self.vxlanStatusDir_.vxlanStatus[ vtiIntf ]
      else:
         vxlanStatus = None
      updateVniToVlanMap( self.bridge_, vxlanStatus, vni, vlanId )

   def modifyBumMcastGrp( self, action, maddr, vlanId ):
      t2( 'modifyBumMcastGrp: %s vlan %d group %s' % ( action, vlanId, maddr ) )
      if action == 'add':
         vxlanHwStatusDir.bumStatus.bumMcastGrp[ vlanId ] = maddr
      elif action == 'del' and \
            vxlanHwStatusDir.bumStatus.bumMcastGrp.has_key( vlanId ):
         del vxlanHwStatusDir.bumStatus.bumMcastGrp[ vlanId ]

   @Tac.handler( 'vArpVtepAddr' )
   def handleVarpVtepAddr( self ):
      updateBpfFilterString( self.bridge_ )
      if vxlanHwStatusDir.bumStatus.bumReplicationMode == \
            BumReplicationMode.headendVcs:
         if not self.bridge_.datapathLearningMode_:
            deleteLearnedRemoteMacs( self.bridge_ )

   @Tac.handler( 'vArpVtepAddr6' )
   def handleVarpVtepAddr6( self ):
      updateBpfFilterString( self.bridge_ )
      if vxlanHwStatusDir.bumStatus.bumReplicationMode == \
            BumReplicationMode.headendVcs:
         if not self.bridge_.datapathLearningMode_:
            deleteLearnedRemoteMacs( self.bridge_ )

   @Tac.handler( 'mlagSharedRouterMacAddr' )
   def handleMlagSharedRouterMacAddr( self ):
      sharedMac = self.notifier_.mlagSharedRouterMacAddr
      t2( 'handleMlagSharedRouterMacAddr %s' % sharedMac )
      self.bridge_.sharedRouterMac_ = sharedMac

   def replicationModeChange( self, vxlanStatus, vxlanStatusReactor, newmode ):
      t2( "vxlan replication Mode change" )

      # On replication mode change, clear stale bumStatus entries
      cleanupMcastGrp()
      if vxlanHwStatusDir.bumStatus.bumReplicationMode != BumReplicationMode.unknown:
         cleanupBumVtepList( self.bridge_ )

      sourceL2RibLocal = ( newmode == BumReplicationMode.headendNonVcs )
      sourceL2Rib = ( sourceL2RibLocal or self.bridge_.controllerControlPlane_ or
                      self.bridge_.controllerClientMode_ )

      if not sourceL2RibLocal:
         if vxlanStatusReactor and vxlanStatusReactor.vxlanStatusFloodListReactor_:
            t2( "Remove vxlanStatusFloodListReactor" )
            vxlanStatusReactor.vxlanStatusFloodListReactor_ = None 
      if not sourceL2Rib:
         if self.bridge_.l2RibVlanFloodSetReactor_ is not None:
            t2( "Remove all reactors within l2RibVlanFloodSetReactor" )
            self.bridge_.l2RibVlanFloodSetReactor_.delAllFloodSetReactors()

      vxlanHwStatusDir.bumStatus.bumReplicationMode = newmode

      # Repopulate bumStatus for new replication mode
      if newmode == BumReplicationMode.multicast:
         self.handleVlanToGroup( vlanId=None )
      elif sourceL2Rib:
         # Make sure we track the flood and learnRestrict lists
         if ( sourceL2RibLocal and vxlanStatusReactor and
              not vxlanStatusReactor.vxlanStatusFloodListReactor_ ):
            t2( "Launch vxlanStatusFloodListReactor" )
            vxlanStatusReactor.vxlanStatusFloodListReactor_ = \
                VxlanStatusFloodListReactor( vxlanStatus, self, self.bridge_ )
         # Iterate across L2RibSetOutput
         if self.bridge_.l2RibVlanFloodSetReactor_ is not None:
            self.bridge_.l2RibVlanFloodSetReactor_.handleVlanFloodSets()

   def computeReplicationMode( self ):
      t2( "VtiStatusReactor::ComputeReplicationMode" )
      vxlanStatus = None
      vxlanStatusReactor = None
      if self.vxlanStatusDirReactor_:
         if 'Vxlan1' in self.vxlanStatusDirReactor_.vxlanStatus_:
            vxlanStatus = self.vxlanStatusDirReactor_.vxlanStatus_[ 'Vxlan1' ]
         if 'Vxlan1' in self.vxlanStatusDirReactor_.vxlanStatusReactor_: 
            vxlanStatusReactor = \
                self.vxlanStatusDirReactor_.vxlanStatusReactor_[ 'Vxlan1' ]

      multicast = self.vtiStatus_.floodMcastGrp != '0.0.0.0' or \
                  self.vtiStatus_.vlanToGroup

      if self.vtiStatus_.linkStatus != 'linkUp':
         newmode = BumReplicationMode.unknown
         cleanupRemoteMacs( self.bridge_ )
      elif self.bridge_.controllerClientMode_ or \
           self.bridge_.controllerControlPlane_:
         newmode = BumReplicationMode.headendVcs
      elif bool( l2RibFloodSet.source & vxlanStaticSource ) and \
            ( not multicast or l2RibFloodSet.vlanFloodSet ):
         newmode = BumReplicationMode.headendNonVcs
      elif multicast:
         newmode = BumReplicationMode.multicast
      else:
         newmode = BumReplicationMode.unknown

      t2( "Vxlan replication Mode old %s new %s " % ( 
         vxlanHwStatusDir.bumStatus.bumReplicationMode, newmode ) )

      if vxlanHwStatusDir.bumStatus.bumReplicationMode != newmode:
         self.replicationModeChange( vxlanStatus, vxlanStatusReactor, newmode )

   @Tac.handler( "vtepToVtepBridging" )
   def handleVtepToVtepBridging( self ):
      if self.bridge_.vxlanPort_.shim:
         self.bridge_.vxlanPort_.shim.suppressInputPortValue = (
            not self.notifier_.vtepToVtepBridging )

   @Tac.handler( "localVtepAddr" )
   @Tac.handler( "mlagVtepAddr" )
   @Tac.handler( "localVtepAddr6" )
   @Tac.handler( "mlagVtepAddr6" )
   def handleAddrChange( self ):
      updateBpfFilterString( self.bridge_ )

class BumStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Vxlan::BumStatus'
   def __init__( self, learnStatus, bumStatus, bridge ):
      t2( 'BumStatusReactor init' )
      self.bridge_ = bridge
      self.learnStatus_ = learnStatus
      self.bumStatus_ = bumStatus
      self.remoteVtepAddrReactor_ = {}
      Tac.Notifiee.__init__( self, bumStatus )

   @Tac.handler( 'bumVtepList' )
   def handleBumVtepList( self, macVlanPair ):
      t2( 'handleBumVtepList %s' % ( macVlanPair ) )

      if macVlanPair is None:
         for macVlan in self.remoteVtepAddrReactor_:
            t2( 'handleBumVtepList handling existing reactor for macVlanPair=',
                macVlan )
            self.remoteVtepAddrReactor_[ macVlan ].handleRemoteVtepAddr( addr=None )
         return

      # If the key doesn't exist, handleCollectionChange will delete the reactor
      Tac.handleCollectionChange( RemoteVtepAddrReactor, macVlanPair,
                self.remoteVtepAddrReactor_,
                self.notifier_.bumVtepList,
                reactorArgs=( self.learnStatus_, self.bumStatus_, self.bridge_ ),
                reactorTakesKeyArg=True,
                reactorFilter=None )

      t4( 'active reactors exist for macVlanPairs:', \
            self.remoteVtepAddrReactor_ )

   def __del__( self ):
      t2( 'BumStatusReactor::__del__()' )

class RemoteVtepAddrReactor( Tac.Notifiee ):
   notifierTypeName = 'Vxlan::BumVtepList'
   def __init__( self, vlist, macVlanPair, learnStatus, bumStatus, bridge ):
      t2( 'RemoteVtepAddrReactor bum init: macVlanPair=', macVlanPair,
          ' learnStatus=', learnStatus, ' bumStatus=', bumStatus,
          ' bridge=', bridge )
      self.bridge_ = bridge
      self.learnStatus_ = learnStatus
      self.bumStatus_ = bumStatus
      self.macVlan_ = macVlanPair
      Tac.Notifiee.__init__( self, self.bumStatus_.bumVtepList[ macVlanPair] )

   @Tac.handler( 'remoteVtepAddr' )
   @Tac.handler( 'remoteVtepAddr6' )
   def handleRemoteVtepAddr( self, addr ):
      vlanId = self.macVlan_.vlanId
      t2( 'bum handleRemoteVtepAddr: addr=', addr, " vlan=", vlanId )
      if not self.learnStatus_.learnStatusList.has_key( vlanId ):
         t2( 'bum handleRemoteVtepAddr: no learnStatusList' )
         return
      if ( self.learnStatus_.learnStatusList[ vlanId ].learnFrom !=
           'learnFromFloodList' ):
         t2( 'bum handleRemoteVtepAddr: not learnFromFloodList' )
         return

      if addr is None:
         if not self.learnStatus_.learnStatusList.has_key( vlanId ):
            self.learnStatus_.learnStatusList.newMember( vlanId )
         if self.bumStatus_.bumVtepList.has_key( self.macVlan_ ):
            updateLearnStatusFromFlood( vlanId,
                                        self.bumStatus_.bumVtepList[ self.macVlan_ ],
                                        self.learnStatus_.learnStatusList[ vlanId ] )
         else:
            updateLearnStatusFromFlood( vlanId, [],
                                        self.learnStatus_.learnStatusList[ vlanId ] )
         return

      # We are adding or removing an entry from the bumVtepList and
      # the vlan is configured with 'learnFromFloodList'. Hence we add/delete
      # the corresponding host prefix in learnStatusList.
      bumVtepList = self.bumStatus_.bumVtepList[ self.macVlan_ ] if \
          self.bumStatus_.bumVtepList.has_key( self.macVlan_ ) else []

      if not self.learnStatus_.learnStatusList.has_key( vlanId ):
         self.learnStatus_.learnStatusList.newMember( vlanId )

      ls = self.learnStatus_.learnStatusList[ vlanId ]

      v4 = isinstance( addr, str )
      gp = vtepToPrefix( addr )
      if ( not bumVtepList or
           ( v4 and not bumVtepList.remoteVtepAddr.has_key( addr ) ) or
           ( not v4 and not bumVtepList.remoteVtepAddr6.has_key( addr ) ) ):
         # Delete?
         t2( 'bum handleRemoteVtepAddr: check delete addr=', addr )
         if ls.prefixList.has_key( gp ):
            t2( 'bum handleRemoteVtepAddr: delete ', addr )
            del ls.prefixList[ gp ]
            del ls.numMatches[ gp ]
            cleanupRemoteMacsLearnRestrict( self.bridge_, vlanId )
      else:
         # Add?
         t2( 'bum handleRemoteVtepAddr: check add addr=', addr )
         if not ls.prefixList.has_key( gp ):
            t2( 'bum handleRemoteVtepAddr: add ', addr )
            ls.prefixList[ gp ] = True
            ls.numMatches[ gp ] = 0

class VlanToLearnRestrictReactor( Tac.Notifiee ):
   notifierTypeName = 'Vxlan::LearnRestrict'
   def __init__( self, vlist, vlanId, learnStatus, vxlanStatus, bridge ):
      t2( 'VlanToLearnRestrictReactor init vlanId=', vlanId )
      assert vlanId == vlist.vlanId
      self.bridge_ = bridge
      self.vxlanStatus_ = vxlanStatus
      self.learnStatus_ = learnStatus
      self.vlanId_ = vlanId
      self.vlist_ = vlist
      Tac.Notifiee.__init__( self, vlist )
      self.handleLearnFrom()

   @Tac.handler( 'learnFrom' )
   def handleLearnFrom( self ):
      t2( 'handleLearnFrom: vlanId=', self.vlanId_ )
      if not self.vxlanStatus_.vlanToLearnRestrict.has_key( self.vlanId_ ):
         if self.learnStatus_.learnStatusList.has_key( self.vlanId_ ):
            del self.learnStatus_.learnStatusList[ self.vlanId_ ]
         return

      learnFrom = self.vxlanStatus_.vlanToLearnRestrict[ self.vlanId_ ].learnFrom
      if learnFrom == 'learnFromDefault':
         if self.learnStatus_.learnStatusList.has_key( self.vlanId_ ):
            del self.learnStatus_.learnStatusList[ self.vlanId_ ]
         return

      prefixes = []
      vteps = []
      if learnFrom == 'learnFromFloodList':
         vteps = getFloodList( self.vlanId_ )
      elif learnFrom == 'learnFromList':
         if ( self.vxlanStatus_ and
              self.vxlanStatus_.vlanToLearnRestrict.has_key( self.vlanId_ ) ):
            prefixes = self.vxlanStatus_.vlanToLearnRestrict[
               self.vlanId_ ].prefixList

      modifyDefaultLearnRestrict( learnFrom, prefixes, vteps, self.vlanId_ )
      # TBD: Could avoid the cleanup when changing to learnFromAny
      cleanupRemoteMacsLearnRestrict( self.bridge_, self.vlanId_ )

   @Tac.handler( 'prefixList' )
   def handlePrefixList( self, prefix ):
      t2( 'handlePrefixList: vlanId=', self.vlanId_, ' prefix=', prefix )
      if prefix is None:
         if self.learnStatus_.learnStatusList.has_key( self.vlanId_ ):
            ls = self.learnStatus_.learnStatusList[ self.vlanId_ ]
            if ls.learnFrom == 'learnFromDefault':
               del self.learnStatus_.learnStatusList[ self.vlanId_ ]
            else:
               for p in ls.prefixList:
                  del ls.prefixList[ p ]
               for p in ls.numMatches:
                  del ls.numMatches[ p ]
            cleanupRemoteMacsLearnRestrict( self.bridge_, self.vlanId_ )

         for p in self.vlist_.prefixList:
            t2( 'calling handlePrefixList %s' % ( p ) )
            self.handlePrefixList( p )
         return

      if self.vlist_.prefixList.has_key( prefix ):
         if self.learnStatus_.learnStatusList.has_key( self.vlanId_ ):
            ls = self.learnStatus_.learnStatusList[ self.vlanId_ ]
         else:
            ls = self.learnStatus_.learnStatusList.newMember( self.vlanId_ )
            if self.vxlanStatus_.vlanToLearnRestrict.has_key( self.vlanId_ ):
               ls.learnFrom = \
                   self.vxlanStatus_.vlanToLearnRestrict[ self.vlanId_ ].learnFrom
            else:
               ls.learnFrom = 'learnFromDefault'

         ls.prefixList[ prefix ] = True
         ls.numMatches[ prefix ] = 0
         # Don't add with learnFromDefault
         if ls.learnFrom == 'learnFromDefault':
            if self.learnStatus_.learnStatusList.has_key( self.vlanId_ ):
               del self.learnStatus_.learnStatusList[ self.vlanId_ ]
               cleanupRemoteMacsLearnRestrict( self.bridge_, self.vlanId_ )
         else:
            t2( 'handlePrefixList added %s' % ( prefix ) )
      else:
         if self.learnStatus_.learnStatusList.has_key( self.vlanId_ ): 
            ls = self.learnStatus_.learnStatusList[ self.vlanId_ ] 
            if ls.prefixList.has_key( prefix ):
               t2( 'handlePrefixList del %s' % ( prefix ) )
               del ls.prefixList[ prefix ]
            if ls.numMatches.has_key( prefix ):
               del ls.numMatches[ prefix ]
            # Fewer VTEPs/prefixes allowed
            cleanupRemoteMacsLearnRestrict( self.bridge_, self.vlanId_ )

   def __del__( self ):
      t2( 'VlanToLearnRestrictReactor deleted vlanId=', self.vlanId_ )
      del self.learnStatus_.learnStatusList[ self.vlanId_ ]
      cleanupRemoteMacsLearnRestrict( self.bridge_, self.vlanId_ )

class VxlanIntfPort( EbraTestPort ):
   """ Manage the VTI port """

   def __init__( self, bridge, tapDevice, trapDevice, intfConfig, intfStatus ):
      """Initialize the port to its default state."""

      assert tapDevice is None
      assert trapDevice is None
      assert intfConfig
      assert intfStatus

      EbraTestPort.__init__( self, bridge, intfConfig, intfStatus )

      intfName = intfStatus.intfId
      num = re.search( r'Vxlan(\d+)', self.intfName_ ).group( 1 )
      if self.bridge_.inNamespace():
         devName = 'vx%s' % num
      else:
         devName = self.bridge_.name() + '-vx%s' % num

      global vxlanAgentDevs

      # When bringing up the first port, bring up the vxlan agent devices as well.
      if vxlanAgentDevs is None:
         vxlanAgentDevs = VxlanAgentDevs( self.bridge_ )

      self.trapDevice_ = Arnet.Device.Tap( devName, hw=self.bridge_.bridgeMac() )

      ent = bridge.em().entity
      self.trapFile_ = None
      self.tapFile_ = None
      self.tapDevice_ = self.trapDevice_
      self.processFrame_ = None

      # Save the sysdb config reference this interface
      self.vtiConfig_ = ent( 'interface/config/eth/vxlan' ).vtiConfig[ intfName ]

      self.vtiStatus_ = ent( 'interface/status/eth/vxlan' ).vtiStatus[ intfName ]
      self.vxlanStatusDir_ = ent( 'vxlan/status' )
      self.mlagStatus_ = ent( 'mlag/status' )
      self.vtiConfigReactor_ = None
      self.vtiStatusReactor_ = None

      # create bumVtepListReactor to update for learnFromFloodList
      self.bumStatus_ = vxlanHwStatusDir.bumStatus
      self.learnStatus_ = vxlanHwStatusDir.learnStatus

      self.bumStatusReactor_ = None
      self.shim = None

   def onActive( self ):
      global vxlanAgentDevs
      vxlanAgentDevs.onActive()
      self.trapFile_ = Tac.File( self.trapDevice_.fileno(),
                                 self._trapDeviceReadHandler,
                                 self.trapDevice_.name,
                                 readBufferSize=16000 )
      self.tapFile_ = self.trapFile_
      t2( "Vxlan Port initialized: Device is: %s fileno %d" %
          ( self.trapDevice_.name, self.trapDevice_.fileno() ) )

      self.vtiConfigReactor_ = VtiConfigReactor( self.bridge_, self.vtiConfig_ )
      updateBpfFilterString( self.bridge_ )
      self.vtiStatusReactor_ = VtiStatusReactor( self.vtiStatus_, self.bridge_,
                                                 self.vxlanStatusDir_ )

      # instantiate the vxlanHwStatus for this intfName
      vxlanHwStatusDir.vxlanHwStatus.newMember( self.intfName_ )

      t4( 'starting BumStatusReactor' )
      self.bumStatusReactor_ = BumStatusReactor( self.learnStatus_,
                                                 self.bumStatus_, self.bridge_ )

   def _tapDeviceReadHandler( self, data) :
      # handle frame from vxlan agent via txraw after ISL is removed
      ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( data )
      t4( "%s: received TAP packet src %s/dst %s" % \
         ( self.name(), srcMacAddr, dstMacAddr ) )
      self.bridge_.processFrame( data, srcMacAddr, dstMacAddr, self, False )

   def _trapDeviceReadHandler( self, data) :
      # handle frame from vxlan agent via txraw after ISL is removed
      ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( data )
      t4( "%s: received trap packet src %s/dst %s" % \
         ( self.name(), srcMacAddr, dstMacAddr ) )
      self.bridge_.processFrame( data, srcMacAddr, dstMacAddr, self, False )

   def close( self ):
      cleanupRemoteMacs( self.bridge_ )
      vxlanLearnedHost.clear()
      multiVtepLearnedHost.clear()
      self.bridge_.macTableFlushPort( self.name() )
      self.trapFile_.close()
      self.trapDevice_.close()
      self.vtiStatusReactor_ = None
      self.bridge_.vxlanPort_ = None
      self.bridge_.remoteVtepRef_ = {}
      self.bridge_.vxlanVniVlanMap_ = {}
      t2( "VxlanIntfPort", self.name(), "closed" )

   def updateStatusConfig( self, intfStatus, intfConfig ):
      EbraTestPort.updateStatusConfig( self, intfStatus, intfConfig )

   def checkVtepToVtepBridging( self ):
      return self.vtiStatus_.vtepToVtepBridging

   def sendFrame( self, data, srcMacAddr, dstMacAddr, srcPortName, priority, vlanId,
                  vlanAction ):
      """Send the frame up to Vxlan Agent for encap."""
      if self.intfConfig_.enabled and \
         self.intfStatus_.operStatus == 'intfOperUp':
        
         # Discard frames received on mlag peer-link since it would
         # be processed by the other mlag peer on which the frame was
         # originally received.
         if self.mlagStatus_.peerLinkIntf is not None:
            if self.mlagStatus_.peerLinkIntf.intfId == srcPortName:
               t6( "Discarding frame on port %s received from peer link port %s" % \
                   ( self.name(), srcPortName ) )
               return

         # Need some additional information about the packet, such as whether or not
         # to strip the 802.1Q tag and encode the information in the header sent to
         # the agent.
         etherType = ( ord( data[ 12 ] ) << 8 ) | ord( data[ 13 ] )
         vlanInfo = self.bridge_.vlanTagState( srcPortName, False, etherType,
                                               data, dstMacAddr )
         ( _, _, vlanTagNeedsUpdating, isTagged, _ ) = vlanInfo

         flags = VXLAN_CPUPORT_FLAG if srcPortName == 'Cpu' else 0
         srcIfIndex = 0

         # Add mlag flag if source is a MLAG port
         if self.mlagStatus_.intfStatus.get( srcPortName ):
            # add flag to indicate that the src port is from an MLAG interface
            flags |= VXLAN_MLAG_SRCPORT_FLAG

         swintfconfig = self.bridge_.brConfig_.switchIntfConfig.get( srcPortName ) 
         # swintfconfig won't exist for Cpu but Cpu will never be dot1qtunnel
         if swintfconfig:
            flags |= VXLAN_DOT1QTUNNEL_FLAG if \
               swintfconfig.switchportMode == 'dot1qTunnel' else 0

         # This uses vxlan_src_vtep_ext
         flags |= VXLAN_SRC_VTEP_EXT_FLAG

         # Extract and parse the source vtep ip
         context = self.bridge_.packetContext
         srcVtep = 0 if 'vxlanSrcVtep' not in context else context[ 'vxlanSrcVtep' ]
         addrFamily = socket.AF_INET6 if ':' in str( srcVtep ) else socket.AF_INET
         srcVtep = IpAddress( str( srcVtep ), addrFamily=addrFamily ).toNum()
         if srcVtep > sys.maxint:
            srcVtep = sys.maxint
         global encapSeqNo
         if isTagged or vlanTagNeedsUpdating or \
                srcPortName == 'Cpu':
            t6( "removing 802.1Q tag on port %s vlanId %d srcPort %s" %
                  ( self.name(), vlanId, srcPortName ) )
            data = data[ 0:12 ] + \
                     struct.pack( '!HHLLLL', ETH_P_ARISTA_VXLAN_ENCAP, flags, \
                        vlanId, srcIfIndex, encapSeqNo, srcVtep ) + \
                     data[ 16: ]
         else:
            t6( "sending frame on port %s vlanId %d srcPort %s action %s" %
                  ( self.name(), vlanId, srcPortName, vlanAction ) )
            data = data[ 0:12 ] + \
                     struct.pack( '!HHLLLL', ETH_P_ARISTA_VXLAN_ENCAP, flags, \
                        vlanId, srcIfIndex, encapSeqNo, srcVtep ) + \
                     data[ 12: ]
         t8( 'Send data to kernel ', data )
         vxlanAgentDevs.vxlanTapDevice_.send( data )
         encapSeqNo = encapSeqNo + 1

      else:
         t6( "Refusing to transmit on disabled port %s" % self.name() )

   def trapFrame( self, data ):
      """ We shouldn't be trapping the frame, but some other trap handlers can look
      at the ipHdr info (e.g. igmp) and think that a copyToCpu is necessary.  Ignore
      this trap request """
      t0( 'unexpected trap on port %s' % self.name() )
      t6( "trapping frame input on %s" % self.name(), Tracing.HexDump( data ) )

class VxlanIntfDirReactor( object ):
   """We wait until a vxlan intf appears in the interface/{config,status}/eth/intf
   collections before adding it to the test bridge.  We don't just watch the
   interface/{config,status}/eth/vxlan collections, because there is a time delay
   between showing up there and showing up in the 'intf' collections.  The test
   bridge looks in the 'intf' collections, so that's what we watch for.

   This class is a base class for the reactors to each of the config and status
   collections since the code for each is nearly identical.
   """

   def __init__( self, bridge ):
      self.bridge_ = bridge
      self.ethIntfConfigDir_ = bridge.ethIntfConfigDir_
      self.ethIntfStatusDir_ = bridge.ethIntfStatusDir_

   def handleIntf( self, key ):
      t2( "VxlanIntfDirReactor for", key )
      if key is None:
         # In the unlikely event of deletion of multiple interfaces in a
         # multi-attribute notification, we will leave orphan interfaces
         # in the bridge port list, until a given interface gets re-added.
         for i in self.ethIntfConfigDir_:
            self.handleIntf( i )
         return

      if not key.startswith( 'Vxlan1' ): # decap only on Vxlan1
         return

      t2( "VxlanIntfDirReactor checking", key )
      intfConfig = self.ethIntfConfigDir_.get( key )
      intfStatus = self.ethIntfStatusDir_.get( key )
      if intfConfig and intfStatus:
         if not key in self.bridge_.port:
            t2( "VxlanIntfDirReactor adding", key, "to ports" )
            vxlanPort = VxlanIntfPort( self.bridge_, None, None,
                                       intfConfig, intfStatus )
            vxlanPort.onActive()
            shim = self.bridge_.addPort( vxlanPort )
            if EtbaDutToggleLib.toggleFastEtbaEnabled():
               shim.isRemoteValue = True
               shim.suppressInputPortValue = not vxlanPort.checkVtepToVtepBridging()
               vxlanPort.shim = shim
            self.bridge_.vxlanPort_ = vxlanPort
            updateBpfFilterString( self.bridge_ )
            t0( 'vxlan port iniitialized' )
            t0( 'port name %s: ' % self.bridge_.vxlanPort_.intfName_ )
         else:
            # intfStatus object changed.
            port = self.bridge_.port[ key ]
            t2( 'vxlan port', key )
            if self.bridge_.vxlanPort_ is None:
               self.bridge_.vxlanPort_ = port
               updateBpfFilterString( self.bridge_ )
            
            oldIntfStatus = port.intfStatus_

            if intfStatus != oldIntfStatus:
               t2( 'VxlanIntfDirReactor, change in intfStatus, modify port' )
               port.updateStatusConfig( intfStatus, intfConfig )

      elif key in self.bridge_.port:
         # Clean up the underlying port
         t2( "VxlanIntfDirReactor removing", key, "from ports" )
         self.bridge_.delPort( key )
         em = self.bridge_.em()
         vxlanBumStatus = em.entity( 'vxlan/hardware/status' ).bumStatus
         for macvlan in vxlanBumStatus.bumVtepList.keys():
            del vxlanBumStatus.bumVtepList[ macvlan ]
            t2( "VxlanIntfDirReactor removed entry ", macvlan, \
                     "from bumVtepList" )
         for vlan in vxlanBumStatus.bumMcastGrp.keys():
            del vxlanBumStatus.bumMcastGrp[ vlan ]
            t2( "VxlanIntfDirReactor removed entry ", vlan, \
                  "from bumMcasGrp" )
         vxlanBumStatus.bumReplicationMode = BumReplicationMode.unknown

class VxlanIntfStatusReactor( Tac.Notifiee, VxlanIntfDirReactor ):

   notifierTypeName = 'Interface::EthIntfStatusDir'

   def __init__( self, bridge ):
      t0( "VxlanIntfStatusReactor: __init__: ")
      VxlanIntfDirReactor.__init__( self, bridge )
      Tac.Notifiee.__init__( self, bridge.ethIntfStatusDir_ )
      for intf in bridge.ethIntfStatusDir_:
         t0( "VxlanIntfStatusReactor: __init__: handleStatus for %s" % intf )
         self.handleStatus( intf )

   @Tac.handler( 'intfStatus' )
   def handleStatus( self, key ):
      self.handleIntf( key )

class VxlanIntfConfigReactor( Tac.Notifiee, VxlanIntfDirReactor ):

   notifierTypeName = 'Interface::EthIntfConfigDir'

   def __init__( self, bridge ):
      t0( "VxlanIntfConfigReactor: __init__ " )
      VxlanIntfDirReactor.__init__( self, bridge )
      Tac.Notifiee.__init__( self, bridge.ethIntfConfigDir_ )
      for intf in bridge.ethIntfConfigDir_:
         t0( "VxlanIntfConfigReactor: __init__: handleConfig for %s" % intf )
         self.handleConfig( intf )

   @Tac.handler( 'intfConfig' )
   def handleConfig( self, key ):
      self.handleIntf( key )

class VxlanVlanStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Bridging::VlanStatusDir'

   def __init__( self, bridgingVlanStatusDir, vxlanStatusDir, bridge ):
      self.bridge_ = bridge
      self.vxlanStatusDir_ = vxlanStatusDir
      Tac.Notifiee.__init__( self, bridgingVlanStatusDir )

   @Tac.handler( 'vlanStatus' )
   def handleVlanStatus( self, vlanId ):
      # XXX: Just supporting one VTI for now
      if not self.bridge_.vxlanPort_ or \
         self.bridge_.vxlanPort_.vtiStatus_.linkStatus != 'linkUp' or \
         self.bridge_.vxlanPort_.intfStatus_.operStatus != 'intfOperUp':
         t2( 'handleVlanStatus: VTI is down for vlan %d' % vlanId )
         return

      vtiStatus = self.bridge_.vxlanPort_.vtiStatus_
      vxlanStatus = self.vxlanStatusDir_.vxlanStatus[ 'Vxlan1' ]

      if vtiStatus.vlanToVniMap.has_key( vlanId ):
         vniSource = vtiStatus.vlanToVniMap.get( vlanId )
         vni = vniSource.vni if vniSource.vni != \
               Tac.Value( "Vxlan::VniOrNone" ) else None 
      else: 
         vni = None

      if vlanId in self.notifier_.vlanStatus and vni is not None:
         mapVlanId = self.bridge_.vxlanVniVlanMap_.get( vni )
         if mapVlanId == vlanId:
            t2( "handleVlanStatus: vlan %d already mapped" % mapVlanId )
            return
         elif mapVlanId:
            # Not sure how this might happen, but print a trace statement
            # if it does
            t0( "handleVlanStatus: vlan %d/mapvlan %d mismatch" %
                  ( vlanId, mapVlanId ) )
            return
         t2( "handleVlanStatus: vlan %d added" % vlanId )
         updateVniToVlanMap( self.bridge_, vxlanStatus, vni, vlanId )
      elif vni is not None:
         t2( "handleVlanStatus: vlan %d removed" % vlanId )
         updateVniToVlanMap( self.bridge_, vxlanStatus, None, vlanId )

class VxlanFdbConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Vxlan::FdbConfig'
   def __init__( self, vxlanFdbConfigDir, bridge ):
      assert bridge
      self.bridge_ = bridge
      self.configuredHosts_ = vxlanFdbConfigDir.configuredHost
      Tac.Notifiee.__init__( self, vxlanFdbConfigDir )

   @Tac.handler( 'configuredHost' )
   def handleFdbConfig( self, macVlan ):

      t2( 'handleFdbConfig key mac %s vlan %d' %  \
            ( macVlan.macAddr, macVlan.vlanId ) )
      self.handleConfiguredHosts( key=macVlan )

   def handleConfiguredHosts( self, key=None ):
      t2( 'handleConfiguredHosts key ', key )
      # add the configured fdb to both vxlan and etba
      keys = self.configuredHosts_.keys() if key is None else [ key ]
      for k in keys:
         host = self.configuredHosts_.get( k )
         if not host:
            t4( 'deleting host found for key %s/%s' % ( k.macAddr, k.vlanId ) )
            deleteLearnedHost( self.bridge_, k.vlanId, k.macAddr,
                               entryType=BridgingEntryType.configuredRemoteMac,
                               deleteBridgeMac=True )
            continue

         assert not host.remoteVtepAddr.isAddrZero, 'remote vtep is zero'
         addLearnedHost( self.bridge_,
                         host.macVlanPair.vlanId,
                         host.macVlanPair.macAddr,
                         host.remoteVtepAddr,
                         entryType=BridgingEntryType.configuredRemoteMac,
                         moveCount=1 )

class FdbConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'Bridging::FdbConfig'

   def __init__( self, fdbConfig, bridge ):
      Tac.Notifiee.__init__( self, fdbConfig )
      self.bridge_ = bridge
      self.vlanId_ = fdbConfig.fid
      for key in fdbConfig.configuredHost:
         self.handleConfiguredHost( key )

   @Tac.handler( 'configuredHost' )
   def handleConfiguredHost( self, key ):
      t0('handleConfiguredHost key ', key )
      hostEntry = self.notifier_.configuredHost.get( key )
      if not hostEntry:
         return
      macVlanPair = Tac.Value( "Vxlan::MacVlanPair", key, self.vlanId_ )
      if not vxlanLearnedHost.has_key( macVlanPair ):
         return

      # the configured static mac overrides the received remote mac
      deleteLearnedHost( self.bridge_, self.vlanId_, key,
                         BridgingEntryType.receivedRemoteMac )

class EvpnMacTableReactor( Tac.Notifiee ):
   notifierTypeName = 'Bridging::RemoteMacTable'

   def __init__( self, evpnMacTable, bridge ):
      assert bridge
      self.bridge_ = bridge
      self.remoteMacEntry_ = evpnMacTable.remoteMacEntry
      Tac.Notifiee.__init__( self, evpnMacTable )
      self.handleRemoteMacEntry()

   @Tac.handler( 'remoteMacEntry' )
   def handleRemoteMacTable( self, macVlan ):
      t2( 'handleRemoteMacTable key mac %s vlan %d' %  \
            ( macVlan.macaddr, macVlan.vlanId ) )
      self.handleRemoteMacEntry( key=macVlan )

   def handleRemoteMacEntry( self, key=None ):
      # add the entry to both vxlan and etba
      keys = self.remoteMacEntry_.keys() if key is None else [ key ]
      for k in keys:
         host = self.remoteMacEntry_.get( k )
         if not host:
            t4( 'deleting evpn found for key %s/%s' % ( k.macaddr, k.vlanId ) )
            deleteLearnedHost( self.bridge_, k.vlanId, k.macaddr,
                               entryType=BridgingEntryType.evpnDynamicRemoteMac,
                               deleteBridgeMac=True )
            continue

         assert not host.tunnelIpGenAddr.isAddrZero, 'remote vtep is zero'
         t4( 'evpn for key %s/%s tunnelIpAddr %s' % ( host.macaddr, host.vlanId,
                                                      host.tunnelIpGenAddr ) )
         addLearnedHost( self.bridge_,
                         host.vlanId,
                         host.macaddr,
                         host.tunnelIpGenAddr,
                         entryType=BridgingEntryType.evpnDynamicRemoteMac )

# Handle wraparound for U16 moveCount
def geMoveCount( m1, m2 ):
   allf = 0xFFFF
   halfway = 0x8000
   return ( ( m1 - m2 ) & allf ) < halfway

def gtMoveCount( m1, m2 ):
   return m1 != m2 and geMoveCount( m1, m2 )

def updateBpfFilterString( bridge ):
   if not EtbaDutToggleLib.toggleFastEtbaEnabled():
      return
   t6( "Updating bpf filter" )

   if ( not bridge or not bridge.root_ or not bridge.root_.pluginManager or
        "Vxlan" not in bridge.root_.pluginManager.decapPythonBpfFilters ):
      t6( "Fast Etba not initialized" )
      return

   bpfStr = generateBpfFilterString( bridge )
   filterObj = bridge.root_.pluginManager.decapPythonBpfFilters[ "Vxlan" ]

   filterObj.bpfFilterString = bpfStr

def generateBpfFilterString( bridge ):
   if not bridge.vxlanPort_:
      t6( "Bpf: no vxlan port" )
      return ""

   bpfFilter = "udp port %s" % (
      bridge.vxlanPort_.vtiConfig_.udpPort )

   ips = []
   for ip in (
         bridge.vxlanPort_.vtiStatus_.localVtepAddr,
         bridge.vxlanPort_.vtiStatus_.vArpVtepAddr,
         bridge.vxlanPort_.vtiStatus_.mlagVtepAddr,
         bridge.vxlanPort_.vtiStatus_.localVtepAddr6,
         bridge.vxlanPort_.vtiStatus_.vArpVtepAddr6,
         bridge.vxlanPort_.vtiStatus_.mlagVtepAddr6 ):
      ip = Arnet.IpGenAddr( str( ip ) )
      t6( "Bpf: ip ", ip )
      if not ip or ip.af == "ipunknown" or \
         ( ip.af == 'ipv4' and ip.v4Addr == "0.0.0.0" ) or \
         ( ip.af == 'ipv6' and ip.v6Addr == Arnet.Ip6Addr( "::" ) ):
         continue
      ips.append( "ip%s dst %s" % ( "6" if ip.af == "ipv6" else "",
                                    ip.stringValue ) )

   if not ips:
      # If no ips were added, we can't function anyway
      t6( "Bpf: no ips" )
      return ""

   bpfFilter += " and ( %s )" % " or ".join( ips )

   t6( "New bpf filter", bpfFilter )

   return bpfFilter

def validateAndParseVxlanHeader( bridge, data ):
   '''Validate Vxlan port and parse Vxlan Header'''

   retval = ( False, 0, 0, 0, 0 )

   t8( 'Validate Vxlan port and parse header' )
   if not bridge.vxlanPort_ or \
      bridge.vxlanPort_.vtiStatus_.linkStatus != 'linkUp' or \
      bridge.vxlanPort_.intfStatus_.operStatus != 'intfOperUp':
      t7( 'got routed packet but vxlan port does not exist' )
      return retval

   ( pkt, headers, off ) = parsePktStr( data )
   ipHdr = findHeader( headers, "IpHdr" )
   ip6Hdr = findHeader( headers, "Ip6Hdr" )
   if ipHdr is None and ip6Hdr is None:
      t7( 'ignoring non ip packet' )
      return retval

   if ipHdr is not None:
      if ipHdr.protocolNum != 'ipProtoUdp':
         t7( 'ignoring non udp frame' )
         return retval
      srcVtep = ipHdr.src
      dstVtep = ipHdr.dst
   else:
      if ip6Hdr.protocolNum != 'ipProtoUdp':
         t7( 'ignoring non udp frame' )
         return retval
      srcVtep = ip6Hdr.src
      dstVtep = ip6Hdr.dst
   t8( 'validateAndParseVxlanHeader srcVtep', srcVtep, 'dstVtep', dstVtep )
   # validate if destination IP is mine or is multicast IP address in case of
   # multicast replication mode. If not, return
   uDst =  UnderlayAddrType( str( dstVtep ) ) 
   if dstVtep != bridge.vxlanPort_.vtiStatus_.localVtepAddr and \
      dstVtep != bridge.vxlanPort_.vtiStatus_.vArpVtepAddr and \
      dstVtep != bridge.vxlanPort_.vtiStatus_.mlagVtepAddr and \
      dstVtep != bridge.vxlanPort_.vtiStatus_.localVtepAddr6 and \
      dstVtep != bridge.vxlanPort_.vtiStatus_.vArpVtepAddr6 and \
      dstVtep != bridge.vxlanPort_.vtiStatus_.mlagVtepAddr6 and \
      not uDst.isMulticast :
      t8( 'received packet, not intended for my Vtep IP %s ' % dstVtep )
      return retval
   elif uDst.isMulticast:
      t8( 'received mcast destIp packet %s ' % dstVtep  )
      # check dstVtep is in decapConfig underlay routes, supported for ipv4 only
      pimSsmDecap = UnderlayRouteType( defaultVrfId, uSrcV4, uDst,
            'tunnelTypePimSsm' )
      pimSmDecap = UnderlayRouteType( defaultVrfId, uSrcV4, uDst,
            'tunnelTypePimSm' )
      if pimSsmDecap not in decapConfig.decapRoute and \
         pimSmDecap not in decapConfig.decapRoute:
         t8( 'mcast destIp not in decap config, dropping it' )
         return retval 

   if ipHdr is not None:
      # validate ip checksum and ttl.
      # Note: we are not asserting here on checksum mismatch
      checksum = ipHdr.computedChecksum
      if ( checksum != 0 or ipHdr.ttl == 0 ):
         t0( 'Invalid IP Checksum %d or zero TTL %d' % \
               ( checksum, ipHdr.ttl ) )
         return retval

   udpHdr = findHeader( headers, 'UdpHdr' )
   if udpHdr is None:
      t7( 'Udp protocol but no udpHdr' )
      return retval
   if udpHdr.dstPort != bridge.vxlanPort_.vtiConfig_.udpPort:
      t7( 'Udp port mismatch got %d expected %d' % \
            ( udpHdr.dstPort, bridge.vxlanPort_.vtiConfig_.udpPort ) )
      return retval
   
   vxlanHdr = Tac.newInstance( "Arnet::VxlanHdrWrapper", pkt, off )

   if not vxlanHdr.flagsVniValid:
      t0( 'got invalid vni %d' % vxlanHdr.vni )
      return retval
   vni = vxlanHdr.vni

   try:
      vlan = bridge.vxlanVniVlanMap_[ vni ]
   except KeyError:
      t0( 'Vni %d not found in VniVlan map' % vni )
      return retval

   off += VxlanHdrSize
   srcVtep = Arnet.IpGenAddr( str( srcVtep ) )
   dstVtep = Arnet.IpGenAddr( str( dstVtep ) )
   return ( True, srcVtep, dstVtep, vlan, off )

def checkLearnFromVtep( vlanId, vtep, stats=True ):
   t8( "checkLearnFromVtep: vlan %d vtep %s" % ( vlanId, vtep ) )

   learnStatus = vxlanHwStatusDir.learnStatus
   bumStatus = vxlanHwStatusDir.bumStatus
   if not learnStatus.learnStatusList.has_key( vlanId ):
      t6( 'checkLearnFromVtep: NO VLAN vlan %d vtep %s' % ( vlanId, vtep ) )
      return True

   ls = learnStatus.learnStatusList[ vlanId ]
   if ls.learnFrom == 'learnFromAny' or ls.learnFrom == 'learnFromDefault':
      t6( 'checkLearnFromVtep: matchAny vlan %d vtep %s' % ( vlanId, vtep ) )
      if stats:
         ls.numMatchAny += 1
      return True

   elif ls.learnFrom == 'learnFromFloodList':
      macVlan = Tac.Value( "Vxlan::MacVlanPair",
                           vxlanHwStatusDir.bumStatus.bumMacDefault, vlanId )
      if bumStatus.bumVtepList.has_key( macVlan ) and \
         ( vtep.v4Addr in bumStatus.bumVtepList[ macVlan ].remoteVtepAddr or
           vtep.v6Addr in bumStatus.bumVtepList[ macVlan ].remoteVtepAddr6 ):
         t6( 'checkLearnFromVtep: matchFlood vlan %d vtep %s' % ( vlanId, vtep ) )
         if stats:
            ls.numMatchFloodList += 1
            ls.numMatches[ vtepToPrefix( vtep.v4Addr ) ] += 1
         return True
      t6( 'checkLearnFromVtep: rejectFlood vlan %d vtep %s' % ( vlanId, vtep ) )
      if stats:
         ls.numRejectFloodList += 1
      return False

   elif ls.learnFrom == 'learnFromList':
      for p in ls.prefixList:
         if p.ipPrefix.contains( vtep ):
            t6( 'checkLearnFromVtep: matchList vlan %d vtep %s' % ( vlanId, vtep ) )
            if stats:
               ls.numMatchList += 1
               ls.numMatches[ p ] += 1
            return True
      t6( 'checkLearnFromVtep: rejectList vlan %d vtep %s' % ( vlanId, vtep ) )
      if stats:
         ls.numRejectList += 1
      return False

   t6( 'checkLearnFromVtep: unknown learnFrom %s' % ls.learnFrom )
   return False

def perVtepLearning( srcVtep ):
   """Check if perVtepLearning is enabled for a given vtep"""
   if srcVtep in remoteVtepConfigDir.vtepConfig.keys() :
      return remoteVtepConfigDir.vtepConfig[ srcVtep ].learningEnabled
   return True

def processInnerFrame( bridge, srcVtep, vlan, off, data ):
   """Extract inner frame data and learn the remote address if needed"""

   vxlanIntfCfg = bridge.brConfig_.switchIntfConfig.get( 'Vxlan1' ) 
   if vxlanIntfCfg and not vxlanIntfCfg.macLearningEnabled:
      t8( 'processInnerFrame: vxlan mac learning is disabled, not learning' )
      return

   if not bridge.datapathLearningMode_:
      t8( 'processInnerFrame: datapath learning is disabled, not learning' )
      return

   if not checkLearnFromVtep( vlan, srcVtep ):
      t8( 'processInnerFrame: rejected by checkLearnFromVtep' )
      return

   if not perVtepLearning( srcVtep ):
      t8( 'processInnerFrame: datapath learning disabled for %s' % srcVtep )
      return

   addr = struct.unpack( '>HHH', data[ off+6 : off+12 ] )
   macAddr = Tac.Value( "Arnet::EthAddr", addr[0], addr[1], addr[2] )

   t8( 'processInnerFrame: learning mac: %s, vlan: %d, vtep: %s in mode: %s' %
         ( macAddr, vlan, srcVtep, vxlanHwStatusDir.bumStatus.bumReplicationMode ) )
   addLearnedHost( bridge, vlan, macAddr, srcVtep, 
                   entryType=BridgingEntryType.learnedRemoteMac )

def incRemoteVtepRef( bridge, vtep):
   # increment/set the reference count for this vtep
   if bridge.remoteVtepRef_.has_key( vtep ):
      bridge.remoteVtepRef_[ vtep ] += 1
   else:
      bridge.remoteVtepRef_[ vtep ] = 1

def decRemoteVtepRef( bridge, vtep ):
   if bridge.remoteVtepRef_.has_key( vtep ):
      t2( "Vxlan dropping refcnt %s" % bridge.remoteVtepRef_[ vtep ] )
      if bridge.remoteVtepRef_[ vtep ] > 1:
         bridge.remoteVtepRef_[ vtep ] -= 1
         return bridge.remoteVtepRef_[ vtep ]

      bridge.remoteVtepRef_[ vtep ] = 0
      return 0
   return None

def mapBridgingEntryType( bridgingEntryType ):
   if bridgingEntryType == BridgingEntryType.configuredRemoteMac or \
         bridgingEntryType == BridgingEntryType.configuredStaticMac or \
         bridgingEntryType == BridgingEntryType.evpnConfiguredRemoteMac or \
         bridgingEntryType == BridgingEntryType.peerConfiguredRemoteMac:
      return VxlanEntryType.configuredMac
   elif bridgingEntryType == BridgingEntryType.receivedRemoteMac or \
        bridgingEntryType == BridgingEntryType.peerReceivedRemoteMac:
      return VxlanEntryType.receivedMac
   elif bridgingEntryType == BridgingEntryType.evpnDynamicRemoteMac:
      return VxlanEntryType.evpnMac
   else:
      return VxlanEntryType.learnedMac

def mapVxlanEntryType( vxlanEntryType ):

   if vxlanEntryType == VxlanEntryType.configuredMac:
      return BridgingEntryType.configuredRemoteMac
   elif vxlanEntryType == VxlanEntryType.receivedMac:
      return BridgingEntryType.receivedRemoteMac
   elif vxlanEntryType == VxlanEntryType.evpnMac:
      return BridgingEntryType.evpnDynamicRemoteMac
   else:
      return BridgingEntryType.learnedRemoteMac

def getHostMoveDetails( bridge, vlan, macAddr ):
   bridgingFdbStatusDir = bridge.em().entity( 'bridging/status' ).fdbStatus
   fdbStatus = bridgingFdbStatusDir.get( vlan )
   if fdbStatus:
      learnedHosts = fdbStatus.learnedHost
      if learnedHosts and learnedHosts.has_key( macAddr ):
         host = learnedHosts[ macAddr ]
         return ( host.moves, host.lastMoveTime, host.intf )
   return ( None, None, None )
         
def maybeAddToLearnedRemoteHostStatus( key, intf, vtepIp, entryType, lastMoveTime, 
                                       moves ):
   t8( 'maybeAddToLearnedRemoteHostStatus key=', key, ' intf=', intf, 
       ' vtepIp=', vtepIp, 'entryType=', entryType, ' lastMoveTime=', lastMoveTime )
     
   if entryType != BridgingEntryType.learnedRemoteMac:
      maybeDelFromLearnedRemoteHostStatus( key )
      return
   macVlanPair = Tac.Value( "Bridging::MacVlanPair", key.macAddr, key.vlanId )
   tacCiDest = Tac.Value( 'L2Rib::CiDest', 'destTypeVxlan' )   
   tacCiDest.intf = intf
   tacCiDest.vtepAddr[ 0 ] = Tac.Value( 'Arnet::IpGenAddr', vtepIp.stringValue )
   host = Tac.Value( 'L2Rib::Host', macVlanPair )
   host.ciDest = tacCiDest
   host.seqNo = Tac.Value( "Ark::Seqno32", moves )
   host.lastMoveTime = lastMoveTime
   host.preference = Tac.Type( "VxlanController::PreferenceV2" ).switchDynamic
   host.entryType = entryType
   vxlanDynamicL2RibHostInput.addHost( host )

def maybeDelFromLearnedRemoteHostStatus( key ):
   t8( 'maybeDelFromLearnedRemoteHostStatus key=', key )   
   macVlanPair = Tac.Value( "Bridging::MacVlanPair", key.macAddr, key.vlanId )
   if not vxlanDynamicL2RibHostInput.host.has_key( macVlanPair ):
      return
   del vxlanDynamicL2RibHostInput.host[ macVlanPair ]   

def addLearnedHost( bridge, vlan, macAddr, vtep, intf=None,
                    entryType=BridgingEntryType.learnedDynamicMac,
                    moveCount=None ):
   t0( 'addLearnedHost vlan:', vlan, 'macAddr:', macAddr, 'vtep:', vtep,
         'intf:', intf, 'type:', entryType, 'moveCount:', moveCount )

   if not hasattr( vtep, '__iter__' ):
      vtep = [ vtep ]
   # make sure vteps are of the correct type
   vtep = [ Arnet.IpGenAddr( str( v ) ) for v in vtep ]

   # Check that link is up and the VLAN-VNI map exists
   if not bridge.vxlanPort_ or \
      bridge.vxlanPort_.vtiStatus_.linkStatus != 'linkUp':
      t2( 'addLearnedHost: VTI is down - returning' )
      return
   else:
      vtiStatus = bridge.vxlanPort_.vtiStatus_
      vni = None
      if vlan in vtiStatus.vlanToVniMap:
         vniSource = vtiStatus.vlanToVniMap[ vlan ]
         if vniSource and vniSource.vni != Tac.Value( "Vxlan::VniOrNone" ):
            vni = vniSource.vni
      # Check that reverse map exists
      if vni is None or vni not in bridge.vxlanVniVlanMap_:
         t0( 'addLearnedHost: VLAN-VNI map is missing or is not active - returning' )
         return

   vxlanEntryType = mapBridgingEntryType( entryType )
   macVlanPair = Tac.Value( "Vxlan::MacVlanPair", macAddr, vlan )

   # Do not learn VARP MAC
   if str( macVlanPair.macAddr ) == str( vrMacStatus.varpVirtualMac ) and \
      vxlanEntryType != VxlanEntryType.configuredMac:
      t2( "macAddr is a varpMac. Skip learning ", macAddr )
      return
   
   # Get host move details for MAC move handling
   hostMovesCount, hostMoveTime, hostIntf = getHostMoveDetails( bridge, vlan, 
                                                                macAddr )
   updateLearnedHost = False
   # Check for remote to remote MAC moves or entry type changes
   if macVlanPair in vxlanLearnedHost:
      lh = vxlanLearnedHost[ macVlanPair ]
      mvlh = multiVtepLearnedHost.get( macVlanPair )
      if mvlh:
         mvlh = mvlh.remoteVtepList.keys()
      else:
         # For ease of comparison, fake a single multi-vtep entry if none exists
         mvlh = [ lh.remoteVtepAddr ]

      t2( "lh entryType: %s vxlanEntryType: %s" % ( lh.entryType, vxlanEntryType ) )

      # Update if learned hosts have changed
      remoteToRemoteMove = set( mvlh ) != set( vtep )
      # Update when learned VTEP(s) have not changed, but control plane indicates
      # move on learned VTEP(s)
      moveCountUpdate = not remoteToRemoteMove and not moveCount is None and \
            moveCount != hostMovesCount
      # Update for non-configured -> configured MAC move
      confEntry = vxlanEntryType == VxlanEntryType.configuredMac
      confMacMove = lh.entryType != VxlanEntryType.configuredMac and confEntry
      # Update for EVPN <-> received MAC move 
      evpnRcvdMacMove = ( lh.entryType == VxlanEntryType.receivedMac and \
                              vxlanEntryType == VxlanEntryType.evpnMac ) or \
                        ( lh.entryType == VxlanEntryType.evpnMac and \
                              vxlanEntryType == VxlanEntryType.receivedMac )

      # Determine whether MAC move has occurred
      if remoteToRemoteMove or confMacMove or evpnRcvdMacMove or moveCountUpdate:
         t0( "updating learned host - mac: %s, vlan: %s, vtep: %s, entryType: %s" \
               "moveCount: %s, lastMoveTime: %s" % ( macAddr, vlan, vtep,
                  vxlanEntryType, moveCount, hostMoveTime ) )
         # MAC move or entry type change results in removing entry from FdbStatus
         # This breaks remote <-> remote MAC move logic, so update move count here
         if moveCount is None and remoteToRemoteMove and not confEntry:
            moveCount = hostMovesCount + 1
         deleteLearnedHost( bridge, vlan, macAddr,
                            mapVxlanEntryType( lh.entryType ),
                            deleteBridgeMac=True )
         updateLearnedHost = True
   else:
      updateLearnedHost = True

   t2( "updateLearnedHost: %r " % ( updateLearnedHost ) )
   if updateLearnedHost:
      if len( vtep ) > 1:
         newMvlh = multiVtepLearnedHost.newMember( macVlanPair )
         newMvlh.entryType = vxlanEntryType
         for v in vtep:
            newMvlh.remoteVtepList[ v ] = True

      if entryType == "evpnIntfDynamicMac" or \
         entryType == 'evpnIntfStaticMac':
         remoteVtepAddr = Arnet.IpGenAddr( "0.0.0.0" )
      else:
         remoteVtepAddr = vtep[ 0 ]

      lh = Tac.Value( 'Vxlan::LearnedHost', macVlanPair, remoteVtepAddr,
                      vxlanEntryType )
      if bridge.vxlanPort_:
         if entryType == 'evpnIntfDynamicMac' or \
            entryType == 'evpnIntfStaticMac':
            hostIntf = intf
         else:
            hostIntf = bridge.vxlanPort_.intfName_
         # EbraTestBridge already handles incrementing move count when FdbStatus
         # contains MacVlan pair, so local to remote moves are covered
         bridge.learnAddr( vlan, str( macAddr ), hostIntf, entryType,
                           moves=moveCount )
         ( hostMoveCount, hostMoveTime, _ ) = getHostMoveDetails( bridge, vlan,
                                                                  macAddr )
      else:
         t0( 'Vxlan port does not exist. Do not learn addr.' )

      t0( 'added lastMoveTime %s' % hostMoveTime )
      if hostMoveTime:
         lh.lastMoveTime = hostMoveTime
      if not hostMoveCount:
         hostMoveCount = 0

      vxlanLearnedHost.addMember( lh )
      maybeAddToLearnedRemoteHostStatus( macVlanPair, bridge.vxlanPort_.intfName_, 
                                         remoteVtepAddr, entryType, 
                                         lh.lastMoveTime, hostMoveCount )

      for v in vtep:
         incRemoteVtepRef( bridge, v )
         updateRemoteVtep( bridge, v, macVlanPair.vlanId, add=True )
         t0( 'added %s mac %s vlan %d vtep %s refcnt %d' % \
             ( entryType, macVlanPair.macAddr, macVlanPair.vlanId, v,
               bridge.remoteVtepRef_[ v ] ) )
   else:
      t7( "not updating host mac: %s, vlan: %d) " % ( macAddr, vlan ),
            "vtep: %s, entryType: %s" % ( vtep, entryType ) )

# Handler is called when new entry is added to bridging::fdbStatus
# Handles remote to local MAC moves, as remote to remote and local to
# remote MAC moves are handled in preTunnelHandler
def macLearningHandler( vlanId, macAddr, portName, entryType ):
   t4( 'macLearningHandler mac %s vlan %d portName %s entryType %s' % \
      ( macAddr, vlanId, portName, entryType ) )
   
   peerRemote = isPeerRemoteEntry( entryType )
   if vxlanAgentDevs is None:
      return 'dontLearn' if peerRemote else None

   # If peer remote mac, then learn it via vxlan plugin 
   # and return 'dontLearn' so that Etba skips processing this entry
   # portName for peer remote macs is encoded as 
   # "<intf>:<vtepIp>" OR "<intf>:<vtepIp>:<preference>"
   # where "intf" is a vxlan interface such as "Vxlan1"
   #        "vtepIp" is the remote vtep IP and
   #        "preference" is the preference associated with the mac 
   #           entry VxlanController::PreferenceV2
   portNameSplit = portName.split(":")
   if peerRemote:
      # Note that this callback will also be called again while Vxlan plugin
      # is in the process of installing this mac. In that situation, the portName 
      # would have already been appropriately pruned to "Vxlan". If so, just return
      # and let etba do the real learning of peer remote macs
      if len( portNameSplit ) > 1:
         return 'dontLearn'
      else:
         return None

   if entryType == BridgingEntryType.learnedDynamicMac:
      if ( mlagStatusMount.peerLinkIntf and
           mlagStatusMount.peerLinkIntf.intfId == portName ):
         # MAC was learned over the MLAG peer-link. Learning will be ignored
         # by MlagDut therefore MAC should not be deleted from the vxlan entry
         return None
      # Delete the vxlan Entry if this is a remote to local move
      deleteLearnedHost( vxlanAgentDevs.bridge_, vlanId, macAddr )
   return None

def deleteLearnedHost( bridge, vlanId, macAddr,
                       entryType=BridgingEntryType.learnedDynamicMac,
                       deleteBridgeMac=False ):
   t4( 'deleteLearnedHost mac %s vlan %d entryType %s' % \
      ( macAddr, vlanId, entryType ) )

   vxlanEntryType = mapBridgingEntryType( entryType )
   macVlan = Tac.Value( "Vxlan::MacVlanPair", macAddr, vlanId )
   if macVlan in vxlanLearnedHost:
      lh = vxlanLearnedHost[ macVlan ]
      # Do not remove configured MAC
      if ( lh.entryType == VxlanEntryType.configuredMac and 
            vxlanEntryType != VxlanEntryType.configuredMac ):
         t2( "Vxlan not removing host type %s, mac: %s, vlan: %d" % \
               ( lh.entryType, macAddr, vlanId ) )
         return

      t2( "Vxlan removing learned entry mac: %s, vlan: %d, entryType: %s" % \
            ( macAddr, vlanId, lh.entryType ) )

      maybeDelFromLearnedRemoteHostStatus( macVlan )
      del vxlanLearnedHost[ macVlan ]

      # We'll get called back recursively from the bridge code (since this is also
      # the aging handler), but since we've already deleted the learned host, it will
      # be a no-op.
      if deleteBridgeMac:
         fid = bridge.brConfig_.vidToFidMap.get( vlanId )
         if not fid:
            fid = vlanId

         if fid in bridge.brStatus_.fdbStatus:
            bridge.deleteMacAddressEntry( vlanId, str( macAddr ), entryType )

      mvlh = multiVtepLearnedHost.get( macVlan )
      if mvlh:
         vtepList = mvlh.remoteVtepList.keys()
      else:
         vtepList = [ lh.remoteVtepAddr ]
      for vtep in vtepList:
         if decRemoteVtepRef( bridge, vtep ) == 0:
            # VTEP is only removed from VLAN flood-lists when there are no 
            # MACs learned behind VTEP for *ANY* VLAN
            updateRemoteVtep( bridge, vtep, vlanId, add=False )

   if macVlan in multiVtepLearnedHost:
      del multiVtepLearnedHost[ macVlan ]

def updateRemoteVtep( bridge, vtep, vlanId, add=True ):
   """ If the Vxlan device exists, add the remote vtep to the config """
   if bridge.vxlanPort_ and \
         vxlanHwStatusDir.vxlanHwStatus.has_key( bridge.vxlanPort_.intfName_ ):
      hwstatus = vxlanHwStatusDir.vxlanHwStatus[ bridge.vxlanPort_.intfName_ ]
      t2( "%s remote vtep %s vlan %s" %
          ( "adding" if add else "removing", vtep, vlanId ) )
      if add:
         # vxlanHwStatus.remoteVtep indicates the set of active remote VTEPs and 
         # the VLANs they are active in. This set is used to construct dynamic 
         # VXLAN flood-lists when configured.  For a vtep to be configured,
         # vlanList[ vlanId ] has to be set
         newVtep = hwstatus.newRemoteVtep( vtep )
         newVtep.vlanList[ vlanId ] = True
         vtepStatus = vtepHwStatus.vtepStatus.get( VxlanTunnelType.vxlanTunnel )
         if not vtepStatus:
            vtepStatus = vtepHwStatus.newVtepStatus( VxlanTunnelType.vxlanTunnel )
         vtepStatus.vtepList[ vtep ] = 1
      else:
         del hwstatus.remoteVtep[ vtep ]
         vtepStatus = vtepHwStatus.vtepStatus.get( VxlanTunnelType.vxlanTunnel )
         if vtepStatus:
            del vtepStatus.vtepList[ vtep ]

def updateVniToVlanMap( bridge, vxlanStatus, vni, vlanId ):
   t2( "updateVniToVlanMap: vni=", vni, " vlan=", vlanId,
       ' vxlanStatus=', vxlanStatus )

   if vni is not None:
      t2( "updateVniToVlanMap: adding reverse lookup vni %d vlan %d" %
           ( vni, vlanId ) )
      for previousVni in bridge.vxlanVniVlanMap_:
         if bridge.vxlanVniVlanMap_[ previousVni ] == vlanId:
            t2( "deleting previous entry for vni", previousVni, "vlan", vlanId )
            del bridge.vxlanVniVlanMap_[ previousVni ]
            break

      bridge.vxlanVniVlanMap_[ vni ] = vlanId
      if vxlanStatus and vxlanStatus.vlanToLearnRestrict.has_key( vlanId ):
         learnFrom = vxlanStatus.vlanToLearnRestrict[ vlanId ].learnFrom
         prefixes = vxlanStatus.vlanToLearnRestrict[ vlanId ].prefixList
      else:
         learnFrom = 'learnFromDefault'
         prefixes = []

      if vlanId in l2RibFloodSet.vlanFloodSet:
         vteps = getFloodList( vlanId )
      else:
         vteps = []

      if vxlanStatus:
         modifyDefaultLearnRestrict( learnFrom, prefixes, vteps, vlanId )
   else:
      for vniKey in bridge.vxlanVniVlanMap_.keys():
         if bridge.vxlanVniVlanMap_[ vniKey ] == vlanId:
            cleanupRemoteMacs( bridge, vlanId )
            # now its safe to delete the reverse lookup map 
            t2( "updateVniToVlanMap: removing reverse lookup vni %d vlan %d" %
                ( vniKey, vlanId ) )
            del bridge.vxlanVniVlanMap_[ vniKey ]
      modifyDefaultLearnRestrict( 'learnFromDefault', [] , [], vlanId )

   # Some vnis could be removed - might need to unlearn from vteps/mac addresses
   cleanupRemoteMacsLearnRestrict( bridge, vlanId )

   if vni is not None:
      if bridge.vxlanFdbConfigReactor_ is not None:
         bridge.vxlanFdbConfigReactor_.handleConfiguredHosts()
      if bridge.mlagHostTableReactor_ is not None:
         bridge.mlagHostTableReactor_.handleMlagHostEntry( vlanId=vlanId )
      # Hosts learned during VLAN flap must be processed again
      if bridge.l2RibOutputReactor_ is not None:
         bridge.l2RibOutputReactor_.handleLearnedRemoteHosts()

def isPeerRemoteEntry( entryType ):
   if entryType == BridgingEntryType.peerConfiguredRemoteMac or \
         entryType == BridgingEntryType.peerReceivedRemoteMac or \
         entryType == BridgingEntryType.peerLearnedRemoteMac:
      return True
      
   return False

class MlagHostTableReactor( Tac.Notifiee ):
   notifierTypeName = 'Mlag::HostTable'
   def __init__( self, mlagHostTable, bridge ):
      t0( 'MlagHostTableReactor init ' )
      self.mlagHostTable_ = mlagHostTable
      self.bridge_ = bridge
      Tac.Notifiee.__init__( self, mlagHostTable )
      self.handleMlagHostEntry()

   @Tac.handler( 'hostEntry' )
   def handleMlagHostEntry( self, hostKey=None, vlanId=None ):
      if hostKey is None:
         for hKey in self.mlagHostTable_.hostEntry.keys():
            self.handleMlagHostEntry( hKey, vlanId )
         return

      t0( "MlagHostTableReactor::handleMlagHostEntry: mac = %s vlanId = %d" % \
            ( hostKey.address, hostKey.vlanId ) )
      
      if vlanId != None and vlanId != hostKey.vlanId:
         return

      macVlan = Tac.Value( "Vxlan::MacVlanPair",
                           hostKey.address, hostKey.vlanId )
      if hostKey in self.mlagHostTable_.hostEntry.keys():
         hostEntry = self.mlagHostTable_.hostEntry[ hostKey ]
         
         t0( "   handleMlagHostEntry: evaluating add entryType = %s intf=%s" % \
               ( hostEntry.entryType, hostEntry.intf ) )

         if not isPeerRemoteEntry( hostEntry.entryType ):
            return

         # interface name for peer remote macs is encoded as
         # '<intf>:<vtepIp>:<preference>', where preference is optional.
         # IPv6 vtep address will be encoded in full format of
         # 'xxxx:xxxx:xxxx:xxxx:xxxx:xxxx:xxxx:xxxx'. Thus the sub list of
         # the intfNameSplit[ 1 : 9 ] will contain the IPv6 vtep address
         intfNameSplit = hostEntry.intf.split( ":" )
         if len( intfNameSplit ) > 8: # IPv6 address
            addr = ':'.join( intfNameSplit[ 1 : 9 ] )
         else:
            addr = intfNameSplit[ 1 ]
         vtep = Arnet.IpGenAddr( addr )
         addLearnedHost( self.bridge_, hostEntry.vlanId, hostEntry.address,
                         vtep, entryType=hostEntry.entryType,
                         moveCount=hostEntry.moves )
      else:
         fid = self.bridge_.brConfig_.vidToFidMap.get( hostKey.vlanId )
         if not fid:
            fid = hostKey.vlanId

         fdbStatus = self.bridge_.brStatus_.fdbStatus.get( fid )
         if not fdbStatus:
            return
         hostEntry = fdbStatus.learnedHost.get( hostKey.address )
         if ( hostEntry and isPeerRemoteEntry( hostEntry.entryType ) and
              macVlan in vxlanLearnedHost.keys() ):
            lh = vxlanLearnedHost[ macVlan ]
            deleteLearnedHost( self.bridge_, hostKey.vlanId, hostKey.address,
                               mapVxlanEntryType( lh.entryType ),
                               deleteBridgeMac=True )  

def cleanupBumVtepList( bridge ):
   learnStatus = vxlanHwStatusDir.learnStatus
   for macVlan in vxlanHwStatusDir.bumStatus.bumVtepList.keys():
      t2( "bum vtepList del macVlan %s " % macVlan )
      del vxlanHwStatusDir.bumStatus.bumVtepList[ macVlan ]

      vlanId = macVlan.vlanId
      if vlanId in learnStatus.learnStatusList and \
       learnStatus.learnStatusList[ vlanId ].learnFrom == 'learnFromFloodList':
         updateLearnStatusFromFlood( vlanId, [], learnStatus.learnStatusList )
         cleanupRemoteMacsLearnRestrictFlood( bridge, vlanId )

def cleanupMcastGrp():
   for vlan in vxlanHwStatusDir.bumStatus.bumMcastGrp.keys():
      t4( "bum mcast group del vlanId %d " % vlan )
      del vxlanHwStatusDir.bumStatus.bumMcastGrp[ vlan ]

def cleanupReceivedRemoteMacs( bridge, vlanId=None ):
   t4( "cleanupReceivedRemoteMacs" )
   for macVlan in vxlanLearnedHost.keys():
      lh = vxlanLearnedHost[ macVlan ]
      if lh.entryType == VxlanEntryType.receivedMac:
         if ( vlanId is None ) or ( macVlan.vlanId == vlanId ):
            deleteLearnedHost( bridge, macVlan.vlanId, macVlan.macAddr,
                  BridgingEntryType.receivedRemoteMac,
                  deleteBridgeMac=True )  

def cleanupRemoteMacs( bridge, vlanId=None ):
   t4( "cleanupRemoteMacs vlan=", vlanId )
   for macVlan in vxlanLearnedHost.keys():
      if ( vlanId is None or vlanId == macVlan.vlanId ):
         lh = vxlanLearnedHost[ macVlan ]
         deleteLearnedHost( bridge, macVlan.vlanId, macVlan.macAddr,
                            mapVxlanEntryType( lh.entryType ),
                            deleteBridgeMac=True )

def deleteLearnedRemoteMacs( bridge ):
   t4( "deleteLearnedRemoteMacs" )
   for macVlan in vxlanLearnedHost.keys():
      lh = vxlanLearnedHost[ macVlan ]
      if lh.entryType == VxlanEntryType.learnedMac: 
         deleteLearnedHost( bridge, macVlan.vlanId, macVlan.macAddr,
                            mapVxlanEntryType( lh.entryType ),
                            deleteBridgeMac=True )

def modifyDefaultFloodList( vteps, vlanId ):
   t2( 'modifyDefaultFloodList: vlan %d vtep %s' % ( vlanId, [ v for v in vteps ] ) )
   macVlan = Tac.Value( "Vxlan::MacVlanPair",
               vxlanHwStatusDir.bumStatus.bumMacDefault, vlanId )
   if vteps:
      del vxlanHwStatusDir.bumStatus.bumVtepList[ macVlan ]
      vlist = vxlanHwStatusDir.bumStatus.bumVtepList.newMember( macVlan )
      for vtep in vteps:
         vtep = Arnet.IpGenAddr( str( vtep ) )
         # pylint: disable-msg=R1703
         if vtep.af == 'ipv4':
            vlist.remoteVtepAddr[ vtep.v4Addr ] = True
         else:
            vlist.remoteVtepAddr6[ vtep.v6Addr ] = True
   elif vxlanHwStatusDir.bumStatus.bumVtepList.has_key( macVlan ):
      del vxlanHwStatusDir.bumStatus.bumVtepList[ macVlan ]

def getFloodList( vlanId ):
   t2( 'getFloodList: vlan =', vlanId )
   bumStatus = vxlanHwStatusDir.bumStatus
   bumVtepList = bumStatus.bumVtepList
   macVlan = Tac.Value( 'Vxlan::MacVlanPair', bumStatus.bumMacDefault, vlanId )
   if bumVtepList.has_key( macVlan ):
      return bumVtepList[ macVlan ].remoteVtepAddr.keys() + \
             bumVtepList[ macVlan ].remoteVtepAddr6.keys()
   else:
      return []

# Helper functions

def vtepToPrefix( vtep ):
   vtep = Arnet.IpGenAddr( str( vtep ) )
   nmask = '/32' if vtep.af == 'ipv4' else '/128'
   return Tac.Value( 'Vxlan::VtepPrefix',
                     Tac.Value( 'Arnet::IpGenPrefix', str( vtep ) + nmask ) )

def updateLearnStatusFromFlood( vlanId, vteps, learnStatusList ):
   t2( 'updateLearnStatusFromFlood vlanId %d, vteps %s, learnStatusList %s' %
       ( vlanId, vteps, learnStatusList ) )
   # Which ones need to be deleted?
   if learnStatusList[ vlanId ]:
      for old in learnStatusList[ vlanId ].prefixList:
         t2( 'updateLearnStatusFromFlood: check remove prefix=', old )
         if not old.ipPrefix.v4Prefix.address in vteps:
            t4( 'updateLearnStatusFromFlood vlanId %d deleting %s' %
                ( vlanId, old ) )
            del learnStatusList[ vlanId ].prefixList[ old ]
            del learnStatusList[ vlanId ].numMatches[ old ]
   # Which ones to add?
   if vteps:
      for vl in vteps:
         tp = vtepToPrefix( vl )
         t2( 'updateLearnStatusFromFlood: check add addr=', vl )
         if not learnStatusList[ vlanId ].prefixList or \
                not tp in learnStatusList[ vlanId ].prefixList:
            t4( 'updateLearnStatusFromFlood vlanId %d adding %s aka %s' %
                ( vlanId, vl, tp ) )
            learnStatusList[ vlanId ].prefixList[ tp ] = True
            learnStatusList[ vlanId ].numMatches[ tp ] = 0

# Add/replace any existing info
def addDefaultLearnRestrict( learnFrom, prefixes, vteps, vlanId ):
   t2( 'addDefaultLearnRestrict: vlan %d learnFrom %s prefixes %s vteps %s' % \
          ( vlanId, learnFrom, prefixes, vteps ) )
   if vxlanHwStatusDir.learnStatus.learnStatusList.has_key( vlanId ):
      del vxlanHwStatusDir.learnStatus.learnStatusList[ vlanId ]
   if learnFrom == 'learnFromDefault':
      return

   ls = vxlanHwStatusDir.learnStatus.learnStatusList.newMember( vlanId )
   ls.learnFrom = learnFrom
   if learnFrom == 'learnFromList':
      for p in prefixes:
         t2( 'addDefaultLearnRestrict: vlan %d add prefix %s' % ( vlanId, p ) )
         ls.prefixList[ p ] = True
         ls.numMatches[ p ] = 0
   elif learnFrom == 'learnFromFloodList':
      for v in vteps:
         p = vtepToPrefix( v )
         t2( 'addDefaultLearnRestrict: vlan %d add vtep %s' % ( vlanId, p ) )
         ls.prefixList[ p ] = True
         ls.numMatches[ p ] = 0

def modifyDefaultLearnRestrict( learnFrom, prefixes, vteps, vlanId ):
   t2( 'modifyDefaultLearnRestrict: vlan %d learnFrom %s prefixes %s vteps %s' % \
          ( vlanId, learnFrom, prefixes, vteps ) )
   learnList = vxlanHwStatusDir.learnStatus.learnStatusList
   if not learnList.has_key( vlanId ):
      # New entry
      t4( 'modifyDefaultLearnRestrict: new vlan %d' % ( vlanId ) )
      addDefaultLearnRestrict( learnFrom, prefixes, vteps, vlanId )
      return

   if learnFrom == 'learnFromDefault':
      del learnList[ vlanId ]
   elif learnFrom != learnList[ vlanId ].learnFrom:
      # learnFrom changed - replace all
      t4( 'modifyDefaultLearnRestrict: learnFrom change from %s to %s' %
          ( learnList[ vlanId ].learnFrom, learnFrom ) )
      addDefaultLearnRestrict( learnFrom, prefixes, vteps, vlanId )
   elif learnList[ vlanId ].learnFrom == 'learnFromFloodList':
      t4( 'modifyDefaultLearnRestrict: learnFromFloodList' )
      learnList[ vlanId ].learnFrom = learnFrom
      updateLearnStatusFromFlood( vlanId, vteps, learnList )
   elif learnList[ vlanId ].learnFrom == 'learnFromList':
      t4( 'modifyDefaultLearnRestrict: learnFromList' )

      learnList[ vlanId ].learnFrom = learnFrom
      # What needs to be deleted?
      for old in learnList[ vlanId ].prefixList:
         if not old in prefixes:
            t4( 'modifyDefaultLearnRestrict: delete vlan %d prefix %s' %
                ( vlanId, old ) )
            del learnList[ vlanId ].prefixList[ old ]
            del learnList[ vlanId ].numMatches[ old ]
         # What needs to be added?
      for prefix in prefixes:
         if not prefix in learnList[ vlanId ].prefixList:
            t4( 'modifyDefaultLearnRestrict: add vlan %d prefix %s' %
                ( vlanId, prefix ) )
            learnList[ vlanId ].prefixList[ prefix ] = True
            learnList[ vlanId ].numMatches[ prefix ] = 0

# Remove any learned entries which matches VTEPs which we no longer accept
def cleanupRemoteMacsLearnRestrict( bridge, vlanId=None ):
   t2( 'cleanupRemoteMacsLearnRestrict vlan %s' %  vlanId )
   learnStatus = vxlanHwStatusDir.learnStatus
   # Determine which vlans are completely gone or have learnFromList with an empty
   # list. Those will be completely removed from vxlanLearnHost.
   # The vlans with learnFromAny will be skipped.
   deleteVlans = set()
   skipVlans = set()
   for vlan in vxlanHwStatusDir.learnStatus.learnStatusList:
      if learnStatus.learnStatusList[ vlan ].learnFrom == 'learnFromAny':
         skipVlans.add( vlan )
         t4( 'cleanupRemoteMacsLearnRestrict skipVlans %s' %  skipVlans )
      elif learnStatus.learnStatusList[ vlan ].learnFrom == 'learnFromList' and \
             not learnStatus.learnStatusList[ vlan ].prefixList:
         deleteVlans.add( vlan )
         t4( 'cleanupRemoteMacsLearnRestrict deleteVlans %s' %  deleteVlans )

   for macVlan in vxlanLearnedHost.keys():
      if ( vlanId is None or vlanId == macVlan.vlanId ) and \
             not macVlan.vlanId in skipVlans:
         lh = vxlanLearnedHost[ macVlan ]
         delete = False
         if macVlan.vlanId in deleteVlans:
            delete = True
         elif not learnStatus.learnStatusList.has_key( macVlan.vlanId ):
            deleteVlans.add( macVlan.vlanId )
            delete = True
         else:
            delete = not checkLearnFromVtep(
                  macVlan.vlanId, lh.remoteVtepAddr, stats=False )

         t5( 'cleanupRemoteMacsLearnRestrict vlan %d vtep %s delete %s' %  \
                ( macVlan.vlanId, lh.remoteVtepAddr, delete ) )
         if delete:
            deleteLearnedHost( bridge, macVlan.vlanId, macVlan.macAddr,
                               mapVxlanEntryType( lh.entryType ),
                               deleteBridgeMac=True )

# vlanId has learnFromFloodList and the flood list changed. Remove any
# entries we should not have learned with new flood list.
def cleanupRemoteMacsLearnRestrictFlood( bridge, vlanId ):
   t2( 'cleanupRemoteMacsLearnRestrictFlood vlan %s' %  vlanId )

   for macVlan in vxlanLearnedHost.keys():
      if vlanId == macVlan.vlanId:
         lh = vxlanLearnedHost[ macVlan ]
         if not checkLearnFromVtep(
               macVlan.vlanId, lh.remoteVtepAddr, stats=False ):
            t5( 'cleanupRemoteMacsLearnRestrictFlood vlan %d vtep %s delete' % \
                   ( macVlan.vlanId, lh.remoteVtepAddr ) )
            deleteLearnedHost( bridge, macVlan.vlanId, macVlan.macAddr,
                               mapVxlanEntryType( lh.entryType ),
                               deleteBridgeMac=True )

# check of vxlan encapsulated arp and redirect to cpu
def redirectArpToCpu( vlanId, offset, pkt ):
   ( _, headers, _ ) = parsePktStr( pkt[ offset : ] )
   ethHdr = findHeader( headers, "EthHdr" )
   if ethHdr and ethHdr.ethType == "ethTypeArpIp":
      t2( "vxlan encapsulated arp frame. Redirecting to cpu" )
      redirectedPkt = pkt[ 0:12 ] + \
          struct.pack( '!HH', ETH_P_ARISTA_VXLAN_ARP, vlanId ) + \
          pkt[ 12: ]
      vxlanAgentDevs.vxlanTapDevice_.send( redirectedPkt )
      return True
   return False

# check of vxlan encapsulated bfd and redirect to cpu
def redirectBfdToCpu( offset, pkt ):
   ( _, headers, _ ) = parsePktStr( pkt[ offset : ] )
   udpHdr = findHeader( headers, 'UdpHdr' )
   if udpHdr and udpHdr.dstPort == 3784:
      ethHdr = findHeader( headers, "EthHdr" )
      if ethHdr and ethHdr.dst == vxlanAgentDevs.vxlanTapDevice_.hw_:
         t2( "vxlan encapsulated bfd frame. Redirecting inner bfd to cpu" )
         vxlanAgentDevs.vxlanTapDevice_.send( pkt[ offset : ] )
         return True
      else:
         t2( "vxlan encapsulated bfd frame not addressed to this cpu" )
   return False

def handlePreTunnelPkt( bridge, dstMacAddr, data, srcPort ):
   """
   If a vxlan unicast encapsulated packet, decap and return the inner packet with 
   a modified source port of the vxlan interface so learning is properly handled.
   """

   t8( "Vxlan handler for preTunnel packet" )
   retval = ( False, False, False )

   if not Ethernet.isUnicast( dstMacAddr ) and \
         not Ethernet.isMulticast( dstMacAddr ):
      t8( 'received potential BUM packet' )
      return retval

   ( valid, srcVtep, dstVtep, vlan, off ) = \
            validateAndParseVxlanHeader( bridge, data )

   if not valid:
      t8( 'received invalid packet, dropping it' )
      return retval

   # When McastEvpn underlay multicast is configured, etba should bridge
   # only BULL packets.
   # Multicast data packets are bridged/routed by BessMgr, so etba should drop them.
   if Ethernet.isMulticast( dstMacAddr ):
      ( _, headers, _ ) = parsePktStr( data[ off : ] )
      ethHdr = findHeader( headers, "EthHdr" )
      if Ethernet.isMulticast( ethHdr.dst ):
         ipHdr = findHeader( headers, "IpHdr" )
         if ipHdr is None:
            t8( 'received multicast packet with non-ipv4 multicast encapsulation, '
                'dropping it' )
            return retval

         if isinstance( ipHdr.dst, str ):
            ipDst = Arnet.IpAddr( ipHdr.dst )
         else:
            ipDst = ipHdr.dst

         if not ipDst.isReservedMulticast:
            t8( 'received multicast packet with multicast encapsulation, drop it' )
            return retval

   intfName = bridge.vxlanPort_.intfName_
   vtiStatus = \
         bridge.em().entity( 'interface/status/eth/vxlan' ).vtiStatus[ intfName ]

   processInnerFrame( bridge, srcVtep, vlan, off, data )
  
   # txraw now gets complete packet from VxlanSwFwd, it is checked against
   # Vxlan port to not send back to VxlanSwFwd
   if srcPort != bridge.vxlanPort_ and redirectArpToCpu( vlan, off, data ): 
      t8( 'packet is redirected to cpu' )
      return ( False, False, True )

   if srcPort != bridge.vxlanPort_ and redirectBfdToCpu( off, data ):
      t8( 'packet is redirected to cpu' )
      return ( False, False, True )

   # Verify that packet is encapsulated with VARP VTEP IP if and only if the inner
   # source mac is VARP MAC.
   addr = struct.unpack( '>HHH', data[ off : off+6 ] )
   innerDstMacAddr = Tac.Value( "Arnet::EthAddr", addr[0], addr[1], addr[2] )
   if vtiStatus.vArpVtepAddr != '0.0.0.0' and dstVtep.af == 'ipv4':
      dstVtep = dstVtep.v4Addr
      if dstVtep == vtiStatus.vArpVtepAddr and \
         str( innerDstMacAddr ) != str( vrMacStatus.varpVirtualMac ):
         t8( 'dropping packet' )
         t8( 'dstVtep %s is varp vtep %s, but inner dmac %s not varp mac %s' %
               ( dstVtep, vtiStatus.vArpVtepAddr, innerDstMacAddr, 
                 vrMacStatus.varpVirtualMac ) )
         return ( False, False, True )

      if dstVtep != vtiStatus.vArpVtepAddr and \
         str( innerDstMacAddr ) == str( vrMacStatus.varpVirtualMac ):
         t8( 'dropping packet' )
         t8( 'dstVtep %s is not varp vtep %s, but inner dmac %s varp mac %s' %
               ( dstVtep, vtiStatus.vArpVtepAddr, innerDstMacAddr, 
                 vrMacStatus.varpVirtualMac ) )
         return ( False, False, True )

   if not vtiStatus.vArpVtepAddr6.isZero and dstVtep.af == 'ipv6':
      dstVtep = dstVtep.v6Addr
      if dstVtep == vtiStatus.vArpVtepAddr6 and \
         str( innerDstMacAddr ) != str( vrMacStatus.varpVirtualMac ):
         t8( 'dropping packet' )
         t8( 'dstVtep %s is varp vtep %s, but inner dmac %s not varp mac %s' %
               ( dstVtep.stringValue, vtiStatus.vArpVtepAddr6.stringValue,
                 innerDstMacAddr, vrMacStatus.varpVirtualMac ) )
         return ( False, False, True )

      if dstVtep != vtiStatus.vArpVtepAddr6 and \
         str( innerDstMacAddr ) == str( vrMacStatus.varpVirtualMac ):
         t8( 'dropping packet' )
         t8( 'dstVtep %s is not varp vtep %s, but inner dmac %s varp mac %s' %
               ( dstVtep.stringValue, vtiStatus.vArpVtepAddr6.stringValue,
                 innerDstMacAddr, vrMacStatus.varpVirtualMac ) )
         return ( False, False, True )

   decaped = data[ off: ]
   vlanPri = 0
   vlanAct = PKTEVENT_ACTION_ADD
   t8( "applyVlanChanges pri=%s id=%s act=%s" % ( vlanPri, vlan, vlanAct ) )
   data = applyVlanChanges( decaped, vlanPri, vlan, vlanAct )

   bridge.packetContext[ 'vxlanSrcVtep' ] = srcVtep
   return ( data, bridge.vxlanPort_, False )

def floodSuppress( bridge, finalIntfs, srcPort=None, dropReasonList=None,
                   vlanId=None, dstMacAddr=None, data=None ):
   srcVtep = bridge.packetContext.get( 'vxlanSrcVtep' )
   if not srcVtep:
      # Not a VXLAN packet.
      return

   if not dstMacAddr:
      return

   if Ethernet.isUnicast( dstMacAddr ):
      # TODO: With VXLAN GPE support, the B bit can make a more accurate assessment
      # of whether this was known or unknown unicast on the source VTEP.
      if bridge.destLookup( vlanId, dstMacAddr )[ 0 ]:
         # Destination is known. Assume known unicast on the source VTEP too, in
         # which case finalIntfs is already correct.
         return

   vlanFloodStatus = evpnStatus.vlanFloodStatus.get( vlanId )
   if vlanFloodStatus:
      peerFloodStatus = vlanFloodStatus.peerFilter.get( srcVtep )

      # Mutate existing list in a delete-over-iterator safe way.
      for intf in finalIntfs.keys():
         if intf in vlanFloodStatus.floodFilter or (
               peerFloodStatus and intf in peerFloodStatus.floodFilter ):
            del finalIntfs[ intf ]

def handleDecapPkt( bridge, routed, vlanId, dstMacAddr, data, srcPortName ):
   """ If the received packet is intended for Vxlan decap, learn the remote
   address """

   retval = ( False, False )

   t8( 'Vxlan Decap packet processing' )
   if not routed:
      t7( 'received non-decap packet' )
      return retval

   ( valid, srcVtep, _, vlan, off ) = \
         validateAndParseVxlanHeader( bridge, data )
   if not valid:
      return retval

   processInnerFrame( bridge, srcVtep, vlan, off, data )
   return retval 

def rewriteSharedRouterDstMac( bridge, routed, vlanId,
                               dstMacAddr, data, srcPortName ):
   """If the received packet dst MAC is MLAG shared router MAC,
      rewrite it to bridge MAC"""

   retval = ( False, False )

   t8( 'MLAG Shared Router MAC processing' )
   if bridge.sharedRouterMac_ and dstMacAddr == bridge.sharedRouterMac_:
      t2( 'Frame with shared router MAC destination. Sending to bridgeMac instead' )
      newDstMac = struct.pack( '>BBBBBB', *( int( n, 16 ) for n in
                                          bridge.bridgeMac_.split( ':' ) ) )
      data = newDstMac + data[ 6: ]
      retval = ( data, bridge.bridgeMac_ )
   return retval

def destLookup( bridge, vlanId, destMac, data ):
   fabricMacs = { vrMacStatus.varpVirtualMac: 'varpMac',
                  bridge.sharedRouterMac_: 'sharedRouterMac' }
   if destMac in fabricMacs:
      t2( "dest mac matches the", fabricMacs[ destMac ], destMac )
      return( True, [ 'Cpu', ] )
   return( False, None )

def learning( vlanId, macAddr, portName, entryType ):
   # disable learning varpMac
   if macAddr == vrMacStatus.varpVirtualMac:
      t2( "macAddr is a varpMac. Skip learning ", macAddr )
      return 'dontLearn'
   return 'learn'

class L2RibDestReactor( Tac.Notifiee ):
   notifierTypeName = 'L2Rib::DestOutput'

   def __init__( self, l2RibDest_, hostReactor ):
      t8( 'L2RibDestReactor initialized' )
      self.l2RibDest_ = l2RibDest_
      self.hostReactor = hostReactor
      Tac.Notifiee.__init__( self, self.l2RibDest_ )

   def __del__( self ):
      t8( 'L2RibDestReactor destroyed' )
   
   @Tac.handler( 'dest' )
   def handleDest( self, key ):
      t8( 'handle destId ', key )
      if key not in self.hostReactor.destIdBacklog:
         t8( 'no hosts' )
         return
      backlog = self.hostReactor.destIdBacklog[ key ]
      self.hostReactor.destIdBacklog[ key ] = []
      for host in backlog:
         t8( 'update host', host )
         self.hostReactor.handleLearnedRemoteHosts( key=host )

class L2RibLBReactor( Tac.Notifiee ):
   notifierTypeName = 'L2Rib::LoadBalanceOutput'

   def __init__( self, l2RibLoadBalance_, hostReactor ):
      t8( 'L2RibLBReactor initialized' )
      self.l2RibLoadBalance_ = l2RibLoadBalance_
      self.hostReactor = hostReactor
      Tac.Notifiee.__init__( self, self.l2RibLoadBalance_ )

   def __del__( self ):
      t8( 'L2RibLBReactor destroyed' )
   
   @Tac.handler( 'lb' )
   def handleLoadBalance( self, key ):
      t8( 'handle lbId ', key )
      if key not in self.hostReactor.hostsByLbId:
         t8( 'no hosts' )
         return
      for host in self.hostReactor.hostsByLbId[ key ]:
         t8( 'update host', host )
         self.hostReactor.handleLearnedRemoteHosts( key=host )

class L2RibOutputReactor( Tac.Notifiee ):
   notifierTypeName = 'L2Rib::HostOutput'

   def __init__( self, l2RibOutput_, l2RibLoadBalance_, l2RibDest_, bridge ):
      assert bridge
      t8( 'L2RibOutputReactor initialized' )
      self.bridge_ = bridge
      self.l2RibOutput_ = l2RibOutput_
      self.hosts_ = self.l2RibOutput_.host
      self.l2RibLoadBalance_ = l2RibLoadBalance_
      self.loadBalance_ = self.l2RibLoadBalance_.lb
      self.l2RibDest_ = l2RibDest_
      self.dest_ = self.l2RibDest_.dest
      Tac.Notifiee.__init__( self, self.l2RibOutput_ )
      self.hostsByLbId = {}
      self.lbIdByHost = {}
      self.lbReactor = L2RibLBReactor( l2RibLoadBalance_, self )
      self.destIdBacklog = {}
      self.destReactor = L2RibDestReactor( l2RibDest_, self )
      self.hostSource = []
      self.handleLearnedRemoteHosts()

   def __del__(self):   
      t8( 'L2RibOutputReactor destroyed' )

   def getHostEntryType( self, key=None ):
      fid = self.bridge_.brConfig_.vidToFidMap.get( key.vlanId )
      if not fid:
         fid = key.vlanId

      fdbStatus = self.bridge_.brStatus_.fdbStatus[ fid ]
 
      learnedHosts = fdbStatus.learnedHost
      hostEntry = learnedHosts.get( key.macaddr )
      if hostEntry:
         return hostEntry.entryType
      else:
         return None

   @Tac.handler( 'host' )
   def handleHost( self, hostKey ):
      t8( 'handleHost key mac %s vlan %d' % ( hostKey.macaddr, hostKey.vlanId ) )
      self.handleLearnedRemoteHosts( key=hostKey )

   def checkDest( self, nt, key ):
      vtepAddrs = []
      intf = None
      destType = 'notVxlan'
      d = self.dest_.get( nt.destId() )
      if not d:
         t0( 'dest entry not ready yet' )
         if not nt.destId() in self.destIdBacklog:
            self.destIdBacklog[ nt.destId() ] = []
         self.destIdBacklog[ nt.destId() ].append( key )
      elif d.destType == 'destTypeVxlan':
         destType = d.destType
         vtepAddrs.append( d.vxlan.vtepAddr )
      elif d.destType == 'destTypeIntf':
         destType = d.destType
         intf = d.intf.intfId
      return ( vtepAddrs, destType, intf )

   def checkLoadBalance( self, nt, key ):
      vtepAddrs = []
      destType = 'notVxlan'
      lb = self.loadBalance_.get( nt.lbId() )
      if not lb:
         if not nt.lbId() in self.hostsByLbId:
            self.hostsByLbId[ nt.lbId() ] = []
         t8( "Adding host %s to lbId" % key, nt.lbId() )
         self.hostsByLbId[ nt.lbId() ].append( key )
         self.lbIdByHost[ key ].append( nt.lbId() )
         t0( 'lb entry not ready yet' )
         return ( vtepAddrs, destType )
      for n in lb.lb.next.values():
         if n.tableType == 'tableTypeLoadBalance':
            vts, dT = self.checkLoadBalance( n, key )
         elif n.tableType == 'tableTypeDest':
            vts, dT, _ = self.checkDest( n, key )
         else:
            continue
         vtepAddrs.extend( vts )
         destType = dT
      if destType == 'destTypeVxlan':
         if not nt.lbId() in self.hostsByLbId:
            self.hostsByLbId[ nt.lbId() ] = []
         t8( "Adding host %s to lbId" % key, nt.lbId() )
         self.hostsByLbId[ nt.lbId() ].append( key )
         self.lbIdByHost[ key ].append( nt.lbId() )
      return ( vtepAddrs, destType )

   
   def handleLearnedRemoteHosts( self, key=None ):
      keys = self.hosts_.keys() if key is None else [ key ]

      for k in keys:
         # verify that the host has been removed from l2RibOutput
         # if so remove it from fdbStatus
         host = self.hosts_.get( k )
         t8( 'handleLearnedRemote host', k )
         if not host:
            if k in self.hostSource:
               t5( 'Deleting evpn found for key %s/%s' % ( k.macaddr,
                                                           k.vlanId ) )
               fid = self.bridge_.brConfig_.vidToFidMap.get( k.vlanId )
               if not fid:
                  fid = k.vlanId

               if fid in self.bridge_.brStatus_.fdbStatus:
                  entryType = self.getHostEntryType( k )
                  t5( 'Host entry type for key %s/%s is %s' % ( k.macaddr,
                                                                k.vlanId,
                                                                entryType ) )
                  if entryType and isEntryTypeController( entryType ):
                     deleteLearnedHost(
                        self.bridge_, k.vlanId, k.macaddr,
                        entryType, deleteBridgeMac=True )
               self.hostSource.remove( k )
            t8( 'skipping host, not found' )
            continue
         
         vtepAddrs = []
         intf = None
         destType = 'notVxlan'
         # clear old lb info
         lbs = self.lbIdByHost.get( key )
         if lbs:
            for lb in lbs:
               hosts = self.hostsByLbId.get( lb )
               hosts.remove( key )
         self.lbIdByHost[ key ] = []
         # Traverse the multi-table model if next is valid
         # L2Rib guarantees that the install order is correct, i.e. a host is not
         # installed if the ObjTuple that  host.next points to isn't also installed,
         # so we don't have to worry about incomplete L2Rib routes.
         if host.next.isValid():
            t0( 'Checking multitable host for Vxlan' )
            nextTuple = host.next
            if nextTuple.tableType == 'tableTypeLoadBalance':
               vtepAddrs, destType = self.checkLoadBalance( nextTuple, key )
            elif nextTuple.tableType == 'tableTypeDest':
               vtepAddrs, destType, intf = self.checkDest( nextTuple, key )
         else:
            destType = host.ciDestType
            vtepAddrs = host.ciDest.vtepAddr.values()
            intf = host.ciDest.intf
         t5( 'host source %s destType %s entryType %s' % ( host.source,
             destType, host.entryType ) )
         if ( host.source in ( 'sourceBgp', 'sourceVcs' ) and
              destType in ( 'destTypeVxlan', 'destTypeIntf' ) ):
            assert( host.entryType == 'evpnDynamicRemoteMac' or
                    host.entryType == 'evpnConfiguredRemoteMac' or
                    host.entryType == 'evpnIntfDynamicMac' or
                    host.entryType == 'evpnIntfStaticMac' or
                    host.entryType == 'receivedRemoteMac' )
            self.hostSource.append( key )
            vtepAddrs = [ addr for addr in vtepAddrs if not addr.isAddrZero ] 
            if destType == 'destTypeVxlan':
               assert vtepAddrs, 'remote vtep is zero'
            t4( 'source %s for key %s/%s tunnelIpAddr %s' % ( 
                  host.source, host.macAddr, host.vlanId, vtepAddrs ) )

            addLearnedHost( self.bridge_,
                            host.vlanId,
                            host.macAddr,
                            vtepAddrs,
                            intf=intf,
                            entryType=host.entryType,
                            moveCount=host.seqNo if host.seqNo else None )

class L2RibVlanFloodSetReactor( Tac.Notifiee ):
   notifierTypeName = 'L2Rib::FloodSetOutput'
   def __init__( self, l2RibFloodSetOutput, bridge ):
      t2( 'L2RibVlanFloodSetReactor::init' )
      self.l2RibFloodSet = l2RibFloodSetOutput
      self.bridge_ = bridge
      self.l2RibFloodSetReactor_ = {}
      Tac.Notifiee.__init__( self, self.l2RibFloodSet )
      self.handleVlanFloodSets()

   @Tac.handler( 'source' )
   def handleSource( self ):
      t2( 'L2RibVlanFloodSetReactor::handleSource %s' % self.l2RibFloodSet.source )
      if self.bridge_.vxlanPort_ and self.bridge_.vxlanPort_.vtiStatusReactor_:
         self.bridge_.vxlanPort_.vtiStatusReactor_.computeReplicationMode()

   @Tac.handler( 'vlanFloodSet' )
   def handleVlanFloodSet( self, vlanId ):
      t2( 'L2RibVlanFloodSetReactor::handleVlanFloodSet for vlan=%s' % vlanId )
      if vlanId not in self.l2RibFloodSet.vlanFloodSet:
         # If flood-list is empty, we may need to transition to multicast
         if not self.l2RibFloodSet.vlanFloodSet:
            if self.bridge_.vxlanPort_ and self.bridge_.vxlanPort_.vtiStatusReactor_:
               self.bridge_.vxlanPort_.vtiStatusReactor_.computeReplicationMode()
         return

      t2( 'handleVlanFloodSet create a new reactor for vlan=%d' % \
          ( vlanId ) )

      # When vlanFoodSet is deleted, L2RibFloodSetReactor will be destroyed
      Tac.handleCollectionChange( L2RibFloodSetReactor, vlanId,
                                  self.l2RibFloodSetReactor_,
                                  self.notifier_.vlanFloodSet,
                                  reactorArgs=( self.bridge_, ),
                                  reactorTakesKeyArg=True,
                                  reactorFilter=None )

   def handleVlanFloodSets( self ):
      t2( 'L2RibVlanFloodSetReactor::handleVlanFloodSets' )
      for vlanId in self.l2RibFloodSet.vlanFloodSet:
         self.handleVlanFloodSet( vlanId )

   def delAllFloodSetReactors( self ):
      t2( 'L2RibVlanFloodSetReactor::delAllFloodSetReactors' )
      #pylint: disable=consider-iterating-dictionary
      for fs in self.l2RibFloodSetReactor_.keys():
         del self.l2RibFloodSetReactor_[ fs ]

   def __del__( self ):
      t2( 'L2RibVlanFloodSetReactor::del' )
      self.delAllFloodSetReactors()

class L2RibFloodSetReactor( Tac.Notifiee ):
   notifierTypeName = 'L2Rib::VlanFloodSet'
   def __init__( self, vlanFloodSet, vlanId, bridge ):
      t2( 'L2RibFloodSetReactor::init' )
      self.vlanFloodSet = vlanFloodSet
      self.vlanId = vlanId
      self.bridge_ = bridge
      self.bumStatus_ = vxlanHwStatusDir.bumStatus
      self.l2RibDestSetReactor_ = {}
      Tac.Notifiee.__init__( self, self.vlanFloodSet )
      self.handleFloodSets()

   @Tac.handler( 'floodSet' )
   def handleFloodSet( self, macAddr ):
      t2( 'L2RibFloodSetReactor::handleFloodSet for mac=%s' % macAddr )
      if macAddr not in self.vlanFloodSet.floodSet:
         return

      t2( 'handleFloodSet create a new reactor for vlan=%d macAddr=%s' % \
          ( self.vlanId, macAddr ) )
      # When floodSet is deleted, L2RibDestSetReactor will be destroyed
      Tac.handleCollectionChange( L2RibDestSetReactor, macAddr,
                                  self.l2RibDestSetReactor_,
                                  self.notifier_.floodSet,
                                  reactorArgs=( self.vlanId, 
                                                self.bridge_ ),
                                  reactorTakesKeyArg=True,
                                  reactorFilter=None )

      # Handle updates to learnStatus
      learnStatus = vxlanHwStatusDir.learnStatus
      if self.vlanId in learnStatus.learnStatusList and \
         learnStatus.learnStatusList[ self.vlanId ].learnFrom == \
            'learnFromFloodList':
         # Get vteps for flood-list
         vteps = []
         if macAddr in self.vlanFloodSet.floodSet:
            for dest in self.vlanFloodSet.floodSet[ macAddr ].destSet:
               vteps.append( dest.vxlan.vtepAddr.stringValue )
         updateLearnStatusFromFlood( self.vlanId, vteps, 
               learnStatus.learnStatusList )
         cleanupRemoteMacsLearnRestrictFlood( self.bridge_, self.vlanId )

   def handleFloodSets( self ):
      t2( 'L2RibFloodSetReactor::handleFloodSets' )
      for macAddr in self.vlanFloodSet.floodSet:
         self.handleFloodSet( macAddr )

   def delAllDestSetReactors( self ):
      t2( 'L2RibFloodSetReactor::delAllDestSetReactors' )
      #pylint: disable=consider-iterating-dictionary
      for ds in self.l2RibDestSetReactor_.keys():
         del self.l2RibDestSetReactor_[ ds ]

   def close( self ):
      t2( 'L2RibFloodSetReactor::close' )
      # Remove bumVtepList entries when SM is deleted
      for fs in self.vlanFloodSet.floodSet:
         macVlan = Tac.Value( 'Vxlan::MacVlanPair', fs, self.vlanId )
         bumVtepList = self.bumStatus_.bumVtepList
         if macVlan in bumVtepList:
            bumVtepList[ macVlan ].remoteVtepAddr.clear()
            bumVtepList[ macVlan ].remoteVtepAddr6.clear()
            del bumVtepList[ macVlan ]
      # Remove destSet reactors
      self.delAllDestSetReactors()
      t2( 'L2RibFloodSetReactor::close complete' )
      Tac.Notifiee.close( self )

class L2RibDestSetReactor( Tac.Notifiee ):
   notifierTypeName = 'L2Rib::FloodSet'
   def __init__( self, floodSet, macAddr, vlanId, bridge ):
      t2( 'L2RibDestSetReactor::init' )
      self.floodSet = floodSet
      self.macVlan_ = Tac.Value( 'Vxlan::MacVlanPair', macAddr, vlanId )
      self.bridge_ = bridge
      self.bumStatus_ = vxlanHwStatusDir.bumStatus
      self.bumVtepList_ = self.bumStatus_.bumVtepList.get( self.macVlan_ )
      Tac.Notifiee.__init__( self, self.floodSet )
      self.handleFloodSets()

   @Tac.handler( 'destSet' )
   def handleFloodSet( self, dest ):
      t2( 'L2RibDestSetReactor::handleFloodSet for dest=%s' % dest )

      if dest.destType != 'destTypeVxlan':
         return
      # If static flood-lists are added while we are in multicast mode,
      # recompute replication mode so that we transition to HER
      if vxlanHwStatusDir.bumStatus.bumReplicationMode in ( 
            BumReplicationMode.multicast, BumReplicationMode.unknown ):
         t2( 'L2RibDestSetReactor::handleFloodSet recompute replication mode' )
         # Transition to HER if vxlanStatic source is set and vlanFloodSets exist
         if l2RibFloodSet.vlanFloodSet and bool( 
               l2RibFloodSet.source & vxlanStaticSource ):
            cleanupMcastGrp()
            vxlanHwStatusDir.bumStatus.bumReplicationMode = \
                  BumReplicationMode.headendNonVcs
            t2( ( 'L2RibDestSetReactor::handleFloodSet replication mode recomputed'
                  ' as headendNonVcs' ) )
         else:
            return

      ip = dest.vxlan.vtepAddr
      vtepType = dest.vxlan.vtepType
      t2( 'vtepType=%s' % vtepType )
      if dest in self.floodSet.destSet:
         if not self.bumVtepList_:
            self.bumVtepList_ = self.bumStatus_.bumVtepList.newMember(
            self.macVlan_ )
         t2( ( 'L2RibDestSetReactor::handleFloodSet add vtep=%s to bumVtepList '
               'for mac=%s vlan=%s' ) % ( ip, self.macVlan_.macAddr,
                                          self.macVlan_.vlanId ) )
         # pylint: disable-msg=R1703
         if ip.af == 'ipv4':
            self.bumVtepList_.remoteVtepAddr[ ip.v4Addr ] = True
         else:
            self.bumVtepList_.remoteVtepAddr6[ ip.v6Addr ] = True
      else:
         if self.bumVtepList_:
            # Check whether another vtepType is present in the destSet collection
            # and only delete if no other vtepType exists for a remote vtep IP before
            # deleting it from the BumVtepList.  We need to do this because L2Rib
            # blindly copies all input flood set dests to the output flood set
            # which means that  we can have multiple VTEP types for a single VTEP IP
            vtepTypeNode = Tac.typeNode( "L2Rib::VtepType" )
            for attr in vtepTypeNode.attributeQ:
               vtepType = attr.enumValue
               otherVtepTypeDest = Tac.ValueConst(
                  'L2Rib::Dest', destIdNotInSmash, destType='destTypeVxlan',
                  vxlan=Tac.ValueConst( 'L2Rib::DestDataVxlan', ip, vtepType ) )
               if otherVtepTypeDest in self.floodSet.destSet:
                  t2( 'Dest with vtepType %s still exists, not deleting' %
                      attr.name )
                  return

            t2( ( 'L2RibDestSetReactor::handleFloodSet remove vtep=%s from '
                  'bumVtepList for macVlan=%s' ) % (
                     ip, self.macVlan_ ) )
            if ip.af == 'ipv4':
               del self.bumVtepList_.remoteVtepAddr[ ip.v4Addr ]
            else:
               del self.bumVtepList_.remoteVtepAddr6[ ip.v6Addr ]
            if ( len( self.bumVtepList_.remoteVtepAddr ) +
                 len( self.bumVtepList_.remoteVtepAddr6 ) == 0 ):
               t2( ( 'L2RibDestSetReactor::handleFloodSet removing bumVtepList for'
                     ' mac=%s vlan=%s' ) % ( self.macVlan_.macAddr, 
                                           self.macVlan_.vlanId ) )
               del vxlanHwStatusDir.bumStatus.bumVtepList[ self.macVlan_ ]

   def handleFloodSets( self ):
      t2( 'L2RibDestSetReactor::handleFloodSets' )
      for dest in self.floodSet.destSet:
         self.handleFloodSet( dest )

   def __del__( self ):
      t2( 'L2RibDestSetReactor::del' )

class EtbaStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Bridging::Etba::Status'
   def __init__( self, etbaStatus, bridge ):
      t0( 'EtbaStatusReactor: __init__ ' )
      assert bridge
      self.etbaStatus_ = etbaStatus
      self.bridge_ = bridge
      Tac.Notifiee.__init__( self, etbaStatus )
      self.handleEtbaStatusInitialized()

   @Tac.handler( 'initialized' )
   def handleEtbaStatusInitialized( self ):
      t0( 'handleEtbaStatusInitialized initialized %d' %  \
            ( self.etbaStatus_.initialized ) )

      if not self.etbaStatus_.initialized:
         return

      bridge = self.bridge_
      em = bridge.em()
      assert em

      # Don't set up a port until the VTI is up
      bridge.vxlanIntfStatusReactor_ = VxlanIntfStatusReactor( bridge )
      bridge.vxlanIntfConfigReactor_ = VxlanIntfConfigReactor( bridge )

      brConfig = em.entity( 'bridging/config' )
      bridge.vxlanBrFdbConfigReactor_ = \
                     Tac.collectionChangeReactor( brConfig.fdbConfig,
                                 FdbConfigReactor, reactorArgs=( bridge, ) )

      bridge.vxlanFdbConfigReactor_ = \
        VxlanFdbConfigReactor( vxlanConfigDir.fdbConfig, bridge )

      bridge.vxlanVlanStatusReactor_ = \
         VxlanVlanStatusReactor( em.entity( 'bridging/vlan/status' ), 
               em.entity( 'vxlan/status' ), bridge )

      bridge.mlagHostTableReactor_ = \
            MlagHostTableReactor( em.entity( 'mlag/hostTable' ), bridge )

      bridge.evpnMacTableReactor_ = \
        EvpnMacTableReactor( em.entity( 'vxlan/input/macaddr/IpRib' ), bridge )

      bridge.l2RibOutputReactor_ = L2RibOutputReactor( l2RibOutput, 
                                                       l2RibLoadBalance, 
                                                       l2RibDest, bridge )

      bridge.l2RibVlanFloodSetReactor_ = \
        L2RibVlanFloodSetReactor( em.entity( 'bridging/l2Rib/floodOutput' ), bridge )

def bridgeInit( bridge ):
   t2( "Vxlan plugin bridgeInit" )
   assert bridge
   assert mountCompleted, 'Vxlan mounts not completed '

   # initialize Vxlan specific data structures attached to the bridge
   bridge.vxlanVniVlanMap_ = {}
   bridge.remoteVtepRef_ = {}
   bridge.vxlanPort_ = None
   bridge.vxlanFdbConfigReactor_ = None
   bridge.vxlanVlanStatusReactor_ = None
   bridge.controllerControlPlane_ = False
   bridge.mlagHostTableReactor_ = None
   bridge.evpnMacTableReactor_ = None
   bridge.l2RibOutputReactor_ = None
   bridge.l2RibVlanFloodSetReactor_ = None
   bridge.l2RibOutput_ = l2RibOutput
   bridge.vrMacStatus_ = vrMacStatus
   bridge.sharedRouterMac_ = None

   em = bridge.em()
   assert em
   bridge.etbaStatusReactor_ = \
         EtbaStatusReactor( em.entity( 'bridging/etba/status' ), bridge )
   bridge.mlagStatus_ = em.entity( 'mlag/status' )

def vxlanMountCompleted():
   t0( 'All mounts completed ' )
   global mountCompleted
   mountCompleted = True

   # Finish setting up the globals
   global vxlanLearnedHost, multiVtepLearnedHost
   vxlanLearnedHost = vxlanHwStatusDir.fdbStatus.learnedHost
   multiVtepLearnedHost = vxlanHwStatusDir.fdbStatus.multiVtepLearnedHost
   assert vxlanLearnedHost is not None
   assert multiVtepLearnedHost is not None
   bridgingHwCapabilities.vxlanMcastGroupSupported = True
   bridgingHwCapabilities.vxlanBfdSupported = True
   bridgingHwCapabilities.vxlanLogicalRouterSupported = True
   bridgingHwCapabilities.vxlanNdSnoopingSupported = True
   bridgingHwCapabilities.vxlanIpv6UnderlaySupported = True
   bridgingHwCapabilities.vxlanTxRawSupported = True
   bridgingHwCapabilities.vxlanEncapIpv6ConfigSupported = True
   bridgingHwCapabilities.vxlanVtepToVtepBridgingSupported = True
   bridgingHwCapabilities.vtepClassificationBridgingSupported = True
   if Toggles.VxlanToggleLib.toggleMultiVxlanTunnelInterfacesEnabled():
      bridgingHwCapabilities.vxlanMaxTunnelInterfacesSupported = 8
   bridgingHwCapabilities.vxlanMultiVtepMlagSupported = True
   bridgingHwCapabilities.vxlanDecapFilterSupported = False

def agentInit( em ):
   t2( "Vxlan plugin agentInit" )

   mg = em.mountGroup()
   shmemEm = SharedMem.entityManager( sysdbEm=em )

   # interface/{config,status}/eth/intf is mounted by the etba, no need to do it here

   # React to the mlag Host table
   mg.mount( 'mlag/hostTable', 'Mlag::HostTable', "rO" )

   # React to this dir to get the vxlan1 up event
   mg.mount( "interface/status/eth/vxlan", "Vxlan::VtiStatusDir", "r" )

   # React to vlan changes
   mg.mount( "bridging/vlan/status", "Bridging::VlanStatusDir", "r" )

   global vrMacStatus
   vrMacStatus = mg.mount( "routing/fhrp/vrMacStatus",
                           "Routing::Fhrp::VirtualRouterMacStatus", "r" )

   # this is set by us to learn the user configured vxlan dmac+vlans
   # needs to be set during init time and during when decap
   global vxlanHwStatusDir
   vxlanHwStatusDir = \
         mg.mount( "vxlan/hardware/status", "Vxlan::VxlanHwStatusDir", "w" )

   global bridgingHwCapabilities
   bridgingHwCapabilities = \
       mg.mount( "bridging/hwcapabilities", "Bridging::HwCapabilities", "w" )

   global vxlanConfigDir
   vxlanConfigDir = mg.mount( "vxlan/config", "Vxlan::VxlanConfigDir", "r" )

   mg.mount( "vxlan/status", "Vxlan::VxlanStatusDir", "r" )

   mg.mount( "vxlan/input/macaddr/IpRib", "Bridging::RemoteMacTable", "rO" )

   # React to etba status
   mg.mount( "bridging/etba/status", "Bridging::Etba::Status", "r" )

   # React to Mlag status
   global mlagStatusMount
   # Mount mlag/status, Mlag::Status and its dependent paths
   mlagStatusMount = MlagMountHelper.mountMlagStatus( mg )

   global vxlanDynamicL2RibHostInput
   smashTableLen = 16 * 1024
   vxlanDynamicL2RibHostInput = \
      shmemEm.doMount( "bridging/l2Rib/vxlanDynamic/hostInput", "L2Rib::HostInput",
                       Smash.mountInfo( 'writer', [ ( "host", smashTableLen ) ] ) )

   global l2RibOutput
   l2RibOutput = shmemEm.doMount( "bridging/l2Rib/hostOutput", "L2Rib::HostOutput",
                                  Smash.mountInfo( 'keyshadow' ) )
   global l2RibLoadBalance
   l2RibLoadBalance = shmemEm.doMount( "bridging/l2Rib/lbOutput",
                                   "L2Rib::LoadBalanceOutput",
                                   Smash.mountInfo( 'keyshadow' ) )
   global l2RibDest
   l2RibDest = shmemEm.doMount( "bridging/l2Rib/destOutput",
                            "L2Rib::DestOutput",
                            Smash.mountInfo( 'keyshadow' ) )
   global decapConfig
   decapConfig = shmemEm.doMount( "routing/multicast/tunnel/ip/decapConfig",
                              "Routing::Multicast::IpTunnelDecapConfig",
                              Smash.mountInfo( 'keyshadow' ) )
   global l2RibFloodSet
   l2RibFloodSet = mg.mount( 'bridging/l2Rib/floodOutput',
                             'L2Rib::FloodSetOutput', 'r' )

   global evpnStatus
   evpnStatus = mg.mount( 'evpn/status', 'Evpn::EvpnStatus', 'r' )

   global remoteVtepConfigDir
   remoteVtepConfigDir = mg.mount( 'vxlan/remoteVtepHwConfig',
                                   'Vxlan::RemoteVtepHwConfigDir', 'r' )

   global vtepHwStatus
   vtepHwStatus = mg.mount( 'vxlan/vtepHwStatus',
                            'Vxlan::VtepHwStatusDir', 'w' )

   def onMountComplete():
      vxlanMountCompleted()
   mg.close( onMountComplete )

def Plugin( ctx ):
   t2( "Vxlan plugin registering" )
   ctx.registerInterfaceHandler( 'Vxlan::VtiStatus', VxlanIntfPort )
   ctx.registerBridgeInitHandler( bridgeInit )
   ctx.registerAgentInitHandler( agentInit )
   ctx.registerPostAgingHandler( deleteLearnedHost )
   ctx.registerLearningHandler( macLearningHandler )
   bpfFilterInfo = ( "Vxlan", "" )
   if not EtbaDutToggleLib.toggleFastEtbaEnabled():
      ctx.registerRewritePacketOnTheWayToTheCpuHandler( handleDecapPkt )
      ctx.registerRewritePacketOnTheWayToTheCpuHandler( rewriteSharedRouterDstMac )
      ctx.registerDestLookupHandler( destLookup )
      ctx.registerPacketReplicationHandler( floodSuppress )
      bpfFilterInfo = None

   # bpf filter will be generated when Vxlan port becomes active
   ctx.registerPreTunnelHandler( handlePreTunnelPkt, bpfFilterInfo=bpfFilterInfo )
   ctx.registerLearningHandler( learning )
