#!/usr/bin/env python
# Copyright (c) 2008, 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import struct
import Tac, Tracing, random
from Arnet import PktParserTestLib
from eunuchs.if_ether_h import ETH_HLEN
from if_ether_arista import ETH_P_ARISTA_SFLOW_ETBA, \
                            SFLOW_TRAILER_EGRESS_PKT, \
                            SFLOW_TRAILER_INPUT_INTERFACE_PRESENT, \
                            SFLOW_TRAILER_OUTPUT_INTERFACE_PRESENT
import SflowConst
import array
from Toggles import EtbaDutToggleLib

handle = Tracing.Handle( 'SflowEtbaPlugin' )
t0 = handle.trace0
t2 = handle.trace2
t4 = handle.trace4
t8 = handle.trace8

def syncStatusSampleRate():
   """Calculate 'kinda' the closest power of two that the hardware platform
   supposedly uses.  Sflow samples sent to the collector 
   will feed off the hwStatus value."""
   sampleRate = sflowHwConfig.sampleRate
   closestPower = 1
   bits = range ( 1, 31 )
   bits.reverse()
   # keep only most significant bit
   # it's a Floor op, not actually 'closest'
   for bit in bits:
      if ( sampleRate >> bit ) & 0x1:
         closestPower = 1 << bit
         break

   sflowHwStatus.sampleRate = closestPower

def performSampling( bridge, data, srcPort, finalIntfs ):
   #Irregardless of what happens next,
   #we want to increment Total_Packets (sFlow V 5, pg 6)
   #XXX - probably Hw status... sflowStatus.totalPackets += 1

   if not sflowHwConfig.enabled:
      t8( 'Hardware not configured to run sflow. not sampling' )
      return

   if not finalIntfs:
      ( pkt, headers, offset ) = PktParserTestLib.parsePktStr( data )
      ethHdr = PktParserTestLib.findHeader( headers, "EthHdr" )
      if not ethHdr.dst == bridge.bridgeMac_:
         return # drop packet to simulate hardware behavior

   t4( 'performSampling...', sflowHwConfig.sampleRate, sflowConfig.sampleRate,
       'port', srcPort.name() )

   syncStatusSampleRate()

   def needSample():
      return random.random() <= ( 1.0 / sflowHwConfig.sampleRate )

   if needSample():
      if srcPort.name() in sflowHwConfig.runIntf:
         t8( 'sampling ingress Packet: ', Tracing.HexDump( data ) )
         #increment hardware trigger
         sflowHwStatus.hwSamples += 1
         srcPort.trapFrame( formatPacket( data ) )      
      elif sflowHwConfig.runEgressIntf:
         intfDetails = Tac.newInstance( "IntfSnmp::IntfDetails" )
         intfStatus = bridge.ethIntfStatusDir_.intfStatus[ srcPort.name() ]
         intfDetails.status = intfStatus
         srcIfIndex = intfDetails.ifIndex
         for intf in finalIntfs:
            if intf in sflowHwConfig.runEgressIntf:
               t8( 'sampling for egress port: ', intf,
                   ' Packet: ', Tracing.HexDump( data ) )
               sflowHwStatus.hwSamples += 1
               intfStatus = bridge.ethIntfStatusDir_.intfStatus[ intf ]
               intfDetails.status = intfStatus
               dstPort = bridge.port.get( intf )
               dstPort.trapFrame( formatPacket( data, egressSample=True,
                                                srcIfIndex=srcIfIndex,
                                                dstIfIndex=intfDetails.ifIndex ) )
               # not a very accurate sampling with multiple egress intfs,
               # but close enough and works well for the sampleRate = 1, use case
               if not needSample():
                  break
   else:
      t8( 'not sampling.'  )

   
def formatPacket( data, egressSample=False, srcIfIndex=0, dstIfIndex=0 ):
   """Wrap the data Packet with an ethernet header with Protocol set
   to ETH_P_ARISTA_SFLOW_ETBA """
   packet = array.array( 'c', data )
   
   aristaHeader = packet[ :ETH_HLEN ]

   #Set EtherType
   aristaHeader[ 12 ] = chr( ( ETH_P_ARISTA_SFLOW_ETBA >> 8 ) & 0xFF )
   aristaHeader[ 13 ] = chr( ETH_P_ARISTA_SFLOW_ETBA & 0xFF )

   aristaHeader.extend( packet )

   # prepare the sflow trailer
   flags = 0
   if srcIfIndex:
      flags |= SFLOW_TRAILER_INPUT_INTERFACE_PRESENT
   if dstIfIndex:
      flags |= SFLOW_TRAILER_OUTPUT_INTERFACE_PRESENT
   if egressSample:
      flags |= SFLOW_TRAILER_EGRESS_PKT
   sflowTrailer = struct.pack( '<IIBIIHQ', flags, dstIfIndex, 0,
                               0, srcIfIndex, 0, 0 )

   aristaHeader.extend( sflowTrailer )
   sflowData = aristaHeader.tostring()
   
   # Truncation is no longer done
   #return sflowData[ :SFLOW_TRUNCATION_SIZE + ETH_HLEN ]
   return sflowData

def agentInit( em ):
   t2( 'Sflow Etba agentInit' )
   global sflowHwConfig, sflowHwStatus, sflowConfig, sflowStatus
   sflowConfig = em.mount( 'sflow/config', 'Sflow::Config', 'r' )
   sflowStatus = em.mount( 'sflow/status', 'Sflow::Status', 'rO' )
   sflowHwConfig = em.mount( 'sflow/hwconfigdir/sflow', 'Sflow::HwConfig', 'rO' )
   sflowHwStatus = em.mount( 'sflow/hwstatus/etba', 'Sflow::HwStatus', 'wc' )

def bridgeInit( bridge ):
   t0( 'BRIDGE INIT' )
   # Tell the Cli this system supports Sflow.
   hwStatus = bridge.em().entity( 'sflow/hwstatus/etba' )
   hwStatus.sflowSupported = True
   hwStatus.sflowAndMirrorSameIntfSupported = False
   hwStatus.minSampleRate = SflowConst.dangMinSampleRate
   hwStatus.maxSampleRate = SflowConst.dangMaxSampleRate
   hwStatus.egressUnmodifiedSflowIntfSupported = True

def Plugin( ctx ):
   if EtbaDutToggleLib.toggleFastEtbaEnabled():
      # C++ plugin handles this
      return
   t0( 'Entering Plugin(ctx) in Sflow.py' )
   ctx.registerAgentInitHandler( agentInit )
   ctx.registerBridgeInitHandler( bridgeInit )
   ctx.registerSflowHandler( performSampling )
