#!/usr/bin/env python
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import struct
from if_ether_arista import ETH_P_ARISTA_PTP
import Tac, Tracing
from PtpPacketLib import ptpPacketType
import EbraTestBridgeConstants

t2 = Tracing.trace2
t3 = Tracing.trace3
t6 = Tracing.trace6
t7 = Tracing.trace7
t8 = Tracing.trace8

ptpStatusDir = None
ptpConfig = None

noMatch = ( False, False, False )
forward = ( True, False, False )
trap = ( True, EbraTestBridgeConstants.TRAP_ANYWAY, False )
mirror = ( True, False, True )

def trapLookupHandler( bridge, routed, vlanId, destMac, data, srcPortName ):
   if ptpConfig.ptpMode == "ptpDisabled":
      t8( "PTP not trapping because it's disabled" )
      return noMatch

   packetType = ptpPacketType( data )
   if packetType is None:
      t8( "PTP not trapping because packet isn't PTP" )
      return noMatch

   if ptpConfig.ptpMode == "ptpBoundaryClock":
      t3( "PTP trapping because of boundary clock mode" )
      return trap
   elif ptpConfig.ptpMode == "ptpGeneralized":
      t3( "PTP trapping because of gPTP clock mode" )
      return trap
   elif ptpConfig.ptpMode == "ptpEndToEndTransparentClock":
      if packetType == "messageSync" or packetType == "messageDelayReq":
         t3( "PTP mirroring to CPU" )
         return mirror
      elif packetType == "messageFollowUp" or packetType == "messageDelayResp":
         t3( "PTP trapping to CPU" )
         return trap
      else:
         t3( "PTP forwarding packet normally" )
         return forward
   elif ptpConfig.ptpMode == "ptpPeerToPeerTransparentClock":
      if packetType == "messageSync":
         t3( "PTP mirroring to CPU" )
         return mirror
      elif packetType in ( "messageFollowUp",
                           "messageDelayReq",
                           "messageDelayResp",
                           "messagePDelayReq",
                           "messagePDelayResp",
                           "messagePDelayRespFollowup" ):
         t3( "PTP trapping to CPU" )
         return trap
      else:
         t3( "PTP forwarding packet normally" )
         return forward
   else:
      t3( "Unknown PTP Mode!" )
   return noMatch

def rewriteHandler( bridge, routed, vlanId, destMac, data, srcPortName ):
   if ptpPacketType( data ) != None:
      # Not a great timestamp for real PTP purposes, but should be sufficient
      # for protocol verification.
      timestamp = int( Tac.now() * 1e9 )
      t7( "Adding timestamp", hex( timestamp ) )
      ptpHeader = struct.pack( "!HQ", ETH_P_ARISTA_PTP, timestamp )
      # Stick the header between the DA/SA and the rest of the packet.
      data = data[ 0 : 12 ] + ptpHeader + data[ 12 : ]
      return ( data, destMac )
   else:
      t8( "PTP not rewriting packet" )
      return ( None, None )

def bridgeInit( bridge ):
   t2( 'Ptp bridgeInit' )

   ptpStatus = ptpStatusDir.newEntity( 'Ptp::HwStatus', 'etba' )

   ptpStatus.ptpSupported = True
   # TODO: This is only partial support as it enables PTP agent in etba dut
   # to send PTP packets with designated mac addresses of G8275.1, snooping packets
   # support has not been added yet.
   ptpStatus.ptpG82751Supported = True

   # We need to explicitly set clockQuality here for etba dut, especially the value 
   # of clock accuracy to be 0x30, because in a few ptp ptests, like Ptp/PtpSmoke.py
   # , Ptp/GPtpTriangleTest.py and Ptp/GPtpCompetingMastersTest.py, the assumption 
   # that namespace dut (like walta) has better clockQuality than Sand and Strata 
   # dut has been made. This assumption is used in test to determine which dut should
   # be master when priority1 is in tie.
   # A better way is to change those tests and explicitly set the priority1, now we 
   # fix this with this workaround.
   ptpStatus.clockQuality = Tac.Value( 
         'Ptp::ClockQuality', 
         248, 
         0x30, 
         Tac.Value( 'Arnet::NetU16', 0xFFFF ) 
         )
   
   for port in bridge.port:
      if not port.startswith( 'Ethernet' ):
         continue
      intfStatus = ptpStatus.newIntfStatus( port )
      intfStatus.twoStepClock = True
      intfStatus.timestampMultiplier = 1
      intfStatus.timestampDivider = 1
      intfStatus.timestampAtPacketEnd = False
      intfStatus.hwReadyForPacket = True
   
def agentInit( em ):
   t2( 'Ptp agentInit' )
   global ptpConfig, ptpStatusDir
   ptpConfig = em.mount( 'ptp/hwConfig', 'Ptp::HwConfig', 'rO' )
   ptpStatusDir = em.mount( 'ptp/hwStatus', 'Tac::Dir', 'w' )

def Plugin( ctx ):
   ctx.registerBridgeInitHandler( bridgeInit )
   ctx.registerAgentInitHandler( agentInit )
   ctx.registerTrapLookupHandler( trapLookupHandler,
                                  priority=ctx.TRAP_PRIORITIES[ "PTP" ] )
   ctx.registerRewritePacketOnTheWayToTheCpuHandler( rewriteHandler )
