#!/usr/bin/env python
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet, Tracing, Tac, Cell
from Arnet import PktParserTestLib
from EbraTestBridge import IP_MCAST_BASE
from EbraTestBridge import IP_MCAST_END
import QuickTrace
from MrouteConsts import routingMulticastStatusSysdbPath
import SharedMem
import Smash
import MfibSmashTestLib
from IpLibConsts import DEFAULT_VRF
from Toggles import EtbaDutToggleLib

handle = Tracing.Handle( 'Mroute' )
t2 = handle.trace2
t8 = handle.trace8

qtInitialized = False

def qTrace():
   '''Initializes QuickTrace once, return the module itself'''
   global qtInitialized
   if not qtInitialized:
      QuickTrace.initialize( 'etba-routingHandlers.qt' )
      qtInitialized = True
   return QuickTrace

def qv( *args ):
   qTrace().Var( *args )

def qt0( *args ):
   qTrace().trace0( *args )

intfVlanIdTable = Tac.newInstance( "IntfVlanId::Table", "intfVlanIdTable" )

# How do I get these flags from the tac model?
flagForward = 1
flagAccept = 2

shmemEm = None

mrouteHandler = 'Mroute:'

def initVrf( bridge, vrfName ):
   if vrfName in bridge.mrouteBidir_:
      return
   t2( "Mount bidir routestatus for vrf", vrfName )

   shmemEm.doMount( "routing/multicast/routestatus/%s/sparsemode" % vrfName,
                    "Smash::Multicast::Fib::Status", Smash.mountInfo( 'reader' ) )
   shmemEm.doMount( "routing6/multicast/routestatus/%s/sparsemode" % vrfName,
                    "Smash::Multicast::Fib::Status", Smash.mountInfo( 'reader' ) )

   shmemEm.doMount( "routing/multicast/routestatus/%s/bidir" % vrfName,
                    "Smash::Multicast::Fib::Status", Smash.mountInfo( 'reader' ) )
   shmemEm.doMount( "routing/multicast/bidirstatus/%s" % vrfName,
                    "Smash::Multicast::Fib::BidirStatus",
                    Smash.mountInfo( 'reader' ) )
   shmemEm.doMount( "routing/multicast/bitmapperstatus/%s/bidir" % vrfName,
                    "BitMapper::SmashStatus", Smash.mountInfo( 'reader' ) )

   bridge.mroutePimsm_[ vrfName ] = bridge.sEm().getTacEntity(
         'routing/multicast/routestatus/%s/sparsemode' % vrfName )
   bridge.mrouteStatus_ = bridge.mroutePimsm_[ vrfName ]
   bridge.mroute6Pimsm_[ vrfName ] = bridge.sEm().getTacEntity(
         'routing6/multicast/routestatus/%s/sparsemode' % vrfName )
   bridge.mroute6Status_ = bridge.mroute6Pimsm_[ vrfName ]

   bridge.mrouteBidir_[ vrfName ] = bridge.sEm().getTacEntity(
         'routing/multicast/routestatus/%s/bidir' % vrfName )
   bridge.bidirStatus_[ vrfName ] = bridge.sEm().getTacEntity(
         'routing/multicast/bidirstatus/%s' % vrfName )
   bridge.bitMapperSmashBidir_[ vrfName ] = bridge.sEm().getTacEntity(
         'routing/multicast/bitmapperstatus/%s/bidir' % vrfName )

def RouteKey( s, g ):
   if s == None:
      s = Arnet.IpGenPrefix( '0.0.0.0/0' )
   else:
      s = Arnet.IpGenPrefix( s + '/32' )
   return Tac.Value( "Routing::Multicast::Fib::IpGenRouteKey",
                     s, Arnet.IpGenPrefix( g +'/32' ) )

def IpGenRouteKey( s, g ):
   if s == None:
      s = Arnet.IpGenPrefix( '0.0.0.0/0' )
   else:
      s = Arnet.IpGenPrefix( s + '/32' )
   return Tac.Value( "Routing::Multicast::Fib::IpGenRouteKey",
                     s, Arnet.IpGenPrefix( g +'/32' ) )


def lookup( bridge, s, g, bidir=True, vrfName='default' ):
   if bidir:
      routingTable = bridge.mrouteBidir_[vrfName].route
      s = None
   else:
      routingTable = bridge.mroutePimsm_[vrfName].route

   rt = routingTable.get( IpGenRouteKey(s, g) )
   # BUG106170 - Support *, G and prefix routes
   #  *, G routes not supported yet
   # if not rt:
   #    rt = routingTable.get( RouteKey( None, g ) )
   return rt

def prefixLen( p ):
   return int( p.split( '/' )[ 1 ] )

def hasActiveIpAddr( bridge, intfName ):
   ipIntfStatus = bridge.ipIntfStatus_.get( intfName )
   return ipIntfStatus and ( ipIntfStatus.activeAddrWithMask.address != "0.0.0.0" )

def getVrfNameFromIntf( bridge, intfName ):
   ipIntfStatus = bridge.ipIntfStatus_.get( intfName )
   if ipIntfStatus:
      return ipIntfStatus.vrf
   return 'default'

def iif( r ):
   return r.iif

def iifFrr( r ):
   return r.iifFrr

def notify( r ):
   return r.notify

def isBidir( bridge, grpAddr, vrfName='default' ):
   # check group address in bidir range
   bidirGroup = bridge.bidirStatus_[vrfName].bidirGroup
   for prefix in bidirGroup:
      if prefix.contains( Arnet.IpGenAddr( grpAddr ) ):
         return True
   return False

def iAmDf( bridge, intf, rpaId, vrfName='default' ):
   dfProfile = bridge.bidirStatus_[vrfName].dfProfile.get( intf )
   if not dfProfile:
      return False
   if rpaId > 64:
      return False
   dfBitmask = 1 << rpaId
   return ( dfProfile.dfBitmap & dfBitmask ) != 0

def oifs( bridge, r, bidir, vrfName='default' ):
   outgoingIntfIds = set()
   oifIndexNull = Tac.Value( "Smash::Multicast::Fib::OifIndex", 512 )
   curOifIndex = oifIndexNull
   curOifIndex = r.nextOif( curOifIndex )
   if bidir:
      mfibSmashTestEnv = MfibSmashTestLib.MfibSmashTestEnv(
         mfibSmash=bridge.mrouteBidir_[vrfName],
         bitMapperSmashStatus=bridge.bitMapperSmashBidir_[vrfName] )
      return set( mfibSmashTestEnv.oifs( r.key ) )
   else:
      oifIndexDict = bridge.mrouteStatus_.oifIndexIntf

   while curOifIndex < 512:
      if oifIndexDict.get( curOifIndex ):
         intfId = oifIndexDict[ curOifIndex ].intf
         outgoingIntfIds.add( intfId )
      curOifIndex = r.nextOif( curOifIndex )
   return outgoingIntfIds

toCpuFlag = 1
# If we're not in forwardingStyleKernel or forwardingStyleBess, then
# remove the Cpu from the output interface list.
def floodsetIncludesCpu( bridge, vlanId, dstMacAddr, data ):
   if dstMacAddr < IP_MCAST_BASE or dstMacAddr > IP_MCAST_END:
      t8( 'floodsetIncludesCpu not claiming dest ', dstMacAddr )
      return None
   if ( bridge.mrouteEtbaStatus_.forwardingStyle in
           [ 'forwardingStyleKernel', 'forwardingStyleBess' ] ):
      return True
   # The rest of the function mimics the HW (FocalPoint) behavior
   intfName = intfVlanIdTable.vlanIdToIntfId( vlanId )
   if not hasActiveIpAddr( bridge, intfName ):
      return None

   # pylint: disable-msg=W0612
   ( _pkt, headers, _offset ) = PktParserTestLib.parsePktStr( data )

   ipHdr = PktParserTestLib.findHeader( headers, "IpHdr" )
   if not ipHdr:
      t8( 'Invalid ipHdr' )
      return None
   # BUG106169 - tracks adding this back
   # if intfName in bridge.mrouteStatus_.notifyLocalSources:
   #    ipIntfStatus = bridge.ipIntfStatus_[ intfName ]
   #    for addr in [ ipIntfStatus.activeAddrWithMask ] + \
   #        ipIntfStatus.activeSecondaryWithMask.keys():
   #       subnet = Subnet( '%s/%s' % ( addr.address, addr.len ) )
   #       if subnet.containsAddr( ipHdr.src ):
   #          t8( 'Including cpu due to notifyLocalSource' )
   #          return True
   #    return None
   vrfName = getVrfNameFromIntf( bridge, intfName )
   initVrf( bridge, vrfName )

   bidir = isBidir( bridge, ipHdr.dst, vrfName )
   r = lookup( bridge, ipHdr.src, ipHdr.dst, bidir, vrfName )
   t8( 'Multicast pkt s, g: ', ipHdr.src, ipHdr.dst )
   if not r:
      t8( 'No matching route in mfib' )
      return True
   if intfName != iif( r ) and intfName != iifFrr( r ) and not bidir:
      # RPF failure. At some point we may want to send the
      # ETH_P_ARISTA_RPF_FAILURE pkt as well.
      t8( 'RPF failure %s, intfName %s, iif( r ) %s, iifFrr( r ) %s'
            % ( r.key, intfName, iif( r ), iifFrr( r ) ) )
      return True
   if r.routeFlags.toCpu:
      t8( '(toCpu) route wants the pkt to be copied to cpu.' )
      return True
   return None
      
# Currently we check the routeConfig object and check the incoming
# interface has an IP address.  Also currently we forward on all the
# egress interfaces regardless of their IP configuration.
def route( bridge, vlanId, destMac, data ):
   noMatch = [ ( None, {} ) ]
   
   if bridge.mrouteEtbaStatus_.forwardingStyle == 'forwardingStyleKernel':
      t8( 'deferring to kernel due to forwardingStyleKernel' )
      qt0( mrouteHandler, 'deferring to kernel due to forwardingStyleKernel' )
      return noMatch
   if bridge.mrouteEtbaStatus_.forwardingStyle == 'forwardingStyleBess':
      t8( 'deferring to BESS due to forwardingStyleBess' )
      qt0( mrouteHandler, 'deferring to BESS due to forwardingStyleBess' )
      return noMatch

   # Compare destMac to virtual Mac for Ip routing
   if destMac < IP_MCAST_BASE or destMac > IP_MCAST_END :
      t8( 'not claiming dest ', destMac )
      qt0( mrouteHandler, 'not claiming dest ', qv( destMac ) )
      return noMatch
   # Get Ip headers
   # pylint: disable-msg=W0612
   ( pkt, headers, _offset ) = PktParserTestLib.parsePktStr( data )
   ipHdr = PktParserTestLib.findHeader( headers, "IpHdr" )
   if not ipHdr:
      t8( 'no ip headers' )
      qt0( mrouteHandler, 'no ip headers' )
      return noMatch
   if not Arnet.Subnet( '224.0.0.0/4' ).containsAddr( ipHdr.dst ):
      t8( 'not ip multicast dest ',  ipHdr.dst )
      qt0( mrouteHandler, 'not ip multicast dest ', qv( ipHdr.dst ) )
      return noMatch
   if ipHdr.ttl <= 1:
      t8( 'packet not routed because of ttl of 1' )
      qt0( mrouteHandler, 'packet not routed because of ttl of 1' )
      return noMatch
   intfName = intfVlanIdTable.vlanIdToIntfId( vlanId )
   if not hasActiveIpAddr( bridge, intfName ):
      t8( 'no active ip on incoming interface' )
      qt0( mrouteHandler, 'no active ip on incoming interface' )
      return noMatch
   vrfName = DEFAULT_VRF
   if bridge.ipIntfStatus_[ intfName ]:
      vrfName = bridge.ipIntfStatus_[ intfName ].vrf

   vrfName = getVrfNameFromIntf( bridge, intfName )
   t8( 'init vrf', vrfName )
   initVrf( bridge, vrfName )
   bidir = isBidir( bridge, ipHdr.dst, vrfName )
   t8( 'init vrf', vrfName, 'for bidir', bidir )
   r = lookup( bridge, ipHdr.src, ipHdr.dst, bidir, vrfName )
   if not r:
      t8( 'no route match s, g: ', ipHdr.src, ipHdr.dst )
      qt0( mrouteHandler, 'no route match s, g: ', qv( ipHdr.src ), qv( ipHdr.dst ) )
      return noMatch
   if not bidir:
      routeStatus = bridge.mrouteHwStatus_.vrfStatus[ vrfName ].route.get( \
         RouteKey( ipHdr.src, ipHdr.dst ) )
      if routeStatus:
         routeStatus.lastPollTime = Tac.now()
   
   rpfIntfName = iif( r )
   rpfFrr = iifFrr( r )
   t8( 'match', r.key.s, r.key.g, 'iif', rpfIntfName, '/', intfName,
       'iifFrr', rpfFrr )

   if intfName == rpfFrr and not bidir:
      t8( 'Packet arrives on rpfFrr', rpfFrr, ', drop the pkt', r.key )
      qt0( mrouteHandler, 'Packet arrives on rpfFrr', qv( rpfFrr ),
           'drop the pkt', qv( r.key ) )
      return noMatch

   if intfName != rpfIntfName and not bidir:
      t8( 'RPF failure, incoming', intfName,
          ', expecting', rpfIntfName, ', not routing the pkt', r.key )
      qt0( mrouteHandler, 'RPF failure, incoming', qv( intfName ),
           ', expecting', qv( rpfIntfName ), ', not routing the pkt', qv( r.key ) )
      return noMatch

   intfNameSet = oifs( bridge, r, bidir, vrfName )

   if bidir:
      # This is BiDIR route, remove ingress interface
      intfNameSet.discard( intfName )
      # check dfProfile
      if not iAmDf( bridge, intfName, r.rpaId, vrfName ):
         t8( 'Not DF for interface', intfName, ', rpaId', r.rpaId, ', drop packet' )
         qt0( mrouteHandler, 'Not DF for interface', qv( intfName ),
              ', rpaId', qv( r.rpaId ), ', drop packet' )
         return noMatch

   if not intfNameSet:
      t8( 'Empty oifs:', r.key )
      qt0( mrouteHandler, 'Empty oifs:', qv( r.key ) )
      return noMatch

   if r.routeFlags.toCpu:
      # Mimic the HW (FocalPoint) behavior.
      t8( 'route with toCpuFlag, so not routing the pkt' )
      qt0( mrouteHandler, 'route with toCpuFlag, so not routing the pkt' )
      return noMatch

   # Edit the packet src MAC and ttl
   pkt = Tac.newInstance( "Arnet::Pkt" )
   pkt.stringValue = data
   
   ( headers, _offset ) = PktParserTestLib.parsePkt( pkt )
   hdrNames = [ i[ 0 ] for i in headers ]
   ethHdr = headers[ hdrNames.index( 'EthHdr' ) ][ 1 ]
   ipHdr = headers[ hdrNames.index( 'IpHdr' ) ][ 1 ]
   
   # update the src MAC address 
   ethHdr.src = bridge.bridgeMac_
   # if the ttl is less or equal to 1 we don't route
   if ipHdr.ttl <= 1:
      t8( 'not routing packet with ttl of ', ipHdr.ttl )
      qt0( mrouteHandler, 'not routing packet with ttl of ', qv( ipHdr.ttl ) )
      return noMatch
   ipHdr.ttl = ipHdr.ttl - 1
   # recompute the ip header checksum
   ipHdr.checksum = 0
   ipHdr.checksum =  ipHdr.computedChecksum
         
   vlanIdSet = set( intfVlanIdTable.intfIdToVlanId( intfName )
                    for intfName in intfNameSet )

   # Its possible that we don't have an intfId to vlanId mapping as yet (or at all
   # as in the case of loopbacks)
   if 0 in vlanIdSet:
      vlanIdSet.remove( 0 )

   if not vlanIdSet:
      t8( 'Empty vlanIdSet:', r.key )
      qt0( mrouteHandler, 'Empty vlanIdSet:', qv( r.key ) )
      return noMatch

   data = pkt.stringValue
   t8( 'vlans', vlanIdSet, 'data:', data )
   return [ ( data, vlanIdSet ) ]

def bridgeInit( bridge ):

   t2( 'bridgeInit' )
   t2( 'Mounting Smashed Mfib' )

   bridge.mroutePimsm_ = {}
   bridge.mroute6Pimsm_ = {}

   bridge.mrouteBidir_ = {}
   bridge.bitMapperSmashBidir_ = {}
   bridge.bidirStatus_ = {}

   bridge.mroutePimsm_[ 'default' ] = bridge.sEm().getTacEntity(
         'routing/multicast/routestatus/default/sparsemode' )
   bridge.mroute6Pimsm_[ 'default' ] = bridge.sEm().getTacEntity(
         'routing6/multicast/routestatus/default/sparsemode' )

   bridge.mrouteBidir_[ 'default' ] = bridge.sEm().getTacEntity(
         'routing/multicast/routestatus/default/bidir' )
   bridge.bitMapperSmashBidir_[ 'default' ] = bridge.sEm().getTacEntity(
         'routing/multicast/bitmapperstatus/default/bidir' )
   bridge.bidirStatus_[ 'default' ] = bridge.sEm().getTacEntity(
         'routing/multicast/bidirstatus/default' )
   bridge.mrouteEtbaStatus_ = bridge.em().entity( 'routing/multicast/etba/status' )

   bridge.mrouteStatus_ = bridge.mroutePimsm_[ 'default' ]
   bridge.mroute6Status_ = bridge.mroute6Pimsm_[ 'default' ]

   bridge.mrouteHwStatus_ = bridge.em().entity( 'routing/hardware/multicast/status' )
   bridge.mroute6HwStatus_ = \
      bridge.em().entity( 'routing6/hardware/multicast/status' )

   bridge.ipIntfStatus_ = bridge.em().entity( 'ip/status' ).ipIntfStatus
   bridge.allVrfStatusLocal_ = bridge.em().entity(
         Cell.path( 'ip/vrf/status/local' ) )

   etbaMrouteRoot = Tac.root.newEntity( "Mroute::Etba::Root", "EtbaMira" )
   etbaMrouteRoot.mira = ( bridge.mrouteEtbaStatus_, bridge.mrouteStatus_,
                           bridge.allVrfStatusLocal_, bridge.mrouteHwStatus_ )
   etbaMrouteRoot.mira6 = ( bridge.mrouteEtbaStatus_, bridge.mroute6Status_,
                            bridge.allVrfStatusLocal_, bridge.mroute6HwStatus_ )

   bridge.routingHwStatus_ = bridge.em().entity( 'routing/hardware/status' )
   bridge.routingHwStatus_.multicastRoutingSupported = True
   bridge.routingHwStatus_.pimBidirectionalSupported = True
   bridge.routingHwStatus_.multicastBoundarySupported = True
   bridge.routingHwStatus_.multicastRoutingSupportsIpEgressAcl = True

   bridge.routing6HwStatus_ = bridge.em().entity( 'routing6/hardware/status' )
   bridge.routing6HwStatus_.multicastRoutingSupported = True
 
   bridgingConfig = bridge.em().entity( 'bridging/config' )
   bridge.intfVlanIdSm_ = Tac.newInstance(
      "IntfVlanId::ConfigSm", bridgingConfig, None, intfVlanIdTable )

def agentInit( em ):
   t2( 'agentInit' )
   global shmemEm
   shmemEm = SharedMem.entityManager( sysdbEm=em )

   shmemEm.doMount( "routing/multicast/routestatus/default/sparsemode",
                    "Smash::Multicast::Fib::Status", Smash.mountInfo( 'reader' ) )
   shmemEm.doMount( "routing6/multicast/routestatus/default/sparsemode",
                    "Smash::Multicast::Fib::Status", Smash.mountInfo( 'reader' ) )

   shmemEm.doMount( "routing/multicast/bitmapperstatus/default/bidir",
                    "BitMapper::SmashStatus", Smash.mountInfo( 'reader' ) )
   shmemEm.doMount( "routing/multicast/routestatus/default/bidir",
                    "Smash::Multicast::Fib::Status", Smash.mountInfo( 'reader' ) )
   shmemEm.doMount( "routing/multicast/bidirstatus/default",
                    "Smash::Multicast::Fib::BidirStatus",
                    Smash.mountInfo( 'reader' ) )

   shmemEm.doMount( "routing/multicast/status", "Smash::Multicast::Fib::Status",
                    Smash.mountInfo( 'shadow' ) )
   shmemEm.doMount( "routing6/multicast/status", "Smash::Multicast::Fib::Status",
                    Smash.mountInfo( 'shadow' ) )

   # mounting config and status.
   em.mount( routingMulticastStatusSysdbPath,
             'Routing::Multicast::Fib::Status', 'r' )
   em.mount( 'routing/multicast/etba/status',
             'Routing::Multicast::Etba::Status', 'r' )

   em.mount( 'routing/hardware/multicast/status',
             'Routing::Multicast::Fib::Hardware::Status', 'w' )
   em.mount( 'routing6/hardware/multicast/status',
             'Routing::Multicast::Fib::Hardware::Status', 'w' )

   em.mount( 'bridging/config', 'Bridging::Config', 'r' )
   mg = em.activeMountGroup()
   Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, False )
   # this is to enable multicast routing support
   em.mount( 'routing/hardware/status', 'Routing::Hardware::Status', 'w' )
   em.mount( 'routing6/hardware/status', 'Routing6::Hardware::Status', 'w' )
   em.mount( Cell.path( 'ip/vrf/status/local' ), 'Ip::AllVrfStatusLocal', 'r' )

def Plugin( ctx ):
   ctx.registerBridgeInitHandler( bridgeInit )
   ctx.registerAgentInitHandler( agentInit )
   if not EtbaDutToggleLib.toggleFastEtbaEnabled():
      ctx.registerRoutingHandler( route )
   ctx.registerFloodsetIncludesCpuHandler( floodsetIncludesCpu )
