# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import defaultdict
from Arnet import IpGenAddr
from GenericReactor import GenericReactor
from ForwardingHelper import (
      Af,
      getNhgSize,
   )
from MplsEtbaLib import (
      ARP_SMASH_DEFAULT_VRF_ID,
      getIntfVlan,
      getL2Intf,
   )
from TypeFuture import TacLazyType
from IpLibConsts import DEFAULT_VRF
import Arnet
import Tac
import Tracing

# pkgdeps: library IraEtba

handle = Tracing.Handle( 'EbraTestBridgeMpls' )
t8 = handle.trace8

AdjType = TacLazyType( 'Smash::Fib::AdjType' )
DynamicTunnelIntfId = TacLazyType( 'Arnet::DynamicTunnelIntfId' )
EthAddr = TacLazyType( 'Arnet::EthAddr' )
FecId = TacLazyType( 'Smash::Fib::FecId' )
FecIdIntfId = TacLazyType( 'Arnet::FecIdIntfId' )
IpAddr = TacLazyType( 'Arnet::IpAddr' )
LfibViaType = TacLazyType( 'Mpls::LfibViaType' )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )
MplsLabelAction = TacLazyType( 'Arnet::MplsLabelAction' )
NexthopGroupIntfIdType = TacLazyType( 'Arnet::NexthopGroupIntfId' )
NhgId = TacLazyType( 'Routing::NexthopGroup::NexthopGroupId' )
NhgL2Via = TacLazyType( 'Routing::Hardware::NexthopGroupL2Via' )
NhgType = TacLazyType( 'Routing::NexthopGroup::NexthopGroupType' )
NhgVia = TacLazyType( 'Routing::Hardware::NexthopGroupVia' )
RoutingNexthopGroupType = TacLazyType( 'Routing::NexthopGroup::NexthopGroupType' )
TunnelId = TacLazyType( 'Tunnel::TunnelTable::TunnelId' )
TunnelType = TacLazyType( 'Tunnel::TunnelTable::TunnelType' )
NhFecIdType = Tac.Type( 'Routing::NexthopStatusFecIdType' )

class L2Hop( object ):
   def __init__( self, vlanId_, macAddr_ ):
      '''
      Returns an L2Hop object with the vlanId and macAddr passed in.
      Note: The macAddr will automatically have L2Hop.macAddrToString() called on it
      and will assert if that value isn't a valid Arnet.EthAddr.
      '''
      self.vlanId = vlanId_
      self.macAddr = L2Hop.macAddrToString( macAddr_ )

   def __repr__( self ):
      return "L2Hop( {}, '{}' )".format( self.vlanId, self.macAddr )

   def __eq__( self, other ):
      return ( isinstance( other, self.__class__ ) and
               self.__dict__ == other.__dict__ )

   def __ne__( self, other ):
      return not self.__eq__( other )

   def __hash__( self ):
      return hash(( self.vlanId, self.macAddr ))

   @staticmethod
   def macAddrToString( macAddr_ ):
      '''
      Returns the colon separated string version of the macAddr.
      If bool( macAddr ) == False, it defaults to returning a zero macAddr.
      '''
      macAddr = macAddr_ or EthAddr.ethAddrZero
      assert L2Hop.isMacAddrValid( macAddr ), 'Invalid macAddr {}'.format( macAddr_ )

      # Convert to a EthAddr object to convert all string representations to colon
      # separated version. Also need guard to prevent a double conversion to EthAddr.
      if not isinstance( macAddr, EthAddr ):
         macAddr = Arnet.EthAddr( macAddr )
      return str( macAddr )

   @staticmethod
   def isMacAddrValid( macAddr ):
      'Returns True if macAddr can successfully convert to Arnet.EthAddr'
      try:
         Arnet.EthAddr( str( macAddr ) )
         return True
      except IndexError:
         return False

   @staticmethod
   def isMacAddrNonZero( macAddr ):
      'Returns True if macAddr is a valid EthAddr and is not 0.0.0'
      return ( L2Hop.isMacAddrValid( macAddr ) and
               bool( Arnet.EthAddr( str( macAddr ) ) ) )

   @staticmethod
   def isVlanIdValid( vlanId ):
      return isinstance( vlanId, int ) and vlanId > 0

   def isValid( self ):
      'Returns True if the both the macAddr and vlanId are non-zero'
      return ( L2Hop.isMacAddrNonZero( self.macAddr ) and
               L2Hop.isVlanIdValid( self.vlanId ) )

class NhgAdjSmHelper( object ):
   '''
   This is a helper object that contains all of the relevant mappings needed for
   the NhgAdjSms. It will also facilitate the adding of unresolved dstIps to the
   NhVrfConfig.nexthop collection for processing by the Ira L3 Nexthop Resolver.

   Lastly, it also will also contain references to the bridging config/status as
   well as the arpSmash in order to allow the other SMs to externally update the
   nhg adjacencies based on changes in resolved L3 or L2 information.

   Example mappings:

   config = {
      1.1.1.1: {
         'nhg1': set([ 0, 1 ]),
      }
      2222::: {
         'nhg2': set([ 2 ])
      }
   }
   status = {
      1.1.1.1: NhgVia() {
                  hop: 10.0.0.2,
                  l3Intf: Ethernet1,
                  l2Via: L2NhgVia() {
                            vlanId: 1006
                            l2Intf: 'Ethernet1'
                            macAddr: '00:00:00:01:00:02'
                         }
               }
   }
   l3HopToDstIp = {
      10.0.0.2: set([ 1.1.1.1, 1.2.3.4 ]),
      20.0.0.2: set([ 2.2.2.2 ])
      2001::2: set([ 2222:: ])
   }
   l2HopToL3Hop = {
      ( 1006, 00:00:00:01:00:02 ): set([ 10.0.0.2, 1001::2 ])
      ( 1007, 00:01:00:02:00:03 ): set([ 20.0.0.2, 2001::2 ])
   }
   nhgAdjSm = {
      'nhg1': NhgAdjSm()
   }
   nhgName = {
      1: 'nhg1',
      2: 'nhg2'
   }

   :type brConfig: Bridging::Config
   :type brStatus: Bridging::Status
   :type nhVrfConfig: Routing::NexthopVrfConfig
   :type arpSmash: Arp::Table::Status
   :type proactiveArpNexthopConfig: Arp::ProactiveArp::ClientConfig
   '''
   def __init__( self, brConfig, brStatus, nhVrfConfig, arpSmash,
                 proactiveArpNexthopConfig ):
      self._traceName = 'NhgAdjSmHelper'
      self.nhVrfConfig = nhVrfConfig
      self._brConfig = brConfig
      self._brStatus = brStatus
      self._arpSmash = arpSmash
      self._proactiveArpNexthopConfig = proactiveArpNexthopConfig

      self.config = defaultdict( lambda: defaultdict( set ) )
      self.status = {}
      self.l3HopToDstIp = defaultdict( set )
      self.l2HopToL3Hop = defaultdict( set )
      self.adjSm = {}

      # Need to clear the nhVrfConfig since it might contain old nexthop entries
      # that weren't cleared (e.g. if agent crashed)
      nhVrfConfig.nexthop.clear()

   def configEntryIs( self, dstIp, nhgName, index ):
      nhce = Tac.newInstance( 'Routing::NexthopConfigEntry', dstIp )
      self.nhVrfConfig.addNexthop( nhce )
      self.config[ dstIp ][ nhgName ].add( index )

   def configEntryDel( self, dstIp, nhgName, entry ):
      self.config[ dstIp ][ nhgName ].remove( entry )

      # If no one else cares about this dstIp, remove the dstIp entirely from
      # both the config map and nhVrfConfig
      if not self.config[ dstIp ][ nhgName ]:
         if len( self.config[ dstIp ] ) == 1:
            del self.config[ dstIp ]
            del self.nhVrfConfig.nexthop[ dstIp ]
         else:
            del self.config[ dstIp ][ nhgName ]

   def l3HopToDstIpIs( self, l3Hop, dstIp ):
      func = self._traceName + '.l3HopToDstIpIs:'
      if l3Hop.isAddrZero or dstIp.isAddrZero:
         msg  = "{} Can't add invalid l3Hop {} dstIp {}".format( func, l3Hop, dstIp )
         assert False, msg

      t8( func, 'Adding new l3Hop', l3Hop, 'dstIp', dstIp )
      self.l3HopToDstIp[ l3Hop ].add( dstIp )

   def l3HopToDstIpDel( self, l3Hop, dstIp ):
      func = self._traceName + '.l3HopToDstIpDel:'
      l3HopToDstIp = self.l3HopToDstIp
      if l3Hop not in l3HopToDstIp or not dstIp in l3HopToDstIp[ l3Hop ]:
         t8( func, "Skip deleting non-existent l3Hop", l3Hop, 'dstIp', dstIp )
         return

      t8( func, "Deleting l3Hop", l3Hop, 'dstIp', dstIp )
      l3HopToDstIp[ l3Hop ].remove( dstIp )
      if not l3HopToDstIp[ l3Hop ]:
         del l3HopToDstIp[ l3Hop ]

   def l2HopToL3HopIs( self, l2Hop, l3Hop ):
      func = self._traceName + '.l2HopToL3HopIs:'
      if not l2Hop.isValid():
         msg  = "{} Can't add invalid l2Hop {} l3Hop {}".format( func, l2Hop, l3Hop )
         assert False, msg

      t8( func, 'Adding new l2Hop', l2Hop, 'l3hop', l3Hop )
      self.l2HopToL3Hop[ l2Hop ].add( l3Hop )

   def l2HopToL3HopDel( self, l2Hop, l3Hop ):
      func = self._traceName + '.l2HopToL3HopDel:'
      l2HopToL3Hop = self.l2HopToL3Hop
      if l2Hop not in l2HopToL3Hop or not l3Hop in l2HopToL3Hop[ l2Hop ]:
         t8( func, "Skip deleting non-existent l2Hop", l2Hop, 'hop', l3Hop )
         return

      t8( func, "Deleting l2Hop", l2Hop, 'hop', l3Hop )
      l2HopToL3Hop[ l2Hop ].remove( l3Hop )
      if not l2HopToL3Hop[ l2Hop ]:
         del l2HopToL3Hop[ l2Hop ]

   def arpEntry( self, l3Hop, intf ):
      'Retrieves an ARP or ND entry from the arpSmash table given an l3 intf and hop'

      arpKey = Tac.Value( 'Arp::Table::ArpKey', ARP_SMASH_DEFAULT_VRF_ID,
                          l3Hop, intf )
      afGetFunc = {
            Af.ipv4: self._arpSmash.arpEntry.get,
            Af.ipv6: self._arpSmash.neighborEntry.get,
         }
      return afGetFunc[ l3Hop.af ]( arpKey )

   def updateViaL2Info( self, dstIp, macAddr, vlanId=None ):
      '''
      Given the new mac addr, the L3 info is copied from the previous status entry's
      via, while the other L2 info is retrieved and copied into a new NhgVia
      '''
      func = self._traceName + '.updateViaL2Info:'
      t8( func, 'Updating the NhgVia L2 info for dstIp', dstIp )

      # Get old L3 info
      oldVia = self.status[ dstIp ]
      l3Intf = oldVia.l3Intf
      l3Hop = oldVia.hop
      route = oldVia.route

      # Get new L2 info
      if not vlanId:
         # getIntfVlan returns None instead of 0, but NhgL2Via expects a U32
         vlanId = getIntfVlan( self._brConfig, l3Intf ) or 0
      if L2Hop.isMacAddrNonZero( macAddr ):
         l2Intf = getL2Intf( self._brStatus, vlanId, macAddr ) if vlanId else ''
      nhgL2Via =  NhgL2Via() if not macAddr else NhgL2Via( vlanId, l2Intf, macAddr )
      nhgVia = NhgVia( route, l3Hop, l3Intf, nhgL2Via )

      self.status[ dstIp ] = nhgVia

   # Updates all the nhg entries in the config map using the corresponding status via
   def updateAdjEntries( self, dstIp ):
      nhgs = self.config.get( dstIp, {} )
      via = self.status.get( dstIp, NhgVia() )
      for nhg in nhgs:
         for index in nhgs[ nhg ]:
            self.adjSm[ nhg ].adjViaIs( index, via )

   def updateProactiveArpNexthop( self, oldNhgVia, nhgVia ):
      func = 'NhgAdjSmHelper.updateProactiveArpNexthop:'

      # Remove the old L3 hop if it's no longer in the l3HopToDstIp map
      isOldL3InfoValid = not oldNhgVia.hop.isAddrZero and oldNhgVia.l3Intf
      if isOldL3InfoValid and oldNhgVia.hop not in self.l3HopToDstIp:
         t8( func, 'Deleting old L3 hop', oldNhgVia.hop )
         oldEntry = Tac.ValueConst( 'Arp::ProactiveArp::Entry',
                                    oldNhgVia.hop, oldNhgVia.l3Intf )
         del self._proactiveArpNexthopConfig.request[ oldEntry ]

      # Add new L3 hop if it's valid
      isL3InfoValid = not nhgVia.hop.isAddrZero and nhgVia.l3Intf
      if isL3InfoValid:
         failMsg = "Valid L3 Hop {} should be in L3 map".format( nhgVia.hop )
         assert nhgVia.hop in self.l3HopToDstIp, failMsg

         t8( func, 'Adding new l3Hop', nhgVia.hop )
         entry = Tac.ValueConst( 'Arp::ProactiveArp::Entry',
                                 nhgVia.hop, nhgVia.l3Intf )
         request = Tac.ValueConst( 'Arp::ProactiveArp::Request', entry )
         self._proactiveArpNexthopConfig.addRequest( request )

class NhgBridgingStatusSm( Tac.Notifiee ):
   '''
   Reacts to updates to learned hosts in the Sysdb BridgingStatus by updating the L2
   info of the NhgVias that use both the vlanId and macAddr of the learned host. It
   will also trigger an update to all relevent NHG adj entries.
   '''
   notifierTypeName = 'Bridging::Status'

   def __init__( self, nhgAdjSmHelper, brStatus ):
      self._traceName = 'NhgBridgingStatusSm'
      self._helper = nhgAdjSmHelper
      self._brStatus = brStatus
      self._fdbReactor = {}
      Tac.Notifiee.__init__( self, brStatus )
      for vlanId in self._brStatus.fdbStatus:
         self.handleFdbStatus( vlanId )

   def updateViaL2Intf( self, dstIp, macAddr, vlanId_ ):
      self._helper.updateViaL2Info( dstIp, macAddr, vlanId=vlanId_ )

   def handleLearnedHost( self, reactor, macAddr ):
      func = self._traceName + '.handleLearnedHost:'
      fdbStatus = reactor.notifier()
      vlanId = fdbStatus.fid
      l2Hop = L2Hop( vlanId, macAddr )

      if l2Hop not in self._helper.l2HopToL3Hop:
         t8( func, 'L2Hop', l2Hop, 'not found in l2HopToL3Hop map' )
         return

      t8( func, 'Updating the L2Intf for all NHG adj entries using L2Hop', l2Hop )
      for l3Hop in self._helper.l2HopToL3Hop[ l2Hop ]:
         for dstIp in self._helper.l3HopToDstIp[ l3Hop ]:
            self.updateViaL2Intf( dstIp, macAddr, vlanId )
            self._helper.updateAdjEntries( dstIp )

   @Tac.handler( 'fdbStatus' )
   def handleFdbStatus( self, vlanId ):
      '''
      Reacts to changes in fdbStatus entries by creating/deleting reactors to monitor
      learnedHost entries for that specific fdbStatus.
      '''
      func = self._traceName + '.handleFdbStatus:'
      fdbStatus = self._brStatus.fdbStatus.get( vlanId )
      if not fdbStatus:
         if vlanId in self._fdbReactor:
            t8( func, 'Deleting fdbStatus reactor with vlanId', vlanId )
            del self._fdbReactor[ vlanId ]
      else:
         t8( func, 'Creating fdbStatus reactor with vlanId', vlanId )
         self._fdbReactor[ vlanId ] = GenericReactor( fdbStatus, [ 'learnedHost' ],
                                                     self.handleLearnedHost )
         for macAddr in fdbStatus.learnedHost:
            self.handleLearnedHost( self._fdbReactor[ vlanId ], macAddr )

class NhgArpSm( Tac.Notifiee ):
   '''
   Reacts to ARP/ND updates by updating the L2 info of the NhgVias that use the L3Hop
   of the ARP/ND entry, as well as trigger an update to all relevant NHG adj entries.
   '''
   notifierTypeName = 'Arp::Table::Status'

   def __init__( self, nhgAdjEtbaHelper, arpSmash ):
      self._traceName = 'NhgArpSm'
      self._helper = nhgAdjEtbaHelper
      self._arpSmash = arpSmash
      Tac.Notifiee.__init__( self, arpSmash )
      for arpKey in arpSmash.arpEntry:
         self.handleArpEntry( arpKey )
      for ndKey in arpSmash.neighborEntry:
         self.handleNeighborEntry( ndKey )

   def getVia( self, l3Hop ):
      'Returns a NhgVia with the given L3Hop'
      if l3Hop not in self._helper.l3HopToDstIp:
         return None

      # Get arbitrary dstIp from set since any entry will have the same l2 info
      dstIp = next( iter( self._helper.l3HopToDstIp[ l3Hop ] ) )
      nhgVia = self._helper.status[ dstIp ]
      return nhgVia

   def updateL2HopToL3Hop( self, oldNhgVia, l3Hop ):
      func = self._traceName + '.updateL2HopToL3Hop:'
      t8( func, 'Updating l2HopToL3Hop with new l3hop', l3Hop )
      newNhgVia = self.getVia( l3Hop )
      newL2Hop = L2Hop( newNhgVia.l2Via.vlanId, newNhgVia.l2Via.macAddr )
      oldL2Hop = L2Hop( oldNhgVia.l2Via.vlanId, oldNhgVia.l2Via.macAddr )

      self._helper.l2HopToL3HopDel( oldL2Hop, oldNhgVia.hop )
      if newL2Hop.isValid():
         self._helper.l2HopToL3HopIs( newL2Hop, l3Hop )

   def updateNhgViasAndL2HopToL3Hop( self, arpKey, arpEntry ):
      l3Hop = arpKey.addr
      oldNhgVia = self.getVia( l3Hop )
      # OldNhgVia will only exist if this l3Hop is used by any NHG entries
      if oldNhgVia:
         for dstIp in self._helper.l3HopToDstIp[ l3Hop ]:
            macAddr = arpEntry.ethAddr if arpEntry else None
            self._helper.updateViaL2Info( dstIp, macAddr )
            self._helper.updateAdjEntries( dstIp )
         self.updateL2HopToL3Hop( oldNhgVia, l3Hop )

   @Tac.handler( 'arpEntry' )
   def handleArpEntry( self, arpKey ):
      func = self._traceName + '.handleArpEntry:'
      t8( func, 'Handling arp entry for L3Hop', arpKey.addr )
      arpEntry = self._arpSmash.arpEntry.get( arpKey )
      self.updateNhgViasAndL2HopToL3Hop( arpKey, arpEntry )

   @Tac.handler( 'neighborEntry' )
   def handleNeighborEntry( self, ndKey ):
      func = self._traceName + '.handleNeighborEntry:'
      t8( func, 'Handling neighbor entry for L3Hop', ndKey.addr )
      ndEntry = self._arpSmash.neighborEntry.get( ndKey )
      self.updateNhgViasAndL2HopToL3Hop( ndKey, ndEntry )

class NhgNhBacklogSm( Tac.Notifiee ):
   notifierTypeName = 'Routing::NexthopBacklog'

   def __init__( self, fwdHelper, nhgAdjSmHelper, nhrStatus, brConfig, brStatus,
                 backlog ):
      self._traceName = 'NhgNhBacklogSm'
      self._fwdHelper = fwdHelper
      self._helper = nhgAdjSmHelper
      self._nhrStatus = nhrStatus
      self._backlog = backlog
      self._brConfig = brConfig
      self._brStatus = brStatus
      Tac.Notifiee.__init__( self, backlog )
      self.handleBacklogVersion()

   def getValidViaInfo( self, nhse, fec ):
      '''
      Iterates through all FEC vias and returns the hop/intf pair for the first via
      with: a valid intf, MPLS null label, no NHG id and no tunnelId.
      '''
      for via in fec.via.values():
         # If the relevant fields are valid, return the hop and intfId
         if( via.mplsLabel == MplsLabel.null and
             not via.nexthopGroupId and
             not via.tunnelId and
             via.intfId ):
            hop = IpGenAddr( str( via.hop ) )
            hop = nhse.addr if hop.isAddrZero else hop
            assert not hop.isAddrZero, 'Both FEC via and NHSE are zero addrs'
            return hop, via.intfId
      return None, None

   def createNhgVia( self, nhse, fec ):
      '''
      Using the information stored in the Ira NexthopStatusEntry (nhse) and the fec
      for some dstIp, this method will create and return a NhgVia.
      '''
      func = self._traceName + '.createNhgVia:'

      # Get the route and the L3Hop based on whether the fec via has a valid hop
      route = nhse.route
      l3Hop, l3Intf = self.getValidViaInfo( nhse, fec )
      if not l3Hop:
         t8( func, 'No unlabeled IP routes found, returning blank NHG via' )
         return NhgVia()

      # Fetch the L2 info if available and create the NhgL2Via
      arpEntry = self._helper.arpEntry( l3Hop, l3Intf )
      if arpEntry:
         macAddr = arpEntry.ethAddr
         vlanId = getIntfVlan( self._brConfig, l3Intf ) or 0
         l2Intf = getL2Intf( self._brStatus, vlanId, macAddr ) if vlanId else ''
         nhgL2Via = NhgL2Via( vlanId, l2Intf, macAddr )
      else:
         nhgL2Via = NhgL2Via()

      nhgVia = NhgVia( route, l3Hop, l3Intf, nhgL2Via )

      t8( func, 'Created nhgVia for route', route, 'fecId', fec.fecId )
      return nhgVia

   def updateMappings( self, oldNhgVia, nhgVia, dstIp ):
      func = 'NhgNhBacklogSm.updateMappings:'

      # Remove old mappings for the oldNhgVia
      if not oldNhgVia.hop.isAddrZero:
         oldL3Hop = oldNhgVia.hop
         oldL2Hop = L2Hop( oldNhgVia.l2Via.vlanId, oldNhgVia.l2Via.macAddr )
         t8( func, 'Removing old NHG via L3Hop', oldL3Hop,
                   'L2Hop', oldL2Hop, 'from mappings' )
         self._helper.l3HopToDstIpDel( oldL3Hop, dstIp )
         if oldL3Hop not in self._helper.l3HopToDstIp:
            self._helper.l2HopToL3HopDel( oldL2Hop, oldL3Hop )

      # Add new entries if resolved and update the status map
      if not nhgVia.hop.isAddrZero:
         l3Hop = nhgVia.hop
         t8( func, 'Adding resolved L3Hop', l3Hop )
         self._helper.l3HopToDstIpIs( l3Hop, dstIp )

         l2Hop = L2Hop( nhgVia.l2Via.vlanId, nhgVia.l2Via.macAddr )
         if l2Hop.isValid():
            t8( func, 'Adding resolved L2Hop', l2Hop )
            self._helper.l2HopToL3HopIs( l2Hop, nhgVia.hop )
         self._helper.status[ dstIp ] = nhgVia
      elif dstIp in self._helper.status:
         del self._helper.status[ dstIp ]

   def handleBacklogNexthop( self, dstIp ):
      '''
      Performs three tasks:
      1) Processes the nexthop by pulling the resolved L3 information from the FIB
         and updating the L3 map, as well as the L2 map if any of the L2 information
         has already been resolved for the L3 hop.
      2) Updates the status map with the newly created NhgVia and triggers an update
         to all NHG adj entries using this dstIp.
      3) Updates the proactive ARP nexthop config collection by removing unresolved
         L3 hops and adding newly resolved L3 hops.
      '''
      func = self._traceName + '.handleBacklogNexthop:'

      # There should not be an update if the dstIp isn't being used by some NHG
      if dstIp not in self._helper.config:
         t8( func, 'Ignoring nexthop update since not in NHG config map' )
         return

      nhgVia = NhgVia()
      nhse = self._nhrStatus.vrf[ DEFAULT_VRF ].nexthop.get( dstIp )
      if nhse:
         fecId = nhse.fecId
         fec = self._fwdHelper.resolveFecId( fecId )
         # Although the route may be resolved (NHSE is present), there may be a
         # transient state where the nhse.fecId is not present in FIB. So, check this
         if fec:
            nhgVia = self.createNhgVia( nhse, fec )

      oldNhgVia = self._helper.status.get( dstIp, NhgVia() )
      self.updateMappings( oldNhgVia, nhgVia, dstIp )
      self._helper.updateAdjEntries( dstIp )
      self._helper.updateProactiveArpNexthop( oldNhgVia, nhgVia )

   @Tac.handler( 'backlogVersion' )
   def handleBacklogVersion( self ):
      '''
      At least one new entry has shown up in the backlog nexthops. Iterate through
      and process each one.
      '''
      func = self._traceName + '.handleBacklogVersion:'
      for dstIp in self._backlog.backlog:
         t8( func, 'Processing backlog nexthop', str( dstIp ) )
         self.handleBacklogNexthop( dstIp )
         del self._backlog.backlog[ dstIp ]

class NhgAdjSm( Tac.Notifiee ):
   '''
   Reacts to any changes in the nexthopGroupConfigEntry's destinationIp collection.
   Updates to the nhgAdj vias are done externally through the NhgAdjSmHelper by the
   other SMs (such as the NhgNhBacklogSm).

   This does not need to handle the case where the nexthopGroupConfig is deleted,
   since this will also cause the associated FEC to disappear, which will trigger the
   cleanup of this SM via the NhgAdjManagerSm.
   '''
   notifierTypeName = 'Routing::NexthopGroup::NexthopGroupConfigEntry'

   def __init__( self, nhgAdjSmHelper, nexthopGroup, nhgAdj ):
      self._traceName = 'NhgAdjSm[ {} ]'.format( nexthopGroup.name )
      self._nexthopGroup = nexthopGroup
      self._nhgName = nexthopGroup.name
      self._helper = nhgAdjSmHelper
      self._nhgAdj = nhgAdj

      # A local map of index -> dstIp needs to be maintained in the case that actual
      # nexthopGroup entries (or the entire entity) is deleted. In the latter case,
      # all revelant entries can be quickly deleted by iterating through the
      # map, otherwise you'd need to search through every dstIp in the configMap.
      self._currNhgEntries = {}
      self._nhgAdj.size = getNhgSize( self._nexthopGroup )
      t8( self._traceName + ':',
          'Initializing SM. NhgAdj size set to', self._nhgAdj.size )
      Tac.Notifiee.__init__( self, self._nexthopGroup )
      for index in range( self._nhgAdj.size ):
         self.handleDstIp( index, isInit=True )

   def cleanup( self ):
      func = self._traceName + '.cleanup:'
      t8( func, 'Removing nhg adj' )
      for index in self._currNhgEntries.keys():
         self.entryDel( index )

   def entryDel( self, index ):
      func = self._traceName + '.entryDel:'
      t8( func, 'Deleting nhgAdj via', index )

      # Delete the config entry and local entry, and the adj entry if it exists
      dstIp = self._currNhgEntries[ index ]
      self._helper.configEntryDel( dstIp, self._nhgName, index )
      del self._currNhgEntries[ index ]
      del self._nhgAdj.via[ index ]

   @Tac.handler( 'destinationIpIntf' )
   def handleDstIp( self, index, isInit=False ):
      func = self._traceName + '.handleDstIp:'
      if not isInit:
         # Need to manually calculate the new size since it can completely change
         # E.g. NHG can have entries 0 and 6 populated. If 6 gets deleted, size goes
         #      from 7 down to 1 since entries 1-5 are unpopulated (zero addresses)
         newSize = getNhgSize( self._nexthopGroup )
         if self._nhgAdj.size != newSize:
            t8( func, 'Nhg size changed from', self._nhgAdj.size, 'to', newSize )
            self._nhgAdj.size = newSize

      # Three cases to handle:
      # 1) Entry was deleted, array size decreased and the dstIp entry doesn't exist
      # 2) Entry was simply deleted, which shows up as a zero address dstIp
      # 3) Entry changed to a different dstIp, which means old dstIp must be deleted
      dstIp = self._nexthopGroup.destinationIp( index )
      if not dstIp.isAddrZero:
         oldDstIp = self._currNhgEntries.get( index )
         if oldDstIp:
            t8( func, 'DstIp for entry', index, 'changed from', oldDstIp, 'to',
                dstIp )
            self.entryDel( index )
         else:
            t8( func, 'Entry', index, 'has new dstIp', dstIp )

         self._currNhgEntries[ index ] = dstIp
         self._helper.configEntryIs( dstIp, self._nhgName, index )

         # Populate the adj via with whatever might be in status
         nhgVia = self._helper.status.get( dstIp )
         self.adjViaIs( index, nhgVia )
      elif not isInit:
         t8( func, 'NhgConfig entry', index, 'deleted.' )
         self.entryDel( index )

   def adjViaIs( self, index, nhgVia ):
      if nhgVia:
         self._nhgAdj.via[ index ] = nhgVia
      else:
         del self._nhgAdj.via[ index ]

class NhgAdjManagerSm():
   '''
   This SM contains reactors for: both the V4 and V6 forwardingStatuses, the
   system LFIB, the NHG Config entities in Sysdb and the NHG Smash entities.
   A NhgAdjSm is created only when a route is created to an existing NHG. In
   other words, the equation for creating the NhgAdjSm is:

   NHG FEC (NhgId) + NHG Smash Entry (NhgId, NhgName) + NHG Sysdb Entry (NhgName)

   or

   NHG LFIB route (NhgId) + NHG Smash Entry (NhgId, NhgName) + NHG Sysdb Entry
   (NhgName)

   Conversely, if any of the above entries are deleted, the corresponding NhgAdjSm
   will also get deleted (assuming it exists).
   '''
   def __init__( self, fwdHelper, nhgAdjSmHelper, nhgHwDefaultVrfStatus,
                 lfib ):
      self._traceName = 'NhgAdjManagerSm'
      t8( self._traceName + ':', 'Initializing SM' )
      self._fwdHelper = fwdHelper
      self._helper = nhgAdjSmHelper
      self._nhgHwDefaultVrfStatus = nhgHwDefaultVrfStatus
      self._nhgIdToName = {}
      self._nhgNameToId = {}
      self._fecIdToNhgId = defaultdict( set )
      self._nhgIdToFecId = defaultdict( set )
      self._nhgIdToRouteKey = defaultdict( set )
      self._routeKeyToNhgId = defaultdict( set )
      # Entities to react to
      self._fs = self._fwdHelper.forwardingStatus_
      self._fs6 = self._fwdHelper.forwarding6Status_
      self._lfib = lfib
      self._nhgConfig = Tac.root.get( 'ira-etba-root' ).nexthopGroupConfig
      self._nhgEntryStatus = fwdHelper.nhgEntryStatus_
      # Reactors
      self._fsReactor = GenericReactor( self._fs, [ 'fec' ], self.handleFec )
      self._fs6Reactor = GenericReactor( self._fs6, [ 'fec' ], self.handleFec )
      self._nhgSmashReactor = GenericReactor( self._nhgEntryStatus,
                                              [ 'nexthopGroupEntry' ],
                                              self.handleNhgSmashEntry )
      self._nhgConfigReactor = GenericReactor( self._nhgConfig, [ 'nexthopGroup' ],
                                               self.handleNhgConfigEntry )

      self._lfibTrackingStatus = Tac.newInstance( 'Mpls::LfibTrackingStatus',
                                                  'NhgAdjManager-TrackingStatus' )
      self._lfibTrackingSm = Tac.newInstance(
            'Mpls::LfibTrackingSm',
            self._lfib,
            self._lfibTrackingStatus,
            Tac.Type( 'Mpls::LfibTrackingMode' ).shadowRouteKeys_,
            'NhgAdjManager-TrackingSm' )
      self._mplsLfibRouteReactor = GenericReactor( self._lfibTrackingStatus,
                                                   [ 'routeVersion' ],
                                                   self.handleMplsLfibRoute )

      # Walk the FIB and LFIB and create NhgAdjSms for existing NHGs
      for fecKey in self._fs.fec:
         self.handleFec( self._fsReactor, fecKey )
      for fecKey in self._fs6.fec:
         self.handleFec( self._fs6Reactor, fecKey )
      for routeKey in self._lfib.lfibRoute:
         self.handleMplsLfibRoute( self._mplsLfibRouteReactor, routeKey )

   def _createNhgAdjSm( self, nhgId=None, nhgName=None ):
      '''
      If an MPLS NHG config is found in Sysdb using the provided info, this will
      create the NHG adj, the NhgAdjSm and trigger the update for the helper maps.
      '''
      if nhgName in self._nhgNameToId:
         return
      func = self._traceName + '._createNhgAdjSm:'
      nhgConfigEntry = self._fwdHelper.getNhgFromSysdb( nhgId=nhgId,
                                                        nhgName=nhgName )
      if nhgConfigEntry.type != RoutingNexthopGroupType.mpls:
         t8( func, 'Unsupported NHG type', nhgConfigEntry.type )
         return

      keyStr = 'NhgId: {}, NhgName: {}'.format( nhgId, nhgName )
      t8( func, 'Creating NhgAdjSm with', keyStr )
      adj = self._nhgHwDefaultVrfStatus.nexthopGroupAdjacency.newMember( nhgId )
      adjSm = NhgAdjSm( self._helper, nhgConfigEntry, adj )
      self._helper.adjSm[ nhgConfigEntry.name ] = adjSm
      self._nhgNameToId[ nhgConfigEntry.name ] = nhgId
      self._nhgIdToName[ nhgId ] = nhgConfigEntry.name

   def _delNhgAdjSm( self, nhgId=None, nhgName=None ):
      '''
      If both (possibly derived) nhgId and nhgName exist in helper mappings, the
      NhgAdjSm will be deleted via the helper and the corr. adj will be removed.
      '''
      func = self._traceName + '._delNhgAdjSm:'
      if not nhgId and not nhgName:
         keyStr = 'NhgId: {}, NhgName: {}'.format( nhgId, nhgName )
         assert False, "{} Can't delete invalid NHG with {}".format( func, keyStr )

      nhgId = nhgId or self._nhgNameToId.get( nhgName )
      nhgName = nhgName or self._nhgIdToName.get( nhgId )
      if nhgId in self._nhgIdToName and nhgName in self._nhgNameToId:
         keyStr = 'NhgId: {}, NhgName: {}'.format( nhgId, nhgName )
         t8( 'Deleting NhgAdjSm with', keyStr )
         self._helper.adjSm[ nhgName ].cleanup()
         del self._helper.adjSm[ nhgName ]
         del self._nhgNameToId[ nhgName ]
         del self._nhgIdToName[ nhgId ]
         del self._nhgHwDefaultVrfStatus.nexthopGroupAdjacency[ nhgId ]

   def handleEntry( self, nhgId=None, nhgName=None ):
      '''
      This method is called whenever an entry in the FEC tables or the NHG
      Sysdb/Smash collections is added or deleted.

      If all entries exist, the NhgAdjSm will be created. If any entry is missing
      instead, it is assumed that an entry has been removed and a deletion will
      be attempted.
      '''
      func = self._traceName + '.handleEntry:'
      if not nhgId and not nhgName:
         t8( func, 'Neither NhgId:', nhgId, 'nor NhgName:', nhgName, 'are valid' )
         return

      # Ensure there is a NhgSmashEntry. This should be checked first since it's the
      # only entity with both the nhgId and nhgName, and both are needed
      nhgSmashEntry = self._fwdHelper.getNhgSmashEntry( nhgId=nhgId,
                                                        nhgName=nhgName )
      if not nhgSmashEntry:
         keyStr = 'NhgId: {}, NhgName: {}'.format( nhgId, nhgName )
         t8( func, 'No NHG Smash Entry found with', keyStr )
         self._delNhgAdjSm( nhgId=nhgId, nhgName=nhgName )
         return
      nhgId = nhgId or nhgSmashEntry.nhgId
      nhgName = nhgName or nhgSmashEntry.key.nhgName()

      if( not self._nhgIdToFecId[ nhgId ] and not
          self._nhgIdToRouteKey[ nhgId ] ):
         t8( func, 'No NHG FEC or LFIB route found with NhgId:', nhgId )
         self._delNhgAdjSm( nhgId=nhgId, nhgName=nhgName )
         return

      # Ensure there is a NHG Config Entry in Sysdb wih this nhgName
      nhgConfigEntry = self._fwdHelper.getNhgFromSysdb( nhgName=nhgName )
      if not nhgConfigEntry:
         keyStr = 'NhgId: {}, NhgName: {}'.format( nhgId, nhgName )
         t8( func, 'No NHG Config Entry found with', keyStr )
         self._delNhgAdjSm( nhgId=nhgId, nhgName=nhgName )
         return

      self._createNhgAdjSm( nhgId=nhgId, nhgName=nhgName )

   def handleMplsLfibRoute( self, reactor, routeKey ):
      func = self._traceName + '.handleMplsLfibRoute:'

      route = self._lfib.lfibRoute.get( routeKey )

      def removeNhgIdMappings( nhgId, routeKey ):
         self._routeKeyToNhgId[ routeKey ].remove( nhgId )
         if not self._routeKeyToNhgId[ routeKey ]:
            del self._routeKeyToNhgId[ routeKey ]
         assert routeKey in self._nhgIdToRouteKey[ nhgId ]
         self._nhgIdToRouteKey[ nhgId ].remove( routeKey )
         if not self._nhgIdToRouteKey[ nhgId ]:
            del self._nhgIdToRouteKey[ nhgId ]

      # Route was deleted
      if route is None:
         if routeKey in self._routeKeyToNhgId:
            oldNhgIds = set( self._routeKeyToNhgId[ routeKey ] )
            for nhgId in oldNhgIds:
               removeNhgIdMappings( nhgId, routeKey )
               self.handleEntry( nhgId=nhgId )
         return

      viaSetKey = route.viaSetKey
      if viaSetKey.viaType != LfibViaType.viaTypeMplsIp:
         return

      # Route was added and is mplsIp adjacency type
      viaSet = self._lfib.viaSet.get( viaSetKey )

      # Mark route to be processed later when a corresponding adjacency is found
      if viaSet is None:
         t8( func, 'LFIB route', routeKey, 'has no existing via set', viaSetKey,
             'so processing is deferred until the via set is added.' )
         return

      vias = []
      for vk in viaSet.viaKey.itervalues():
         via = self._lfib.mplsVia.get( vk )
         if via is None:
            t8( func, 'LFIB route', routeKey, 'has no existing via', vk,
                'so processing is deferred until the via is added.' )
            return
         vias.append( via )


      # Adjacency exists, so process route
      oldNhgIds = set( self._routeKeyToNhgId[ routeKey ] )
      for via in vias:
         if not FecIdIntfId.isNexthopGroupIdIntfId( via.intf ):
            continue
         nhgId = FecIdIntfId.intfIdToNexthopGroupId( via.intf )
         t8( func, 'Handling NHG route', routeKey, 'with NhgId', nhgId )
         if nhgId not in self._routeKeyToNhgId[ routeKey ]:
            self._routeKeyToNhgId[ routeKey ].add( nhgId )
            self._nhgIdToRouteKey[ nhgId ].add( routeKey )
         else:
            oldNhgIds.discard( nhgId )
         self.handleEntry( nhgId=nhgId )

      # Remove the stale nhgIds
      for oldNhgId in oldNhgIds:
         removeNhgIdMappings( oldNhgId, routeKey )
         self.handleEntry( nhgId=oldNhgId )

   def handleFec( self, reactor, fecKey ):
      fecId = Tac.Value( 'Smash::Fib::FecId', fecKey )
      func = self._traceName + '.handleFec:'
      if ( fecId.adjType() != AdjType.fibV4Adj ) and \
         ( fecId.adjType() != AdjType.fibV6Adj ):
         return

      def removeNhgIdMappings( nhgId, fecKey ):
         self._fecIdToNhgId[ fecKey ].remove( nhgId )
         if not self._fecIdToNhgId[ fecKey ]:
            del self._fecIdToNhgId[ fecKey ]
         assert fecKey in self._nhgIdToFecId[ nhgId ]
         self._nhgIdToFecId[ nhgId ].remove( fecKey )
         if not self._nhgIdToFecId[ nhgId ]:
            del self._nhgIdToFecId[ nhgId ]

      fec = self._fs.fec.get( fecId ) or self._fs6.fec.get( fecId )
      if not fec or fec.via is None:
         if fecKey in self._fecIdToNhgId:
            oldNhgIds = set( self._fecIdToNhgId[ fecKey ] )
            for nhgId in oldNhgIds:
               removeNhgIdMappings( nhgId, fecKey )
               self.handleEntry( nhgId=nhgId )
         return

      # Update the forward and reverse mappings for nhgId to fecId
      # with updated info from the FIB
      oldNhgIds = set( self._fecIdToNhgId[ fecKey ] )
      for i in range( len( fec.via ) ):
         nhgId = None
         viaIntfId = fec.via[ i ].intfId
         if DynamicTunnelIntfId.isDynamicTunnelIntfId( viaIntfId ):
            viaTunnelId = DynamicTunnelIntfId.tunnelId( viaIntfId )
            if TunnelId( viaTunnelId ).tunnelType() == TunnelType.nexthopGroupTunnel:
               tunnelFibEntry = self._fwdHelper.tunnelFib_.entry.get( viaTunnelId )
               if not tunnelFibEntry or not tunnelFibEntry.tunnelVia:
                  continue
               nhgId = NexthopGroupIntfIdType.nexthopGroupId(
                           tunnelFibEntry.tunnelVia[ 0 ].intfId )
         if ( not NexthopGroupIntfIdType.isNexthopGroupIntfId( viaIntfId )
              and nhgId is None ):
            continue
         nhgId = ( NexthopGroupIntfIdType.nexthopGroupId( viaIntfId )
                   if nhgId is None else nhgId )
         t8( func, 'Handling NHG FEC', fecKey, 'with NhgId', nhgId )
         if nhgId not in self._fecIdToNhgId[ fecKey ]:
            self._fecIdToNhgId[ fecKey ].add( nhgId )
            self._nhgIdToFecId[ nhgId ].add( fecKey )
         else:
            oldNhgIds.remove( nhgId )

      # Remove the stale nhgIds from the forward and reverse mappings
      for oldNhgId in oldNhgIds:
         removeNhgIdMappings( oldNhgId, fecKey )
         self.handleEntry( nhgId=oldNhgId )

      # Call handleEntry on the updated nhgIds for this FEC
      for nhgId in self._fecIdToNhgId[ fecKey ]:
         self.handleEntry( nhgId=nhgId )

   def handleNhgSmashEntry( self, reactor, nhgKey ):
      nhgId = None
      nhgName = nhgKey.nhgName()
      entry = self._nhgEntryStatus.nexthopGroupEntry.get( nhgKey )
      if entry:
         nhgId = entry.nhgId
      self.handleEntry( nhgId=nhgId, nhgName=nhgName )

   def handleNhgConfigEntry( self, reactor, nhgName ):
      self.handleEntry( nhgName=nhgName )

class NhgL3ResolverSm( Tac.Notifiee ):
   '''
   Responsible for creating the FibMonitorHelperSm once the default nhVrfStatus shows
   up in Sysdb.
   '''
   notifierTypeName = 'Routing::NexthopStatus'

   def __init__( self, nhrStatus, fs, fs6, backlog, scheduler ):
      self._traceName = 'NhgL3ResolverSm:'
      self._nhrStatus = nhrStatus
      self._backlog = backlog
      self._scheduler = scheduler
      self._fs = fs
      self._fs6 = fs6
      self._fibMonitorHelperSm = None
      Tac.Notifiee.__init__( self, self._nhrStatus )
      for vrfName in self._nhrStatus.vrf:
         self.handleVrf( vrfName )

   @Tac.handler( 'vrf' )
   def handleVrf( self, vrfName ):
      if vrfName != DEFAULT_VRF:
         return

      vrf = self._nhrStatus.vrf.get( vrfName )
      if vrf:
         self._fibMonitorHelperSm = Tac.newInstance( 'Routing::FibMonitorHelperSm',
                                                    'etba',
                                                    vrf,
                                                    self._backlog,
                                                    self._fs,
                                                    self._fs6,
                                                    self._scheduler )
         t8( self._traceName, 'Created new fibMonitorHelperSm for vrf', vrfName )
      else:
         self._fibMonitorHelperSm = None
         t8( self._traceName, 'Deleted fibMonitorHelperSm for vrf', vrfName )

class MplsNhgHardwareStatusSm( object ):
   '''
   This SM simply instantiates and encapsulates everything pertaining to the creation
   of NexthopGroup Adjacencies in Sysdb. Refer to AID4354 for implementation details.
   '''

   def __init__( self,
                 # outputs
                 routingHwStatus,
                 nhgHwDefaultVrfStatus,
                 nhrConfig,
                 proactiveArpNexthopConfig,
                 # inputs
                 nhrStatus,
                 brConfig,
                 brStatus,
                 fs,
                 fs6,
                 arpSmash,
                 fwdHelper,
                 lfib ):
      # Only NHGs with MPLS Push routes are currently supported
      routingHwStatus.nexthopGroupMplsLabelAction.add( MplsLabelAction.push )
      routingHwStatus.nexthopGroupMplsLabelStackSize = 6
      routingHwStatus.nexthopGroupTypeSupported[ NhgType.mpls ] = True
      self.routingHwStatus = routingHwStatus

      # Need to create iraEtbaRoot in case this is created before the ira plugin's
      # agent init is called
      iraEtbaRoot = Tac.root.newEntity( 'Ira::IraEtbaRoot', 'ira-etba-root' )
      if not iraEtbaRoot.nexthopGroupConfig:
         iraEtbaRoot.nexthopGroupConfig = ()

      nhVrfConfig = nhrConfig.newVrf( DEFAULT_VRF, NhFecIdType.fecId )
      nhgAdjSmHelper = NhgAdjSmHelper( brConfig, brStatus, nhVrfConfig, arpSmash,
                                       proactiveArpNexthopConfig )

      scheduler = Tac.Type( 'Ark::TaskSchedulerRoot' ).findOrCreateScheduler()
      backlog = Tac.newInstance( 'Routing::NexthopBacklog' )
      self.nhgL3ResolverSm = NhgL3ResolverSm( nhrStatus, fs, fs6, backlog, scheduler)
      self.nhgNhBacklogSm = NhgNhBacklogSm( fwdHelper, nhgAdjSmHelper, nhrStatus,
                                            brConfig, brStatus, backlog )

      self.nhgArpSm = NhgArpSm( nhgAdjSmHelper, arpSmash )
      self.nhgBridgingStatusSm = NhgBridgingStatusSm( nhgAdjSmHelper, brStatus )
      self.nhgAdjManagerSm = NhgAdjManagerSm( fwdHelper, nhgAdjSmHelper,
                                              nhgHwDefaultVrfStatus, lfib )

factoryInstances = []
def mplsNhgHardwareStatusSmFactory( bridge ):
   em = bridge.em()
   defaultVrf = DEFAULT_VRF
   path = 'routing/hardware/nexthopgroup/status/'
   nhgHwDefaultVrfStatus = em.entity( path ).vrfStatus[ defaultVrf ]

   factory = MplsNhgHardwareStatusSm(
         routingHwStatus=em.entity( 'routing/hardware/status' ),
         nhgHwDefaultVrfStatus=nhgHwDefaultVrfStatus,
         nhrConfig=bridge.nhrConfig,
         proactiveArpNexthopConfig=bridge.proactiveArpNhgNexthopConfig,
         nhrStatus=bridge.nhrStatus,
         brConfig=bridge.brConfig,
         brStatus=bridge.brStatus,
         fs=bridge.forwardingStatus_,
         fs6=bridge.forwarding6Status_,
         arpSmash=bridge.arpSmash_,
         fwdHelper=bridge.fwdHelper,
         lfib=bridge.lfib_,
         )

   factoryInstances.append( factory )
   return factory
