# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import collections

from MplsEtbaLib import (
      getIntfVlan,
      getL2Intf,
      isRoutedPort,
   )
from TypeFuture import TacLazyType
import Tac
import Tracing

ArnetIntfId = TacLazyType( 'Arnet::IntfId' )
DynamicTunnelIntfId = TacLazyType( 'Arnet::DynamicTunnelIntfId' )
EthAddr = TacLazyType( 'Arnet::EthAddr' )
HwAdjBaseKey = TacLazyType( "Mpls::Hardware::AdjacencyBaseKey" )
FecIdIntfId = Tac.Type( 'Arnet::FecIdIntfId' )
HwL2Via = TacLazyType( 'Mpls::Hardware::L2Via' )
HwL2ViaSet = TacLazyType( 'Mpls::Hardware::L2ViaSet' )
ExtFecAdjacency = TacLazyType( 'Mpls::LFib::ExtFecAdjacency' )
HwRoute = TacLazyType( 'Mpls::Hardware::Route' )
RouteKey = TacLazyType( 'Mpls::RouteKey' )
HwVia = TacLazyType( 'Mpls::Hardware::Via' )
IpGenAddr = TacLazyType( 'Arnet::IpGenAddr' )
IpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
LfibViaSetType = TacLazyType( 'Mpls::LfibViaSetType' )
ViaType = TacLazyType( 'Mpls::LfibViaType' )
AdjType = TacLazyType( 'Mpls::AdjacencyType' )

handle = Tracing.Handle( 'MplsEtbaHardwareStatusSm' )
t1 = handle.trace1
t2 = handle.trace2
t9 = handle.trace9

class EntityReactor( Tac.Notifiee ):
   """
   Reacts to any attribute of the given entity or any attribute of any
   sub entity changing.
   """

   notifierTypeName = "*"

   def __init__( self, entity, callback ):
      Tac.Notifiee.__init__( self, entity, filtered=False )
      self.childReactors = collections.defaultdict( dict )
      for attrName in entity.attributes:
         if attrName == 'parent':
            continue
         attr = entity.tacType.attr( attrName )
         if attr.instantiating:
            if attr.isCollection:
               for key, value in getattr( entity, attrName ).iteritems():
                  self.childReactors[ attrName ][ key ] = EntityReactor( value,
                                                                         callback )
            else:
               child = getattr( entity, attrName )
               if child is not None:
                  self.childReactors[ attrName ][ None ] = EntityReactor( child,
                                                                          callback )
      self.entity = entity
      self.callback = callback

   def onAttribute( self, attr, key ):
      attrName = attr.name
      if attr.isCollection:
         value = getattr( self.entity, attrName ).get( key )
         if attr.instantiating:
            if value is not None:
               self.childReactors[ attrName ][ key ] = EntityReactor( value,
                                                                      self.callback )
            else:
               del self.childReactors[ attrName ][ key ]
      else:
         key = None
      t9( 'entity: %s attr: %s key: %s changed!' %
          ( self.notifier_, attrName, key ) )
      self.callback()

class FdbStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Bridging::FdbStatus'

   def __init__( self, fdbStatus, callback ):
      self.callback = callback
      self.host = {}
      self.fdbStatus = fdbStatus
      Tac.Notifiee.__init__( self, fdbStatus )

   @Tac.handler( 'learnedHost' )
   def handleHost( self, key ):
      lh = self.fdbStatus.learnedHost.get( key, None )
      if lh is None:
         # Deletion
         if key in self.host:
            del self.host[ key ]
         self.callback()
         return
      intf = self.host.get( key, None )
      if intf != lh.intf:
         self.host[ key ] = lh.intf
         self.callback()
         return

class BridgingStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Bridging::Status'

   def __init__( self, brStatus, callback ):
      self.callback = callback
      self.brStatus = brStatus
      self.fdbReactor = {}
      Tac.Notifiee.__init__( self, brStatus )

   @Tac.handler( 'fdbStatus' )
   def handleFdbStatus( self, key ):
      fdbStatus = self.brStatus.fdbStatus.get( key, None )
      if not fdbStatus:
         if key in self.fdbReactor:
            del self.fdbReactor[ key ]
      else:
         self.fdbReactor[ key ] = FdbStatusReactor( fdbStatus, self.callback )
      self.callback()

class AgentRootReactor( Tac.Notifiee ):
   notifierTypeName = "MplsEtbaAgent::MplsEtbaAgentRoot"

   def __init__( self, agentRoot, handleSyncRequest, handleInterval ):
      super( AgentRootReactor, self ).__init__( agentRoot )
      self.agentRoot_ = agentRoot
      self.handleSyncRequest_ = handleSyncRequest
      self.handleInterval_ = handleInterval

   @Tac.handler( "syncMplsHardwareStatusSmRequest" )
   def handleSyncRequest( self ):
      value = self.agentRoot_.syncMplsHardwareStatusSmRequest
      self.agentRoot_.syncMplsHardwareStatusSmComplete = (
            self.handleSyncRequest_( value ) )

   @Tac.handler( "hwStatusSmInterval" )
   def handleInterval( self ):
      value = self.agentRoot_.hwStatusSmInterval
      self.handleInterval_( value )

def createHwViaFromVia( via, vlanId=None, l2Intf=None, macAddr=None ):
   '''Creates a Mpls::Hardware::L2Via from a Mpls::LfibMplsVia'''
   hwL2Via = HwL2Via()
   hwL2Via.labelAction = via.labelAction
   # Only a single label push/swap is supported here.
   if via.labelStack.stackSize:
      hwL2Via.outLabel = via.labelStack.topLabel()
   hwL2Via.payloadType = via.payloadType
   hwL2Via.ttlMode = via.ttlMode
   hwL2Via.dscpMode = via.dscpMode
   if vlanId:
      hwL2Via.vlanId = vlanId
   if l2Intf:
      hwL2Via.l2Intf = l2Intf
   if macAddr:
      hwL2Via.macAddr = macAddr
   hwL2Via.skipEgressAcl = False

   hwVia = HwVia()
   hwVia.recursiveNextHop = via.nextHop
   hwVia.resolvedNextHop = via.nextHop
   hwVia.l3Intf = via.intf
   hwVia.route = IpGenPrefix()
   hwVia.l2Via = hwL2Via
   return hwVia

class HwStatusUpdater( object ):
   """
   An interface to minimize changes to Mpls::Hardware::Status in Sysdb.

   When we are instantiated, keep the "real" collection (the one mounted from
   Sysdb), and an empty shadow.  When we add routes or adjacencies, we add them
   to the shadow.  When we are asked to reconcile, we look at the state of Sysdb,
   and only modify the values that have changed, and remove the entries
   that have not been added to the shadow.

   This allows the state to be built from scratch as needed, without causing
   excessive EntityLog overhead in Sysdb from deleting and recreating collection
   members.
   """
   def __init__( self, mplsHwStatus ):
      self.realMplsHwStatus = mplsHwStatus
      self.newMplsHwStatus = Tac.newInstance( "Mpls::Hardware::Status", "shadow" )

   def reconcileWithSysdb( self ):
      t9( 'reconcileWithSysdb for', self.realMplsHwStatus.name,
          self.realMplsHwStatus )
      routeStaleSet = set( self.realMplsHwStatus.route )
      adjacencyStaleSet = set( self.realMplsHwStatus.adjacency )
      l2AdjacencyStaleSet = set( self.realMplsHwStatus.l2Adjacency )

      for routeKey in self.newMplsHwStatus.route:
         t9( 'processing shadow route', routeKey )
         shadowRoute = self.newMplsHwStatus.route[ routeKey ]
         realRoute = None
         try:
            realRoute = self.realMplsHwStatus.route[ routeKey ]
            if realRoute.adjBaseKey != shadowRoute.adjBaseKey:
               del self.realMplsHwStatus.route[ routeKey ]
               realRoute = None
            elif realRoute.unprogrammed != shadowRoute.unprogrammed:
               del self.realMplsHwStatus.route[ routeKey ]
               realRoute = None
         except KeyError:
            pass

         if ( not realRoute or
              ( realRoute.metric != shadowRoute.metric or
                realRoute.requestedFecId != shadowRoute.requestedFecId or
                realRoute.installedFecId != shadowRoute.installedFecId ) ):
            del self.realMplsHwStatus.route[ routeKey ]
            self.realMplsHwStatus.route.addMember( shadowRoute )

         routeStaleSet.discard( routeKey )

      for adjIndex in self.newMplsHwStatus.l2Adjacency:
         t9( 'processing shadow l2adjacency', adjIndex )
         shadowL2Adjacency = self.newMplsHwStatus.l2Adjacency[ adjIndex ]
         realL2Adjacency = self.realMplsHwStatus.l2Adjacency.newMember( adjIndex )
         realL2Adjacency.l2ViaSet = shadowL2Adjacency.l2ViaSet

         l2AdjacencyStaleSet.discard( adjIndex )

      for adjIndex in self.newMplsHwStatus.adjacency:
         t9( 'processing shadow adjacency', adjIndex )
         shadowAdjacency = self.newMplsHwStatus.adjacency[ adjIndex ]
         realAdjacency = self.realMplsHwStatus.adjacency.newMember( adjIndex )
         realAdjacency.viaSetType = shadowAdjacency.viaSetType
         if shadowAdjacency.l2Adjacency:
            realAdjacency.l2Adjacency = self.realMplsHwStatus.l2Adjacency[
               shadowAdjacency.l2Adjacency.index ]
         else:
            realAdjacency.l2Adjacency = None
         oldVias = set( realAdjacency.via )
         for via in shadowAdjacency.via:
            realAdjacency.via[ via ] = True
            oldVias.discard( via )
         for via in oldVias:
            del realAdjacency.via[ via ]
         adjacencyStaleSet.discard( adjIndex )

      for label in routeStaleSet:
         t9( "Remove stale route", label )
         del self.realMplsHwStatus.route[ label ]
      for adjIdx in adjacencyStaleSet:
         t9( "Remove stale adjacency", adjIdx )
         del self.realMplsHwStatus.adjacency[ adjIdx ]
      for adjIdx in l2AdjacencyStaleSet:
         t9( "Remove stale l2Adjacency", adjIdx )
         del self.realMplsHwStatus.l2Adjacency[ adjIdx ]

      clearHwMplsStatus( self.newMplsHwStatus )

   def addRoute( self, route ):
      routeKey = route.routeKey
      t9( "Add route", routeKey, self.realMplsHwStatus )
      return self.newMplsHwStatus.route.addMember( route )

   def newAdjacency( self, adjIndex ):
      t9( "New adjacency", adjIndex, self.realMplsHwStatus )
      return self.newMplsHwStatus.adjacency.newMember( adjIndex )

   def newL2Adjacency( self, adjIndex ):
      t9( "New l2Adjacency", adjIndex, self.realMplsHwStatus )
      return self.newMplsHwStatus.l2Adjacency.newMember( adjIndex )

def clearHwMplsStatus( entity ):
   entity.l2Adjacency.clear()
   entity.adjacency.clear()
   entity.route.clear()

class LfibViaSetContainer( object ):
   '''A container object for an Lfib ViaSet and all of its vias.
   This is necessary, since they are not contained in a single object in smash.
   '''
   def __init__( self, viaSetKey, vias ):
      self.viaSetKey = viaSetKey
      self.vias = vias

class LfibRouteContainer( object ):
   def __init__( self, routeKey, metric, viaSetContainer ):
      self.routeKey = routeKey
      self.metric = metric
      self.viaSet = viaSetContainer
      self.backupViaSet = None

class MplsHardwareStatusSm( object ):
   def __init__( self,
                 # outputs
                 routingHardwareStatus,
                 mplsHwStatus,
                 mplsHwTunnelStatus,
                 mplsBackupHwStatus,
                 mplsBackupHwTunnelStatus,
                 proactiveArpNexthopConfig,
                 # inputs
                 lfib,
                 arpSmash,
                 bridgingConfig,
                 bridgingStatus,
                 fwdHelper,
                 agentRoot=None,
                 ):

      if agentRoot:
         self.interval_ = agentRoot.hwStatusSmInterval
      else:
         self.interval_ = 1
      self.count = 1
      self.routingHwStatus_ = routingHardwareStatus

      # The raw types here should not be written to directly.
      # they should be written to by the HwStatusUpdater class
      # They can safely be used for read only access to the
      # associated Mpls::Hardware::Status
      self._rawHwStatus = mplsHwStatus
      self._rawHwTunnelStatus = mplsHwTunnelStatus
      self._rawBackupHwStatus = mplsBackupHwStatus
      self._rawBackupHwTunnelStatus = mplsBackupHwTunnelStatus

      clearHwMplsStatus( mplsHwStatus )
      clearHwMplsStatus( mplsHwTunnelStatus )
      clearHwMplsStatus( mplsBackupHwStatus )
      clearHwMplsStatus( mplsBackupHwTunnelStatus )
      self.proactiveArpNexthopConfig_ = proactiveArpNexthopConfig
      self.lfib_ = lfib
      self.extFecHwStatus = {}
      self.arpSmash_ = arpSmash
      self.brConfig_ = bridgingConfig
      self.brStatus_ = bridgingStatus
      self.fwdHelper = fwdHelper
      self.agentRoot_ = agentRoot

      self.routingHwStatus_.mplsPushSupported = True
      self.adjIndex = 1
      # Shared for both Hw adj and L2 adj
      self.viaSetKeyToHwAdjIdx = {}
      self.seenViaSetKeysSinceUpdate = set()
      # All currently used ViaSetContainers keyed by ViaSetKey
      self.viaSetContainers = {}

      # self.timer_ must be created before any of the reactors that use kickTimer.
      self.timer_ = Tac.ClockNotifiee( self.timerEvent, timeMin=Tac.endOfTime )
      self.timerEvent()

      self.mplsReactor = EntityReactor( self.lfib_, self.kickTimer )
      self.arpReactor = EntityReactor( self.arpSmash_, self.kickTimer )
      self.brConfigReactor = EntityReactor( self.brConfig_, self.kickTimer )

      if self.agentRoot_:
         self.agentRootReactor_ = AgentRootReactor( self.agentRoot_,
                                                    self.handleSyncRequest,
                                                    self.handleHwStatusSmInterval )

      # FdbStatus has the timestamp for the learned hosts, that gets updated on
      # packets forwarding, we don't want to re-evaluate the Mpls route when
      # just the timestamp changes. So, we end up having a custom reactor
      # for Bridging::Status
      self.fdbReactor = BridgingStatusReactor( self.brStatus_, self.kickTimer )

   def handleSyncRequest( self, value ):
      t9( "MplsHardwareStatusSm.handleSyncRequest request:", value )
      self.timerEvent()
      t9( "MplsHardwareStatusSm.handleSyncRequest set response:", value )
      return value

   def handleHwStatusSmInterval( self, value ):
      t9( "MplsHardwareStatusSm.handleHwStatusSmInterval:", value )
      self.interval_ = value
      self.timer_.timeMin = ( Tac.now() + self.interval_ )

   def kickTimer( self ):
      if self.timer_.timeMin != Tac.endOfTime:
         return
      self.timer_.timeMin = ( Tac.now() + self.interval_ )

   @staticmethod
   def isTunnelIntf( intfId ):
      return DynamicTunnelIntfId.isDynamicTunnelIntfId( intfId )

   def updateProactiveArpNexthopConfig( self ):
      """
      Updates the ARP resolver collection by reconciling with the current set of
      adjacencies.  Safe access for _rawHwStatus (read-only).
      """
      resolvedNexthops = set()
      for adj in self._rawHwStatus.adjacency.values():
         for via in adj.via:
            isL3InfoValid = not via.resolvedNextHop.isAddrZero and via.l3Intf
            if not isL3InfoValid:
               continue
            entry = Tac.ValueConst( 'Arp::ProactiveArp::Entry',
                                    via.resolvedNextHop, via.l3Intf )
            resolvedNexthops.add( entry )

      delSet = ( set( self.proactiveArpNexthopConfig_.request.keys() ) -
                 resolvedNexthops )
      addSet = ( resolvedNexthops -
                 set( self.proactiveArpNexthopConfig_.request.keys() ) )
      for i in addSet:
         request = Tac.Value( 'Arp::ProactiveArp::Request', i )
         self.proactiveArpNexthopConfig_.addRequest( request )
      for i in delSet:
         del self.proactiveArpNexthopConfig_.request[ i ]

   def _cleanViaSetKeyToHwAdjIdx( self ):
      '''To be called post-update, to clear out all ViaSetKeys that are no longer
      in use.
      '''
      unused = ( set( self.viaSetKeyToHwAdjIdx.keys() ) -
                 self.seenViaSetKeysSinceUpdate )
      for vsk in unused:
         del self.viaSetKeyToHwAdjIdx[ vsk ]

   def _hwIndexForViaSetKey( self, vsk ):
      self.seenViaSetKeysSinceUpdate.add( vsk )
      if vsk not in self.viaSetKeyToHwAdjIdx:
         func = 'MplsHardwareStatusSm._hwIndexForViaSetKey'
         self.viaSetKeyToHwAdjIdx[ vsk ] = self.adjIndex
         t1( func, "ViaSet", vsk, " now bound to hwAdjIdx", self.adjIndex )
         self.adjIndex += 1
      return self.viaSetKeyToHwAdjIdx[ vsk ]

   def _viaSetTypeToAdjType( self, viaSetKey ):
      func = 'MplsHardwareStatusSm._viaSetTypeToAdjType'
      if viaSetKey.viaType == ViaType.viaTypeMplsIp:
         return AdjType.mplsIp
      elif viaSetKey.viaType == ViaType.viaTypeExternalFec:
         return AdjType.mplsExternalFec
      else:
         t1( func, 'found incompatible viaType', viaSetKey )
         # We should not find any other viaType here
         assert False
         return None

   def _resolveMplsIpViaSet( self, viaSet, hwStatus ):
      '''
      Takes an LfibViaSetContainer of an entry in the transit LFIB and creates the
      appropriate HW status objects, and will return the associated HW adjacency
      object.
      '''
      func = 'MplsHardwareStatusSm._resolveMplsIpViaSet'
      vsk = viaSet.viaSetKey
      hwAdj = hwStatus.newAdjacency( self._hwIndexForViaSetKey( vsk ) )
      tunnelVia = []
      for via in viaSet.vias:
         intfId = via.getRawAttribute( 'intf' )
         if self.isTunnelIntf( intfId ):
            # For any viaSet with any tunnel via, just create the adjacency object
            # without any VIAs.
            tunnelVia.append( via )

      if tunnelVia:
         t1( func, 'Tunnel viaSet', vsk, 'with',
             len( tunnelVia ), 'vias' )
         return hwAdj, tunnelVia

      t1( func, 'Resolve viaSet', vsk )
      hwL2ViaSet = set()

      hwAdj.viaSetType = vsk.viaSetType
      t1( func, '    Setting viaSetType to', hwAdj.viaSetType,
         'from', vsk.viaSetType )

      for via in viaSet.vias:
         t1( func, '  processing via', via )
         if not via.intf:
            t1( func, '    via has no L3 interface' )
            continue

         if FecIdIntfId.isNexthopGroupIdIntfId( via.intf ):
            hwVia = HwVia()
            hwL2Via = HwL2Via()
            hwVia.l3Intf = via.intf
            hwL2Via.labelAction = via.labelAction
            hwVia.l2Via = hwL2Via
            hwAdj.via[ hwVia ] = True
            t1( func, 'added hwAdj for NHG', via.intf )
            continue

         vlanId = getIntfVlan( self.brConfig_, via.intf )
         if vlanId is None:
            t1( func, '    unable to get intf VLAN ID for intf', via.intf )
            continue

         dstMac = self.fwdHelper.resolveL2Nexthop( via.intf, via.nextHop )
         if isRoutedPort( via.intf ):
            l2Intf = via.intf
         else:
            if dstMac:
               l2Intf = getL2Intf( self.brStatus_, vlanId, dstMac )
               if l2Intf == ArnetIntfId():
                  t1( func, '    note: no L2 intf for', dstMac, 'on vlan', vlanId )
            else:
               t1( func, '    note: no L2 nexthop for', via.nextHop, 'on', via.intf )
               l2Intf = ArnetIntfId()

         hwVia = createHwViaFromVia( via, vlanId=vlanId, l2Intf=l2Intf,
                                     macAddr=dstMac )
         hwL2Via = hwVia.l2Via
         if dstMac:
            t1( func, '    resolved to', hwVia.l3Intf, hwL2Via.l2Intf,
                hwL2Via.macAddr )
            hwL2ViaSet.add( Tac.const( hwL2Via ) )
         else:
            t1( func, '    unresolved with', hwVia.l3Intf, 'l2Intf',
                repr( hwL2Via.l2Intf ), 'macAddr', hwL2Via.macAddr )
         hwAdj.via[ hwVia ] = True

      if hwL2ViaSet:
         # When there are any l2Vias, create an l2Adjacency object with the
         # l2 via set.
         hwL2Adjacency = hwStatus.newL2Adjacency( self._hwIndexForViaSetKey( vsk ) )
         for v in hwL2ViaSet:
            l2ViaSetObj = HwL2ViaSet()
            l2ViaSetObj.l2Via[ v ] = True
            hwL2Adjacency.l2ViaSet = l2ViaSetObj
         hwAdj.l2Adjacency = hwL2Adjacency
      return hwAdj, None

   def _resolveExternalFecViaSet( self, viaSet, hwExtStatus ):
      '''
      Takes an LfibViaSetContainer for a viaSet in the transit LFIB and creates
      the appropriate HW status objects, and will return the associated HW adjacency
      object.
      '''
      adjIndex = self._hwIndexForViaSetKey( viaSet.viaSetKey )
      hwAdj = hwExtStatus.newAdjacency( adjIndex )
      return hwAdj

   def _addTunnelFwdStatus( self, routeKey, viaSet, viaList, hwTunnelStatus ):
      func = 'MplsHardwareStatusSm._addTunnelFwdStatus'
      hwAdjacency = hwTunnelStatus.newAdjacency( viaSet.viaSetKey.index )
      hwL2ViaSet = HwL2ViaSet()
      for via in viaList:
         hwVia = createHwViaFromVia( via )
         hwAdjacency.via[ hwVia ] = True
         hwL2ViaSet.l2Via[ hwVia.l2Via ] = True
      if viaList:
         hwL2Adjacency = hwTunnelStatus.newL2Adjacency( viaSet.viaSetKey.index )
         hwL2Adjacency.l2ViaSet = hwL2ViaSet
         hwAdjacency.l2Adjacency = hwL2Adjacency

      hwAdjKey = HwAdjBaseKey( viaSet.viaSetKey.index, 'mplsUnknown' )
      route = HwRoute( routeKey, hwAdjKey, False )
      t1( func, 'add tunnel route', routeKey, '->', hwAdjKey )
      hwTunnelStatus.addRoute( route )

   def _getLfibViaSetContainer( self, viaSetKey, vias ):
      if viaSetKey in self.viaSetContainers:
         viaSetCont = self.viaSetContainers[ viaSetKey ]
         viaSetCont.vias = vias
      else:
         viaSetCont = LfibViaSetContainer( viaSetKey, vias )
         self.viaSetContainers[ viaSetKey ] = viaSetCont
      return viaSetCont

   def _getLfibRouteContainer( self, route, backupViaSet=False,
                               routeContainer=None ):
      '''Returns an LfibRouteContainer'''
      func = 'MplsHardwareStatusSm._getLfibRoute'
      lfib = self.lfib_
      viaSetKey = route.viaSetKey
      viaSet = lfib.viaSet.get( viaSetKey )
      if viaSet is None:
         # This is a transient condition that the route does not have a
         # viaSet. Will process again when the route gets updated.
         t1( func, 'no viaSet', viaSetKey, 'for route', route.key )
         return None

      if backupViaSet:
         # In case backupViaSet, reset viaSetKey and viaSet
         viaSetKey = viaSet.backupViaSetKey
         viaSet = None
         if viaSetKey:
            viaSet = lfib.viaSet.get( viaSetKey )
         if viaSet is None:
            # This is a transient condition that the route does not have a
            # viaSet. Will process again when the route gets updated.
            t1( func, 'no backupViaSet', viaSetKey, 'for route', route.key )
            return None

      vias = []
      if viaSetKey.viaType == ViaType.viaTypeMplsIp:
         for vk in viaSet.viaKey.itervalues():
            via = lfib.mplsVia.get( vk )
            if via is None:
               t1( func, 'no via', vk )
               return None
            vias.append( via )
      elif viaSetKey.viaType == ViaType.viaTypeExternalFec:
         if len( viaSet.viaKey ) != 1:
            t1( func, 'invalid number of vias for external FEC via. vsk:', viaSetKey,
                'vias:', len( viaSet.viaKey ) )

         vk = viaSet.viaKey[ 0 ]
         via = lfib.extFecVia.get( vk )
         if via is None:
            t1( func, 'no via', vk )
            return None

         vias.append( via )
      else:
         t1( func, 'ignored non-mpls or ext fec via set', viaSetKey )
         return None

      viaSetCont = self._getLfibViaSetContainer( viaSetKey, vias )
      if not backupViaSet:
         routeContainer = LfibRouteContainer( route.key, route.metric, viaSetCont )
         # Recursive call once to populate the backupViaSet if it exists
         self._getLfibRouteContainer( route, backupViaSet=True,
                                      routeContainer=routeContainer )
      else:
         assert routeContainer
         routeContainer.backupViaSet = viaSetCont

      return routeContainer

   def _updateMplsIpHwStatusLfib( self ):
      func = 'MplsHardwareStatusSm._updateMplsIpHwStatusLfib'
      lfib = self.lfib_
      hwStatus = HwStatusUpdater( self._rawHwStatus )
      backupHwStatus = HwStatusUpdater( self._rawBackupHwStatus )
      hwExtStatus = hwStatus
      hwTunnelStatus = HwStatusUpdater( self._rawHwTunnelStatus )
      backupHwTunnelStatus = HwStatusUpdater( self._rawBackupHwTunnelStatus )
      self.seenViaSetKeysSinceUpdate.clear()
      self.viaSetContainers.clear()
      lfibRoutes = {}
      resolveMplsViaSets = set()
      resolveBackupMplsViaSets = set()
      resolveExtViaSets = set()
      self.extFecHwStatus.clear()

      for route in lfib.lfibRoute.itervalues():
         t1( func, 'process route', route.key )
         lfibRouteCont = self._getLfibRouteContainer( route )
         if lfibRouteCont is not None:
            lfibRoutes[ route.key ] = lfibRouteCont
            viaSetKey = route.viaSetKey
            if viaSetKey.viaSetType == LfibViaSetType.multicast:
               t1( func, 'Found multicast viaSet' )
            if viaSetKey.viaType == ViaType.viaTypeMplsIp:
               resolveMplsViaSets.add( lfibRouteCont.viaSet )
               if lfibRouteCont.backupViaSet:
                  resolveBackupMplsViaSets.add( lfibRouteCont.backupViaSet )
            elif viaSetKey.viaType == ViaType.viaTypeExternalFec:
               resolveExtViaSets.add( lfibRouteCont.viaSet )
            else:
               # For all other viaTypes, we should have received None for the
               # lfibRouteCont
               assert False, "Unexpected viaType: " + viaSetKey.viaType

      viaSetToHwAdj = {}
      tunnelViaSetToVias = {}
      for viaSet in resolveMplsViaSets:
         viaSetToHwAdj[ viaSet ], tunnelViaList = \
               self._resolveMplsIpViaSet( viaSet, hwStatus )
         if tunnelViaList:
            tunnelViaSetToVias[ viaSet ] = tunnelViaList

      # For backup via sets
      backupViaSetToHwAdj = {}
      backupTunnelViaSetToVias = {}
      for viaSet in resolveBackupMplsViaSets:
         backupViaSetToHwAdj[ viaSet ], tunnelViaList = \
               self._resolveMplsIpViaSet( viaSet, backupHwStatus )
         if tunnelViaList:
            backupTunnelViaSetToVias[ viaSet ] = tunnelViaList

      extViaSet = set()
      for viaSet in resolveExtViaSets:
         viaSetToHwAdj[ viaSet ] = self._resolveExternalFecViaSet( viaSet,
                                                                   hwExtStatus )
         extViaSet.add( viaSet )

      # Finally create HW route for each LFIB route
      for routeKey in sorted( lfibRoutes ):
         t1( func, 'finish route', routeKey )
         lfibRouteCont = lfibRoutes[ routeKey ]
         metric = lfibRouteCont.metric
         viaSet = lfibRouteCont.viaSet
         hwAdj = viaSetToHwAdj[ viaSet ]
         resolved = False
         t1( func, '  metric', metric, 'hwAdj', hwAdj.index )
         if hwAdj.l2Adjacency:
            t1( func, '    resolved' )
            resolved = True
         elif viaSet in tunnelViaSetToVias:
            t1( func, '    resolved using tunnel' )
            resolved = True
         elif viaSet in extViaSet:
            t1( func, '    resolved using ext FEC' )
            resolved = True
         else:
            t1( func, '    unresolved (no L2 adjacency)' )

         vsk = viaSet.viaSetKey
         adjType = self._viaSetTypeToAdjType( vsk )
         if resolved:
            if viaSet in tunnelViaSetToVias:
               resolvedViaList = tunnelViaSetToVias[ viaSet ]
               self._addTunnelFwdStatus( routeKey, viaSet, resolvedViaList,
                                         hwTunnelStatus )
            elif viaSet in extViaSet:
               hwAdjKey = HwAdjBaseKey( viaSet.viaSetKey.index, adjType )
               extRoute = HwRoute( routeKey, hwAdjKey, False )
               self.extFecHwStatus[ routeKey.topLabel ] = viaSet
               hwExtStatus.addRoute( extRoute )
         hwAdjKey = Tac.Value( "Mpls::Hardware::AdjacencyBaseKey",
                               self._hwIndexForViaSetKey( vsk ), adjType )
         route = HwRoute( routeKey, hwAdjKey, False )
         route.metric = metric
         t1( func, 'added route', routeKey, '->', vsk )
         hwStatus.addRoute( route )

         # Populate the backup via set
         backupViaSet = lfibRouteCont.backupViaSet
         backupHwAdj = backupViaSetToHwAdj.get( backupViaSet )
         if backupHwAdj:
            if backupViaSet in backupTunnelViaSetToVias:
               resolvedViaList = backupTunnelViaSetToVias[ backupViaSet ]
               self._addTunnelFwdStatus( routeKey, backupViaSet, resolvedViaList,
                                         backupHwTunnelStatus )
            vsk = backupViaSet.viaSetKey
            backupHwAdjKey = Tac.Value( "Mpls::Hardware::AdjacencyBaseKey",
                               self._hwIndexForViaSetKey( vsk ), adjType )
            backupRoute = HwRoute( routeKey, backupHwAdjKey, False )
            backupRoute.metric = metric
            t1( func, 'added backup route', routeKey, '->', vsk )
            backupHwStatus.addRoute( backupRoute )

      hwStatus.reconcileWithSysdb()
      # hwExtStatus.reconcileWithSysdb() above hwExtStatus = hwStatus
      hwTunnelStatus.reconcileWithSysdb()
      backupHwStatus.reconcileWithSysdb()
      backupHwTunnelStatus.reconcileWithSysdb()

      self._cleanViaSetKeyToHwAdjIdx()

   def timerEvent( self ):
      t9( 'MplsHardwareStatusSm.timerEvent: Start' )
      self.count += 1
      self.timer_.timeMin = Tac.endOfTime
      self._updateMplsIpHwStatusLfib()
      self.updateProactiveArpNexthopConfig()
      t9( 'MplsHardwareStatusSm.timerEvent: End' )

def mplsHardwareStatusSmFactory( bridge ):
   em = bridge.em()
   return MplsHardwareStatusSm(
         routingHardwareStatus=em.entity( 'routing/hardware/status' ),
         mplsHwStatus=em.entity( 'routing/hardware/mpls/status' ),
         mplsHwTunnelStatus=em.entity( 'routing/hardware/mpls/tunnel-status' ),
         mplsBackupHwStatus=em.entity( 'routing/hardware/mpls/backup-status' ),
         mplsBackupHwTunnelStatus=em.entity( 
                                 'routing/hardware/mpls/backup-tunnel-status' ),
         proactiveArpNexthopConfig=bridge.proactiveArpNexthopConfig,
         lfib=bridge.lfib_,
         arpSmash=bridge.arpSmash_,
         bridgingConfig=bridge.brConfig,
         bridgingStatus=bridge.brStatus,
         fwdHelper=bridge.fwdHelper,
         )
