# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
'''
This file adds some utilities for use across any decap plugin
for Etba.
'''
from IpLibConsts import DEFAULT_VRF
import Tac
import Tracing
from TypeFuture import TacLazyType

th = Tracing.Handle( 'EbraTestBridgeMplsDecapUtils' )
terr = th.trace0
tinfo = th.trace1
tverb = th.trace2
tfunc = th.trace8

PacketFormat = Tracing.HexDump

EthIntfId = TacLazyType( 'Arnet::EthIntfId' )
LfibViaSetType = TacLazyType( 'Mpls::LfibViaSetType' )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )
PortChannelIntfId = TacLazyType( 'Arnet::PortChannelIntfId' )
VlanIntfId = TacLazyType( 'Arnet::VlanIntfId' )

def _ipLookupVia( vrfName, addressFamily ):
   key = Tac.newInstance( 'Mpls::LfibViaKey' )
   via = Tac.newInstance( 'Mpls::LfibIpLookupVia', key, vrfName, addressFamily )
   return via

def handleExpNull( bridge, label ):
   func = 'handleExpNull'
   tfunc( func, label )
   if label == MplsLabel.explicitNullIpv4:
      tfunc( func, 'matched IPv4 Explicit NULL for VRF', DEFAULT_VRF )
      via = _ipLookupVia( DEFAULT_VRF, 'ipv4' )
   elif label == MplsLabel.explicitNullIpv6:
      tfunc( func, 'matched IPv6 Explicit NULL for VRF', DEFAULT_VRF )
      via = _ipLookupVia( DEFAULT_VRF, 'ipv6' )
   else:
      tfunc( func, 'no match for label', label )
      via = None
   return via

def findDecapRoute( bridge, label, viaSetType=LfibViaSetType.unicast ):
   func = 'findDecapRoute'
   tfunc( func, label )
   if viaSetType == LfibViaSetType.unicast:
      # For mldp (multicast viaSets, we would not have explicit null labels
      via = handleExpNull( bridge, label )
      if via:
         tverb( func, 'found route', label, via.vrfName, via.payloadType )
         return via
   routeKey = Tac.Type( 'Mpls::RouteKey' ).fromLabel( label )
   route = bridge.decapLfib_.lfibRoute.get( routeKey )
   if route is None:
      tinfo( func, 'no route for label', label )
      return None
   viaSetKey = route.viaSetKey
   if route.viaSetKey.viaType != 'viaTypeIpLookup':
      tinfo( func, 'no viaTypeIpLookup route for label', label )
      return None
   viaSet = bridge.decapLfib_.viaSet.get( viaSetKey )
   if viaSet is None:
      terr( func, 'missing viaSet for key', viaSetKey )
      return None
   tinfo( func, 'viaSetType is ', viaSet.viaSetType )
   if viaSet.viaSetType != viaSetType:
      tinfo( func, 'Ignoring viaSet as the types dont match' )
      return None
   if len( viaSet.viaKey ) != 1:
      terr( func, 'invalid ipLookup via, unexpected len', len( viaSet.viaKey ) )
      return None
   viaKey = viaSet.viaKey[ 0 ]
   if viaKey.viaType != 'viaTypeIpLookup':
      terr( func, 'invalid viaKey', viaKey )
      return None
   via = bridge.decapLfib_.ipLookupVia.get( viaKey )
   if via is None:
      terr( func, 'missing via for key', viaKey )
      return None
   tverb( func, 'found route', label, via.vrfName, via.payloadType )
   return via

def findIntfPortInVrf( bridge, vrfName, mplsVrfInterfaces ):
   '''Use the VRF interface information collected by the IpStatusReactor
      to find a port in the VRF that we can use as the source port for
      the decapsulated packet.'''
   func = 'findIntfPortInVrf'
   tfunc( func, 'vrf', vrfName )
   interfaces = mplsVrfInterfaces[ vrfName ]
   tinfo( func, 'vrf', vrfName, 'interfaces', interfaces )
   if not interfaces:
      return None, None
   for intfName in interfaces:
      vlanId = None
      if VlanIntfId.isVlanIntfId( intfName ):
         vlanId = VlanIntfId.vlanId( intfName )
         phyIntfName = getVlanPhyIntf( bridge, vlanId )
         if phyIntfName is None:
            terr( func, 'vrf', vrfName, 'no intf for vlan', vlanId )
            continue
      else:
         phyIntfName = intfName
      port = bridge.port.get( phyIntfName )
      if port:
         tinfo( func, 'vrf', vrfName, 'using', phyIntfName, 'vlanId', vlanId )
         return port, vlanId
      terr( func, 'vrf', vrfName, 'no port for intf with name', phyIntfName )
   return None, None

def getVlanPhyIntf( bridge, vlanId ):
   vlanStatus = bridge.brVlanStatus_.vlanStatus.get( vlanId )
   if not vlanStatus:
      return None
   for intfId in vlanStatus.vlanPortStatus:
      if intfId.startswith( 'MplsTrunk' ):
         return intfId
   return None

class IpStatusReactor( Tac.Notifiee ):
   '''IpStatusReactor is used to maintain the per-VRF interface list'''
   notifierTypeName = 'Ip::Status'

   def __init__( self, entity, bridge, mplsIntfs ):
      terr( 'init IpStatusReactor' )
      Tac.Notifiee.__init__( self, entity )
      self.bridge = bridge
      self.intfToVrfName = {}
      self.mplsVrfInterfaces = mplsIntfs
      # Process existing interfaces
      for intfName in self.notifier().ipIntfStatus:
         self.updateMapForIntf( intfName )

   def updateMapForIntf( self, intfName ):
      func = 'IpStatusReactor.updateMapForIntf'
      terr( func, 'intfName', intfName )

      oldVrfName = self.intfToVrfName.get( intfName )
      if oldVrfName:
         tinfo( func, 'remove intf', intfName, 'from VRF', oldVrfName )
         self.mplsVrfInterfaces[ oldVrfName ].remove( intfName )
         del self.intfToVrfName[ intfName ]

      ipIntfStatus = self.notifier().ipIntfStatus.get( intfName )
      if ipIntfStatus:
         newVrfName = ipIntfStatus.vrf
         tinfo( func, 'add intf', intfName, 'to VRF', newVrfName )
         self.intfToVrfName[ intfName ] = newVrfName
         self.mplsVrfInterfaces[ newVrfName ].add( intfName )

   @Tac.handler( 'ipIntfStatus' )
   def handleIpIntfStatus( self, intfName=None ):
      self.updateMapForIntf( intfName )
