#!/usr/bin/env python
# Copyright (c) 2009-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Tracing
import MlagMountHelper

handle = Tracing.Handle( 'EbraMlagDut' )

t0 = handle.trace0
t1 = handle.trace1
t2 = handle.trace2
t8 = handle.trace8
t9 = handle.trace9

mlagStatus_ = None
mlagHwStatus_ = None
mlagConfig_ = None
hostTable_ = None
brStatus = None
reactors_ = []

def isPeerMac( entryType ):
   if entryType == 'peerStaticMac' or \
         entryType == 'peerDynamicMac' or \
         entryType == 'peerConfiguredRemoteMac' or \
         entryType == 'peerLearnedRemoteMac' or \
         entryType == 'peerReceivedRemoteMac':
      return True 
   else:
      return False

def learningEnabled( vlanId, macAddr, portName, entryType ):
   if mlagStatus_.mlagState == 'inactive' or mlagStatus_.mlagState == 'disabled':
      t8( 'Skip learn check in mlag state %s' % mlagStatus_.mlagState )
      return 'learn'
   if entryType == 'configuredStaticMac':
      # Want to configure static entry on peer link
      # go ahead
      return 'learn'
   if( mlagStatus_.peerLinkIntf is not None ):
      peerLinkIntfName = mlagStatus_.peerLinkIntf.intfId
      t8( 'learningEnabled on port', portName,
          '? peerLinkIntf is', peerLinkIntfName )
      if not entryType or not isPeerMac( entryType ):
         # Do not ignore the peer entries that we are programming
         # because they are programmed on peer.
         if( peerLinkIntfName == portName ):
            t8( 'dont learn on this port' )
            return 'dontLearn'
   fdbStatus = brStatus.fdbStatus.get( vlanId )
   if fdbStatus:
      macEntry = fdbStatus.learnedHost.get( macAddr )
      if macEntry and macEntry.entryType == 'peerStaticMac':
         if entryType == 'learnedDynamicMac':
            # Do not overwrite a peer static entry with a local
            # dyamic entry
            return 'dontLearn'
   return 'learn'


def mlagFloodSuppress( bridge, finalIntfs, srcPort=None, dropReasonList=None,
                       vlanId=None, dstMacAddr=None, data=None ):
   t9( 'call to mlagFloodSuppress' )

   if not mlagConfig_.peerLinkIntfId:
      return

   if not hasattr( srcPort, 'intfConfig_' ):
      t9( 'No intfConfig_ (probably cpu port)' )
      return
   if not srcPort or not srcPort.intfConfig_ \
          or srcPort.intfConfig_.intfId != mlagConfig_.peerLinkIntfId:
      t9( 'Packet did not arrive from the peer link.' )
      return

   t9( 'Packet arrived on peer link; suppress flooding to active mlags' )
   for intf in finalIntfs.keys():
      intfStatus = mlagStatus_.intfStatus.get( intf )
      if intfStatus and intfStatus.status == 'linkActive':
         t8( "Flooding suppressed by Mlag on", intf )
         del finalIntfs[ intf ]


def bridgeInit( bridge ):
   t2( 'Mlag bridgeInit' )
   mlagHwStatus_.mlagSupported = True
   reactors_.append( HostTableReactor( hostTable_, bridge ) )
   reactors_.append( MlagStateReactor( mlagStatus_, bridge ) )

class MlagStateReactor( Tac.Notifiee ):
   "When a mlag is formed, we need to clear the mac entries formed on the peerLink"
   notifierTypeName = 'Mlag::Status'

   def __init__( self, mlagStatus, bridge ):
      self.mlagStatus_ = mlagStatus
      self.bridge_ = bridge
      self.hostTableReactor = None
      Tac.Notifiee.__init__( self, mlagStatus )

   @Tac.handler( 'mlagState' )
   def handleMlagState( self ):
      t2( 'Mlag formed in state %s' % self.mlagStatus_.mlagState )
      if self.mlagStatus_.mlagState in [ 'primary', 'secondary' ]:
         self.bridge_.macTableFlushPort( self.mlagStatus_.peerLinkIntf.intfId )
         self.hostTableReactor = HostTableReactor( hostTable_, self.bridge_ )
      else:
         t2( "Mlag destroyed clean up host table reactor" )
         if self.hostTableReactor:
            self.hostTableReactor.close()
            self.hostTableReactor = None
      
class HostTableReactor( Tac.Notifiee ):
   """When a hostTable entry is added, add its mac address to the local table."""
   notifierTypeName = 'Mlag::HostTable'
   def __init__( self, hostTable, bridge ):
      t0( 'created HostTableReactor, hostTable', hostTable )
      self.hostTable_ = hostTable
      self.bridge_ = bridge
      Tac.Notifiee.__init__( self, hostTable )

   @Tac.handler( 'hostEntry' )
   def handleHostEntry( self, key ):
      def learnKey( self, key ):
         t0( "Mac Entry ", key, "added to peer" )
         hostEntry = self.hostTable_.hostEntry[ key ]
         address = hostEntry.hostKey.address
         intfName = hostEntry.intf
         vlanId = hostEntry.hostKey.vlanId
         # Look for existing learnedHost in fdbStatus.
         fdbStatus = brStatus.fdbStatus.get( vlanId )
         if fdbStatus:
            fdb = fdbStatus.learnedHost.get( address )
            # In the case of competing dynamic mac entries, the later
            # last moved time wins.
            if fdb and fdb.entryType == "learnedDynamicMac" and \
                   hostEntry.entryType == "peerDynamicMac" and \
                   fdb.lastMoveTime >= hostEntry.lastMoveTime:
               t0( "Keep existing learnedHost" )
               return
         self.bridge_.learnAddr( vlanId, address, intfName, hostEntry.entryType )
      def unlearnKey( self, key ):
         t0( "Mac Entry", key, " removed from peer" )
         fdbStatus = brStatus.fdbStatus.get( key.vlanId )
         if not fdbStatus:
            # When the last mac gets aged, the fdbStatus is deleted.
            t1( "Mlag host deleetion of", key.address, "but fdbStatus is gone" )
            return
         hostEntry = fdbStatus.learnedHost.get( key.address )
         if hostEntry and isPeerMac( hostEntry.entryType ):
            self.bridge_.deleteMacAddressEntry( key.vlanId, key.address )
         else:
            t0( "Mac Entry", key.address, "overwritten by a packet" )
      if( key is None ):
         for key in self.hostTable_.hostEntry:
            learnKey( self, key )
      else:
         if( self.hostTable_.hostEntry.has_key( key ) ):
            learnKey( self, key )
         else:
            unlearnKey( self, key )

   def close( self ):
      for key in self.hostTable_.hostEntry:
         hostEntry = self.hostTable_.hostEntry[ key ]
         address = hostEntry.hostKey.address
         vlanId = hostEntry.hostKey.vlanId
         if hostEntry and isPeerMac( hostEntry.entryType ):
            self.bridge_.deleteMacAddressEntry( vlanId, address )
      Tac.Notifiee.close( self )

def addPeerAged( ctx, vlanId, macAddr ):
   hostEntryKey = Tac.Value( "Mlag::HostEntryKey", address=macAddr, vlanId=vlanId )
   hostEntry = hostTable_.hostEntry.get( hostEntryKey )
   if hostEntry:
      ctx.learnAddr( vlanId, macAddr, hostEntry.intf, hostEntry.entryType )
      
def agentInit( em ):
   global mlagStatus_
   global mlagHwStatus_
   global mlagConfig_
   global hostTable_
   global brStatus
   # Mount mlag/status, Mlag::Status and its dependent paths
   mlagStatus_ = MlagMountHelper.mountMlagStatus( em )
   mlagHwStatus_ = em.mount( 'mlag/hardware/status', 'Mlag::Hardware::Status', 'w' )
   mlagConfig_ = em.mount( 'mlag/config', 'Mlag::Config', 'r' )
   hostTable_ = em.mount( 'mlag/hostTable', 'Mlag::HostTable', 'rO' )
   brStatus = em.mount( 'bridging/status', 'Bridging::Status', 'w' )
   
def Plugin( ctx ):
   t2( 'MlagDut plugin registering' )
   ctx.registerAgentInitHandler( agentInit )
   ctx.registerPacketReplicationHandler( mlagFloodSuppress )
   ctx.registerBridgeInitHandler( bridgeInit )
   ctx.registerLearningHandler( learningEnabled )
   ctx.registerPostAgingHandler( addPeerAged )
