#!/usr/bin/env python
# Copyright (c) 2008, 2009 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import Cell
from EbraTestBridge import PKTEVENT_ACTION_NONE

# This file contains simulation behavioural changes for etba dut. For changes
# related to EbraBridgeTestLib, look into EbraBridgeTestPlugin
handle = Tracing.Handle('EbraTestBridgeMirroring')
t4 = handle.trace4
t5 = handle.trace5
t4( 'Plugin called' )

class AgentDirReactor( Tac.Notifiee ):
   notifierTypeName = 'Tac::Dir'

   def __init__( self, agentDir, mirroringHwCapability ):
      self.mirroringHwCapability = mirroringHwCapability
      Tac.Notifiee.__init__( self, agentDir )
      self.handleAgentDir( None )

   @Tac.handler( 'entryState' )
   def handleAgentDir( self, key ):
      if not key:
         for k in self.notifier_.entryState:
            self.handleAgentDir( k )
         return

      if key == 'Mirroring':
         # The Mirroring agent is running, so enable mirroring-related CLI commands.
         self.mirroringHwCapability.maxSupportedSessions = 4
         self.mirroringHwCapability.aclSupported = True
         self.mirroringHwCapability.ip4AclSupported = True
         self.mirroringHwCapability.macAclSupported = True
         self.mirroringHwCapability.cpuDestSupported = True
         self.mirroringHwCapability.lagDestSupported = True
         self.mirroringHwCapability.multiDestSupported = True
         self.mirroringHwCapability.subIfAsSrcSupported = True

def agentInit( em ):
   t5( 'agent Init function called' )
   em.mount( 'mirroring/hwconfig', 'Mirroring::Config', 'r' )
   em.mount( 'mirroring/hwcapability', 'Mirroring::HwCapability', 'w' )

def bridgeInit( bridge ):
   bridge.mirroringHwConfig = bridge.em().entity( 'mirroring/hwconfig' )
   bridge.mirroringAgentDirReactor_ = AgentDirReactor(
      bridge.em().entity( Cell.path( 'agent/status' ) ),
      bridge.em().entity( 'mirroring/hwcapability' ) )

def destModify( bridge, finalIntfs, srcPort=None, dropReasonList=None, vlanId=None,
                dstMacAddr=None, data=None ):
   srcPortName = srcPort.name() if srcPort else "unspecified"
   t5( 'destModify: finalInfs', finalIntfs, 'srcPort', srcPortName )
   if dropReasonList:
      # Packet marked for drop copy it to a Mirroring Destination if
      # destination is availaible
      # Removing all the interfaces from the o/p list, packet was marked as dropped
      t5( 'Dropping packet', dropReasonList[0], 'on source', srcPortName )
      while finalIntfs.keys():
         finalIntfs.clear()

      # Remove all the reasons from dropReasons so the packet is not dropped
      # and copied onto mirroring destination
      while dropReasonList:
         dropReasonList.pop()
   # Try to see if the source or destination port is configured for Mirroring
   candidateMirror = []
   for ( sessionName, session ) in bridge.mirroringHwConfig.session.items():
      for targetIntfName in session.targetIntf.iterkeys():
         if targetIntfName.startswith( "Gre" ):
            # no support for gre tunnel yet
            continue
         for ( srcIntfName, srcIntf ) in session.srcIntf.items():
            if srcPort is not None:
               if ( ( srcIntfName == srcPort.name() ) and
                    ( srcIntf.direction=='directionRx' or
                      srcIntf.direction=='directionBoth' ) ):
                  candidateMirror.append( ( sessionName, srcIntfName,
                     targetIntfName ) )
            if ( ( srcIntfName in finalIntfs ) and
                 ( srcIntf.direction =='directionTx' or
                   srcIntf.direction =='directionBoth' ) ):
               t5( 'printing' , finalIntfs )
               candidateMirror.append( ( sessionName, srcIntfName,
                  targetIntfName ) )
   t5( 'candidateMirror', candidateMirror )

   # In case of more than one interface configured for Mirroring of the frame,
   # frame gets copied to the destination of the highest session number

   if len( candidateMirror ) > 0:
      candidateMirror.sort()
      candidateMirror.reverse()
      t5( 'mirroring packet on mirror source', candidateMirror[ 0 ][ 1 ],
          'on destination', candidateMirror[ 0 ][ 2 ] )
      finalIntfs[ candidateMirror[ 0 ][ 2 ] ] =  PKTEVENT_ACTION_NONE

def Plugin( ctx ):
   ctx.registerAgentInitHandler( agentInit )
   ctx.registerBridgeInitHandler( bridgeInit )
   ctx.registerPacketReplicationHandler( destModify )
