#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Tracing
import SharedMem, SmashLazyMount
import Cell
# import Arnet

handle = Tracing.Handle( 'EbraL2Rib' )
t9 = handle.trace9
t5 = handle.trace5

#BUG153172
class hostPreference( object ):
   switchDynamic = 16      

def mapSourceToBrEntry( sourceEntry ):
   if sourceEntry == 'sourceBgp':
      return 'evpnDynamicRemoteMac'
   elif sourceEntry == 'sourceVcs':
      return 'configuredRemoteMac'
   elif sourceEntry == 'sourceVxlanStatic':
      return 'configuredRemoteMac'
   elif sourceEntry == 'sourceVxlanDynamic':
      return 'learnedDynamicMac'
   elif sourceEntry == 'sourceLocalDynamic':
      return 'learnedDynamicMac'
   elif sourceEntry == 'sourceLocalStatic':
      return 'configuredStaticMac'
   elif sourceEntry == 'sourceMlag':
      return 'peerDynamicMac'
   elif sourceEntry == 'sourceRouterMac':
      return 'evpnDynamicRemoteMac'
   return None

class EtbaStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Bridging::Etba::Status'
   def __init__( self, l2RibEtba ):
      t9( 'EtbaStatusReactor: __init__ ' )
      self.l2Rib_ = l2RibEtba
      self.etbaStatus_ = self.l2Rib_.etbaStatus
      Tac.Notifiee.__init__( self, self.etbaStatus_ )
      self.handleEtbaStatusInitialized()

   @Tac.handler( 'initialized' )
   def handleEtbaStatusInitialized( self ):
      t9( 'handleEtbaStatusInitialized initialized %d' %  \
            ( self.etbaStatus_.initialized ) )

      if not self.etbaStatus_.initialized:
         return

      bridge = self.l2Rib_.bridge
      assert bridge

class L2RibEtba( object ):

   def __init__( self, ctx ):
      self.ctx = ctx
      self.em = None
      self.mg = None
      self.shmemEm = None
      self.l2RibFloodSetReactor = None
      self.bridge = None
      self.dynSources = None
      self.hostInput = None
      self.smashPath = 'bridging/l2Rib/%s/hostInput' % 'localDynamic'
      self.smashTableLen = 64 * 1024
      self.vrMacStatus = None
      self.etbaStatus = None
      self.l2RibOutput = None
      self.mlagStatus = None

      # Register handlers
      ctx.registerBridgeInitHandler( self.handleBridgeInit )
      ctx.registerAgentInitHandler( self.handleInit )
      ctx.registerLearningHandler( self.macLearningHandler )
      ctx.registerPostAgingHandler( self.postAgingHandler )

   def _entityManagerIs( self, em ):
      self.em = em
      self.mg = em.mountGroup()
      self.shmemEm = SharedMem.entityManager( sysdbEm=em )

   def _doMountsComplete( self ):
      redStatus = self.em.entity( Cell.path( 'redundancy/status' ) )
      # Don't write into the paths below if we are not in active mode
      if redStatus.mode != 'active':
         t9( 'L2Rib _doMountsComplete in standby mode, not writing to',
              self.dynSources )
         return
      self.dynSources.hostTableSmashPath = self.smashPath
      self.dynSources.hostTableLen = self.smashTableLen
      self.dynSources.source = 'sourceLocalDynamic'
      self.dynSources.initialized = True

   def handleInit( self, em ):
      t9( 'L2Rib Init' )
      self._entityManagerIs( em )

      self.dynSources = self.mg.mount( "bridging/l2Rib/inputDir/localDynamic",
                                        "L2Rib::SourceDirEntry", 'wc' )
      self.hostInput = self.shmemEm.doMount(
         self.smashPath,
         'L2Rib::HostInput',
         SmashLazyMount.mountInfo( 'writer', [ ( 'host', self.smashTableLen ) ] ) )
      self.etbaStatus = self.mg.mount( "bridging/etba/status",
                     "Bridging::Etba::Status", "r" )

      # Commented out. See note in macLearning Handler section
      # self.vrMacStatus = self.mg.mount( "routing/fhrp/vrMacStatus",
      #                "Routing::Fhrp::VirtualRouterMacStatus", "r" )
      # self.mlagStatus = self.mg.mount( "mlag/status", "Mlag::Status", "r" )
      self.mg.close( self._doMountsComplete )

   def handleBridgeInit( self, bridge ):
      t9( 'L2Rib plugin BridgeInit' )
      self.bridge = bridge
      bridge.l2RibEtbaStatusReactor_ = EtbaStatusReactor( self )

   def macLearningHandler( self, vlanId, macAddr, portName, entryType ):

      # Currently this plugin is meant to work in conjuction with Vxlan Etba
      # plugin. Will need to get away from that eventually since L2Rib will serve
      # lot more than Vxlan. But for now, to not run into dependency issues with
      # myriad packages, Im doing this. One of the myriad packages being Mirroring
      # that has a breadth test that excercises etba and doesnt end up finding Mlag
      # status and vrMacStatus. Most Likely, this plugin needs to move into some
      # Test/Dut related package.
      if not hasattr( self.bridge, 'vrMacStatus_' ) or \
         not hasattr( self.bridge, 'l2RibOutput_' ) or \
         not hasattr( self.bridge, 'mlagStatus_' ) :
         return
      else:
         if not self.vrMacStatus or not self.l2RibOutput or \
            not self.mlagStatus :
            self.vrMacStatus = self.bridge.vrMacStatus_
            self.l2RibOutput = self.bridge.l2RibOutput_
            self.mlagStatus = self.bridge.mlagStatus_

      if self.mlagStatus.peerLinkIntf:
         if entryType != 'peerStaticMac' and entryType != 'peerDynamicMac':
            if self.mlagStatus.peerLinkIntf.intfId == portName:
               return

      t9( 'L2Rib Etba macLearningHandler for', vlanId, macAddr, portName, entryType )

      if macAddr == '00:00:00:00:00:00':
         t9( 'L2Rib refusing to learn the all-zero mac address' )
         return

      key = Tac.Value( 'Bridging::MacVlanPair', macAddr, vlanId )
      curInput = self.hostInput.host.get( key )
      if entryType != 'learnedDynamicMac':
         if curInput:
            assert curInput.ciDest.destType == 'destTypeIntf'
            t9( 'removing existing localDynamic input for %s (%s)' %
                ( key, curInput.ciDest.intf ) )
            del self.hostInput.host[ key ]
         else:
            t9( 'Nothing interesting to do with key %s' % key )
         return


      curOutput = self.l2RibOutput.host.get( key )

      # If the winning entry in the L2Rib output is not ours but it
      # has the same preference as us then even if there is no change
      # in our input we need to update it in order to bump the SeqNo
      # to beat them.
      beatSeqNo = ( curOutput and
                    curOutput.source != 'sourceLocalDynamic' and
                    curOutput.preference == hostPreference.switchDynamic and
                    ( curInput is None or
                      curInput.seqNo <= curOutput.seqNo ) )
      newInfo = ( curInput is None or
                  curInput.ciDest.destType != 'destTypeIntf' or
                  curInput.ciDest.intf != portName )
      updateInput = newInfo or beatSeqNo

      if not updateInput:
         t9( 'localDynamic input is up-to-date and seqNo will not',
             'change winner, ignore this event' )
         return

      # OK, so now we've decided we need to program it.  Figure out
      # what seqno we need to use
      seqNo = 1
      if( curOutput and seqNo <= curOutput.seqNo ):
         seqNo = curOutput.seqNo + 1

      if( curInput and seqNo <= curInput.seqNo ):
         seqNo = curInput.seqNo + 1

      t9( 'programming host into LD input with seqNo', seqNo )
      newDest = Tac.Value( 'L2Rib::CiDest', 'destTypeIntf' )
      newDest.intf = Tac.Value( 'Arnet::IntfId', portName )
      newHost = Tac.Value( 'L2Rib::Host', key )
      newHost.entryType = 'learnedDynamicMac'
      newHost.preference = hostPreference.switchDynamic
      newHost.ciDest = newDest
      newHost.lastMoveTime = Tac.now()
      newHost.seqNo = seqNo
      self.hostInput.addHost( newHost )

   def postAgingHandler( self, bridge, vlanId, macAddr ):
      key = Tac.Value( 'Bridging::MacVlanPair', macAddr, vlanId )
      t9( 'L2Rib postAging Handler with key %s' % key )
      if key in self.hostInput.host:
         t5( 'Deleting key %s from L2Rib host input' % key )
         del self.hostInput.host[ key ]

def Plugin( ctx ):
   _ = L2RibEtba( ctx )

