#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing, Tac, Cell

# pkgdeps: library IraEtba

handle = Tracing.Handle( 'EbraTestBridge' )
t2 = handle.trace2

class AllVrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::AllVrfStatusLocal'

   def __init__( self, allVrfStatusLocal, seqStatusDir ):
      Tac.Notifiee.__init__( self, allVrfStatusLocal )
      self.allVrfStatusLocal_ = allVrfStatusLocal
      self.seqStatusDir_ = seqStatusDir
      for vrfName in self.allVrfStatusLocal_.vrf:
         self.handleVrf( vrfName )

   @Tac.handler( 'vrf' )
   def handleVrf( self, vrf ):
      func = 'AllVrfStatusLocalReactor.handleVrf'
      if self.allVrfStatusLocal_.vrf.get( vrf ):
         if not self.seqStatusDir_.get( vrf ):
            t2( func, 'vrfName', vrf, 'creating entry in seqStatusDir' )
            self.seqStatusDir_.newEntity( "Routing::SeqStatus", vrf )
      else:
         if self.seqStatusDir_.get( vrf ):
            t2( func, 'vrfName', vrf, 'deleting entry in seqStatusDir' )
            self.seqStatusDir_.deleteEntity( vrf )

def agentInit( em ):
   t2( 'Ira agentInit' )
   mg = em.mountGroup()
   routingHwStatus = mg.mount( 'routing/hardware/status',
                               'Routing::Hardware::Status', 'w' )
   routing6HwStatus = mg.mount( 'routing6/hardware/status',
                               'Routing6::Hardware::Status', 'w' )

   nexthopGroupCliConfig = mg.mount( 'routing/nexthopgroup/input/cli',
                                     'Routing::NexthopGroup::ConfigInput', 'r' )
   nexthopGroupConfigInput = mg.mount( "routing/nexthopgroup/input/config",
                                       "Tac::Dir", "r" )
   Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, True )
   ipStatus = mg.mount( "ip/status", "Ip::Status", "r" )
   ip6Status = mg.mount( "ip6/status", "Ip6::Status", "r" )

   mg.mount( Cell.path( "ip/vrf/status/local" ), "Ip::AllVrfStatusLocal", "r" )
   # this path is mounted using f flag to support switchover for RedSupDut which is
   # derived from EtbaDut.
   mg.mount( Cell.path( "routing/sequence/vrf/status" ), "Tac::Dir", "wif" )

   def onMountComplete():
      routingHwStatus.maxEcmp = 128
      routing6HwStatus.maxEcmp = 128
      routingHwStatus.maxUcmp = 128
      routing6HwStatus.maxUcmp = 128
      routingHwStatus.staticTunIntfPlatformCapabilityDeclared = True

      t2( 'starting Routing::NexthopGroup SMs' )
      nexthopGroupStatus = Tac.newInstance( "Routing::NexthopGroup::Status",
                                            "nexthopgroup-status" )
      iraEtbaRoot = Tac.root.newEntity( 'Ira::IraEtbaRoot', 'ira-etba-root' )
      iraEtbaRoot.nexthopGroupConfig = ()
      cEm = em.cEntityManager()
      NexthopGroupSm = Tac.Type( 'Ira::NexthopGroupSm' )
      isWriter = True
      smashNhgStatus = NexthopGroupSm.smashMountNhgStatus( cEm, isWriter )
      iraEtbaRoot.nexthopGroupSm = ( iraEtbaRoot.nexthopGroupConfig,
                                     nexthopGroupStatus,
                                     ipStatus,
                                     ip6Status,
                                     routingHwStatus,
                                     smashNhgStatus )
      iraEtbaRoot.nexthopGroupConfigMergeSm = ( iraEtbaRoot.nexthopGroupConfig,
                                                nexthopGroupCliConfig,
                                                nexthopGroupConfigInput )

   mg.close( onMountComplete )

def iraBridgeInit( bridge ):
   em = bridge.em()

   allVrfStatusLocal_ = em.entity( Cell.path( "ip/vrf/status/local" ) )
   seqStatusDir_ = em.entity( Cell.path( "routing/sequence/vrf/status" ) )
   bridge.allVrfStatusLocalReactor_ = AllVrfStatusLocalReactor(
                                                   allVrfStatusLocal_,
                                                   seqStatusDir_ )

def Plugin( ctx ):
   t2( 'Registering Ira plugin' )
   ctx.registerAgentInitHandler( agentInit )
   ctx.registerBridgeInitHandler( iraBridgeInit )
   ctx.registerSwitchoverBridgeInitHandler( iraBridgeInit )
