#!/usr/bin/env python
# Copyright (c) 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet, Tracing, Tac
from Arnet import PktParserTestLib
from EbraTestBridge import IP_MCAST_BASE
from EbraTestBridge import IP_MCAST_END
from EbraTestBridge import isIEEELinkConstrained

# For functionality related to EbraBridgeTestLib( for running product tests ),
# look for files in EbraBridgeTestPlugin rather than ones in EbraTestBridgePlugin

handle = Tracing.Handle( 'EbraTestBridge' )
t2 = handle.trace2
t8 = handle.trace8

def destLookup( bridge, vlanId, destMac, data ):
   # Comparing MAC addresses as strings works in the canonical format.
   # Flood if it's outside the IP Multicast range.
   if destMac < IP_MCAST_BASE or destMac > IP_MCAST_END:
      t8( 'IgmpSnooping not claiming dest', destMac )
      return ( False, None )
   if not bridge.igmpSnoopingStatus_:
      # This only happens if we're not running Etba, so agentInit is not called.
      # This should not be here long term, as we should figure out how to get
      # agentInit called in all cases.
      t8( "agentInit didn't get called" )
      return ( False, None )
   # If IGMP Snooping is enabled on the VLAN, flood to the router
   # interfaces plus the interfaces with group members
   vlan = bridge.igmpSnoopingStatus_.vlanStatus.get( vlanId )
   if vlan:
      if vlan.useVlanFloodset:
         # Flood normally if we use the VLAN floodset
         return ( False, None )
      (pkt, headers, offset) = PktParserTestLib.parsePktStr( data )
      ipHdr = PktParserTestLib.findHeader( headers, "IpHdr" )
      if ipHdr:
         if( ipHdr.protocolNum == 'ipProtoIgmp' ):
            # A race condition exists if IgmpSnooping has been disabled on the vlan
            # but the entity logging has not caught up to delete the vlanStatus yet.
            # In this case, we will not trap the packet because the check will see
            # igmpsnooping is disabled, causing us to get here.
            assert not isVlanSnoopingEnabled( bridge.igmpSnoopingConfig_, 
                  bridge.igmpSnoopingSwForceConfig_, vlanId )
            return ( False, None )

         assert ipHdr.protocolNum != 'ipProtoIgmp'
         # Igmp is trapped in the trapLookup handler below.
         if Arnet.Subnet( '224.0.0.0/24' ).containsAddr( ipHdr.dst ):
            # If it's in 224.0.0/24, then flood to the whole VLAN by
            # failing to claim this destination.
            t8( 'vlan', vlanId, 'IGMP snooping flooding to 224.0.0/24 dst',
                ipHdr.dst )
            return ( False, None )
      # If it's not a 224 packet, then pick the proper floodset.
      intfs = set( vlan.routerIntf )
      if vlan.ethGroup.has_key( destMac ):
         intfs.update( set( vlan.ethGroup[ destMac ].intf ) )
      overlayVlanGroup = Tac.Value( "Routing::Multicast::OverlayVlanGroup", vlanId,
            Tac.Value( "Arnet::IpGenAddr" ) )
      if overlayVlanGroup in bridge.ipTunnelVlanGroupStatus_.vlanToMulticastGroup:
         intfs.add( 'Cpu' )
      if vlanId in bridge.evpnStatus_.vlanFloodStatus:
         intfs = intfs.difference(
               bridge.evpnStatus_.vlanFloodStatus[ vlanId ].floodFilter.keys() )
      t8( 'vlan', vlanId, 'IGMP snooping reduced forwarding set to', intfs )
      return ( True, list( intfs ) )
   # IGMP Snooping isn't enabled on this VLAN, so allow flooding.
   t8( 'IGMP Snooping not enabled on vlan', vlanId )
   return ( False, None )

def isVlanSnoopingEnabled( config, swForceConfig, vlanId ):
   if 1 < vlanId < 4095:
      vlan = config.vlanConfig.get( vlanId )
      if vlan and vlan.enabled == 'vlanStateDisabled':
         return False 
   return config.vlanDefaultEnabled

def trapLookup( bridge, routed, vlanId, destMac, data, srcPortName ):
   if isIEEELinkConstrained( destMac ) or routed:
      t8( "Packet is routed or isn't for IGMP snooping, not trapping" )
      return ( False, False, False )

   vlanCount = len( bridge.igmpSnoopingStatus_.vlanStatus )
   if vlanCount == 0:
      return (False, False, False)
   
   (pkt, headers, offset) = PktParserTestLib.parsePktStr( data )
   ipHdr = PktParserTestLib.findHeader( headers, "IpHdr" )
   snoopingEnabled = isVlanSnoopingEnabled( bridge.igmpSnoopingConfig_, 
         bridge.igmpSnoopingSwForceConfig_, vlanId )
   hwConfig = bridge.igmpSnoopingHwConfig_
   if ipHdr:
      # Depending upon the attributes set in hwConfig, igmp and link local 
      # packets are either trapped or forwarded by etba itself
      if ipHdr.protocolNum == 'ipProtoIgmp':
         if snoopingEnabled or hwConfig.igmpSwFloodingWhenSnoopingDisabled:
            t8( 'vlan', vlanId, 'IGMP snooping "trapping" IGMP packet' )
            return (True, True, False)

         t8( 'vlan', vlanId, 'Etba forwarding IGMP packet' )
         return (True, False, False)

      if ipHdr.protocolNum == 'ipProtoPim':
         if ( snoopingEnabled and hwConfig.pimSwForwarding ) \
           or ( not snoopingEnabled and hwConfig.pimSwFloodingWhenSnoopingDisabled ):
            t8( 'vlan', vlanId, 'IGMP Snooping "trapping" PIM packet' )
            return (True, True, False)

         t8( 'vlan', vlanId, 'Etba forwarding PIM packet' )
         return (True, False, True)

      if Arnet.Subnet( '224.0.0.0/24' ).containsAddr( ipHdr.dst ):
         if ( snoopingEnabled and hwConfig.linkLocalMulticastSwForwarding ) \
            or ( not snoopingEnabled and \
            hwConfig.linkLocalMulticastSwFloodingWhenSnoopingDisabled ):
            t8( 'vlan', vlanId, 'IGMP Snooping "trapping" link local packet' )
            return (True, True, False)

         t8( 'vlan', vlanId, 'Etba forwarding link local packet' )
         return (True, False, True)

   return (False, False, False)
   
def bridgeInit( bridge ):
   t2( 'IgmpSnooping bridgeInit' )
   bridge.igmpSnoopingStatus_ = bridge.em().entity(
         'bridging/igmpsnooping/forwarding/status' )
   bridge.igmpSnoopingConfig_ = bridge.em().entity( 'bridging/igmpsnooping/config' )
   bridge.igmpSnoopingSwForceConfig_ = bridge.em().entity(
         'bridging/igmpsnooping/swForceConfig' )
   bridge.igmpSnoopingHwConfig_ = bridge.em().entity(
         'bridging/igmpsnooping/hardware/config' )
   bridge.ipTunnelVlanGroupStatus_ = bridge.em().entity(
      Tac.Type( "Routing::Multicast::IpTunnelVlanGroupStatus" ).mountPath( \
         "ipv4", "irbLocalStatus" ) )
   bridge.evpnStatus_ = bridge.em().entity( 'evpn/status' )

def agentInit( em ):
   t2( 'IgmpSnooping agentInit' )
   em.mount( 'bridging/igmpsnooping/forwarding/status',
             'Bridging::IgmpSnooping::Status', 'rO' )
   em.mount( 'bridging/igmpsnooping/config',
             'Bridging::IgmpSnooping::Config', 'r' )
   em.mount( 'bridging/igmpsnooping/swForceConfig',
             'Bridging::IgmpSnooping::SwForceConfig', 'r' )
   em.mount( 'bridging/igmpsnooping/hardware/config',
             'Bridging::IgmpSnooping::HwConfig', 'r' )
   em.mount( Tac.Type( "Routing::Multicast::IpTunnelVlanGroupStatus" ).mountPath( \
         "ipv4", "irbLocalStatus" ),
         'Routing::Multicast::IpTunnelVlanGroupStatus', 'r' )
   em.mount( 'evpn/status', 'Evpn::EvpnStatus', 'r' )

def Plugin( ctx ):
   ctx.registerBridgeInitHandler( bridgeInit )
   ctx.registerAgentInitHandler( agentInit )
   ctx.registerDestLookupHandler( destLookup )
   ctx.registerTrapLookupHandler( trapLookup )
