# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ForwardingHelper import (
      forwardingHelperFactory,
      RouteOrFecChangeReactor,
   )
from IpLibConsts import DEFAULT_VRF
from FibUtils import (
      forwardingStatusInfo,
      routeStatusInfo,
   )
import Cell
import SmashLazyMount
import SharedMem
import Smash
import Tac
import Tracing
from TypeFuture import TacLazyType

# Following import is needed because we mount arp/... in sysdb
# pkgdeps: rpm Arp-lib

handle = Tracing.Handle( 'EbraTestBridgeFwd' )
t1 = handle.trace1
t2 = handle.trace2

TunnelProgrammingStatus = TacLazyType( "Tunnel::Hardware::TunnelProgrammingStatus" )
TunnelStatus = Tac.Type( 'Tunnel::Hardware::TunnelStatus' )
clientMountName = 'etba'
TunnelTableIdentifier = TacLazyType( "Tunnel::TunnelTable::TunnelTableIdentifier" )
TunnelTableMounter = TacLazyType( "Tunnel::TunnelTable::TunnelTableMounter" )
TunnelType = TacLazyType( "Tunnel::TunnelTable::TunnelType" )
TunnelViaStatus = TacLazyType( 'Tunnel::Hardware::TunnelViaStatus' )

# Return True if any primary/backup tunnelVia of a tunnelFibEntry is usable, i.e.
# at least one interface in via collection is in 'linkUp' state. Additionally check
# if there is a bfd tracked peer on this interface, then bfdPeerStatus is 'up'.
# If 'primary' is True, check for tunnelVia in tunnelFibEntry
# If 'primary' is False, check for backupTunnelVia in tunnelFibEntry
def tunnelViaUsable( tunnelFibEntry, intfStatus, intfBfdStatus, primary=True ):
   via = tunnelFibEntry.tunnelVia if primary else tunnelFibEntry.backupTunnelVia
   # A tunnelFib entry also defines a flag 'burnPrimaryVias', which determines
   # whether or not we can revert back to primary via once backup becomes active.
   # Ale looks at 'burnPrimaryVias' as well to determie if primary via is usable,
   # but since currently no tunneling protocol fills this up and Etba does not look
   # at it for traffic forwarding, we skip that check here as well.
   arePrimaryViasUsable = tunnelFibEntry.arePrimaryViasUsable if primary else True
   for i in xrange( len( via ) ):
      intfId = via[ i ].intfId
      # intf is linkUp if intfStatus has the entry and its linkStatus is 'linkUp'
      isLinkUp = ( intfId in intfStatus and
                   intfStatus[ intfId ].linkStatus == 'linkUp' )
      # intf is bfdUp if it is bfd tracked and bfd status is 'up'. If intf is not
      # present in intfBfdStatus collection, it means bfd is not enabled for any
      # neighbor on that interface, and we don't need to consider bfdUp to determine
      # if via is usable.
      isBfdUp = ( intfId not in intfBfdStatus or intfBfdStatus[ intfId ] == 'up' )
      t2( "tunnelViaUsable: intfId: %s, isLinkUp: %d, isBfdUp: %d" %
          ( intfId, isLinkUp, isBfdUp ) )
      if isLinkUp and isBfdUp and arePrimaryViasUsable:
         return True
   return False

def updateTunnelProgrammingStatus( tunnelId, tunnelFib, intfStatus, intfBfdStatus,
                                   tunnelProgrammingStatus ):
   t2( "updateTunnelProgrammingStatus: ", tunnelId )
   tunnelFibEntry = tunnelFib.entry.get( tunnelId )
   if tunnelFibEntry:
      # If no primary via in tunnelFibEntry is usable, then mark the
      # tunnelProgrammingStatus entry as usingPrimaryVias. Else if no backup in
      # tunnelFibEntry is usable, then mark as usingBackupVias. Else mark as
      # unresolved.
      status = TunnelViaStatus.unresolved
      if tunnelViaUsable( tunnelFibEntry, intfStatus, intfBfdStatus, primary=True ):
         status = TunnelViaStatus.usingPrimaryVias
      elif tunnelViaUsable( tunnelFibEntry, intfStatus, intfBfdStatus,
                            primary=False ):
         status = TunnelViaStatus.usingBackupVias
      tunnelStatus = TunnelStatus( tunnelId, status )
      # Fill tunnelFibEntry's seqNo into tunnelStatus entry.
      tunnelStatus.seqNo = tunnelFibEntry.seqNo
      tunnelProgrammingStatus.addTunnelStatus( tunnelStatus )
   elif tunnelId in tunnelProgrammingStatus.tunnelStatus:
      del tunnelProgrammingStatus.tunnelStatus[ tunnelId ]

class AllVrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::AllVrfStatusLocal'

   def __init__( self, bridge, avsl ):
      self._bridge = bridge
      self._vrfReactor = {}
      Tac.Notifiee.__init__( self, avsl )
      for vrfName in avsl.vrf:
         self.handleVrf( vrfName )

   def bridge( self ):
      return self._bridge

   @Tac.handler( 'vrf' )
   def handleVrf( self, vrfName ):
      func = 'AllVrfStatusLocalReactor.handleVrf'
      avsl = self.notifier_
      create = vrfName in avsl.vrf
      t1( func, 'vrfName', vrfName, 'create', create )
      # mount routingStatus for the vrf
      bridge = self.bridge()
      if create and vrfName != DEFAULT_VRF:
         # mount the smash paths
         shmemEm = SharedMem.entityManager( sysdbEm=bridge.em() )
         routeInfo = routeStatusInfo( 'keyshadow' )
         shmemEm.doMount( "routing/vrf/status/" + vrfName,
                          "Smash::Fib::RouteStatus", routeInfo )
         shmemEm.doMount( "routing6/vrf/status/" + vrfName,
                          "Smash::Fib6::RouteStatus", routeInfo )

         sEm = bridge.sEm()
         routingStatus = sEm.getTacEntity( 'routing/vrf/status/' + vrfName )
         routing6Status = sEm.getTacEntity( 'routing6/vrf/status/' + vrfName )

         bridge.vrfRoutingStatus_[ vrfName ] = routingStatus
         bridge.vrfRouting6Status_[ vrfName ] = routing6Status
         # create the tries
         trie = Tac.root.newEntity( "Routing::Trie", "trie" + vrfName )
         v6trie = Tac.root.newEntity( "Routing6::Trie", "v6trie" + vrfName )
         # pylint: disable=unused-variable
         trieBuilder = Tac.newInstance( "Routing::TrieBuilder", routingStatus, trie )
         v6trieBuilder = Tac.newInstance( "Routing6::TrieBuilder",
                                          routing6Status, v6trie )
         bridge.vrfTrie_[ vrfName ] = trie
         bridge.vrfTrie6_[ vrfName ] = v6trie
         bridge.vrfTrieBuilder_[ vrfName ] = trieBuilder
         bridge.vrfTrie6Builder_[ vrfName ] = v6trieBuilder
         bridge.vrfRouteOrFecChangeReactor[ vrfName ] = \
               RouteOrFecChangeReactor( trie,
                                        v6trie, bridge.forwardingStatus_,
                                        bridge.forwarding6Status_, None )
      elif not create and vrfName != DEFAULT_VRF:
         del bridge.vrfTrie_[ vrfName ]
         del bridge.vrfTrie6_[ vrfName ]
         del bridge.vrfTrieBuilder_[ vrfName ]
         del bridge.vrfTrie6Builder_[ vrfName ]
         del bridge.vrfRouteOrFecChangeReactor[ vrfName ]
         del bridge.vrfRoutingStatus_[ vrfName ]
         del bridge.vrfRouting6Status_[ vrfName ]
         # delete the tries
         trie = Tac.root.deleteEntity( "trie" + vrfName )
         v6trie = Tac.root.deleteEntity( "v6trie" + vrfName )

class TunnelFibReactor( Tac.Notifiee ):
   notifierTypeName = "Tunnel::TunnelFib::TunnelFib"
   def __init__( self, bridge ):
      t2( "TunnelFibReactor: __init__: ")
      self.bridge = bridge
      self.tunnelFib = bridge.tunnelFib_
      bridge.intfIdToTunnelId_ = {}
      # This is a cache of tunnelId to intfId mapping. This mapping is already
      # present in tunnelFib smash table, but we need to maintain a cache of this
      # to handle the change in a tunnelFib entry's mapping so that old
      # intfIdToTunnelId mappings can be removed.
      # { tunnelId : oldTunnelFibEntry }
      self.tunnelIdToIntfIdCache = {}

      Tac.Notifiee.__init__( self, self.tunnelFib )
      # Process all existing tunnelFib entries.
      for tunnelId in self.tunnelFib.entry:
         t2( "TunnelFibReactor: __init__: handleEntry for %s" % tunnelId )
         self.handleEntry( tunnelId )
      # Cleanup every tunnelProgrammingStatus entry for which tunnelFib entry
      # does not exist.
      for tunnelId in self.bridge.tunnelProgrammingStatus_.tunnelStatus:
         if not self.tunnelFib.entry.get( tunnelId ):
            del self.bridge.tunnelProgrammingStatus_.tunnelStatus[ tunnelId ]

   @Tac.handler( 'entry' )
   def handleEntry( self, tunnelId ):
      t2( "TunnelFibReactor: handleEntry: ", tunnelId )
      self.updateIntfIdToTunnelIdMap( tunnelId )
      # If a tunnel entry is present in TunnelFib, create corresponding entry in
      # tunnelProgrammingStatus. Else delete the entry from tunnelProgrammingStatus.
      updateTunnelProgrammingStatus( tunnelId,
                                     self.tunnelFib,
                                     self.bridge.ethIntfStatusDir_.intfStatus,
                                     self.bridge.intfIdToBfdStatus_,
                                     self.bridge.tunnelProgrammingStatus_ )

   # Keep the reverse mapping of intfIds to their corresponding tunnelIds up-to-date.
   def updateIntfIdToTunnelIdMap( self, tunnelId ):
      t2( "updateIntfIdToTunnelIdMap: ", tunnelId )
      # Tunnel vias might have changed. It's safer to always remove now possibly
      # stale intfIdToTunnelId entries, and recerate them everytime.
      oldTunnelFibEntry = self.tunnelIdToIntfIdCache.get( tunnelId )
      if oldTunnelFibEntry:
         for via in [ oldTunnelFibEntry.tunnelVia,
                      oldTunnelFibEntry.backupTunnelVia ]:
            for i in xrange( len( via ) ):
               intfId = via[ i ].intfId
               if intfId in self.bridge.intfIdToTunnelId_:
                  if tunnelId in self.bridge.intfIdToTunnelId_[ intfId ]:
                     self.bridge.intfIdToTunnelId_[ intfId ].remove( tunnelId )
                  # If no tunnelId mappings for intfId, remove intfId entry.
                  if not self.bridge.intfIdToTunnelId_[ intfId ]:
                     del self.bridge.intfIdToTunnelId_[ intfId ]

      # Now if the tunnel is added or changed, recreate the intfIdToTunnelId map
      # based on tunnel's latest contents.
      if self.tunnelFib.entry.get( tunnelId ):
         t2( "updateIntfIdToTunnelIdMap: add/change", tunnelId )
         # Tunnel added. Add the correspoinding intfId-->tunnelId entries to
         # intfIdToTunnelId[ intfId ].
         tunnelFibEntry = self.tunnelFib.entry.get( tunnelId )
         for via in [ tunnelFibEntry.tunnelVia, tunnelFibEntry.backupTunnelVia ]:
            for i in xrange( len( via ) ):
               intfId = via[ i ].intfId
               # If no intfId indexed collection present then first create one.
               if not self.bridge.intfIdToTunnelId_.get( intfId ):
                  self.bridge.intfIdToTunnelId_[ intfId ] = []
               # Now add tunnelId to the collection indexed by intfId.
               if tunnelId not in self.bridge.intfIdToTunnelId_[ intfId ]:
                  self.bridge.intfIdToTunnelId_[ intfId ].append( tunnelId )
         # Now update the cache tunnelIdToIntfIdCache
         self.tunnelIdToIntfIdCache[ tunnelId ] = tunnelFibEntry
      else:
         t2( "updateIntfIdToTunnelIdMap: del", tunnelId )
         # Remove the old tunnel entry from cache tunnelIdToIntfIdCache
         del self.tunnelIdToIntfIdCache[ tunnelId ]

      t2( "updateIntfIdToTunnelIdMap: intfIdToTunnelId_ map is:" )
      for intf in self.bridge.intfIdToTunnelId_:
         t2( intf, ":" )
         for tunnel in self.bridge.intfIdToTunnelId_[ intf ]:
            t2( "---->", tunnel )

class EthIntfStatusDirReactor( Tac.Notifiee ):
   notifierTypeName = 'Interface::EthIntfStatusDir'
   def __init__( self, bridge ):
      t2( "EthIntfStatusDirReactor: __init__: ")
      self.bridge = bridge
      self.ethIntfStatusDir = bridge.ethIntfStatusDir_
      self.ethIntfStatusReactor = {}
      Tac.Notifiee.__init__( self, self.ethIntfStatusDir )
      for intf in self.ethIntfStatusDir:
         t2( "EthIntfStatusDirReactor: __init__: handleStatus for %s" % intf )
         self.handleStatus( intf )

   @Tac.handler( 'intfStatus' )
   def handleStatus( self, key ):
      t2( "EthIntfStatusDirReactor: handleStatus: ", key )
      if self.ethIntfStatusDir.intfStatus.get( key ):
         intfStatus = self.ethIntfStatusDir.intfStatus[ key ]
         self.ethIntfStatusReactor[ key ] = EthIntfStatusReactor(
                                               self.bridge, intfStatus )
      else:
         del self.ethIntfStatusReactor[ key ]

class EthIntfStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Interface::EthIntfStatus'
   def __init__( self, bridge, intfStatus ):
      t2( "EthIntfStatusReactor: __init__: ")
      Tac.Notifiee.__init__( self, intfStatus )
      self.bridge = bridge
      self.intfIdToTunnelId = bridge.intfIdToTunnelId_
      self.handleLinkStatus()

   @Tac.handler( 'linkStatus' )
   def handleLinkStatus( self ):
      intfId = self.notifier_.intfId
      t2( "EthIntfStatusReactor: handleLinkStatus: ", intfId )
      # Update tunnelProgrammingStatus for matching tunnels.
      for tunnelId in self.intfIdToTunnelId.get( intfId, [] ):
         updateTunnelProgrammingStatus( tunnelId,
                                        self.bridge.tunnelFib_,
                                        self.bridge.ethIntfStatusDir_.intfStatus,
                                        self.bridge.intfIdToBfdStatus_,
                                        self.bridge.tunnelProgrammingStatus_ )

class BfdAllPeerStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Bfd::AppStatus'
   def __init__( self, bridge ):
      t2( "BfdAllPeerStatusReactor: __init__: ")
      self.bridge = bridge
      self.bfdAllPeerStatus = bridge.bfdAllPeerStatus
      bridge.intfIdToBfdStatus_ = {}
      self.bfdPeerStatusReactor = {}
      Tac.Notifiee.__init__( self, self.bfdAllPeerStatus )
      for peer in self.bfdAllPeerStatus.peerStatus:
         t2( "BfdAllPeerStatusReactor: __init__: handlePeerStatus for %s" % peer )
         self.handlePeerStatus( peer )

   @Tac.handler( 'peerStatus' )
   def handlePeerStatus( self, key ):
      t2( "BfdAllPeerStatusReactor: handlePeerStatus: ", key )
      if self.bfdAllPeerStatus.peerStatus.get( key ):
         peerStatus = self.bfdAllPeerStatus.peerStatus[ key ]
         self.bridge.intfIdToBfdStatus_[ key.intf ] = peerStatus.status
         self.bfdPeerStatusReactor[ key ] = BfdPeerStatusReactor(
                                               self.bridge, peerStatus )
      else:
         if key in self.bfdPeerStatusReactor:
            del self.bfdPeerStatusReactor[ key ]
         if key.intf in self.bridge.intfIdToBfdStatus_:
            del self.bridge.intfIdToBfdStatus_[ key.intf ]

class BfdPeerStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Bfd::BfdPeerStatus'
   def __init__( self, bridge, peerStatus ):
      t2( "BfdPeerStatusReactor: __init__: ")
      Tac.Notifiee.__init__( self, peerStatus )
      self.bridge = bridge
      self.intfIdToTunnelId = bridge.intfIdToTunnelId_
      self.handleStatus()

   @Tac.handler( 'status' )
   def handleStatus( self ):
      intf = self.notifier_.peer.intf
      t2( "BfdPeerStatusReactor: handleStatus: ", intf )
      self.bridge.intfIdToBfdStatus_[ intf ] = self.notifier_.status
      # Update tunnelProgrammingStatus for matching tunnels.
      for tunnelId in self.intfIdToTunnelId.get( intf, [] ):
         updateTunnelProgrammingStatus( tunnelId,
                                        self.bridge.tunnelFib_,
                                        self.bridge.ethIntfStatusDir_.intfStatus,
                                        self.bridge.intfIdToBfdStatus_,
                                        self.bridge.tunnelProgrammingStatus_ )

def forwardingBridgeInit( bridge, clientName=clientMountName ):
   t2( 'forwardingBridgeInit' )

   em = bridge.em()

   l3Config = em.entity( 'l3/config' )
   bridge.fecModeStatus = Tac.newInstance( 'Smash::Fib::FecModeStatus', 'fms' )
   # fecModeSm is instantiated to generate the correct fecMode in fecModeStatus
   _ = Tac.newInstance( 'Ira::FecModeSm', l3Config, bridge.fecModeStatus )

   bridge.routingVrfInfo_ = em.entity( 'routing/vrf/routingInfo/status' )
   bridge.routing6VrfInfo_ = em.entity( 'routing6/vrf/routingInfo/status' )
   bridge.nhrConfig = em.entity( 'l3/nexthop-resolver/config/%s' %
                                 clientName )
   bridge.nhrStatus = em.entity( 'l3/nexthop-resolver/status/%s' %
                                 clientName )
   sEm = bridge.sEm()
   bridge.tunnelRib = sEm.doMount(
      'tunnel/tunnelRibs/status/0', 'Tunnel::TunnelTable::TunnelRib',
      Smash.mountInfo( 'keyshadow' ) )
   bridge.intfConfigDir_ = em.entity( 'l3/intf/config' )
   bridge.ethIntfStatusDir_ = em.entity( 'interface/status/eth/intf' )

   bridge.tunnelTables = dict()
   for tableId in TunnelTableIdentifier.attributes:
      if tableId != TunnelTableIdentifier.invalidTunnelTable:
         tableInfo = TunnelTableMounter.getMountInfo( tableId ).tableInfo
         tunnelTable = SmashLazyMount.mount(
            em, tableInfo.mountPath, tableInfo.tableType,
            SmashLazyMount.mountInfo( "keyshadow" ) )
         bridge.tunnelTables.setdefault( tunnelTable.tableType, [] ).append(
            tunnelTable )

   bridge.srTeSegmentListTunnelTable = (
      bridge.tunnelTables[ TunnelType.srTeSegmentListTunnel ][ 0 ] )

   bridge.tunnelFib_ = sEm.getTacEntity( 'tunnel/tunnelFib' )
   bridge.routingStatus_ = sEm.getTacEntity( 'routing/status' )
   bridge.routing6Status_ = sEm.getTacEntity( 'routing6/status' )
   if bridge.fecModeStatus.fecMode == 'fecModeUnified':
      bridge.forwardingStatus_ = sEm.getTacEntity( 'forwarding/unifiedStatus' )
      bridge.forwarding6Status_ = sEm.getTacEntity( 'forwarding6/unifiedStatus' )
      bridge.vrfNameStatus_ = em.entity( Cell.path( 'vrf/vrfNameStatus' ) )
   else:
      bridge.forwardingStatus_ = sEm.getTacEntity( 'forwarding/status' )
      bridge.forwarding6Status_ = sEm.getTacEntity( 'forwarding6/status' )
      bridge.vrfNameStatus_ = None
   bridge.srTeForwardingStatus_ = sEm.getTacEntity( 'forwarding/srte/status' )
   bridge.nhgEntryStatus_ = sEm.getTacEntity(
      'routing/nexthopgroup/entrystatus' )
   bridge.arpSmash_ = sEm.getTacEntity( 'arp/status' )

   bridge.trie = Tac.root.newEntity( "Routing::Trie", "trie" )
   bridge.trieBuilder = Tac.newInstance( "Routing::TrieBuilder",
                bridge.routingStatus_, bridge.trie )
   bridge.v6trie = Tac.root.newEntity( "Routing6::Trie", "v6trie" )
   bridge.v6trieBuilder = Tac.newInstance( "Routing6::TrieBuilder",
                bridge.routing6Status_, bridge.v6trie )
   bridge.brStatus = em.entity( 'bridging/status' )
   bridge.brConfig = em.entity( 'bridging/config' )
   bridge.pwRouteIndex = {}
   bridge.routeOrFecChangeReactor_ = RouteOrFecChangeReactor( bridge.trie,
                           bridge.v6trie, bridge.forwardingStatus_,
                           bridge.forwarding6Status_, None )
   bridge.vrfRoutingStatus_ = {}
   bridge.vrfRouting6Status_ = {}
   bridge.vrfTrie_ = {}
   bridge.vrfTrie6_ = {}
   bridge.vrfTrieBuilder_ = {}
   bridge.vrfTrie6Builder_ = {}
   bridge.vrfRouteOrFecChangeReactor = {}
   bridge.vrfRoutingStatus_[ DEFAULT_VRF ] = bridge.routingStatus_
   bridge.vrfRouting6Status_[ DEFAULT_VRF ] = bridge.routing6Status_
   bridge.vrfTrie_[ DEFAULT_VRF ] = bridge.trie
   bridge.vrfTrie6_[ DEFAULT_VRF ] = bridge.v6trie
   bridge.vrfTrieBuilder_[ DEFAULT_VRF ] = bridge.trieBuilder
   bridge.vrfTrie6Builder_[ DEFAULT_VRF ] = bridge.v6trieBuilder

   bridge.vrfRouteOrFecChangeReactor[ DEFAULT_VRF ] = \
            RouteOrFecChangeReactor( bridge.trie,
                                     bridge.v6trie, bridge.forwardingStatus_,
                                     bridge.forwarding6Status_, None )

   bridge.fwdHelper = forwardingHelperFactory( bridge )
   avsl = bridge.em().entity( Cell.path( 'ip/vrf/status/local' ) )
   bridge.vrfRouteStatusSm = AllVrfStatusLocalReactor( bridge, avsl )
   # Maintain following dictionary to quickly find out all the tunnelFib entries that
   # are using a particular interface either in tunnelVia[] or in backupTunnelVia[].
   # This dictionary is maintained by TunnelFibReactor and used for lookup by
   # EthIntfStatusReactor.
   # { intfId : [ tunnelId1, tunnelId2... ] }
   bridge.intfIdToTunnelId_ = {}

   # Currently this only needs to be mounted by the etba agent.
   # When bug/488648 is resolved this conditional can be removed
   if clientName == 'etba':
      bridge.tunnelProgrammingStatus_ = sEm.getTacEntity(
         TunnelProgrammingStatus.mountPath )

      bridge.tunnelFibSm = TunnelFibReactor( bridge )

   bridge.ethIntfStatusDirSm = EthIntfStatusDirReactor( bridge )

   # Maintain intf to bfd peer status mapping. When BfdPeerStatus reactor fires, this
   # mapping speeds up the process of determining if the tunnels corresponding to
   # intf in BfdPeerStatus have primary vias usable or back vias.
   bridge.intfIdToBfdStatus_ = {}
   bridge.bfdAllPeerStatusDir = em.entity( 'bfd/status/peer' )
   bridge.bfdAllPeerStatus = Tac.newInstance( "Bfd::AppStatus", "bfdEtba" )
   bridge.bfdAppListener = Tac.newInstance( "BfdStatAgentLib::BfdAppListener", "",
                                            bridge.bfdAllPeerStatusDir,
                                            bridge.bfdAllPeerStatus, "" )
   bridge.bfdAllPeerStatusSm = BfdAllPeerStatusReactor( bridge )

def mountNexthopResolver( mounter, clientName=clientMountName ):
   mounter.mount( 'l3/nexthop-resolver/config/%s' % clientName,
                  'Routing::NexthopConfig', 'w' )
   mounter.mount( 'l3/nexthop-resolver/status/%s' % clientName,
                  'Routing::NexthopStatus', 'r' )

def forwardingAgentInit( em ):
   t2( 'forwardingAgentInit' )

   sEm = SharedMem.entityManager( sysdbEm=em )
   routeInfo = routeStatusInfo( 'keyshadow' )
   fwdInfo = forwardingStatusInfo( 'keyshadow' )
   sEm.doMount( "routing/status", "Smash::Fib::RouteStatus", routeInfo )
   sEm.doMount( "routing6/status", "Smash::Fib6::RouteStatus", routeInfo )
   sEm.doMount( 'forwarding/unifiedStatus', "Smash::Fib::ForwardingStatus", fwdInfo )
   sEm.doMount( 'forwarding6/unifiedStatus', "Smash::Fib6::ForwardingStatus",
                fwdInfo )
   sEm.doMount( 'forwarding/status', "Smash::Fib::ForwardingStatus", fwdInfo )
   sEm.doMount( 'forwarding6/status', "Smash::Fib6::ForwardingStatus", fwdInfo )
   sEm.doMount( 'forwarding/srte/status', 'Smash::Fib::ForwardingStatus', fwdInfo )
   sEm.doMount( 'arp/status', "Arp::Table::Status", Smash.mountInfo( 'shadow' ) )
   sEm.doMount( 'tunnel/tunnelFib', 'Tunnel::TunnelFib::TunnelFib',
                Smash.mountInfo( 'keyshadow' ) )

   tunProgWriterInfo = Smash.mountInfo( 'writer', [ ( "tunnelStatus", 1024 ) ] )
   sEm.doMount( TunnelProgrammingStatus.mountPath,
                "Tunnel::Hardware::TunnelProgrammingStatus",
                tunProgWriterInfo )

   em.mount( 'l3/intf/config', 'L3::Intf::ConfigDir', 'r' )
   em.mount( 'interface/status/eth/intf', 'Interface::EthIntfStatusDir', 'r' )
   em.mount( Cell.path( 'vrf/vrfNameStatus' ),
             'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )

   em.mount( 'l3/config', 'L3::Config', 'r' )

   em.mount( 'routing/hardware/nexthopgroup/status',
             'Routing::Hardware::NexthopGroupStatus', 'r' )
   em.mount( 'routing/vrf/routingInfo/status', 'Tac::Dir', 'r' )
   em.mount( 'routing6/vrf/routingInfo/status', 'Tac::Dir', 'r' )


   # Ira L3 Resolver etba config and status
   mountNexthopResolver( em )

   em.mount( 'bfd/status/peer', 'Bfd::StatusPeer', 'r' )

def Plugin( ctx ):
   # In breadth tests, the ctx will be None, ignore it in this case
   if ctx is None:
      return

   # forwarding handlers
   ctx.registerBridgeInitHandler( forwardingBridgeInit )
   ctx.registerAgentInitHandler( forwardingAgentInit )
