# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet.NsLib import (
      DEFAULT_NS,
      runMaybeInNetNs,
      )
from EbraTestBridgeLib import (
      applyVlanChanges,
      EtbaCounterConfigReactor,
      macAddrsAsStrings,
      makeTap,
      PKTEVENT_ACTION_ADD,
      PKTEVENT_ACTION_NONE,
      )
from EbraTestBridgePort import EbraTestPortBase
import Cell
import Tac
from TypeFuture import TacLazyType
import Tracing

PacketFormat = Tracing.HexDump
th = Tracing.Handle( 'FwdIntfEtba' )
terr = th.trace0
tinfo = th.trace1
tverb = th.trace2
tfunc = th.trace8

RoutingHandlerCounter = TacLazyType( 'Bridging::Etba::RoutingHandlerCounter' )
VlanIntfId = Tac.Type( 'Arnet::VlanIntfId' )

DEFAULT_VRF = Tac.Type( 'L3::VrfName' ).defaultVrf
FWD_ETHTYPE_DEFAULT = [ "ipv4", "ipv6", "mpls" ]
FWD_DEVICE = "fwd0"
# Allow for an MPLS label stack containing 3 labels to bring us up
# to 1500 octets
MTU_MPLS_ENCAP = 1500 - 3 * 4

fwdIntfRoutingHandlerCounter = {}

class FwdIntfFile( Tac.File ):
   def __init__( self, device, handler, name, vrfName ):
      self._vrfName = vrfName
      Tac.File.__init__( self, device, handler, name, readBufferSize=16000 )

   @Tac.handler( 'readableCount' )
   def handleReadableCount( self ):
      '''Override Tac.File.handleReadableCount to catch the case of the fd
         becoming invalid when the VRF is deleted.'''
      try:
         Tac.File.handleReadableCount( self )
      except OSError, e:
         # just close the socket if exceptions occur
         terr( 'FwdIntfFile.handleReadableCount vrf', self._vrfName,
               "exception", e, "closing fwd0" )
         self.close()

   def close( self ):
      if self.deleted_:
         return
      Tac.File.close( self )

class FwdIntfDevice( EbraTestPortBase ):
   '''Device used by kernel to route packets requiring special handle,
      i.e., corresponding to leaked route or requiring MPLS
      encapsulation. We pick them up in the readHandler and use a
      plugin routingHandler to perform the encapsulation and identify
      the interface on which the encapsulated packet must be
      forwarded.
   '''
   NO_MATCH = ( None, None )

   __routingHandler = []
   @classmethod
   def registerFwdIntfRoutingHandler( cls, func ):
      cls.__routingHandler.append( func )
      fwdIntfRoutingHandlerCounter[ func.__name__ ] = RoutingHandlerCounter()

   def __init__( self, vrfName, bridge, intfName, trapDevice ):
      EbraTestPortBase.__init__( self, intfName, bridge )
      tfunc( "trap", trapDevice.name, trapDevice.fileno() )
      self.trapDevice_ = trapDevice
      self.trapFile_ = FwdIntfFile( trapDevice, self._trapDeviceReadHandler,
                                    trapDevice.name, vrfName )
      self._vrfName = vrfName
      self._bridge = bridge

      self.count = {
         'fwd': 0,
         'drop': 0,
         'no-intf': 0,
      }
      self.handlerMatchCount = {}
      self.enableIpv6()

   def close( self ):
      self.trapFile_.close()
      self.trapFile_ = None
      self.trapDevice_ = None

   @property
   def bridge( self ):
      return self._bridge

   @property
   def vrfName( self ):
      return self._vrfName

   def enableIpv6( self ):
      fname = "/proc/sys/net/ipv6/conf/fwd0/disable_ipv6"
      with file( fname, 'w' ) as f:
         f.write( '0\n' )

   def _trapDeviceReadHandler( self, data ):
      func = 'FwdIntfDevice._trapDeviceReadHandler'
      tfunc( func, 'vrf', self._vrfName, PacketFormat( data ) )

      # Since this frame got to fwd0, it should be handled by special routing
      # handler, i.e., route leaking handler or MPLS handler.
      # This could be basic MPLS push or L3 EVPN.
      newData = None
      for h in self.__routingHandler:
         routeMatch = h( self, data )
         if routeMatch != self.NO_MATCH:
            newData, newIntfId = routeMatch
            newCount = self.handlerMatchCount.get( h, 0 ) + 1
            self.handlerMatchCount[ h ] = newCount
            fwdIntfRoutingHandlerCounter[ h.__name__ ].routed += 1
            break
         else:
            fwdIntfRoutingHandlerCounter[ h.__name__ ].ignored += 1
      if newData is None:
         tinfo( func, 'DROP: vrf', self._vrfName, 'no available route found (DROP)',
                PacketFormat( data ) )
         self.count[ 'drop' ] += 1
         return

      port = self._bridge.port.get( newIntfId )
      if port:
         # sendFrame, apparently, can figure all this out on its own for physical
         # interfaces...
         tverb( func, 'FWD: vrf', self._vrfName, 'on interface', newIntfId )
         port.sendFrame( newData, None, None, None, None, None,
                         PKTEVENT_ACTION_NONE )
         self.count[ 'fwd' ] += 1
         return

      # Didn't find a physical port, so try to bridge it from Cpu on the newIntfId
      # VLAN.
      if VlanIntfId.isVlanIntfId( newIntfId ):
         vlanId = VlanIntfId.vlanId( newIntfId )
         vlanPri = 0
         vlanAct = PKTEVENT_ACTION_ADD
         tverb( func, 'FWD: vrf', self._vrfName, ', bridge on VLAN', vlanId )
         ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( newData )
         vlanData = applyVlanChanges( newData, vlanPri, vlanId, vlanAct )
         srcPort = self._bridge.port[ 'Cpu' ]
         self._bridge.bridgeFrame( vlanData, srcMacAddr, dstMacAddr, srcPort, False )
         self.count[ 'fwd' ] += 1
         return

      tinfo( func, 'DROP: vrf', self._vrfName, 'no port for', newIntfId,
             PacketFormat( data ) )
      self.count[ 'no-intf' ] += 1

   def sendFrame( self, data, srcMacAddr, dstMacAddr, srcPortName,
                  priority, vlanId, vlanAction ):
      assert False, 'FwdIntfDevice.sendFrame should not be called'

   def trapFrame( self, data ):
      assert False, 'FwdIntfDevice.trapFrame should not be called'

def _disableArp( nsName, intfName ):
   tfunc( '_disableArp nsName', nsName, 'intfName', intfName )
   cmd = [ 'ip', 'link', 'set', 'dev', intfName, 'arp', 'off' ]
   runMaybeInNetNs( nsName, cmd )

def createFwdIntf( bridge, vrfName, nsName, intfName, rtHwRouteStatus=True,
                   mtu=MTU_MPLS_ENCAP, ethType=None ):
   func = 'createFwdIntf'
   tfunc( func, 'vrf', vrfName, 'ns', nsName )

   if ethType is None:
      ethType = FWD_ETHTYPE_DEFAULT

   if vrfName in bridge.fwdIntf_:
      tinfo( func, 'vrfName', vrfName, intfName, 'already exists (skip)' )
      return

   tinfo( func, 'vrfName', vrfName, 'nsName', nsName, 'create intfName', intfName )

   # setup routingHWRouteStatus
   if rtHwRouteStatus:
      routingHwRouteStatus = bridge.em().entity( 'routing/hardware/route/status' )
      routingHwRouteStatus.hierarchicalFecsEnabled = True
      vrfStatus = routingHwRouteStatus.newVrfStatus( vrfName )
      if isinstance( ethType, list ):
         for et in ethType:
            vrfStatus.fwd0EthTypeEntry.newMember( et )
            vrfStatus.fwd0EthTypeEntry[ et ].devName = intfName
      elif isinstance( ethType, str ):
         vrfStatus.fwd0EthTypeEntry.newMember( et )
         vrfStatus.fwd0EthTypeEntry[ et ].devName = intfName
      else:
         terr( 'invalid ethType, should be list or string' )

   try:
      trapDevice = makeTap( intfName, netNs=nsName, mtu=mtu )
      _disableArp( nsName, intfName )
   except Tac.SystemCommandError, e:
      # It is possible for this to fail if the namespace is deleted
      # before we can process the VrfStatusLocal notification.
      terr( 'NetNs error', nsName, intfName, "exception", e )
      return

   port = FwdIntfDevice( vrfName, bridge, intfName, trapDevice )
   bridge.fwdIntf_[ vrfName ] = port

def deleteFwdIntf( bridge, vrfName ):
   func = 'deleteFwdIntf'
   tfunc( func, 'vrf', vrfName )

   intfName = FWD_DEVICE
   if vrfName not in bridge.fwdIntf_:
      tinfo( func, 'vrfName', vrfName, intfName, 'does not exist (skip)' )
      return
   bridge.fwdIntf_[ vrfName ].close()
   del bridge.fwdIntf_[ vrfName ]

class VrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::VrfStatusLocal'
   def __init__( self, bridge, vsl ):
      self._bridge = bridge
      Tac.Notifiee.__init__( self, vsl )
      self.handleState()

   def __del__( self ):
      vsl = self.notifier_
      deleteFwdIntf( self._bridge, vsl.vrfName )

   @Tac.handler( 'state' )
   def handleState( self ):
      func = 'VrfStatusLocalReactor.handleState'
      vsl = self.notifier_
      active = ( vsl.state == 'active' )
      tinfo( func, 'vrfName', vsl.vrfName, 'state', vsl.state,
             'nsName', vsl.networkNamespace )
      if active:
         intfName = FWD_DEVICE
         createFwdIntf( self._bridge, vsl.vrfName, vsl.networkNamespace, intfName )
      else:
         deleteFwdIntf( self._bridge, vsl.vrfName )

class AllVrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::AllVrfStatusLocal'

   def __init__( self, bridge, avsl ):
      self._bridge = bridge
      self._vrfReactor = {}
      Tac.Notifiee.__init__( self, avsl )
      for vrfName in avsl.vrf:
         self.handleVrf( vrfName )

   @Tac.handler( 'vrf' )
   def handleVrf( self, vrfName ):
      func = 'AllVrfStatusLocalReactor.handleVrf'
      avsl = self.notifier_
      create = vrfName in avsl.vrf
      vrfReactor = self._vrfReactor.get( vrfName, None )
      tinfo( func, 'vrfName', vrfName, 'create', create, 'vrfReactor', vrfReactor )
      if create:
         if not vrfReactor:
            vsl = avsl.vrf[ vrfName ]
            self._vrfReactor[ vrfName ] = VrfStatusLocalReactor( self._bridge, vsl )
      else:
         if vrfReactor:
            del self._vrfReactor[ vrfName ]
            vrfReactor = None

def fwdIntfBridgeInit( bridge ):
   func = 'fwdIntfBridgeInit'
   tinfo( func )

   bridge.fwdIntf_ = {}
   intfName = FWD_DEVICE
   createFwdIntf( bridge, DEFAULT_VRF, DEFAULT_NS, intfName )
   # The AllVrfRoutingInfoReactor is needed to create the fwd0 in VRFs.
   avsl = bridge.em().entity( Cell.path( 'ip/vrf/status/local' ) )
   bridge.vrfStatusLocalReactor_ = AllVrfStatusLocalReactor( bridge, avsl )
   bridge.fwdIntfCounterConfigReactor_ = EtbaCounterConfigReactor(
      bridge.em().entity( 'bridging/etba/counter/fwdIntf/config' ),
      bridge.em().entity( 'bridging/etba/counter/fwdIntf/status' ),
      fwdIntfRoutingHandlerCounter )

def fwdIntfAgentInit( em ):
   # Mount AllVrfStatusLocal to handle the addition of new VRFs, used
   # to create the fwd0 interface in the VRF.
   em.mount( Cell.path( 'ip/vrf/status/local' ), 'Ip::AllVrfStatusLocal', 'r' )
   em.mount( 'routing/hardware/route/status', 'Routing::Hardware::RouteStatus', 'w' )
   em.mount( 'bridging/etba/counter/fwdIntf/config',
             'Bridging::Etba::RoutingHandlerCounterConfig', 'r' )
   em.mount( 'bridging/etba/counter/fwdIntf/status',
             'Bridging::Etba::RoutingHandlerCounterStatus', 'w' )

def pluginHelper( ctx ):
   ctx.registerBridgeInitHandler( fwdIntfBridgeInit )
   ctx.registerAgentInitHandler( fwdIntfAgentInit )
