#!/usr/bin/env python
# Copyright (c) 2009-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Contains common routines for dealing with EbraTestBridge instances
and the kernel interfaces associated with them."""
import re, sys
import random
import struct
import Tac
import Arnet.Device
from eunuchs.if_ether_h import ETH_P_8021Q

# VlanAction values passed to sendFrame
PKTEVENT_ACTION_NONE = 0
PKTEVENT_ACTION_REMOVE = 1
PKTEVENT_ACTION_ADD = 2
PKTEVENT_ACTION_REPLACE = 3

# Handler priority for ordered execution of registered handlers
HANDLER_PRIO_HIGHEST = 1
HANDLER_PRIO_HIGH = 2
HANDLER_PRIO_NORMAL = 3
HANDLER_PRIO_LOW = 4
HANDLER_PRIO_LOWEST = 5

def tacRunNoCloseFds( argv, **kwargs ):
   # Since we're running a short-lived command, we don't need to use closeFds.
   # The problem with Popen( close_fds=True ) is that it will call close() on
   # all FD numbers from 1 to MAXFD (1M)
   kwargs[ 'closeFds' ] = False
   return Tac.run( argv, **kwargs )

def makeTap( name, **kargs ):
   try:
      return Arnet.Device.Tap(name, **kargs)
   except OSError, e:
      if e.errno == 16:
         sys.stderr.write( "The kernel device named %s is already in use.\n" \
                           "Perhaps another EtbaDut is running?\n" % name )
         sys.exit( 1 )
      raise

def fabricDevice( dutName, bridgeMac, inNamespace=False ):
   """Create the fabric device for dutName with the specified bridge
   mac address"""
   if inNamespace:
      dev = 'fabric'
   else:
      dev = '%s-fabric' % dutName
   return makeTap(dev, hw=bridgeMac)

def ethIntfSuffix( intfName ):
   m = re.search( 'Ethernet(\d+)(?:/(\d+)(?:/(\d+))?)?$', intfName )
   if m:
      levels = list( m.groups() )
      while levels[ -1 ] is None:
         levels.pop( -1 )
      return "_".join( levels )
   return None

def tapDevice( dutName, intfName ):
   """Create the kernel tap ('et') device for interface
   intfName on dutname"""
   suffix = ethIntfSuffix( intfName )
   if suffix:
      return makeTap( "%s-et%s" % ( dutName, suffix ) )
   return None

def trapDevice( dutName, intfName, inNamespace=False, hwAddr=None ):
   """Create the kernel trap ('ed') device for interface
   intfName on dutName.  If we're in a namespace, use the
   'et%d' or 'et%d_%d' name that matches what's used on the hardware."""
   suffix = ethIntfSuffix( intfName )
   if suffix:
      if inNamespace:
         dev = 'et'
      else:
         dev = '%s-ed' % dutName
      dev += suffix
      trapDev = makeTap( dev, hw=hwAddr )
      return trapDev
   elif intfName == 'Cpu':
      if inNamespace:
         dev = 'cpu'
      else:
         dev = '%s-cpu' % dutName
      trapDev = makeTap( dev )
      return trapDev
   else:
      return None

def isTrapDeviceName( tapName ):
   """Return true if tapName corresponds to a trapDevice."""
   return bool( re.search( "(^et|-ed)(\d+)$", tapName ) )

def tapDeviceIndex( tapName ):
   """Translate a tapDevice or trapDevice name like etba1-ed%s to an
   index"""
   match = re.search( "(?:^et|-e[td])(\d+)$", tapName )
   index = match and int( match.group( 1 ))
   return index
   
def rawPhyControlPacket( linkUp=True, src=None ):
   pkt = '011c73ffffff021c73ffffffd28be1ba0000'
   if linkUp:
      pkt += '01'
   else:
      pkt += '00'
   raw = pkt.decode( 'hex' )
   if src is not None:
      raw = raw[ 0:6 ] + src + raw[ 12: ]
   return raw


def macAddrsAsStrings( packet ):
   d = packet[ 0:6 ]
   s = packet[ 6:12 ]
   dstMacAddr = '%02x:%02x:%02x:%02x:%02x:%02x' % tuple( map( ord, d ) )
   srcMacAddr = '%02x:%02x:%02x:%02x:%02x:%02x' % tuple( map( ord, s ) )
   return ( srcMacAddr, dstMacAddr )

def applyVlanChanges( data, priority, vlanId, vlanAction, priorityInner=None,
                      vlanIdInner=None, numTagsToRemove=None ):
   # If necessary, insert, replace or remove a VLAN tag.
   if vlanAction in [ PKTEVENT_ACTION_ADD, PKTEVENT_ACTION_REPLACE ]:
      assert numTagsToRemove is None
      if vlanIdInner is None:
         # single tag manipulation
         newVlanTag = struct.pack( '>HH', ETH_P_8021Q, vlanId | ( priority << 13 ) )
         offset = 16
      else:
         # double tag manipulation
         newVlanTag = struct.pack( '>HHHH', ETH_P_8021Q, vlanId | ( priority << 13 ),
                                   ETH_P_8021Q, vlanIdInner |
                                                ( priorityInner << 13 ) )
         offset = 20
      if vlanAction == PKTEVENT_ACTION_ADD:
         data = data[ 0:12 ] + newVlanTag + data[ 12: ]
      else:
         data = data[ 0:12 ] + newVlanTag + data[ offset: ]
   elif vlanAction == PKTEVENT_ACTION_REMOVE:
      if numTagsToRemove is None:
         numTagsToRemove = 1
      assert numTagsToRemove
      offset = 12 + 4 * numTagsToRemove
      data = data[ 0:12 ] + data[ offset: ]
      minSize = 0x10 * 4
      if len( data ) + 4 < minSize:
         # Pad packets to minimum size with random data.
         padLen = minSize - ( len( data ) + 4 )
         pad = [ chr( random.randint( 0, 255 ) ) for _i in range( padLen ) ]
         data += ''.join( pad )
   return data

class EtbaCounterConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Bridging::Etba::RoutingHandlerCounterConfig"

   def __init__( self, config, status, counter ):
      Tac.Notifiee.__init__( self, config )
      self.counterStatus = status
      self.counter = counter
      self.handleCounterUpdateRequestTime()

   @Tac.handler( 'counterUpdateRequestTime' )
   def handleCounterUpdateRequestTime( self ):
      for handler in self.counter:
         self.counterStatus.routingHandlerCounter[ handler ] = (
            self.counter[ handler ] )
      self.counterStatus.counterUpdateTime = Tac.now() 

   @Tac.handler( 'resetCountersTrigger' )
   def handleResetCountersTrigger( self ):
      for handler in self.counter:
         self.counter[ handler ].routed = 0
         self.counter[ handler ].ignored = 0
