# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# The PythonImpl file contains code specific to the python implementation which can
# be deleted once C++ Etba is complete. Some of the functions/classes are used by
# both implementations (and will need to be moved later).
import os, Cell
import Tac, Tracing, Agent, Arnet, Plugins, SharedMem, Smash
import EbraTestBridgeLib
from EbraTestBridgePythonImpl import * # pylint: disable=wildcard-import
import CEosHelper
from Ark import getPlatform
from eunuchs.if_arp_h import ARPHRD_ETHER
from eunuchs.if_ether_h import ETH_P_ALL
from Toggles import EtbaDutToggleLib
import StageSysdbHelper
import EbraTestBridgeConstants
# e.g. IP_MCAST_BASE
from EbraTestBridgeConstants import * # pylint: disable-msg=wildcard-import
import QuickTrace
import ArfaHelperLib

handle = Tracing.defaultTraceHandle()
t0 = handle.trace0
t4 = handle.trace4    # misc important events
t5 = handle.trace5    # Packet tx/rx, error level
t6 = handle.trace6    # Packet tx/rx, alert level
t7 = handle.trace7    # Packet tx/rx, normal level
t8 = handle.trace8    # Packet tx/rx, debug level
t8b = handle.trace3   # Packet tx/rx, super-debug level
t9 = handle.trace9    # MAC address table aging/flushing

qt0 = QuickTrace.trace0

def etbaTrace( *args ):
   t0( *args )
   qt0( *args )

# During a switchover, Etba agent will take care of platform and hardware
# agents that don't run on a namespace dut so that all switchover stages
# can be marked as complete
agentsAndStagesDict = {}
agentsAndStagesDict[ 'PlxPcie-system' ] = [ 'PCIEAcquired',
                                            'PcieConfigurable' ]
agentsAndStagesDict[ 'ForwardingAgent' ] = [ 'DmaReady',
                                             'HwSyncWaitNormal',
                                             'HwSyncWaitForSlowCards' ]
agentsAndStagesDict[ 'ElectionMgr' ] = [ 'SwitchoverReady' ]
agentsAndStagesDict[ 'Fru' ] = [ 'Fru-Plugins' ]
agentsAndStagesDict[ 'ModularSystem' ] = [ 'ModularSystem-switchover' ]
agentsAndStagesDict[ 'NorCalCard' ] = [ 'CardPowerAgents' ]
agentsAndStagesDict[ 'Pca9555' ] = [ 'CardPowerAgents' ]
agentsAndStagesDict[ 'Ucd9012-system' ] = [ 'CardPowerAgents' ]
agentsAndStagesDict[ 'MactAgent' ] = [ 'InitMacTable',
                                       'L2RibReady' ]
agentsAndStagesDict[ 'SandL3Unicast' ] = [ 'PlatformLfibSync' ]

class StageProgressReactor( object ):
   def __init__( self, progressDir, redundancyStatus ):
      self.progressDir_ = progressDir
      self.redundancyStatus_ = redundancyStatus
      self.sm_ = StageSysdbHelper.ProgressDirSm( self.progressDir_, self._handler )

   def _handler( self, instance, key ):
      t0( 'StageProgressReactor marking switchoverStage', key, 'as complete' )
      self.redundancyStatus_.switchoverStage[ key ] = True

_extraSupportedPlugins = []

def addExtraFastEtbaPlugin( *args ):
   # We only want this to be called once
   assert not _extraSupportedPlugins
   _extraSupportedPlugins.extend( args )
   os.environ[ 'ETBATESTBRIDGE_PLUGINS_EXTRA' ] = (
      ",".join( _extraSupportedPlugins ) )

def loadEbraTestBridgePlugins():
   filenameFilter = None
   if 'SKIP_ETBA_PLUGINS' in os.environ:
      return

   if 'ETBATESTBRIDGE_PLUGINS' in os.environ:
      plugins = os.environ[ 'ETBATESTBRIDGE_PLUGINS' ].split( ',' )
   else:
      plugins = None

      if EtbaDutToggleLib.toggleFastEtbaEnabled():
         # Note: ONLY the agentInit and bridgeInit handlers are
         #       currently called.  The bridgeInit handlers are
         #       called with a trampoline object that tries to
         #       emulate the old behavior of the python version.

         # Our default set is small: only load the ones we know we support, but
         # don't create forward dependencies if they are not installed yet, by
         # dynamically filtering the filenames discoved by loadPlugins().
         # Ira and IpEth: set up state allowing rtr duts to support IP
         # SubIntf: set up state allowing subintfs
         # FwdingEtba: interface with ForwardingHelper
         # MplsEtba: MPLS forwarding
         #   Note that MplsEtba loads FwdIntfEtba explicitly.

         extraPlugins = []
         if 'ETBATESTBRIDGE_PLUGINS_EXTRA' in os.environ:
            extraPlugins = os.environ[ 'ETBATESTBRIDGE_PLUGINS_EXTRA' ].split( "," )

         def supportedPlugin( f ):
            for plugin in [ 'FwdingEtba', 'IpEth', 'Ira', 'MplsEtba',
                            'SubIntf', 'Vxlan', 'Qos' ] + extraPlugins:
               if f.endswith( ( plugin + ".py", plugin + ".pyc" ) ):
                  print f, "plugin permitted"
                  return True
            print f, "plugin denied"
            return False
         filenameFilter = supportedPlugin

   t0( 'loading EbraTestBridgePlugins, plugin list:', plugins )
   Plugins.loadPlugins( 'EbraTestBridgePlugin', plugins=plugins,
                        context=PluginContext(),
                        filenameFilter=filenameFilter )
   os.environ[ EbraTestBridgeConstants.pluginsLoadedEnvVarName ] = '1'

def loadCPlugins( pluginCtx ):
   loader = Tac.newInstance( "Arfa::Plugin::Loader" )
   # TODO change the install location of all the plugins
   loader.loadPlugins( 'EtbaPlugin', pluginCtx, "", "" )

class EbraTestPortTrampoline( object ):
   """ Allow python handlers to call into a port to send a packet.
   All packet modifications are ignored for now.
   """
   def __init__( self, port ):
      self.port_ = port
   
   def name( self ):
      # Method needed in MplsL3EvpnEtbaDecap.py module for a trace
      return self.port_.intfId

   @property
   def intfName_( self ):
      # Property required in decapHandlersTaccCallback() of
      # class EbraTestBridgeTrampoline
      return self.port_.intfId

   @property
   def intfStatus_( self ):
      return self.port_.intfStatus

   @property
   def intfConfig_( self ):
      return self.port_.intfConfig

   def port( self ):
      # Not part of the python API, but used to trampoline back into C++
      return self.port_

   def processFrame( self, pkt ):
      self.port_.processor.processFrame( pkt, self.port_ )

   def sendFrame( self, data, srcMac, dstMac, srcPort, priority,
                  vlanId, vlanAction ):
      pkt = Tac.newInstance( "Arnet::Pkt" )
      pkt.stringValue = data
      _ = srcMac = dstMac

      assert not priority

      translatedVlanAction = {
            EbraTestBridgeLib.PKTEVENT_ACTION_NONE: 'none',
            EbraTestBridgeLib.PKTEVENT_ACTION_REMOVE: 'remove',
            EbraTestBridgeLib.PKTEVENT_ACTION_ADD: 'add',
            EbraTestBridgeLib.PKTEVENT_ACTION_REPLACE: 'replace',
            }[ vlanAction ]

      if srcPort is None:
         srcPort = self.intfName_

      if priority is None:
         priority = 0

      if vlanId is None:
         vlanId = Tac.Value( 'Bridging::VlanId', 1 )

      self.port_.sendFrame( pkt, srcPort, priority, vlanId, translatedVlanAction )

class EbraTestBridgeTrampoline( object ):
   # Trampoline from old python API to new C++ API.
   def __init__( self, entityManager, smashEntityManager, root, etbaAgent,
                 ethIntfConfigDir, ethIntfStatusDir, inNamespace ):
      self.entityManager_ = entityManager
      self.smashEntityManager_ = smashEntityManager
      self.root_ = root
      self.brStatus = etbaAgent.brStatus_
      self.brStatus_ = self.brStatus
      self.brConfig_ = root.bridgeRoot.commonData.brConfig
      self.bridgeMac_ = self.brConfig_.bridgeMacAddr
      self.ethIntfConfigDir_ = ethIntfConfigDir
      self.ethIntfStatusDir_ = ethIntfStatusDir
      self.inNamespace_ = inNamespace
      self.packetContext = None
      self.pythonPorts = {}
      self.etbaAgent_ = etbaAgent
      self.setupPythonCallbacks()

   def setupPythonCallbacks( self ):
      import _Tac
      # pylint: disable=protected-access
      if ebraRoutingHandlers:
         self.root_.pluginManager.routingPythonCallback = _Tac.handle(
            self.routingHandlersTaccCallback )
      if packetReplicationHandlers:
         self.root_.pluginManager.outputIntfsPythonCallback = (
            _Tac.handle( self.outputIntfsFilterHandlersTaccCallback ) )
      self.setupDecapHandlers()

   def setupDecapHandlers( self ):
      if not preTunnelHandler:
         return
      import _Tac
      # pylint: disable=protected-access
      self.root_.pluginManager.decapPythonCallback = _Tac.handle(
         self.decapHandlersTaccCallback )

      mgr = self.root_.pluginManager
      for key, ( bpfFilterStr, func ) in preTunnelHandlerBpfFilter.iteritems():
         name = "%s.%s" % ( func.__module__, func.__name__ )
         obj = Tac.newInstance( "Arnet::BpfFilter", name )
         obj.bpfFilterString = bpfFilterStr
         mgr.decapPythonBpfFilters[ key ] = obj

   def bridgeMac( self ):
      return self.bridgeMac_

   def vlanTagState( self, srcPortName, srcIsFabric, etherType, data, dstMacAddr ):
      etbaPort = self.root_.ports.port[ srcPortName ]
      pkt = Tac.newInstance( "Arnet::Pkt" )
      pkt.stringValue = data
      vlanTagInfo = self.root_.packetProcessor.vlanTagState( pkt, etbaPort )
      return ( vlanTagInfo.vlanId,
               None, # priority?
               False, # vlanTagNeedsUpdating?
               vlanTagInfo.tagged,
               vlanTagInfo.routed )

   def addPort( self, port ):
      import _Tac
      # pylint: disable=protected-access
      pythonPtr = _Tac.handle( port )
      cPort = Tac.newInstance( "Arfa::ArfaPythonPort",
                               port.intfName_, None, None, pythonPtr )
      self.root_.ports.port.addMember( cPort )
      self.pythonPorts[ port.intfName_ ] = port
      return cPort

   def delPort( self, portName ):
      port = self.pythonPorts.pop( portName, None )
      if port:
         port.close()

      del self.root_.ports.port[ portName ]

   def macTableFlushPort( self, intfId ):
      self.root_.bridgeRoot.macFlush.flushPort( intfId )

   def learnAddr( self, vlanId, macAddr, portName,
                  entryType='learnedDynamicMac', moves=1 ):
      if moves is None:
         moves = 1
      fid = self.brConfig_.vidToFidMap.get( vlanId )
      if not fid:
         fid = vlanId
      self.root_.bridgeRoot.pipeline.learnHelper.learnAddrValues(
         fid, vlanId, macAddr, portName, entryType, moves )

   def inNamespace( self ):
      return self.inNamespace_

   def em( self ):
      return self.entityManager_

   def sEm( self ):
      return self.smashEntityManager_

   def decapHandlersTaccCallback( self, pktCtx, handlerKey ):
      # pylint: disable=protected-access
      data = str( pktCtx.pkt.stringValue )
      srcPort = pktCtx.srcPort
      preTunnelHdlr = preTunnelHandlerBpfFilter[ handlerKey ][ 1 ]
      t7( "In python decapHandlersTaccCallback" )

      ( chgData, chgSrcPort, drop ) = preTunnelHdlr(
         self, pktCtx.dstMacAddr, data, srcPort )
      if drop:
         t7( '%s consumed the frame' % preTunnelHdlr.__name__ )
         return Tac.enumValue( "Arfa::DecapPluginResultEnum",
                               "drop" )
      if chgData or chgSrcPort:
         t7( 'data/port %s/%s from %s' % (
            bool( chgData ), bool( chgSrcPort ), preTunnelHdlr.__name__ ) )
         newData = Tac.newInstance( "Arnet::Pkt" )
         newData.stringValue = chgData or data
         newPort = ( self.root_.ports.port[ chgSrcPort.intfName_ ] if
                     chgSrcPort else srcPort )
         newPktCtx = self.root_.packetProcessor.setupPacketContext( newData,
                                                                    newPort )
         mod = Tac.newInstance( "Arfa::RewriteModification",
                                newPktCtx )
         pktCtx.outputModifications.enq( mod )
         return Tac.enumValue( "Arfa::DecapPluginResultEnum",
                               "packetDecapped" )
      return Tac.enumValue( "Arfa::DecapPluginResultEnum",
                            "noActionTaken" )

   def routingHandlersTaccCallback( self, vlanId, tagged, data ):
      _, dstMacAddr = EbraTestBridgeLib.macAddrsAsStrings( data )
      routedPackets = []
      t5( "Python routing handlers" )
      for h in ebraRoutingHandlers:
         t5( "Python handler:", h )
         modifiedPktVlanTupleList = h( self, vlanId, dstMacAddr, data )
         for ( modifiedPkt, vlans ) in modifiedPktVlanTupleList:
            if modifiedPkt:
               t5( "Python modified packet" )
               routedPackets.append( ( modifiedPkt, vlans ) )
      vlanAction = ( EbraTestBridgeLib.PKTEVENT_ACTION_REPLACE if tagged else
                     EbraTestBridgeLib.PKTEVENT_ACTION_ADD )
      for pkt, vlans in routedPackets:
         for vlan in vlans:
            newdata = EbraTestBridgeLib.applyVlanChanges( pkt, 0, vlan,
                                                          vlanAction )
            arnetPkt = Tac.newInstance( "Arnet::Pkt" )
            arnetPkt.stringValue = newdata
            srcPort = self.root_.ports.port[ "Cpu" ]
            pktCtx = self.root_.packetProcessor.setupPacketContext( arnetPkt,
                                                                    srcPort )
            self.root_.bridgeRoot.pipeline.stepSm.processPacket( pktCtx )

      # False if no routing handler did anything, True if a routing handler did
      return bool( routedPackets )

   def outputIntfsFilterHandlersTaccCallback( self, pktCtx ):
      # pylint: disable=protected-access
      finalIntfs = { intf: None for intf in pktCtx.outputIntfs.intf }
      srcPort = pktCtx.srcPort
      dropReasonList = list( pktCtx.dropReasonList.dropReason )
      vlanId = pktCtx.vlanInfo.vlanId
      dstMacAddr = pktCtx.dstMacAddr
      data = str( pktCtx.pkt.stringValue )
      for func in packetReplicationHandlers:
         # These functions better not muck with anything but the finalIntfs, if
         # they do, we don't honor it.
         func( self, finalIntfs, srcPort, dropReasonList, vlanId, dstMacAddr,
               data=data )

      t4( "Old Intfs:", str( set( pktCtx.outputIntfs.intf ) ), "New Intfs:",
          finalIntfs )
      pktCtx.outputIntfs.intf.clear()
      for intf in finalIntfs:
         pktCtx.outputIntfs.intf.add( intf )

      return None # Function doesn't need a return value

   def deleteMacAddressEntry( self, vlanId, macAddr, entryType=None ):
      t9( 'Deleting MAC entry at vlan ', vlanId, ' mac ', macAddr )
      fid = self.brConfig_.vidToFidMap.get( vlanId )
      if not fid:
         fid = vlanId

      key = HostKey( fid, macAddr )
      if self.etbaAgent_.smashBrStatus_.smashFdbStatus.has_key( key ):
         # It is possible that the address was already deleted by a flushing
         # operation.
         hostEntry = self.etbaAgent_.smashBrStatus_.smashFdbStatus[ key ]
         if hostEntry.entryType == 'configuredStaticMac' and \
            entryType != 'configuredStaticMac':
            t4( "Do not Flush static entry" )
         else:
            del self.etbaAgent_.smashBrStatus_.smashFdbStatus[ key ]
            # The python implemtation of rmoving from the aging table doesn't
            # make sense to me. Only delete from the aging table if we deleted
            # from the smash table
            hostKey = Tac.Value( "Bridging::HostKey", fid, macAddr )
            del self.root_.bridgeRoot.agingTable.agingTable[ hostKey ]

   def processFrame( self, data, dc1, dc2, srcPortPython, tracePkt ):
      pkt = Tac.newInstance( "Arnet::Pkt" )
      pkt.stringValue = data
      port = self.root_.ports.port[ srcPortPython.intfName_ ]
      self.root_.packetProcessor.processFrame( pkt, port, tracePkt )

   def destLookup( self, vlanId, macAddr ):
      assert False, "Hoping this isn't used right now as it has been converted"
      intfs = Tac.newInstance( "Arfa::InterfaceSet" )
      result = self.root_.bridge.destLookup( vlanId, macAddr, intfs )
      return result, list( intfs.intf )

   def bridgeFrame( self, data, srcMac, dstMac, srcPort, trace ):
      """ Allow a plugin (e.g., FwdIntfEtba) to bridge a frame using the
      python API. """
      pkt = Tac.newInstance( "Arnet::Pkt" )
      pkt.stringValue = data
      # srcPort is a trampoline port, so port() is the C++ port.
      srcPort.processFrame( pkt )

   @property
   def port( self ):
      # Return a dict of trampolines for all the ports.  The use case here
      # is basically bridge.port[ 'Ethernet32' ].sendFrame()
      portDict = {}
      for ( p, v ) in self.root_.ports.port.items():
         if p in self.pythonPorts:
            # For python ports return python object
            portDict[ p ] = self.pythonPorts[ p ]
         else:
            portDict[ p ] = EbraTestPortTrampoline( v )
      return portDict

class PortFactory( object ):
   def __init__( self, entityManager, bridge ):
      self.entityManager_ = entityManager
      self.bridge_ = bridge
      smash = SharedMem.entityManager( entityManager.sysname(), False ).shmemDir
      self.intfCounters = EtbaIntfCounterHelper(
         smash.entity[ mountHelperTac.basePath ][ 'EtbaDut' ][ 'current' ] )
      self.kernelOperStateRoot = \
         Tac.root.newEntity( 'KernelOperState::Root', 'kernelOperStateRoot' )
      self.kernelOperStateRoot.kosm = ()
      self.kernelOperStateRoot.intfStatusDir = ( 'intfStatusDir', )
      self.kernelOperStateRoot.intfStatusLocalDir = ( 'intfStatusLocalDir', )
      self.kernelOperStateRoot.netNsCollection = ( "kns", )
      self.kernelOperStateRoot.netNsManager = (
         self.kernelOperStateRoot.netNsCollection, )
      self.kernelOperStateRoot.intfStatusOperStatusSm = (
         self.kernelOperStateRoot.intfStatusDir,
         self.kernelOperStateRoot.intfStatusLocalDir,
         self.kernelOperStateRoot.kosm,
         self.kernelOperStateRoot.netNsCollection )

   def newPort( self, intfConfig, intfStatus, intfXcvrStatus, intfStatusLocal,
                tapDevice, trapDevice ):
      em = self.entityManager_

      if not intfConfig:
         # This can happen for some types of remote ports, like Peer ports in
         # the presence of an Mlag, if Etba is restarted
         return None

      if intfStatus.tacType.fullTypeName == 'Interface::VlanIntfStatus':
         # This can happen if a startup-config is being loaded.
         return None

      # BUG460489: get the intfGlobalConfig so that we can specify the
      # counter rate interval.  For now, we specify it as None and
      # use the defaults.
      intfGlobalConfig = None
      if intfStatus.tacType.fullTypeName == 'Interface::EthPhyIntfStatus':
         intfId = intfStatus.intfId
         intfCounter = Tac.Value( 'Interface::EthIntfCounterSmash', intfId )
         self.intfCounters.allIntfCounters.counter.addMember( intfCounter )
         if EtbaDutToggleLib.toggleFastEtbaEnabled():
            if tapDevice.fileno != 0:
               # tapDevice is a Bridging::Etba::Config::TapDevice that refers to
               # a file descriptor we got from our parent.  We turn it into a
               # Pam using TapFdPam.
               tapPam = Tac.newInstance( 'Arnet::TapFdPam', intfId,
                     tapDevice.name, tapDevice.fileno )
               tapPam.mode = 'enabled'
            else:
               # open tapDevice.name using an EthDevPam
               tapPam = Tac.newInstance( 'Arnet::EthDevPam', tapDevice.name )
               tapPam.ethProtocol = ETH_P_ALL
               tapPam.separateDot1QHdr = True
               # This results in a socket buffer about double the default,
               # to absorb bursts of packets.
               tapPam.sockRcvBuf = 5000000
               tapPam.maxRxPktData = 10000
               tapPam.enabled = True
            port = Tac.newInstance( 'Arfa::ArfaPhyPort',
                                    intfId, self.bridge_.packetProcessor,
                                    tapPam, trapDevice,
                                    intfConfig,
                                    intfGlobalConfig,
                                    intfStatus,
                                    intfStatusLocal,
                                    self.intfCounters.allIntfCounters,
                                    intfXcvrStatus )
         else:
            port = EbraTestPhyPort( self.bridge_, tapDevice, trapDevice, intfConfig,
                                    intfStatus, intfXcvrStatus, intfStatusLocal,
                                    self.intfCounters, self.entityManager_ )
         ethIntfCounterWriterStatus = em.entity(
               'interface/ethIntfCounter/writerStatus/EtbaDut' )
         ethIntfCounterWriterStatus.intf[ intfStatus.intfId ] = True

         self.kernelOperStateRoot.intfStatusDir.intfStatus.addMember( intfStatus )

         if EtbaDutToggleLib.toggleFastEtbaEnabled():
            # List of EthPhyIntf interfaces for forwarding model resolution
            mngdIntfList = em.getLocalEntity( 'interface/status/managedIntfList' )
            mngdIntfList.intfStatus.addMember( intfStatus )

      elif EtbaDutToggleLib.toggleFastEtbaEnabled():
         return None
      else:
         for ( typeName, klass ) in interfaceHandlers:
            if intfStatus.tacType.fullTypeName == typeName:
               port = klass( self.bridge_, tapDevice, trapDevice,
                             intfConfig, intfStatus )
               break
         else:
            raise Exception( "Can't add an %s to an EbraTestBridge" %
                             intfStatus.tacType.fullTypeName )
      return port

class EtbaFdbStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Smash::Bridging::Status"

   def __init__( self, status, bridgeTrampoline ):
      Tac.Notifiee.__init__( self, status )
      self.bridgeTrampoline_ = bridgeTrampoline

   # Is immediate? If so, should it be a codef?
   @Tac.handler( 'smashFdbStatus' )
   def handleFdbStatus( self, key ):
      vlanId = key.fid
      macAddr = key.addr
      if key in self.notifier_.smashFdbStatus:
         val = self.notifier_.smashFdbStatus[ key ]
         portName = val.intf
         entryType = val.entryType
         for h in learningHandlers:
            # ignore result, what does it even mean?
            h( vlanId, macAddr, portName, entryType )
      else:
         for h in postAgingHandlers:
            h( self.bridgeTrampoline_, vlanId, macAddr )

class EbraBridgingConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Bridging::Config"

   def __init__( self, etbaConfigReactor ):
      etbaTrace( "Creating EbraTestBridge" )
      Tac.Notifiee.__init__( self, etbaConfigReactor.etbaAgent_.brConfig_ )
      self.etbaConfigReactor_ = etbaConfigReactor

   @Tac.handler( 'bridgeMacAddr' )
   def handleBridgeMacAddr( self ):
      etbaTrace( "handleBridgeMacAddr" )
      if self.notifier_.bridgeMacAddr != '00:00:00:00:00:00':
         self.etbaConfigReactor_.createPorts()
         self.etbaConfigReactor_.handleComplete()
         # this should happen only once so we can clean up reactor now
         del self.etbaConfigReactor_.ebraBridgingConfigReactor

class EtbaConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Bridging::Etba::Config"

   def __init__( self, etbaAgentObj, etbaConfig, etbaCliConfig, ethIntfConfigDir,
                 ethIntfStatusDir, ethIntfStatusLocalDir, allEthPhyIntfStatusDir,
                 xcvrStatusDir, entityManager, inNamespace,
                 ebraCounterConfig, ebraCounterStatus ):
      self.etbaAgent_ = etbaAgentObj
      self.etbaConfig_ = etbaConfig
      self.etbaCliConfig_ = etbaCliConfig
      self.ethIntfConfigDir_ = ethIntfConfigDir
      self.ethIntfStatusDir_ = ethIntfStatusDir
      self.ethIntfStatusLocalDir_ = ethIntfStatusLocalDir
      self.allEthPhyIntfStatusDir_ = allEthPhyIntfStatusDir
      self.intfXcvrStatusDir_ = xcvrStatusDir
      self.entityManager_ = entityManager
      self.inNamespace_ = inNamespace
      self.tapDevices = {}
      self.bridge_ = None
      self.root_ = None
      self.bridgeConfigSm_ = None
      self.fabricTapDevice_ = None
      self.shim_ = None
      Tac.Notifiee.__init__( self, etbaConfig )
      self.hwcapabilities_ = self.entityManager_.entity( 'bridging/hwcapabilities' )
      self.noTrapDevice = False
      if CEosHelper.isCeos():
         etbaTrace( 'Skip creating the trapDevices in CEos' )
         self.noTrapDevice = True
      self.ebraCounterConfig_ = ebraCounterConfig
      self.ebraCounterStatus_ = ebraCounterStatus

      self.hwcapabilities_.vxlanSupported = True
      self.hwcapabilities_.vxlanRoutingSupported = True
      # Needed to unblock the CLI configuration guard in Etba for
      # Vxlan interface.
      self.hwcapabilities_.vxlanUnderlayMcastSupported = True
      # Needed to unblock the CLI configuration guard in Etba
      # "switchport mode dot1q-tunnel"
      self.hwcapabilities_.dot1qTunnelSupported = True
      # "switchport trunk native vlan tag"
      self.hwcapabilities_.taggedNativeVlanSupported = True

      # Replicator to create EthPhyIntfStatus under the archer path
      # This SM moved here from "class Etba( Agent.Agent ):"
      # as part of review: http://reviewboard/r/240481
      self._episReplicatorSm = None

      self.ebraBridgingConfigReactor = EbraBridgingConfigReactor( self )
      self.ebraBridgingConfigReactor.handleBridgeMacAddr()


   def isPhyDevice( self, devName ):
      phyPrefix = os.environ.get( 'ETBA_PHY_PREFIX' )
      if phyPrefix is not None and devName.startswith( phyPrefix ):
         return True
      sysClassNet = "/sys/class/net"
      if os.path.isdir( os.path.join( sysClassNet, devName ) ):
         devType = int( file( os.path.join( sysClassNet, devName,
                                            "type" ) ).read() )
         if ( devType == ARPHRD_ETHER and
            os.path.exists( os.path.join( sysClassNet, devName, "device" ) ) ):
            return  True
      return False

   def getMtu( self, devname ):
      sysClassNet = "/sys/class/net"
      mtu = int( file( os.path.join( sysClassNet, devname,
                                     "mtu" ) ).read() )
      return mtu

   def _createNewEtbaBridge( self ):
      # Would use a print here as none of the tracing in this file is enabled by
      # default, but prints don't show up in the agent log
      import sys
      sys.stderr.write( "Using new FastEtba (C++) implementation\n" )

      # TODO: need a better accessor from Agent.py
      # pylint: disable=protected-access
      cAgent = self.etbaAgent_._cAgent()
      if cAgent is None:
         cAgent = self.etbaAgent_.agentRoot_[ self.etbaAgent_.agentName ]
         assert cAgent is not None

      root = ArfaHelperLib.createAll(
         requireAllValid=True,
         brConfig=self.etbaAgent_.brConfig_,
         sbrStatus=self.etbaAgent_.smashBrStatus_,
         ethIntfStatusDir=self.ethIntfStatusDir_,
         topoConfig=self.entityManager_.entity( 'bridging/topology/config' ),
         topoStatus=self.entityManager_.entity( 'bridging/topology/inst/etba' ),
         ethIntfStatusLocal=self.ethIntfStatusLocalDir_,
         entityManager=self.entityManager_,
         cliConfig=self.etbaCliConfig_,
         requestCli=self.entityManager_.entity( 'bridging/flush/request/cli' ),
         replyCli=self.entityManager_.entity( 'bridging/flush/reply/all' ),
         clock=Tac.activityManager.clock,
         cAgent=cAgent )
      self.root_ = root
            
      self.root_.startTasks()
      for handler in self.etbaAgent_.cPluginCtx_.bridgeInitHandlers.values():
         handler.handler( root.pluginManager )

   def createPorts( self ):
      etbaTrace( "createPorts" )

      if not self.notifier_.complete:
         etbaTrace( "createPorts returning due to not complete" )
         return

      sysname = self.entityManager_.sysname()

      etbaTrace( "Creating EbraTestBridge" )
      if EtbaDutToggleLib.toggleFastEtbaEnabled():
         self._createNewEtbaBridge()
      else:
         self.bridge_ = EbraTestBridge( sysname, self.entityManager_,
                                        self.inNamespace_,
                                        self.ethIntfConfigDir_,
                                        self.ethIntfStatusDir_,
                                        self.ethIntfStatusLocalDir_,
                                        self.allEthPhyIntfStatusDir_,
                                        self.ebraCounterConfig_,
                                        self.ebraCounterStatus_ )

      portFactory = PortFactory( self.entityManager_, self.bridge_ or self.root_ )

      bridgeMac = self.entityManager_.entity( 'bridging/config' ).bridgeMacAddr

      for intfName in Arnet.sortIntf( self.ethIntfStatusDir_.intfStatus ):
         if intfName.startswith( "Management" ):
            continue

         # Sort the names so that the tap interfaces tend to come out in
         # order.  We set the MAC address to the bridge
         # MAC in case this becomes a routed port.  There's no harm
         # in having it set if it's not a routed port, since this mac
         # is never otherwise used.
         intfStatus = self.ethIntfStatusDir_[ intfName ]
         intfConfig = self.ethIntfConfigDir_.get( intfName )
         intfXcvrStatus = self.intfXcvrStatusDir_.intfXcvrStatus.get( intfName )
         if not intfXcvrStatus:
            intfXcvrStatus = \
               self.intfXcvrStatusDir_.intfXcvrStatus.newMember( intfName )
         # Look up the local EthIntfStatus, if one exists, generally
         # only true for physical management and front panel interfaces.
         intfStatusLocal = self.ethIntfStatusLocalDir_.intfStatusLocal.get(
            intfName )
         trapDevice = None
         if not self.noTrapDevice:
            if EtbaDutToggleLib.toggleFastEtbaEnabled():
               trapDevFn = EbraTestBridgeLib.trapDevicePam
            else:
               trapDevFn = EbraTestBridgeLib.trapDevice

            trapDevice = trapDevFn( sysname, intfName, self.inNamespace_,
                                    hwAddr=bridgeMac )
            etbaTrace( "createPorts IntfName %s " % intfName )

         if trapDevice or intfName in self.etbaConfig_.extTapDevice:
            # Look up external tap device created for this interface.
            if not os.environ.get( 'ETBA_DEBUG' ):
               tapDevice = self.etbaConfig_.extTapDevice[ intfName ]
            else:
               # Setting ETBA_DEBUG=1 allows manual launch of Etba.
               # e.g. ETBA_DEBUG=1 sudo Etba
               # This creates tap interfaces inside the namespace.
               # The DUT will not be able to send/recv packets from other NS DUTs.
               # Injecting packets from the tap interfaces inside the ns,
               # e.g., into rtrmpls2-et1 after you "netns rtrmpls2".
               print "Locally generating Tap interfaces for ", intfName
               if EtbaDutToggleLib.toggleFastEtbaEnabled():
                  self.tapDevices[ intfName ] = \
                        EbraTestBridgeLib.tapDevicePam( sysname, intfName )
               else:
                  self.tapDevices[ intfName ] = \
                        EbraTestBridgeLib.tapDevice( sysname, intfName )
               tapDevice = Tac.newInstance( 'Bridging::Etba::Config::TapDevice' )
               tapDevice.fileno = self.tapDevices[ intfName ].fileno()
               tapDevice.name = self.tapDevices[ intfName ].name

            # We set the intf deviceName to the trap device, because that is
            # where we're going to send trapped packets.  We put the tap (not
            # trap) device name in the etbaStatus, because that is the device
            # that the test should use.
            if self.noTrapDevice:
               intfStatus.deviceName = tapDevice.name
            else:
               if EtbaDutToggleLib.toggleFastEtbaEnabled():
                  intfStatus.deviceName = trapDevice.deviceName
               else:
                  intfStatus.deviceName = trapDevice.name
            # On vEOS Lab running on certain hypervisors jumbo frames are not
            # supported. Check the mtu setting of the physical device
            devname = tapDevice.name
            if getPlatform() == "veos" and self.isPhyDevice( devname ) :
               mtu = self.getMtu( devname )
               if mtu < self.hwcapabilities_.maxFrameSize:
                  etbaTrace( "Setting max frame size lower : Device %s mtu %d" %
                      ( devname, mtu ) )
                  self.hwcapabilities_.maxFrameSize = mtu
         else:
            # virtual interface, e.g., LAG
            tapDevice = None

         port = portFactory.newPort( intfConfig, intfStatus,
                                     intfXcvrStatus, intfStatusLocal,
                                     tapDevice, trapDevice )
         if port is not None:
            if EtbaDutToggleLib.toggleFastEtbaEnabled():
               self.root_.ports.port.addMember( port )
            else:
               self.bridge_.addPort( port )

      # Add a fabric (a.k.a cpu) tap interface.
      if EtbaDutToggleLib.toggleFastEtbaEnabled():
         self.fabricTapDevice_ = EbraTestBridgeLib.fabricDevicePam(
            sysname, bridgeMac, self.inNamespace_ )
         fabricTrapDevice = EbraTestBridgeLib.trapDevicePam( sysname, 'Cpu',
                                                             self.inNamespace_ )
      else:
         self.fabricTapDevice_ = EbraTestBridgeLib.fabricDevice( sysname, bridgeMac,
                                                                 self.inNamespace_ )
         fabricTrapDevice = EbraTestBridgeLib.trapDevice( sysname, 'Cpu',
                                                          self.inNamespace_ )

      if EtbaDutToggleLib.toggleFastEtbaEnabled():
         # TODO don't need to call ip for this, just need the right helpers
         # C++ Interface::setMtu()
         Tac.run( [ "ip", "link", "set", "dev", self.fabricTapDevice_.deviceName,
                    "mtu", "10000" ] )

         port = Tac.newInstance( "Arfa::ArfaCpuPort",
                                 'Cpu', self.root_.packetProcessor,
                                 self.fabricTapDevice_, fabricTrapDevice )
         self.root_.ports.port.addMember( port )
      else:
         self.fabricTapDevice_.ifconfig( "mtu", "10000" )

         port = EbraTestCpuPort(
            'Cpu', self.bridge_, self.fabricTapDevice_, fabricTrapDevice )
         self.bridge_.addPort( port )

   def _handleNewEtbaComplete( self ):
      # start a shim to populate Bridging::Status from changes in the Smash table
      # for the benefit of plugins that look at brStatus_ directly.  Once these
      # plugins are rewritten to use smash, we can eliminate this reactor and
      # the Bridging::Status altogether.
      self.shim_ = Tac.Type( 'Bridging::ReaderFdbStatusSm' )(
         self.etbaAgent_.brStatus_, self.etbaAgent_.smashBrStatus_ )

      # QQQ: where should this initialization stuff go?
      # In the python it was in the EbraTestBridge class
      topoInstStatus = self.entityManager_.entity( 'bridging/topology/inst/etba' )
      topoInstStatus.maximumTopologiesSupported = 63
      topoInstStatus.hardwareConfigured = True
      # Piling more initializtion stuff here for now
      brHwCapabilities = self.entityManager_.entity( 'bridging/hwcapabilities' )
      # Don't put anything here that isn't true of the
      # code in this directory.  If the feature is implemented
      # by a plugin, then the hwcapabilites setting goes in the
      # plugin too.
      brHwCapabilities.dot1qTunnelSupported = True
      brHwCapabilities.taggedNativeVlanSupported = True
      brHwCapabilities.pvstSupported = True
      brHwCapabilities.backupInterfacesSupported = True
      brHwCapabilities.privateVlansSupported = True
      brHwCapabilities.ieeeReservedMacForwardAddrSupported = True
      brHwCapabilities.staticMcastSupported = True
      # Needed to unblock the CLI configuration guard in Etba for the
      # 'no mac address learning' command for VLANs
      brHwCapabilities.vlanMacLearningSupported = True

      # XXX TODO don't create a new smash entity manager here!
      trampoline = EbraTestBridgeTrampoline(
         self.entityManager_,
         SharedMem.entityManager( self.entityManager_.sysname(),
                                  self.entityManager_.local() ),
         self.root_, self.etbaAgent_, self.ethIntfConfigDir_, self.ethIntfStatusDir_,
         self.inNamespace_ )

      # Huge hack to keep calling into python Vxlan plugin working
      trampoline.packetContext = self.root_.packetProcessor.vxlanPacketContext

      self.etbaAgent_.fdbReactorForPythonPlugins = EtbaFdbStatusReactor(
         self.etbaAgent_.smashBrStatus_, trampoline )

      for f in bridgeInitHandlers:
         try:
            f( trampoline )
         except Exception, e:   # pylint: disable-msg=broad-except
            # Lots of noise here until we get the backwards-compatible
            # API going.
            print "Ignoring %s in %s.%s" % ( e, f.__module__, f.__name__ )

   @Tac.handler( 'complete' )
   def handleComplete( self ):
      etbaTrace( "handleComplete" )

      if not self.notifier_.complete:
         etbaTrace( "handleComplete returning due to not complete" )
         return

      isActive = self.entityManager_.redundancyStatus().mode == "active"
      if not isActive:
         etbaTrace( "handleComplete returning due to not active state" )
         return

      if not self.bridge_ and not self.root_:
         etbaTrace( "handleComplete returning due to bridge not ready" )
         return

      etbaStatus = self.entityManager_.entity( 'bridging/etba/status' )
      etbaStatus.initialized = False # To take care of Etba restart case

      self._episReplicatorSm = Tac.newInstance(
         "Interface::EthPhyIntfStatusDirCreatorAndReplicatorSm",
         self.entityManager_.root().entity[ "interface/status/eth/phy/slice" ],
         self.entityManager_.root().entity[ "interface/archer/status/eth/phy/slice"
         ] )

      if EtbaDutToggleLib.toggleFastEtbaEnabled():
         self._handleNewEtbaComplete()
         etbaStatus.fabricDeviceName = self.fabricTapDevice_.deviceName
         self.setupBridgingPipline()
      else:
         self.bridge_.createHostReactor()
         self.bridge_.addReflector()
         self.bridge_.startRoot()
         self.bridge_.createCounterConfigReactor()
         self.bridge_.runPlugins()
         for v in self.bridge_.port.values():
            v.onActive()
         etbaStatus.fabricDeviceName = self.fabricTapDevice_.name

      etbaStatus.initialized = True

   def setupBridgingPipline( self ):
      print "Using new pipeline"

      # step = Tac.newInstance( "Bridging::Etba::StpBridgingStep",
      #                        self.bridge_.topoConfig )
      #pipelineAgent.steps.addStep( step )

      # step = Tac.newInstance( "Bridging::Etba::OutputIntfStpFilterBridgingStep",
      #                        self.bridge_.topoConfig )
      #pipelineAgent.steps.addStep( step )

# This class handles issues that arise when running on sso active and standby.
# Specifically, when running on active, there are many actions that need to be taken
# that cause a 'Write to read-only proxy' error when done on standby.
#
# This is a wrapper for the EtbaConfigReactor that has two parts. First, whether on
# active or standby, it creates the EtbaConfigRreactor object. That sets up the
# necessary state, but has been stripped of anything that writes to Sysdb. Next, only
# on active, it runs EtbaConfigReactor's handleComplete(), which sets up any state
# that writes to Sysdb.
class EtbaRedundancyStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Redundancy::RedundancyStatus"

   def __init__( self, etbaAgentObj, etbaConfig, etbaCliConfig, ethIntfConfigDir,
                 ethIntfStatusDir, ethIntfStatusLocalDir, allEthPhyIntfStatusDir,
                 xcvrStatusDir, entityManager, inNamespace, redundancyStatus,
                 ebraCounterConfig, ebraCounterStatus, progressDir ):
      Tac.Notifiee.__init__( self, redundancyStatus )
      self.etbaAgent_ = etbaAgentObj
      self.etbaConfig_ = etbaConfig
      self.etbaCliConfig_ = etbaCliConfig
      self.ethIntfConfigDir_ = ethIntfConfigDir
      self.ethIntfStatusDir_ = ethIntfStatusDir
      self.ethIntfStatusLocalDir_ = ethIntfStatusLocalDir
      self.allEthPhyIntfStatusDir_ = allEthPhyIntfStatusDir
      self.intfXcvrStatusDir_ = xcvrStatusDir
      self.entityManager_ = entityManager
      self.inNamespace_ = inNamespace
      self.protocol = redundancyStatus.protocol
      self.ebraCounterConfig_ = ebraCounterConfig
      self.ebraCounterStatus_ = ebraCounterStatus
      self.redundancyStatus_ = redundancyStatus
      self.progressDir_ = progressDir
      self.sPR_ = None
      self.updateReactors()

   @Tac.handler( 'mode' )
   def updateReactors( self ):
      sysMode = self.notifier_.mode
      etbaTrace( 'mode:', sysMode, 'protocol:', self.protocol )
      # If we are not on a modular system, mode will be always set to active, which
      # means that we always create the EtbaConfigReactor, which is the desired
      # behavior. We only care if we have the situation where the dut might be
      # running on standby mode, if so we don't want EtbaConfigReactor to run.

      firstTimeRun = not self.etbaAgent_.etbaConfigReactor_
      if firstTimeRun:
         etbaTrace( "creating EtbaConfigReactor" )
         self.etbaAgent_.etbaConfigReactor_ = EtbaConfigReactor( self.etbaAgent_,
                                                   self.etbaConfig_,
                                                   self.etbaCliConfig_,
                                                   self.ethIntfConfigDir_,
                                                   self.ethIntfStatusDir_,
                                                   self.ethIntfStatusLocalDir_,
                                                   self.allEthPhyIntfStatusDir_,
                                                   self.intfXcvrStatusDir_,
                                                   self.entityManager_,
                                                   self.inNamespace_,
                                                   self.ebraCounterConfig_,
                                                   self.ebraCounterStatus_ )

      if not firstTimeRun and sysMode == 'active':
         self.etbaAgent_.etbaConfigReactor_.handleComplete()
      elif sysMode == 'switchover':
         for agent, stages in agentsAndStagesDict.items():
            sysdbRoot = self.entityManager_.root()
            agentDir = Cell.path( 'stageAgentStatus/switchover/%s' % agent )
            agentStatus = sysdbRoot.entity[ agentDir ]
            for key in stages:
               t0( "EtbaRedundancyStatusReactor Marking stageAgentStatus", key,
                   "as complete" )
               switchoverStatusKey = Tac.Value( 'Stage::AgentStatusKey', agent, key,
                                                'default' )
               agentStatus.complete[ switchoverStatusKey ] = True
            self.sPR_ = StageProgressReactor( self.progressDir_,
                                              self.redundancyStatus_ )
         etbaTrace( "init plugins in switchover" )
         self.etbaAgent_.etbaConfigReactor_.bridge_.runSwitchoverBridgePlugins()

class Etba( Agent.Agent ):

   def __init__( self, entityManager, inNamespace=False ):
      self.inNamespace = inNamespace
      if 'NSNAME' in os.environ:
         self.inNamespace = True

      self.etbaConfigReactor_ = None
      self.redStatusReactor_ = None

      self.brConfig_ = None
      self.brStatus_ = None
      self.brVlanStatus_ = None
      self.brHwCapabilities_ = None
      self.smashBrStatus_ = None
      self.progressDir_ = None

      self.cPluginCtx_ = None

      self._cpuUsageCollectorSm = None

      Agent.Agent.__init__( self, entityManager )

      Tac.activityManager.useEpoll = True

   def doInit( self, entityManager ):
      loadEbraTestBridgePlugins()

      mountGroup = entityManager.mountGroup()

      mountGroup.mount( "interface/status/eth/phy/slice", "Tac::Dir", "ri" )
      mountGroup.mount( "interface/archer/status/eth/phy/slice",
                        "Tac::Dir", "wi" )

      mountGroup.mount( 'interface/counter', 'Tac::Dir', 'wi' ) # BUG6489 workaround
      mountGroup.mount( 'bridging/topology/config',
                        'Bridging::Topology::Config', 'w' )
      mountGroup.mount( 'bridging/topology/inst/etba',
                        'Bridging::Topology::Inst::Status', 'cw' )
      self.brConfig_ = mountGroup.mount( 'bridging/config',
                        'Bridging::Config', 'r' )
      mountGroup.mount( 'bridging/flush/request/cli',
                        'Bridging::HostTableFlushRequestDir', 'r' )
      mountGroup.mount( 'bridging/flush/reply/all',
                        'Bridging::HostTableFlushReplyDir', 'w' )
      self.brStatus_ = mountGroup.mount( 'bridging/status', 'Bridging::Status', 'w' )
      self.brVlanStatus_ = mountGroup.mount( 'bridging/vlan/status',
                                             'Bridging::VlanStatusDir', 'r' )
      self.brHwCapabilities_ = mountGroup.mount( 'bridging/hwcapabilities',
                                                 'Bridging::HwCapabilities', 'w' )
      ethIntfStatusLocalDir = mountGroup.mount( 
                               Cell.path( "interface/status/eth/local" ),
                                          "Interface::EthIntfStatusLocalDir", "wf" )
      mountGroup.mount( 'interface/status/eth/phy/slice',
                        'Tac::Dir', 'wi' )
      mountGroup.mount( 'interface/ethIntfCounter/writerStatus/EtbaDut',
                        'Interface::EthIntfCounterWriterStatus', 'cw' )
      mountGroup.mount( 'bridging/etba/status',
                        'Bridging::Etba::Status', 'w' )
      etbaConfig = mountGroup.mount( 'bridging/etba/config',
                                     'Bridging::Etba::Config', 'r' )
      etbaCliConfig = mountGroup.mount( 'bridging/etba/cli/config',
                                        'Bridging::Etba::CliConfig', 'r' )
      ebraCounterConfig = mountGroup.mount(
         'bridging/etba/counter/ebra/config',
         'Bridging::Etba::RoutingHandlerCounterConfig', 'r' )
      ebraCounterStatus = mountGroup.mount(
         'bridging/etba/counter/ebra/status',
         'Bridging::Etba::RoutingHandlerCounterStatus', 'w' )
      mountGroup.mount( Cell.path( "interface/status/eth/phy/local" ),
                              "Interface::EthPhyIntfStatusLocalDir", "wf" )
      mountGroup.mount( "interface/archer/eth/xcvr/slice/%d" % Cell.cellId(), 
                        "Tac::Dir", "wf" )
      xcvrStatusDir = mountGroup.mount(
         "interface/archer/eth/xcvr/slice/%d/etba" % Cell.cellId(),
         "Interface::EthIntfXcvrStatusDirEntry", "wf" )

      # Paths for SSO completion
      for agent, _ in agentsAndStagesDict.items():
         mountGroup.mount( Cell.path( 'stageAgentStatus/switchover/%s' % agent ),
                           "Stage::AgentStatus", "wcf" )
      self.progressDir_ = mountGroup.mount( Cell.path( 'stage/switchover/progress' ),
                                            "Stage::ProgressDir", "r" )
      mountGroup.mount( Cell.path( 'redundancy/status' ),
                        "Redundancy::RedundancyStatus", "wf" )

      maxNumIntfs = 128
      smashEm = SharedMem.entityManager( entityManager.sysname(),
                                          entityManager.local() )
      mountHelper = Tac.newInstance( 'Interface::EthIntfCounterMountHelper',
                                     smashEm )
      mountHelper.doMountWrite( 'EtbaDut', 'current', maxNumIntfs )

      if EtbaDutToggleLib.toggleFastEtbaEnabled():
         # create the Smash::Bridging::Status entity and initialize it for writing
         writerInfo = Smash.mountInfo(
            'writer',
            collectionInfo=[ ( 'smashFdbStatus', ETBA_SMASH_SIZE ) ],
            cleanup=True
         )

         self.smashBrStatus_ = smashEm.doMount(
            'bridging/status',
            'Smash::Bridging::Status',
            writerInfo
         )

         # TODO clean this up
         # pylint: disable=protected-access
         cAgent = self._cAgent()
         if cAgent is None:
            cAgent = self.agentRoot_[ self.agentName ]
            assert cAgent is not None

         self.cPluginCtx_ = Tac.newInstance( "Arfa::PluginContext",
                                             self.entityManager.cEntityManager(),
                                             cAgent )
         loadCPlugins( self.cPluginCtx_ )


      for f in agentInitHandlers:
         f( entityManager )

      self.createLocalEntity( "EthIntfConfigDir",
                              "Interface::EthIntfConfigDir",
                              "interface/config/eth/intf" )

      self.createLocalEntity( "EthIntfStatusDir",
                              "Interface::EthIntfStatusDir",
                              "interface/status/eth/intf" )

      self.createLocalEntity( "AllEthPhyIntfStatusDir",
                              "Interface::AllEthPhyIntfStatusDir",
                              "interface/status/eth/phy/all" )

      # IntfForwardingModel plugins for EthIntfPhy interfaces
      self.createLocalEntity( "ForwardingModelResolverState",
                              "Interface::LocalForwardingModelResolverState",
                              "interface/status/forwardingModelResolverState" )
      self.createLocalEntity( "ManagedIntfList", "Interface::IntfStatusPtrDir",
                              "interface/status/managedIntfList" )

      self.localEntitiesCreatedIs( True )

      ethIntfConfigDir = entityManager.getLocalEntity( "interface/config/eth/intf" )
      ethIntfStatusDir = entityManager.getLocalEntity( "interface/status/eth/intf" )
      allEthPhyIntfStatusDir = entityManager.getLocalEntity(
                                  "interface/status/eth/phy/all" )


      def _createRedundancyStatusReactor():
         etbaTrace( "creating EtbaRedundancyStatusReactor" )
         # EtbaRedundancyStatusReactor will react to redundancy/status.mode and
         # instantiate EtbaConfigReactor if we are in active mode
         redStatus = entityManager.entity( Cell.path( 'redundancy/status' ) )
         self.redStatusReactor_ = EtbaRedundancyStatusReactor( self, etbaConfig,
                                                           etbaCliConfig,
                                                           ethIntfConfigDir,
                                                           ethIntfStatusDir,
                                                           ethIntfStatusLocalDir,
                                                           allEthPhyIntfStatusDir,
                                                           xcvrStatusDir,
                                                           entityManager,
                                                           self.inNamespace,
                                                           redStatus,
                                                           ebraCounterConfig,
                                                           ebraCounterStatus,
                                                           self.progressDir_ )

      def _finish():

         # BUG27740 - we can't guarantee that all of the inner mount group
         # finish functions have been called at this point. We have to
         # return to the activity loop before continuing. Otherwise we
         # can end up calling into functions registered by plugin that
         # tries to access state that has been mounted but not converted
         # to an Entity from an EntityFuture yet
         Tac.Deferral( _createRedundancyStatusReactor )

         self._cpuUsageCollectorSm = Tac.newInstance(
               "CpuUsage::CpuUsageCollectorSm", "Etba" )

      mountGroup.close( _finish )

def main():
   # Custom toggle overrides just for Etba in agent mode
   import ArTogglesPyAgent
   ArTogglesPyAgent.pyParseToggleOverridesFile( '/tmp/etba-override' )

   QuickTrace.initialize( "Etba.qt" )
   container = Agent.AgentContainer( [ Etba ], passiveMount=True )
   container.addOption( "", "--inNamespace", action="store_true",
                        help="Etba is running in an isolated environment",
                        agentClass=Etba )
   container.parseOptions()
   container.runAgents()
