#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Arnet
import re
from TypeFuture import TacLazyType

HwCapabilities = Tac.Type( "Bridging::HwCapabilities" )
defaultTpid = Tac.Value( 'Bridging::VlanTpidOrNone' ).value

tacL2PtProtocol = TacLazyType( "Ebra::L2Pt::L2PtProtocol" )
tacL2PtAction = TacLazyType( "Ebra::L2Pt::L2PtAction" )
tacL2PtTagFormat = TacLazyType( "Ebra::L2Pt::L2PtTagFormat" )
tacL2ProtocolInfo = TacLazyType( "Ebra::L2Pt::L2ProtocolInfo" )
tacL2ProtocolFwdAction = TacLazyType( "Ebra::L2Pt::L2ProtocolFwdAction" )
tacL2ProtocolFwdInfo = TacLazyType( "Ebra::L2Pt::L2ProtocolFwdInfo" )
tacL2ProtocolMatch = TacLazyType( "Ebra::L2Pt::L2ProtocolMatch" )
tacSequenceNumber = TacLazyType( "Ebra::L2Pt::SequenceNumber" )

l2PtProtocolToMacAddr = {
   tacL2PtProtocol.lacp   : '01:80:c2:00:00:02',
   tacL2PtProtocol.lldp   : '01:80:c2:00:00:0e',
   tacL2PtProtocol.stp    : '01:80:c2:00:00:00',
   tacL2PtProtocol.macsec : '01:80:c2:00:00:03',
   tacL2PtProtocol.pause  : '01:80:c2:00:00:01',
   tacL2PtProtocol.elmi   : '01:80:c2:00:00:07'
}
macAddrTol2PtProtocol = { v : k for k, v in  l2PtProtocolToMacAddr.iteritems() }

l2PtProtocolToCliToken = {
   tacL2PtProtocol.lacp   : 'lacp',
   tacL2PtProtocol.lldp   : 'lldp',
   tacL2PtProtocol.stp    : 'stp',
   tacL2PtProtocol.macsec : 'macsec',
   tacL2PtProtocol.pause  : 'pause',
   tacL2PtProtocol.elmi   : 'e-lmi'
}

cliTokenToL2PtProtocol = { v : k for k, v in l2PtProtocolToCliToken.iteritems() }

pweProtocols = [ tacL2PtProtocol.stp, tacL2PtProtocol.lldp, tacL2PtProtocol.macsec,
                 tacL2PtProtocol.elmi ]

def l2ProtocolFromMacAddress( macAddr ):
   protocol = macAddrTol2PtProtocol.get( macAddr )
   if not protocol:
      assert False, 'Invalid MAC address'
   return protocol

def l2ProtocolMacAddress( l2PtProtocol ):
   macAddrStr = l2PtProtocolToMacAddr.get( l2PtProtocol )
   if not macAddrStr:
      assert False, 'Invalid protocol'
   return Arnet.EthAddr( macAddrStr )

def portChannelNameShort( pcnum ):
   return 'Po%d' % pcnum

def portChannelNameLong( pcnum ):
   return 'Port-Channel%d' % pcnum

def newBridgingConfig( name ):
   return Tac.newInstance( 'Bridging::Config', name )

ieeeReservedAddressList = [ '0180.c200.00%02x' % i for i in range( 0x00, 0x10 ) ]
ieee8021dStpBpduAddress = '0180.c200.0000'
ieeeReservedAddressAllowed = ( set( ieeeReservedAddressList ) -
                               set( [ ieee8021dStpBpduAddress ] ) )
ieeeReservedTokenAll = '0180.c200.0001-0180.c200.000f'
ieeeReservedConfigAll = HwCapabilities.ieeeReservedMacForwardAddrAll

ieeeReservedAddressConfigTokenMap = {
   ieeeReservedConfigAll : ieeeReservedTokenAll,
   '01:80:c2:00:00:01' : '0180.c200.0001',
   '01:80:c2:00:00:02' : '0180.c200.0002',
   '01:80:c2:00:00:03' : '0180.c200.0003',
   '01:80:c2:00:00:04' : '0180.c200.0004',
   '01:80:c2:00:00:05' : '0180.c200.0005',
   '01:80:c2:00:00:06' : '0180.c200.0006',
   '01:80:c2:00:00:07' : '0180.c200.0007',
   '01:80:c2:00:00:08' : '0180.c200.0008',
   '01:80:c2:00:00:09' : '0180.c200.0009',
   '01:80:c2:00:00:0a' : '0180.c200.000a',
   '01:80:c2:00:00:0b' : '0180.c200.000b',
   '01:80:c2:00:00:0c' : '0180.c200.000c',
   '01:80:c2:00:00:0d' : '0180.c200.000d',
   '01:80:c2:00:00:0e' : '0180.c200.000e',
   '01:80:c2:00:00:0f' : '0180.c200.000f',
   }

def ieeeReservedAddressTokenList():
   return ieeeReservedAddressConfigTokenMap.values()

def ieeeReservedAddressConfigList():
   return ieeeReservedAddressConfigTokenMap.keys()

def isIeeeReservedAddressToken( token ):
   return ( token in ieeeReservedAddressTokenList() )

def ieeeReservedAddressConfigToToken( config ):
   return ieeeReservedAddressConfigTokenMap[ config ]

def ieeeReservedAddressTokenToConfig( token ):
   for ( config, tokenMap ) in ieeeReservedAddressConfigTokenMap.items():
      if token == tokenMap:
         return config
   assert 0, ( 'token %s not found in ieeeReservedAddressTokenConfigMap' % token )

def ieeeReservedAddressTokenAll():
   return ieeeReservedTokenAll

def ieeeReservedAddressConfigAll():
   return ieeeReservedConfigAll

ieeeReservedGroupConfigTokenMap = {
   ieeeReservedConfigAll : ieeeReservedTokenAll,
   '01:80:c2:00:00:01' : '0180.c200.0001',
   '01:80:c2:00:00:02' : '0180.c200.0002',
   '01:80:c2:00:00:03' : '0180.c200.0003,0180.c200.000e',
   '01:80:c2:00:00:04' : '0180.c200.0004-0180.c200.000d,0180.c200.000f',
   }

def ieeeReservedGroupTokenList():
   return ieeeReservedGroupConfigTokenMap.values()

def ieeeReservedGroupConfigList():
   return ieeeReservedGroupConfigTokenMap.keys()

def isIeeeReservedGroupToken( token ):
   return ( token in ieeeReservedGroupTokenList() )

def isIeeeReservedGroupConfig( config ):
   return config in ieeeReservedGroupConfigTokenMap

def ieeeReservedGroupConfigToToken( config ):
   return ieeeReservedGroupConfigTokenMap[ config ]

def ieeeReservedGroupTokenToConfig( token ):
   for ( config, tokenMap ) in ieeeReservedGroupConfigTokenMap.items():
      if token == tokenMap:
         return config
   assert 0, ( 'token %s not found in ieeeReservedGroupConfigTokenMap' % token )

def ieeeReservedGroupTokenAll():
   return ieeeReservedTokenAll

def ieeeReservedGroupConfigAll():
   return ieeeReservedConfigAll

def expandIeeeMacAddressStrToEthAddr( macStr ):
   return expandIeeeMacAddressStr( macStr, False )

def ieeeMacAddressStrFmt( macStr, strFormat ):
   if strFormat:
      return macStr
   else:
      return Arnet.EthAddr( macStr )

def expandIeeeMacAddressStr( macStrs, strFormat ):
   """ Do not use expandIeeeMacAddressStr, but use
       its callers instead.

       Expands sequential mac addresses and returns a
       list of mac addresses.

       Expects a mac address string in either format:
         '0000.0000.0000'
         '0000.0000.0000, 0000.0000.00ff'
         '0000.0000.0000 - ffff.ffff.ffff'
         '0000.0000.0000, 0000.0000.00ff-0000.0000.ffff'
   """
   macHexLen = 12
   macs = []
   if not macStrs:
      return macs
   for macStr in macStrs.split( ',' ):
      if '-' in macStr.strip():
         ( lMac, uMac ) = map( lambda x: re.sub( '\.', '', x ), macStr.split( '-' ) )
         for mac in range( int( lMac, 16 ), int( uMac, 16 ) + 1 ):
            fmtMac = '{0:0{1}x}'.format( mac, macHexLen )
            fmtMac = re.sub( '(.{4})', '\\1.', fmtMac )[ :-1 ]
            macs.append( ieeeMacAddressStrFmt( fmtMac, strFormat ) )
      else:
         macs.append( ieeeMacAddressStrFmt( macStr, strFormat ) )

   return macs

def egressVlanXlateKeyIfBoth( egressVlanXlate,
                              ingressVlanXlateKey, ingressVlanXlateMapping ):
   if ingressVlanXlateMapping is None:
      return None

   egressVlanXlateKey = Tac.Value( "Bridging::VlanXlateKey",
         ingressVlanXlateMapping.vlanId,
         ingressVlanXlateKey.outerVid if ingressVlanXlateMapping.tunnel else 0 )
   egressVlanXlateMapping = egressVlanXlate.get( egressVlanXlateKey )
   if egressVlanXlateMapping and \
         egressVlanXlateMapping.vlanId == ingressVlanXlateKey.outerVid and \
         ( ( egressVlanXlateMapping.tunnel and ingressVlanXlateMapping.tunnel ) or
           ( egressVlanXlateMapping.innerVid == ingressVlanXlateKey.innerVid ) ):
      return egressVlanXlateKey

   return None

def privateVlanAllowed( epochStatus ):
   if not epochStatus:
      return True
   return epochStatus.privateVlanEnabled
