# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import cjson
import copy
import os
import select
import StringIO

import CliCommon
from CliShellLib import EapiCliConnector
import FastServUtil

# External api to invoke eAPI going directly to the CLI process. This is similiar
# other eAPI examples using jsonrpclib
#    from EapiClientLib import EapiClient
#    eapiClient = EapiClient()
#    print( eapiClient.runCmds( 1, [ "show version" ] ) )

REQUEST_CONTENT = { 'jsonrpc': '2.0',
                    'method': None,
                    'params': None,
                    'id': 'FastCliSocketLib',
                  }

REMOTE_SOCK_READ_SIZE = 65536
SELECT_TIMEOUT = 5

# pylint: disable-msg=W1401

class EapiException( Exception ):
   def __init__( self, msg=None, code=None, data=None ):
      Exception.__init__( self, msg )
      self.msg = msg
      self.code = code
      self.data = data

   def __str__( self ):
      msg = 'EapiClient error %s: \'%s\'' % ( self.code, self.msg )
      if self.data:
         msg += ' Data: %s' % self.data
      return msg

class _Method( object ):
   def __init__( self, name, sysname, disableAaa, disableGuards, privLevel, env,
                 uid, gid ):
      self.name_ = name
      self.sysname_ = sysname
      self.disableAaa_ = disableAaa
      self.disableGuards_ = disableGuards
      self.env_ = env
      self.privLevel_ = privLevel
      self.uid_ = uid
      self.gid_ = gid
      self.signalSock_ = None
      self.responseSock_ = None
      self.statisticsSock_ = None

   def _disconnect( self ):
      self.signalSock_.close()
      self.signalSock_ = None
      if self.responseSock_:
         self.responseSock_.close()
      self.responseSock_ = None

   def _sendRequest( self, jsonRpcRequest ):
      cliConnector = EapiCliConnector()
      argv = [ '-p=%s' % self.privLevel_, '-s=%s' % self.sysname_ ]
      if self.disableAaa_:
         argv.append( '-A' )
      if self.disableGuards_:
         argv.append( '-G' )
      self.signalSock_, self.responseSock_, requestSock, self.statisticsSock_ = \
            cliConnector.connectToBackend( self.sysname_, argv, self.env_,
                                           self.uid_, self.gid_ )
      FastServUtil.writeString( requestSock, jsonRpcRequest )
      requestSock.close()

   def _getJsonRpcRequest( self, args, kwargs ):
      request = copy.deepcopy( REQUEST_CONTENT )
      request[ 'method' ] = self.name_
      if len( kwargs ) > 0:
         request[ 'params' ] = kwargs
      else:
         request[ 'params' ] = args
      return cjson.encode( request )

   def _readStreamedResponse( self ):
      readableSocks = [ self.signalSock_, self.responseSock_ ]
      try:
         while True:
            try:
               socksToRead = select.select( readableSocks, [], [] )[ 0 ]
            except select.error as e:
               if e.args[ 0 ] == os.errno.EINTR:
                  continue
               raise
            if self.signalSock_ in socksToRead:
               self.signalSock_.recv( 1 )
               if not self.responseSock_:
                  # if we don't have anymore in the response buffer we don't need
                  # to clear it
                  return

               # The FastClid-session has signaled to use that we are done producing
               # output. So we clear the response buffer until it returns nothing
               while True:
                  response = self.responseSock_.recv( REMOTE_SOCK_READ_SIZE )
                  if not response:
                     return
                  yield response
               return
            if self.responseSock_ in socksToRead:
               response = self.responseSock_.recv( REMOTE_SOCK_READ_SIZE )
               if not response:
                  readableSocks.remove( self.responseSock_ )
                  self.responseSock_.close()
                  self.responseSock_ = None
               else:
                  yield response

      except Exception as e: # pylint: disable-msg=W0703
         raise EapiException( str( e ), CliCommon.JsonRpcErrorCodes.INTERNAL_ERROR )
      finally:
         self._disconnect()

   def _readStatistics( self ):
      try:
         requestCount = FastServUtil.readInteger( self.statisticsSock_ )
         commandCount = FastServUtil.readInteger( self.statisticsSock_ )
         return requestCount, commandCount
      except Exception as e: # pylint: disable-msg=W0703
         raise EapiException( str( e ), CliCommon.JsonRpcErrorCodes.INTERNAL_ERROR )
      finally:
         self.statisticsSock_.close()
         self.statisticsSock_ = None

   def _jsonRpcCall( self, args, kwargs ):
      jsonRpcRequest = self._getJsonRpcRequest( args, kwargs )
      try:
         self._sendRequest( jsonRpcRequest )
         responseBuffer = StringIO.StringIO()
         for i in self._readStreamedResponse():
            if i is None:
               continue
            responseBuffer.write( i )
         self._readStatistics()
         response = cjson.decode( responseBuffer.getvalue() )
         responseBuffer.close()
      except EapiException as e:
         raise e
      except Exception as e: # pylint: disable-msg=W0703
         raise EapiException( str( e ), CliCommon.JsonRpcErrorCodes.INTERNAL_ERROR )

      if 'error' in response:
         raise EapiException( response[ 'error' ][ 'message' ],
                              response[ 'error' ][ 'code' ],
                              response[ 'error' ].get( 'data', None ) )
      return response

   def _sendRpcRequest( self, args, kwargs ):
      jsonRpcRequest = None
      if args:
         jsonRpcRequest = args[ 0 ]
      if kwargs:
         jsonRpcRequest = kwargs[ 'jsonRpcRequest' ]
      if jsonRpcRequest is None:
         raise EapiException( '\'jsonRpcRequest\' not specified',
                              CliCommon.JsonRpcErrorCodes.INTERNAL_ERROR )

      try:
         self._sendRequest( jsonRpcRequest )
      except Exception as e: # pylint: disable-msg=W0703
         raise EapiException( str( e ), CliCommon.JsonRpcErrorCodes.INTERNAL_ERROR )

      return self._readStreamedResponse(), self._readStatistics

   def __call__( self, *args, **kwargs ):
      if len( kwargs ) > 0 and len( args ) > 0:
         raise EapiException( 'JSON-RPC does not support both '
                              'positional and keyword arguments.',
                              CliCommon.JsonRpcErrorCodes.INVALID_PARAMS )
      if self.name_ == '_sendRpcRequest':
         return self._sendRpcRequest( args, kwargs )
      else:
         return self._jsonRpcCall( args, kwargs )

class _EapiClientBase( object ):
   def __init__( self, sysname='ar', disableAaa=False, disableGuards=False,
                 privLevel=1, uid=os.getuid(), gid=os.getgid(), aaaAuthnId=None,
                 realTty=None, logName=None, sshConnection=None ):
      self.sysname_ = sysname
      self.disableAaa_ = disableAaa
      self.disableGuards_ = disableGuards
      self.privLevel_ = privLevel
      self.uid_ = uid
      self.gid_ = gid
      self.env_ = {}
      if aaaAuthnId is not None:
         self.env_[ 'AAA_AUTHN_ID' ] = aaaAuthnId
      if realTty is not None:
         self.env_[ 'REALTTY' ] = realTty
      if logName is not None:
         self.env_[ 'LOGNAME' ] = logName
      if sshConnection is not None:
         self.env_[ 'SSH_CONNECTION' ] = sshConnection

class EapiClient( _EapiClientBase ):
   def __init__( self, *args, **kwargs ):
      super( EapiClient, self ).__init__( *args, **kwargs )

   def setPrivLevel( self, privLevel ):
      self.privLevel_ = privLevel

   def __getattr__( self, name ):
      return _Method( name, self.sysname_, self.disableAaa_, self.disableGuards_,
                      self.privLevel_, self.env_, self.uid_, self.gid_ )

class EapiClientCtx( _EapiClientBase ):
   def __init__( self, *args, **kwargs ):
      super( EapiClientCtx, self ).__init__( **kwargs )
      self.signalSock_ = None
      self.responseSock_ = None
      self.requestSock_ = None
      self.statisticsSock_ = None

   def __enter__( self ):
      cliConnector = EapiCliConnector( stateless=False )
      argv = [ '-p=%s' % self.privLevel_, '-s=%s' % self.sysname_ ]
      if self.disableAaa_:
         argv.append( '-A' )
      if self.disableGuards_:
         argv.append( '-G' )
      signalSock, responseSock, requestSock, statisticsSock = \
            cliConnector.connectToBackend( self.sysname_, argv, self.env_,
                                           self.uid_, self.gid_ )
      self.signalSock_ = signalSock
      self.responseSock_ = responseSock
      self.requestSock_ = requestSock
      self.statisticsSock_ = statisticsSock

   def sendRpcRequest( self, jsonRpcRequest ):
      FastServUtil.writeString( self.requestSock_, jsonRpcRequest )
      output = []
      done = False
      while not done:
         try:
            filesReadyToRead, _, _ = select.select( [ self.statisticsSock_,
                                                      self.responseSock_ ],
                                                    [], [],
                                                    SELECT_TIMEOUT )
         except select.error as e:
            if e.args[ 0 ] == os.errno.EINTR:
               # If we get interrupted we just go back to where we were before
               continue
            raise

         if self.responseSock_ in filesReadyToRead:
            buf = self.responseSock_.recv( REMOTE_SOCK_READ_SIZE )
            output.append( buf )
         elif self.statisticsSock_ in filesReadyToRead:
            # now lets read the statistics to make sure everything is correct
            requestCount = FastServUtil.readInteger( self.statisticsSock_ )
            commandCount = FastServUtil.readInteger( self.statisticsSock_ )
            done = True

      return ''.join( output ), requestCount, commandCount

   def __exit__( self, errType, value, tb ):
      # we are going to block until the front-end has disconnected
      self.requestSock_.close()
      self.requestSock_ = None
      self.responseSock_.close()
      self.responseSock_ = None
      self.statisticsSock_.close()
      self.statisticsSock_ = None
      self.signalSock_.recv( 1 )
      self.signalSock_.close()
      self.signalSock_ = None
