#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Tac

class GenericDhcpServiceReactor( Tac.Notifiee ):
   notifierTypeName = '*'

   def __init__( self, config, service ):
      self.config_ = config
      self.service_ = service
      Tac.Notifiee.__init__( self, self.config_ )

   def onAttribute( self, attr, key ):
      Tac.Notifiee.onAttribute( self, attr, key )
      self.service_.sync()

class VendorOptionV4Reactor( GenericDhcpServiceReactor ):
   notifierTypeName = 'DhcpServer::VendorOptionIpv4'

   def __init__( self, config, service ):
      GenericDhcpServiceReactor.__init__( self, config, service )
      self.service_.handleVendorOptionAdded( self.config_.vendorId )

   def close( self ):
      self.service_.handleVendorOptionRemoved( self.config_.vendorId )
      GenericDhcpServiceReactor.close( self )

   @Tac.handler( 'subOptionConfig' )
   def handleChange( self, *args ):
      pass

class ReservationMacAddrIpv4Reactor( GenericDhcpServiceReactor ):
   notifierTypeName = 'DhcpServer::ReservationMacAddrIpv4'

   @Tac.handler( 'ipAddr' )
   def handleChange( self, *args ):
      pass

class SubnetV4ConfigReactor( GenericDhcpServiceReactor ):
   notifierTypeName = 'DhcpServer::SubnetConfig'

   def __init__( self, config, service ):
      GenericDhcpServiceReactor.__init__( self, config, service )
      self.service_.handleSubnetAdded( self.config_.subnetId )

      self.reservationsMacAddrReactor_ = Tac.collectionChangeReactor(
         self.config_.reservationsMacAddr, ReservationMacAddrIpv4Reactor,
         reactorArgs=( self.service_, ) )

   def close( self ):
      self.service_.handleSubnetRemoved( self.config_.subnetId )
      GenericDhcpServiceReactor.close( self )

   @Tac.handler( 'subnetName' )
   @Tac.handler( 'ranges' )
   @Tac.handler( 'dnsServers' )
   @Tac.handler( 'defaultGateway' )
   @Tac.handler( 'leaseTime' )
   @Tac.handler( 'tftpServerOption66' )
   @Tac.handler( 'tftpServerOption150' )
   @Tac.handler( 'tftpBootFileName' )
   @Tac.handler( 'reservationsMacAddr' )
   def handleChange( self, *args ):
      pass

class ServiceV4Reactor( GenericDhcpServiceReactor ):
   notifierTypeName = 'DhcpServer::Config'

   def __init__( self, config, service ):
      GenericDhcpServiceReactor.__init__( self, config, service )
      self.subnetConfigReactor_ = Tac.collectionChangeReactor(
         self.config_.subnetConfigIpv4, SubnetV4ConfigReactor,
         reactorArgs=( self.service_, ) )

      self.vendorOptionReactor_ = Tac.collectionChangeReactor(
         self.config_.vendorOptionIpv4, VendorOptionV4Reactor,
         reactorArgs=( self.service_, ) )

   @Tac.handler( 'clearIdIpv4' )
   def handleClearConfig( self ):
      self.service_.handleClearLease( self.config_.clearIdIpv4 )

class SubnetV6ConfigReactor( GenericDhcpServiceReactor ):
   notifierTypeName = 'DhcpServer::Subnet6Config'

   def __init__( self, config, service ):
      GenericDhcpServiceReactor.__init__( self, config, service )
      self.service_.handleSubnetAdded( self.config_.subnetId )

   def close( self ):
      self.service_.handleSubnetRemoved( self.config_.subnetId )
      GenericDhcpServiceReactor.close( self )

   @Tac.handler( 'subnetName' )
   @Tac.handler( 'ranges' )
   @Tac.handler( 'dnsServers' )
   @Tac.handler( 'leaseTime' )
   def handleChange( self, *args ):
      pass

class ServiceV6Reactor( GenericDhcpServiceReactor ):
   notifierTypeName = 'DhcpServer::Config'

   def __init__( self, config, service ):
      GenericDhcpServiceReactor.__init__( self, config, service )
      self.subnetConfigReactor_ = Tac.collectionChangeReactor(
         self.config_.subnetConfigIpv6, SubnetV6ConfigReactor,
         reactorArgs=( self.service_, ) )

   @Tac.handler( 'clearIdIpv6' )
   def handleClearConfig( self ):
      self.service_.handleClearLease( self.config_.clearIdIpv6 )

class VrfManagerReactor( Tac.Notifiee ):
   notifierTypeName = 'DhcpServer::Config'

   def __init__( self, config, vrfManager ):
      self.config_ = config
      self.vrfManager_ = vrfManager
      Tac.Notifiee.__init__( self, self.config_ )

   @Tac.handler( 'disabled' )
   def handleDisable( self ):
      self.vrfManager_.handleDisable( self.config_.disabled )

   @Tac.handler( 'interfacesIpv4' )
   @Tac.handler( 'interfacesIpv6' )
   def handleChange( self, *args ):
      self.vrfManager_.handleIntfChange()

   @Tac.handler( 'debugLogPath' )
   def handleDebugLog( self ):
      self.vrfManager_.handleDebugLog( self.config_.debugLogPath )

class ManagerReactor( Tac.Notifiee ):
   notifierTypeName = 'DhcpServer::Config'

   def __init__( self, config, vrf, manager ):
      self.config_ = config
      self.vrf_ = vrf
      self.manager_ = manager
      Tac.Notifiee.__init__( self, self.config_ )
      self.handleDhcpServerMode()

   @Tac.handler( 'dhcpServerMode' )
   def handleDhcpServerMode( self ):
      mode = self.config_.dhcpServerMode
      self.manager_.handleDhcpServerMode( mode, self.vrf_ )

class KniIntfReactor( Tac.Notifiee ):
   notifierTypeName = "KernelNetInfo::Root"

   def __init__( self, kniRoot, vrfManager ):
      self.kniRoot_ = kniRoot
      self.vrfManager_ = vrfManager
      Tac.Notifiee.__init__( self, self.kniRoot_ )

   @Tac.handler( 'interfaceByDeviceName' )
   def handleIntfDevName( self, key ):
      self.vrfManager_.handleIntfChange()

class KniReactor( Tac.Notifiee ):
   notifierTypeName = "KernelNetInfo::Status"

   def __init__( self, kniStatus, vrfManager ):
      self.kniStatus_ = kniStatus
      self.vrfManager_ = vrfManager
      Tac.Notifiee.__init__( self, self.kniStatus_ )

      self.kniIntfInfo = Tac.newInstance( 'KernelNetInfo::InterfaceInfo', kniStatus )
      self.vrfManager_.interfaceToKni_ = self.kniIntfInfo.root.interfaceByDeviceName
      self.kniIntfReactor_ = KniIntfReactor( self.kniIntfInfo.root, vrfManager )

      for a in self.kniStatus_.addr.keys():
         self.handleAddr( a, doIntfChange=False )

      self.vrfManager_.handleIntfChange()

   @Tac.handler( 'addr' )
   def handleAddr( self, key, doIntfChange=True ):
      # No other super server plugin looks to see if the addrs are active in this kni
      # ip status objects. So we are going to do that too.
      ipState = self.kniStatus_.addr.get( key )
      if not ipState:
         if key.ifKey in self.vrfManager_.intferfaceToAddrs_:
            self.vrfManager_.intferfaceToAddrs_[ key.ifKey ].discard( key )
            if not self.vrfManager_.intferfaceToAddrs_[ key.ifKey ]:
               del self.vrfManager_.intferfaceToAddrs_[ key.ifKey ]
      else:
         addrs = self.vrfManager_.intferfaceToAddrs_.get( key.ifKey )
         if not addrs:
            addrs = set()
            self.vrfManager_.intferfaceToAddrs_[ key.ifKey ] = addrs
         addrs.add( key )

      if doIntfChange:
         self.vrfManager_.handleIntfChange()

class IntfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = "Interface::IntfStatusLocal"

   def __init__( self, intfStatusLocal, manager ):
      self.manager_ = manager
      self.intfStatusLocal_ = intfStatusLocal
      Tac.Notifiee.__init__( self, intfStatusLocal )
      self.handleNetNsName()

   @Tac.handler( 'netNsName' )
   def handleNetNsName( self ):
      self.manager_.handleIntfChange()

class DhcpRelayHelperConfigReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::Helper::DhcpRelay::Config"

   def __init__( self, dhcpRelayHelperConfig, service ):
      self.service_ = service
      self.dhcpRelayHelperConfig_ = dhcpRelayHelperConfig
      Tac.Notifiee.__init__( self, dhcpRelayHelperConfig )
      self.handleIntfConfig( None )

   @Tac.handler( 'intfConfig' )
   def handleIntfConfig( self, _ ):
      self.service_.handleIntfChange()

   @Tac.handler( 'alwaysOn' )
   def handleAlwaysOn( self ):
      self.service_.handleIntfChange()
