# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Plugins, Tac, Tracing

from AclLib import _addRule, genUniqueId, IPPROTO_OSPF, IPPROTO_VRRP, IPPROTO_AHP, \
                   initializeDefaultCpAclName
from eunuchs.in_h import * # pylint: disable-msg=W0401

t0 = Tracing.trace0

# populate the default config
def _createDefaultConfig( cfg, paramCfg ):
   t0( "Creating %s" % paramCfg.cpAclNameDefault )
   aclConfig = cfg.config[ 'ip' ].newAcl( paramCfg.cpAclNameDefault,
         cfg.config[ 'ip' ].type, False, True )
   version = genUniqueId( )
   aclSubConfig = aclConfig.newSubConfig( version ) 
   aclSubConfig.readonly = True
   aclSubConfig.countersEnabled = True
   # 10 permit icmp any any
   _addRule( aclSubConfig, 10, 'permit', IPPROTO_ICMP )
   # 20 permit ip any any tracked
   _addRule( aclSubConfig, 20, 'permit', IPPROTO_IP, tracked=True )   
   # 30 permit udp any any eq bfd ttl eq 255
   _addRule( aclSubConfig, 30, 'permit', IPPROTO_UDP, 
         dport=[ 'bfd' ], ttl=255 )
   # 40 permit udp any any eq bfd-echo ttl eq 254
   _addRule( aclSubConfig, 40, 'permit', IPPROTO_UDP, 
         dport=[ 'bfd-echo' ], ttl=254 )
   # 50 permit udp any any eq multihop-bfd
   _addRule( aclSubConfig, 50, 'permit', IPPROTO_UDP, 
         dport=[ 'multihop-bfd' ] )   
   # 60 permit udp any any eq micro-bfd
   _addRule( aclSubConfig, 60, 'permit', IPPROTO_UDP, 
         dport=[ 'micro-bfd' ] )   
   # 70 permit udp any any dport eq sbfd
   _addRule( aclSubConfig, 70, 'permit', IPPROTO_UDP,
         dport=[ 'sbfd' ] )
   # 80 permit udp any any sport eq sbfd, dport eq sbfd-initiator
   _addRule( aclSubConfig, 80, 'permit', IPPROTO_UDP,
         sport=[ 'sbfd' ], dport=[ 'sbfd-initiator' ] )
   # 90 permit ospf any any
   _addRule( aclSubConfig, 90, 'permit', IPPROTO_OSPF )
   # 100 permit tcp any any eq ssh telnet www snmp bgp https msdp ldp netconf-ssh
   # gnmi
   _addRule( aclSubConfig, 100, 'permit', IPPROTO_TCP,
             dport=[ 'ssh', 'telnet', 'www', 'snmp', 'bgp', 
                     'https', 'msdp', 'ldp', 'netconf-ssh', 'gnmi' ] )
   # 110 permit udp any any eq bootps bootpc snmp rip ntp
   _addRule( aclSubConfig, 110, 'permit', IPPROTO_UDP,
             dport=[ 'bootps', 'bootpc', 'snmp', 'rip', 'ntp', 'ldp' ] )
   # for security reasons only allow TTL=255 for mlag
   # 120 permit tcp any any eq mlag ttl eq 255
   _addRule( aclSubConfig, 120, 'permit', IPPROTO_TCP,
             dport=[ 'mlag' ], ttl=255 )
   # 130 permit udp any any eq mlag ttl eq 255
   _addRule( aclSubConfig, 130, 'permit', IPPROTO_UDP,
             dport=[ 'mlag' ], ttl=255 )
   # 140 permit vrrp any any
   _addRule( aclSubConfig, 140, 'permit', IPPROTO_VRRP )
   # 150 permit ahp any any
   _addRule( aclSubConfig, 150, 'permit', IPPROTO_AHP )
   # 160 permit pim any any
   _addRule( aclSubConfig, 160, 'permit', IPPROTO_PIM )
   # 170 permit igmp any any
   _addRule( aclSubConfig, 170, 'permit', IPPROTO_IGMP )
   # 180 permit tcp any any range 5900 5910 (VNC)
   _addRule( aclSubConfig, 180, 'permit', IPPROTO_TCP,
             dport=[ '5900', '5910' ], dportOper='range' )
   # 190 permit tcp any any range 50001 50100 (reserved for internal services)
   _addRule( aclSubConfig, 190, 'permit', IPPROTO_TCP,
             dport=[ '%d' % Tac.Value("Arnet::PrivateTcpPorts").tcpMinPort,
                     '%d' % Tac.Value("Arnet::PrivateTcpPorts").tcpMaxPort ],
             dportOper='range' )
   # 200 permit udp any any range 51001 51100 (reserved for internal services)
   _addRule( aclSubConfig, 200, 'permit', IPPROTO_UDP,
             dport=[ '%d' % Tac.Value("Arnet::PrivateUdpPorts").udpMinPort,
                     '%d' % Tac.Value("Arnet::PrivateUdpPorts").udpMaxPort ],
             dportOper='range' )
   # 210 permit tcp any any eq 3333 ( Local License Server )
   _addRule( aclSubConfig, 210, 'permit', IPPROTO_TCP, dport=[ '3333' ] )

   # for security reasons only allow TTL=255 for nat
   # 220 permit tcp any any eq nat ttl eq 255
   _addRule( aclSubConfig, 220, 'permit', IPPROTO_TCP,
             dport=[ 'nat' ], ttl=255 )
   # 230 permit tcp any any eq bgp
   _addRule( aclSubConfig, 230, 'permit', IPPROTO_TCP,
             sport=['bgp'] )
   # 240 permit rsvp any any
   _addRule( aclSubConfig, 240, 'permit', IPPROTO_RSVP )
   # 250 permit tcp any any eq 6040, for gRIBI service
   _addRule( aclSubConfig, 250, 'permit', IPPROTO_TCP, dport=[ '6040' ] )
   # 260 permit tcp any any eq 5541 ttl eq 255, for IgmpSnooping mlag agent to agent
   # sync
   # for security reasons only allow TTL=255
   _addRule( aclSubConfig, 260, 'permit', IPPROTO_TCP,
             dport=[ 5541 ], ttl=255 )
   # 270 permit tcp any any eq 5542 ttl eq 255, for MldSnooping mlag agent to agent
   # sync
   # for security reasons only allow TTL=255
   _addRule( aclSubConfig, 270, 'permit', IPPROTO_TCP,
             dport=[ 5542 ], ttl=255 )


   aclConfig.currCfg = aclSubConfig

# populate the default config
def _createIp6DefaultConfig( cfg, paramCfg ):
   t0( "Creating %s for IPv6" % paramCfg.cpAclNameDefault )
   aclConfig = cfg.config[ 'ipv6' ].newAcl( paramCfg.cpAclNameDefault,
         cfg.config[ 'ipv6' ].type, False, True )
   version = genUniqueId( )
   aclSubConfig = aclConfig.newSubConfig( version ) 
   aclSubConfig.readonly = True
   aclSubConfig.countersEnabled = True
   # 10 permit icmpv6 any any
   _addRule( aclSubConfig, 10, 'permit', IPPROTO_ICMPV6, aclType='ipv6' )
   # 20 permit ipv6 any any tracked
   _addRule( aclSubConfig, 20, 'permit', IPPROTO_IP, tracked=True, aclType='ipv6' )
   # 30 permit udp any any eq bfd ttl eq 255
   _addRule( aclSubConfig, 30, 'permit', IPPROTO_UDP,
         dport=[ 'bfd' ], ttl=255, aclType='ipv6' )
   # 40 permit udp any any eq bfd-echo ttl eq 254
   _addRule( aclSubConfig, 40, 'permit', IPPROTO_UDP,
         dport=[ 'bfd-echo' ], ttl=254, aclType='ipv6' )
   # 50 permit udp any any eq multihop-bfd
   _addRule( aclSubConfig, 50, 'permit', IPPROTO_UDP,
         dport=[ 'multihop-bfd' ], aclType='ipv6' )   
   # 60 permit udp any any eq micro-bfd
   _addRule( aclSubConfig, 60, 'permit', IPPROTO_UDP,
         dport=[ 'micro-bfd' ], aclType='ipv6' )   
   # 70 permit udp any any dport eq sbfd
   _addRule( aclSubConfig, 70, 'permit', IPPROTO_UDP,
         dport=[ 'sbfd' ], aclType='ipv6' )
   # 80 permit udp any any sport eq sbfd, dport eq sbfd-initiator
   _addRule( aclSubConfig, 80, 'permit', IPPROTO_UDP,
         sport=[ 'sbfd' ], dport=[ 'sbfd-initiator' ], aclType='ipv6' )
   # 90 permit ospf any any
   _addRule( aclSubConfig, 90, 'permit', IPPROTO_OSPF, aclType='ipv6' )
   # 100 permit ah any any
   _addRule( aclSubConfig, 100, 'permit', IPPROTO_AH, aclType='ipv6' )
   # 110 permit esp any any
   _addRule( aclSubConfig, 110, 'permit', IPPROTO_ESP, aclType='ipv6' )
   # 120 permit tcp any any eq ssh telnet www snmp bgp https msdp
   _addRule( aclSubConfig, 120, 'permit', IPPROTO_TCP,
             dport=[ 'ssh', 'telnet', 'www', 'snmp', 'bgp', 'https', 'gnmi' ],
             aclType='ipv6' )
   # 130 permit udp any any eq bootps bootpc snmp ntp
   _addRule( aclSubConfig, 130, 'permit', IPPROTO_UDP,
             dport=[ 'bootps', 'bootpc', 'snmp', 'ntp' ], aclType='ipv6' )
   # for security reasons only allow TTL=255 for mlag
   # 140 permit tcp any any eq mlag ttl eq 255
   _addRule( aclSubConfig, 140, 'permit', IPPROTO_TCP,
             dport=[ 'mlag' ], ttl=255, aclType='ipv6' )
   # 150 permit udp any any eq mlag ttl eq 255
   _addRule( aclSubConfig, 150, 'permit', IPPROTO_UDP,
             dport=[ 'mlag' ], ttl=255, aclType='ipv6' )
   # 160 permit tcp any any range 5900 5910 (VNC)
   _addRule( aclSubConfig, 160, 'permit', IPPROTO_TCP,
             dport=[ '5900', '5910' ], dportOper='range', aclType='ipv6' )
   # 170 permit tcp any any range 50001 50100 (reserved for internal services)
   _addRule( aclSubConfig, 170, 'permit', IPPROTO_TCP,
             dport=[ '%d' % Tac.Value("Arnet::PrivateTcpPorts").tcpMinPort,
                     '%d' % Tac.Value("Arnet::PrivateTcpPorts").tcpMaxPort ],
             dportOper='range', aclType='ipv6' )
   # 180 permit udp any any range 51001 51100 (reserved for internal services)
   _addRule( aclSubConfig, 180, 'permit', IPPROTO_UDP,
             dport=[ '%d' % Tac.Value("Arnet::PrivateUdpPorts").udpMinPort,
                     '%d' % Tac.Value("Arnet::PrivateUdpPorts").udpMaxPort ],
             dportOper='range', aclType='ipv6' )
   # 190 permit udp any any range 546 547 (DHCPv6)
   _addRule( aclSubConfig, 190, 'permit', IPPROTO_UDP,
             dport=[ 'dhcpv6-client', 'dhcpv6-server' ],
             aclType='ipv6' )
   
   # 200 permit tcp any any eq bgp
   _addRule( aclSubConfig, 200, 'permit', IPPROTO_TCP,
             sport=['bgp'], aclType="ipv6" )

   # for security reasons only allow TTL=255 for nat
   # 210 permit tcp any any eq nat ttl eq 255
   _addRule( aclSubConfig, 210, 'permit', IPPROTO_TCP,
             dport=[ 'nat' ], ttl=255, aclType='ipv6' )
   # 220 permit udp any any eq nat ttl eq 255
   _addRule( aclSubConfig, 220, 'permit', IPPROTO_UDP,
             dport=[ 'nat' ], ttl=255, aclType='ipv6' )
   # 230 permit rsvp any any
   _addRule( aclSubConfig, 230, 'permit', IPPROTO_RSVP, aclType='ipv6' )
   # 240 permit pim any any
   _addRule( aclSubConfig, 240, 'permit', IPPROTO_PIM, aclType='ipv6' )
   # 250 permit tcp any any eq 6040, for gRIBI service
   _addRule( aclSubConfig, 250, 'permit', IPPROTO_TCP, dport=[ '6040' ],
             aclType='ipv6' )

   aclConfig.currCfg = aclSubConfig

@Plugins.plugin( requires=( 'acl/config/cli',
                            'acl/paramconfig' ) )
def Plugin( entMan ):
   cfg = entMan.lookup( 'acl/config/cli' )
   paramCfg = entMan.lookup( 'acl/paramconfig' ) 
   
   _createDefaultConfig( cfg, paramCfg )
   _createIp6DefaultConfig( cfg, paramCfg )

   initializeDefaultCpAclName( entMan, cfg, paramCfg ) 
