#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import re

import CliCommand
import CliMatcher
import CliParserCommon

class TimeMatcher( CliMatcher.Matcher ):
   # match timestamp in the format hh:mm:ss.
   # The values for hour, min, seconds must be validated by library user
   # This is done to make the matcher generic enough to handle any aaa:bb:cc:dd
   # format of strings, May be with slight modification it can be made to accept
   # MAC address( use [0-9a-f] instead of \d )
   __slots__ = ( 'inputRange_', 'completion_', 'matchRe_', 'completionRe_' )

   def __init__( self, name="hh:mm:ss", numPattern="hh:mm:ss", inputRange=None,
                 helpdesc="Time in hh:mm:ss format", **kargs ):
      # Assuming numPattern is non empty Ex hhh:mm hh:mm mmm yyyy and so on
      # Goal is to get following two regex for hh:mm:ss generically
      # '(\d?\d):(\d?\d):(\d?\d)$' match regex
      # '((\d?\d)(:((\d?\d)(:(\d?\d)?)?)?)?)$' completion regex
      #
      # inputRange: a list of intervals, ex ( ( 0, 24 ), ( 0, 30 ) ). This
      #              will ensure that the first pattern has a value [ 0, 24 )
      #              and the second pattern [ 0, 30 ).
      assert numPattern

      if helpdesc is None:
         helpdesc = "Time in %s format" % numPattern

      if inputRange:
         assert len( inputRange ) == len( numPattern.split( ":" ) )

      self.inputRange_ = inputRange
      self.completion_ = CliParserCommon.Completion( name, helpdesc, False )

      reList = []
      for term in numPattern.split( ":" ):
         curRe = "(" + ( r"\d?" * ( len( term ) - 1 ) ) + r"\d" + ")"  # (\d?\d)q
         reList.append( curRe )
      reStr = ":".join( reList ) + "$"
      self.matchRe_ = re.compile( reStr )

      reList.reverse()
      reStr = ""
      for curRe in reList:
         reStr = "(:(" + curRe + reStr + ")?)?"
      reStr = reStr[ 2 : -3 ] # Remove extra (: prefix and its counter ?)? suffix
      reStr += "$"  # neccessary for proper completion
      self.completionRe_ = re.compile( reStr )
      CliMatcher.Matcher.__init__( self, helpdesc=helpdesc, **kargs )

   def match( self, mode, context, token ):
      m = self.matchRe_.match( token )
      if not m:
         return CliMatcher.noMatch
      if self.inputRange_:
         values = [ int( v ) for v in m.groups() ]
         for val, inputRange in zip( values, self.inputRange_ ):
            if val < inputRange[ 0 ] or val >= inputRange[ 1 ]:
               return CliMatcher.noMatch
      return CliMatcher.MatchResult( tuple( map( int, m.groups() ) ),
                                     token )

   def completions( self, mode, context, token ):
      if not token:
         return [ self.completion_ ]
      m = self.completionRe_.match( token )
      if not m:
         return []
      if self.inputRange_:
         groupNum = 2
         for min_val, max_val in self.inputRange_:
            value = m.group( groupNum )
            if value:
               if int( value ) < min_val or int( value ) >= max_val:
                  return []
            else:
               return [ self.completion_ ]
            groupNum += 3
      return [ self.completion_ ]

# TimeMatcher ends

# A rule to match timestamp in the format hh:mm:ss with validation
validTime = ( ( 0, 24 ), ( 0, 60 ), ( 0, 60 ) )

class ValidTimeMatcher( TimeMatcher ):
   def __init__( self, **kargs ):
      TimeMatcher.__init__( self, inputRange=validTime, **kargs )

# Other rules like that needed to support hh:mm, mm, yyyy can be created easily
# hhmmMatcher = DateTimeRule.TimeMatcher( name="hh:mm", numPattern="hh:mm",
#                                         helpdesc="Time in hh:mm")

# Matches any month of the year( case-insensitive ). We use this instead of a
# simple 'Jan|Feb|...' OrRule of KeywordRule for special completion handling.
# In particular, we want to show only a single completion option when the user
# hasn't typed anything; however once he starts entering a month name, we want
# to show the exact completion options for that prefix.
monthMap = { 'January': 1, 'February': 2, 'March': 3,
             'April': 4, 'May': 5, 'June': 6,
             'July': 7, 'August': 8, 'September': 9,
             'October': 10, 'November': 11, 'December': 12 }

monthMatcher = CliMatcher.DynamicKeywordMatcher(
   # helpdesc doesn't matter when using emptyTokenCompletion
   # so we can return monthMap directly
   lambda mode: monthMap,
   emptyTokenCompletion=[ CliParserCommon.Completion(
      "MONTH", "Month of the year (Jan, Feb, etc.)",
      literal=False ) ],
   value=lambda mode, match: monthMap[ match ] )

"""A rule to match the day range <1..31>. It is the responsibility of the
caller to validate the day value for sanity. """

def dateValue( match, international=False ):
   # returns ( month, day, year )
   g1, g2, g3 = match.groups()
   if international:
      return ( int( g2 ), int( g3 ), int( g1 ) )
   else:
      return ( int( g1 ), int( g2 ), int( g3 ) )

dayMatcher = CliMatcher.IntegerMatcher( 1, 31, helpdesc='Day of the month' )

# An expression that accepts date in mm/dd/yyyy or yyyy-mm-dd format
# returns value in ( mon, day, year ) tuple.
def dateExpression( name, helpdesc="Date" ):
   # make sure the name is unique so we can use DateExpression multiple times
   usDateKey = "US_DATE_" + name
   intDateKey = "INT_DATE_" + name

   usDateMatcher = CliMatcher.PatternMatcher(
      r'(\d{1,2})/(\d{1,2})/(\d{4})$', helpname='mm/dd/yyyy', helpdesc=helpdesc,
      rawResult=True )
   intDateMatcher = CliMatcher.PatternMatcher(
      r'(\d{4})-(\d{1,2})-(\d{1,2})$', helpname='yyyy-mm-dd', helpdesc=helpdesc,
      rawResult=True )

   class DateExpression( CliCommand.CliExpression ):
      expression = usDateKey + " | " + intDateKey
      data = { usDateKey: usDateMatcher,
               intDateKey: intDateMatcher }

      @staticmethod
      def adapter( mode, args, argsList ):
         usDateVal = args.get( usDateKey )
         intDateVal = args.get( intDateKey )
         match = usDateVal or intDateVal
         if match:
            # deal with IterationRule
            if isinstance( match, list ):
               match = match[ 0 ]
            vals = dateValue( match, intDateVal is not None )
            args.pop( usDateKey, None )
            args.pop( intDateKey, None )
            args[ name ] = vals

   return DateExpression
