#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import weakref
import Tracing

class AgentConfigReactor( Tac.Notifiee ):
   notifierTypeName = 'GenericAgent::Config'

   def __init__( self, genericAgentConfig, runnabilitySm ):
      self.name_ = genericAgentConfig.name
      self.runnabilitySm_ = runnabilitySm
      Tac.Notifiee.__init__( self, genericAgentConfig )
      Tracing.trace0("AgentConfigReactor __init__", self.name_)
      self.handleEnabled()

   @Tac.handler( 'enabled' )
   def handleEnabled( self ):
      Tracing.trace0("AgentConfigReactor handleEnabled", self.name_)
      self.runnabilitySm_.checkRunnability( self.name_ )

   def cleanup( self ):
      Tracing.trace0("AgentConfigReactor cleanup", self.name_)
      self.runnabilitySm_.checkRunnability( self.name_ )

class AgentStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'GenericAgent::Status'

   def __init__( self, genericAgentConfig, runnabilitySm ):
      self.name_ = genericAgentConfig.name
      self.runnabilitySm_ = runnabilitySm
      Tac.Notifiee.__init__( self, genericAgentConfig )
      Tracing.trace0("AgentStatusReactor __init__", self.name_)
      self.handleRunning()

   @Tac.handler( 'running' )
   def handleRunning( self ):
      Tracing.trace0("AgentStatusReactor handleEnabled", self.name_)
      self.runnabilitySm_.checkRunnability( self.name_ )

   def cleanup( self ):
      Tracing.trace0("AgentStatusReactor cleanup", self.name_)
      self.runnabilitySm_.checkRunnability( self.name_ )

class AgentConfigDirReactor( Tac.Notifiee ):
   notifierTypeName = 'Tac::Dir'

   def __init__( self, agentConfigDir, runnabilitySm ):
      self.runnabilitySm_ = runnabilitySm
      self.agentConfigReactor_ = {}
      Tac.Notifiee.__init__( self, agentConfigDir )
      Tracing.trace0("AgentConfigDirReactor __init__")
      for key in agentConfigDir.keys():
         self.handleEntityPtr( key )

   @Tac.handler( 'entityPtr' )
   def handleEntityPtr( self, key ):
      agentConfig = self.notifier_.get( key )
      if ( agentConfig and
           key in self.notifier_.entryState ):
         Tracing.trace0("AgentConfigDirReactor handleEntityPtr: add", key)
         self.agentConfigReactor_[ key ] = AgentConfigReactor( agentConfig,
                                                               self.runnabilitySm_ )
      else:
         Tracing.trace0("AgentConfigDirReactor handleEntityPtr: del", key)
         reactor = self.agentConfigReactor_.get( key )
         if reactor:
            reactor.cleanup()
            del self.agentConfigReactor_[ key ]

   def cleanup( self ):
      Tracing.trace0("AgentConfigDirReactor cleanup")
      for reactor in self.agentConfigReactor_:
         reactor.cleanup()
      self.agentConfigReactor_ = {}

class AgentStatusDirReactor( Tac.Notifiee ):
   notifierTypeName = 'Tac::Dir'

   def __init__( self, agentStatusDir, runnabilitySm ):
      self.runnabilitySm_ = runnabilitySm
      self.agentStatusReactor_ = {}
      Tac.Notifiee.__init__( self, agentStatusDir )
      Tracing.trace0("AgentStatusDirReactor __init__")
      for key in agentStatusDir.keys():
         self.handleEntityPtr( key )

   @Tac.handler( 'entityPtr' )
   def handleEntityPtr( self, key ):
      agentStatus = self.notifier_.get( key )
      if ( agentStatus and
           self.notifier_.entryState.has_key( key ) ):
         Tracing.trace0("AgentStatusDirReactor handleEntityPtr: add", key)
         self.agentStatusReactor_[ key ] = AgentStatusReactor( agentStatus,
                                                               self.runnabilitySm_ )
      else:
         Tracing.trace0("AgentStatusDirReactor handleEntityPtr: del", key)
         reactor = self.agentStatusReactor_.get( key )
         if reactor:
            reactor.cleanup()
            del self.agentStatusReactor_[ key ]

   def cleanup( self ):
      Tracing.trace0("AgentStatusDirReactor cleanup")
      for reactor in self.agentStatusReactor_:
         reactor.cleanup()
      self.agentStatusReactor_ = {}

class RunnabilitySm( object ):

   def __init__( self, agentConfigDir, agentStatusDir, runnabilityDir ):
      self.agentConfigDir_ = agentConfigDir
      self.agentStatusDir_ = agentStatusDir
      self.runnabilityDir_ = runnabilityDir
      Tracing.trace0("RunnabilitySm __init__")

      # On startup need to check that nothing needs to be deleted from
      # the runnabilityDir. This can only really happen in an SSO
      # switchover.
      for name in self.runnabilityDir_.keys():
         self.checkRunnability( name )

      self.agentConfigDirReactor_ = AgentConfigDirReactor( agentConfigDir,
                                                           weakref.proxy( self ) )
      self.agentStatusDirReactor_ = AgentStatusDirReactor( agentStatusDir,
                                                           weakref.proxy( self ) )

   def checkRunnability( self, name ):
      agentConfig = self.agentConfigDir_.get( name )
      agentStatus = self.agentStatusDir_.get( name )

      # We use getattr here because it's possible that agentConfig/agentStatus
      # is a Tac::Dir instead of the entity we expect, because we're in the
      # middle of de-registering an agent and we just deleted the entry that
      # was masking the underlying Tac::Dir of the mount point.
      if ( ( agentConfig and getattr( agentConfig, "enabled", None ) ) or
           ( agentStatus and getattr( agentStatus, "running", None ) ) ):
         Tracing.trace0("RunnabilitySm checkRunnability add", name)
         self.runnabilityDir_.newEntity( "Tac::Dir", name )
      else:
         Tracing.trace0("RunnabilitySm checkRunnability del", name)
         self.runnabilityDir_.deleteEntity( name )
