#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import sys

#
# The format of Core Annotation data is as follows:
#
# <core file as produced by core_dumper script>
# <json section containing swi_version information,
#  a list of packages removed from the swi image as booted,
#  and a list of packages added to the image.>
# <magic string + 19 bytes of length info terminated by \n>
#
# Do not change magic string contents or format to allow us
# to read old core annoation data.
#
# This means the last line of the file contain the magic
# string plus the length info for a overall length of 36 bytes
# including the terminating \n.  The length info is always in the
# last 20 bytes of the file.
#
#
magicString = "Annotated core: "
lengthFmt = "%19d"
lengthLen = len( lengthFmt % 1 + "\n" )
recordLen = len( magicString ) + lengthLen


def pkgDiffsToJSON():
   """ Take diff input from stdin and convert to json lists.
   Used in core_annotate_util script."""
   rm = []
   add = []
   output = '{ "AnnotateVersion" : 1, \n'

   for line in sys.stdin:
      line = line.strip()
      if not line:
         continue
      if line[ 0 ] == "<":
         # wrap in double quotes since that is what json wants
         rm.append( "\"" + line[ 1: ].strip() + "\"" )
      elif line[ 0 ] == ">":
         add.append( "\"" + line[ 1: ].strip() + "\"" )
      elif "=" in line: # line of form a=b
         fields = line.partition( "=" )
         output += ' "%s" : "%s", \n' % ( fields[ 0 ], fields[ 2 ] )

   output += "\"RemovedPackages\":[%s],\"AddedPackages\":[%s]}\n" % (
      ",".join( rm ), ",".join( add ) )
   output += ( magicString + lengthFmt ) % ( len( output ) + recordLen )
   print( output )

def readAnnotationData( filePath ):
   """Given a uncompressed core file path, return the core annotation data as
   a python dictionary. Raises ValueError if no annotation data is
   present."""
   import json # avoid doing this when generating core above
   import os


   inputFd = os.open( filePath, os.O_RDONLY )
   try:
      os.lseek( inputFd, -recordLen, os.SEEK_END )
      lengthRecord = os.read( inputFd, recordLen ) # grab appended length record
      if not lengthRecord.startswith( magicString ):
         raise ValueError( "Core file %s does not appear to contain annotation" %
                           filePath )
      length = int( lengthRecord[ lengthLen : ] ) # get the length of annotation data
      os.lseek( inputFd, -length, os.SEEK_END ) # rewind file ptr to annotation start
      annotationStrData = os.read( inputFd, length - recordLen ) # read json part
      returnValue = json.loads( annotationStrData )
   finally:
      os.close( inputFd )
   return returnValue

