# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import Tracing
import QuickTrace
import Tac
import Agent
import uuid
import ControllerdbEntityManager
from GenericReactor import GenericReactor
from ControllerRegistry import ControllerRegistry
from LeaderReactor import LeaderReactor

__defaultTraceHandle__ = Tracing.Handle( 'Controllerdb' )
t0 = Tracing.trace0
t3 = Tracing.trace3
t6 = Tracing.trace6
t8 = Tracing.trace8

Constants = Tac.Type( "Controller::Constants" )

qv = QuickTrace.Var
qt5 = QuickTrace.trace8
 
class ControllerdbAgent( Agent.Agent ):
   def __init__( self, entityMgr, plugins=None, pluginPath=None ):
      t8( "ControllerdbAgent: __init__" )
      self.plugins_ = plugins
      self.pluginPath_ = pluginPath
      self.controllerdbStatus = None
      self.configDir = None
      self.statusDir = None
      self.serviceConfig = None
      self.ipStatus = None
      self.controllerdbPublishStatus = None
      self.serviceMountConfigDir = None
      self.serviceMountStatusDir = None
      self.servicePublishConfigDir = None
      self.servicePublishStatusDir = None
      self.serviceProxyMountConfigDir = None
      self.serviceProxyMountStatusDir = None
      self.controllerRegistry_ = None
      self.configReactor_ = None
      self.controllerdbEntityManager_ = None
      self.servicePluginCtx_ = None
      self.leaderReactor_ = None
      Agent.Agent.__init__( self, entityMgr, agentName="Controllerdb" )
      qtfile = "%s%s.qt" % ( self.agentName, "-%d" if "QUICKTRACEDIR"
                             not in os.environ else "" )
      # QuickTrace.initialize( qtfile, "8,1024,8,8,8,1024,8,8,1024,8", 
      #                        maxStringLen=80 )
      # Averaged out, as per BUG91918.  Hurry up and fix this.
      QuickTrace.initialize( qtfile, "32,32,32,32,32,32,32,32,32,32",
                             maxStringLen=80 )

   def _clusterConfig( self ):
      return self.configDir.config[ Constants.clusterDefaultName ]

   def _clusterStatus( self ):
      return self.statusDir.status[ Constants.clusterDefaultName ]

   def _oobConfig( self ):
      return self._clusterConfig().oobConfig

   def _oobStatus( self ):
      return self._clusterStatus().oobStatus

   def doInit( self, entityMgr ):
      qt5( "ControllerdbAgent: doInit" )
      mg = entityMgr.mountGroup()
      self.controllerdbStatus = mg.mount( "controller/status",
                                          "Controllerdb::Status", "w" )
      self.controllerdbPublishStatus = mg.mount( "controller/switchpublish/status",
                                                 "Controllerdb::PublishStatus", "w" )
      self.configDir = mg.mount( "controller/cluster/configDir",
                                 "ControllerCluster::ClusterConfigDir", "r" )
      self.statusDir = mg.mount( "controller/cluster/statusDir",
                                 "ControllerCluster::ClusterStatusDir", "r" )
      self.serviceConfig = mg.mount( "controller/service/config", 
                                     "Controller::ServiceConfigDir", "r" )
      Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, False )
      self.ipStatus = mg.mount( "ip/status", "Ip::Status", "r" )
      self.serviceMountConfigDir = mg.mount( "controller/mount/config/service", 
                                             "Tac::Dir", "ri" )
      self.serviceMountStatusDir = mg.mount( "controller/mount/status/service", 
                                             "Tac::Dir", "wi" )
      self.servicePublishConfigDir = mg.mount( "controller/publish/config/service", 
                                               "Tac::Dir", "ri" )
      self.servicePublishStatusDir = mg.mount( "controller/publish/status/service", 
                                               "Tac::Dir", "wi" )
      self.serviceProxyMountConfigDir = mg.mount(
                                           "controller/mount/config/proxy/service",
                                           "Tac::Dir", "ri" )
      self.serviceProxyMountStatusDir = mg.mount(
                                           "controller/mount/status/proxy/service",
                                           "Tac::Dir", "wi" )


      def _finish():
         # Change uuid. WARNING, this has to be the first thing Controllerdb
         # does on startup, otherwise, we run the risk of propagating changes
         # to clients ahead of the uuid change; which will break graceful reboot.
         self.controllerdbStatus.controllerUuid = str( uuid.uuid1() )

         # Clear any stale leaderGenId. This too has to be done before
         # Service Agents start mounting Sysdb so that they don't end up
         # looking at a stale leaderGenId.
         self.controllerdbStatus.leaderGenId = ""

         # The publish/mount dirs survive the Controllerdb restart. Clean up the 
         # old publish/mount dirs so that Controllerdb does not publish/mount the
         # stale config. The Service agents when ready, will populate the config 
         self.cleanupOldSysdbState()

         t3( "Starting ControllerdbEntityManager.Local" )
         self.controllerdbEntityManager_ = ControllerdbEntityManager.Local(
                                             entityMgr.sysname() )

         def _onFlushComplete( timedOut=False ):
            assert not timedOut, "EntityLog flush timed out"
            t6( "Instantiating the core state-machines" )
            self.configReactor_ = GenericReactor( self._clusterConfig(),
                                       [ 'enabled' ], self.handleEnabled,
                                       callBackNow=True )

            self.controllerRegistry_ = ControllerRegistry(
                                          self.controllerdbEntityManager_,
                                          self.controllerdbStatus,
                                          self.controllerdbPublishStatus,
                                          self._clusterConfig(),
                                          self._clusterStatus(),
                                          self.serviceConfig,
                                          self.serviceMountConfigDir,
                                          self.serviceMountStatusDir,
                                          self.servicePublishConfigDir,
                                          self.servicePublishStatusDir,
                                          self.serviceProxyMountConfigDir,
                                          self.serviceProxyMountStatusDir )
            self.leaderReactor_ = LeaderReactor(
                  self.controllerdbStatus,
                  self.controllerdbEntityManager_.root(),
                  self.servicePublishConfigDir, self._clusterStatus() )

         # Flush entitylog before listening to make sure that the MountInterface 
         # delete/create has propagated to Sysdb before service agents mount it.
         self.flushEntityLog( _onFlushComplete )

      mg.close( _finish )

   def cleanupOldSysdbState( self ):
      for service in self.serviceMountStatusDir:
         if not service.endswith( Constants.proxyServiceSuffix ):
            serviceMountStatus = self.serviceMountStatusDir[ service ]
            serviceMountStatus.mountedPathPrefix.clear()
            serviceMountStatus.systemStatus.clear()
            serviceMountStatus.initialMountsComplete.clear()
      for service in self.servicePublishStatusDir:
         servicePublishStatus = self.servicePublishStatusDir[ service ]
         servicePublishStatus.servicePublishGroupStatus.clear()
      # Cleaup controllerdbPublishStatus
      self.controllerdbPublishStatus.system.clear()
      self.controllerdbPublishStatus.servicePublishReady.clear()

   def handleEnabled( self, notifiee=None ):
      enabled = self._clusterConfig().enabled
      t8( "Controllerdb::handleEnabled %r" % enabled )
      qt5( "handleEnabled:", qv( enabled ) )
      self.controllerdbStatus.enabled = enabled


def name():
   ''' Call this to establish an explicit dependency on the Controllerdb
   agent executable, to be discovered by static analysis. '''
   return 'Controllerdb'

