# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model
from CliModel import Bool
from CliModel import Int
from CliModel import Float
from CliModel import Str
from Ark import timestampToStr
import Tac
import Arnet

def utcTimestamp( timestamp ):
   if not timestamp:
      # If timestamp is never updated, do not convert it.
      return timestamp
   return timestamp + Tac.utcNow() - Tac.now()

def ipOrHostname( cfgHost ):
   # check whether the given cfgHost is an IP address or a hostname
   try:
      ip = Arnet.IpAddress( cfgHost )
      hostname = ""
   except ValueError:
      ip = Tac.Value( "Arnet::IpAddr" )
      hostname = cfgHost
   return ip, hostname

def invalidHostIp( ip ):
   return ( ip == Arnet.IpAddr( "0.0.0.0" ) or
            ip == Arnet.IpAddr( "255.255.255.255" ) or
            ip.isMulticast )

class ServiceStatus( Model ):
   enabled = Bool( help='If the service is enabled' )
   versionNegotiationComplete = Bool( help='Status of negotiation process' )
   versionCompatible = Bool( help='Version compatibility' )
   negotiatedVersion = Int( help='Negotiated version' )
   
   def statusString( self ):
      return 'Enabled' if self.enabled else 'Disabled'
   
   def versionString( self ):
      if not self.enabled:
         return ''
      if not self.versionNegotiationComplete:
         return 'In progress'
      elif not self.versionCompatible:
         return 'Incompatible'
      else:
         return str( self.negotiatedVersion )

   # Hack to render with different indent levels.
   def display( self, indent='' ):
      print '%sStatus: %s' % ( indent, self.statusString() )
      if self.enabled:
         print '%sNegotiated version: %s' % ( indent, self.versionString() )

class HeartbeatStatus( Model ):
   lastHeartbeatSent = Float( help='Last heartbeat sent.' )
   lastHeartbeatReceived = Float( help='Last heartbeat' )

   def display( self, indent='' ):
      lastSent = timestampToStr( self.lastHeartbeatSent, now=Tac.utcNow() )
      lastReceived = timestampToStr( self.lastHeartbeatReceived, now=Tac.utcNow() )
      print '%sLast heartbeat sent: %s' % ( indent, lastSent )
      print '%sLast heartbeat received: %s' % ( indent, lastReceived )

class MountState( Model ):
   path = Str( help='Mount path.' )
   type = Str( help='Mount type.' )
   state = Str( help='Mount state.' )

   def display( self, indent='' ):
      print '%s%s: (%s) : %s' % ( indent, self.path, self.type, self.state )

