# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Aresolve
import BothTrace
import Logging
import socket
import Tac
import Tracing
import uuid
import weakref
from IpLibConsts import DEFAULT_VRF
from Arnet.NsLib import DEFAULT_NS, socketAt
from ControllerSslReactorLib import ControllerSslProfileStateReactor
from GenericReactor import GenericReactor

__defaultTraceHandle__ = Tracing.Handle( 'ControllerClientConfigMonitor' )
bv = BothTrace.Var
bt0 = BothTrace.tracef0
bt8 = BothTrace.tracef6

# pylint: disable-msg=E0602

Logging.logD( id="CVX_HOSTNAME_LOOKUP_FAIL",
              severity=Logging.logError,
              format="Resolution of CVX hostname '%s' failed",
              explanation="Switch failed to resolve the controller's "
              "hostname. Switch registration cannot proceed.",
              recommendedAction="Check the configured name or use the "
              "controller's IP address." )

Constants = Tac.Type( "Controller::Constants" )
ipAddrZero = Tac.Value( "Arnet::IpAddr" ).ipAddrZero
macAddrZero = Tac.Value( "Arnet::EthAddr" ).stringValue
SslProfileDisableReason = Tac.Type( "Controller::SslProfileStatus::DisableReason" )
ProfileState = Tac.Type( "Mgmt::Security::Ssl::ProfileState" )
OobState = Tac.Type( "Controller::OobState" ) 
VrfState = Tac.Type( "Ip::VrfState" )

def connectionShuttingDown( connectionStatus ):
   return connectionStatus.state == OobState.oobStateShuttingDown

def connectionShutdown( connectionStatus ):
   return connectionStatus.state == OobState.oobStateShutdown

class VrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = "Ip::VrfStatusLocal"
   def __init__( self, vrfStatus, configMonitor ):
      self.configMonitor_ = configMonitor
      Tac.Notifiee.__init__( self, vrfStatus )

   @Tac.handler( 'state' )
   def handleState( self ):
      bt0( "handleState", bv( self.notifier_.state ), "for",
           bv( self.notifier_.vrfName ) )
      self.configMonitor_.handleVrfStateChange( self.notifier_.vrfName )

   def close( self ):
      bt0( "handleClose for", bv( self.notifier_.vrfName ) )
      self.configMonitor_.handleVrfStateChange( self.notifier_.vrfName )
      Tac.Notifiee.close( self )

class ConfigMonitor( object ):
   """
   The ConfigMonitor has two jobs. First, it watches the client
   config for changes in the controllerConfig collection (i.e.,
   controllers being configured/deconfigured). For each
   configured controller it maintains a ControllerConfigManager,
   which will handle controller-specific configuration changes.
   The ConfigMonitor's second job is to watch for config changes
   that are not specific to a single controller (e.g., SSL
   config). When any of this top-level configuration changes, the
   ConfigMonitor cleans up all ControllerConfigManagers and
   reinstantiates them with the new configuration.
   """
   def __init__( self, config, status,
                 overrideServiceConfigDir, serviceConfigDir,
                 mgmtSecSslStatus, bridgingConfig, netConfig,
                 ipStatus, ipConfig, mlagConfig, vrfStatusLocal,
                 controllerSysdbConfig, agentRoot ):
      bt0()
      self.config_ = config
      self.status_ = status
      self.overrideServiceConfigDir_ = overrideServiceConfigDir
      self.serviceConfigDir_ = serviceConfigDir
      self.mgmtSecSslStatus_ = mgmtSecSslStatus
      self.netConfig_ = netConfig
      self.bridgingConfig_ = bridgingConfig
      self.ipStatus_ = ipStatus
      self.ipConfig_ = ipConfig
      self.mlagConfig_ = mlagConfig
      self.vrfStatusLocal_ = vrfStatusLocal
      self.controllerSysdbConfig_ = controllerSysdbConfig
      self.agentRoot_ = agentRoot
      self.sslProfileStateReactor_ = None
      self.sslProfileConfig_ = None

      # The source-interface is configured for all controllers. Keep the reactor at
      # top-level. If the source-interface or the associated IP address is changed,
      # we restart.
      self.srcIntf_ = None
      self.srcIntfIpReactor_ = None
      self.srcIntfIpInUse_ = None
      self.netNsInUse_ = None

      # We should be starting with empty collections in Sysdb.
      self.status_.established.peer.clear()
      self.status_.controllerStatus.clear()

      # We'll create a ControllerConfigManager for each configured "server host"
      self.controllerConfigMan_ = {}

      self.bridgeMacAddrReactor_ = GenericReactor( self.bridgingConfig_,
                                                   [ "bridgeMacAddr" ],
                                                   self._handleBridgeMac )
      self.switchConfigReactor_ = GenericReactor( self.config_.switchHost,
                                                  [ 'sourceIntf', 'vrfName' ],
                                                  self._handleConfigChange )
      self.sslProfileConfigReactor_ = GenericReactor(
                                       self.config_.sslProfileConfig,
                                       [ "sslProfileName" ],
                                       self._handleConfigChange )
      self.controllerConfigReactor_ = GenericReactor( self.config_,
                                                      [ "controllerConfig" ],
                                                      self._maybeHandleController )
      self.vrfStatusLocalReactor_ = Tac.collectionChangeReactor(
                                    self.vrfStatusLocal_.vrf,
                                    VrfStatusLocalReactor,
                                    reactorArgs=( weakref.proxy( self ), ) )

      # Sit tight until we have a valid bridge MAC.
      self._handleBridgeMac()

   def _handleBridgeMac( self, notifiee=None ):
      bt0( "bridgeMac: ", bv( self.bridgingConfig_.bridgeMacAddr ) )
      if self.bridgingConfig_.bridgeMacAddr != macAddrZero:
         self.status_.systemId = Tac.Value( "Controller::SystemId",
                                            self.bridgingConfig_.bridgeMacAddr )
         self._handleConfigChange()

   def _handleSourceIntf( self, notifiee=None ):
      bt0( "sourceIntf: ", bv( self.config_.switchHost.sourceIntf ) )
      srcIntfId = self.config_.switchHost.sourceIntf 
      if srcIntfId:
         if not self.agentRoot_.sourceIntfSm:
            self.agentRoot_.sourceIntfSm = ( self.agentRoot_.sourceIntfDir,
                                             self.ipStatus_ )
         self.srcIntfIpInUse_ = None
         self.srcIntf_ = self.agentRoot_.sourceIntfSm.registerSourceIntf( srcIntfId )
         self.srcIntfIpReactor_ = GenericReactor( self.srcIntf_,
                                                  [ "ipAddr" ],
                                                  self._handleSourceIntfIpAddr )
         self._handleSourceIntfIpAddr()
      else:
         self._maybeHandleControllers()

   def _handleVrf( self, notifiee=None ):
      vrfName = self.config_.switchHost.vrfName
      bt0( "vrfName: ", bv( vrfName ) )
      assert vrfName
      if vrfName != DEFAULT_VRF:
         self.handleVrfStateChange( vrfName )
      else:
         self.netNsInUse_ = DEFAULT_NS
         self._maybeHandleControllers()
      
   def _handleSourceIntfIpAddr( self, notifiee=None ):
      bt0( "sourceIntfIp: ", bv( self.srcIntf_.ipAddr ),
           "ip in use: ", bv( self.srcIntfIpInUse_ ) )
      self.status_.srcIntfChanges += 1 # Used for testing
      if self.srcIntf_.ipAddr != ipAddrZero:
         if self.srcIntfIpInUse_:
            if self.srcIntfIpInUse_ != self.srcIntf_.ipAddr:
               # The source intf IP we've been using changed. Go back
               # to the beginning.
               self._handleConfigChange()
            else:
               # The source intf flapped with no change to its IP addr
               pass
         else:
            # We're seeing an IP on this source intf for the first
            # time. Proceed with our setup.
            self.srcIntfIpInUse_ = self.srcIntf_.ipAddr
            self._maybeHandleControllers()
      else:
         # No IP on this interface; wait for one.
         pass

   def handleVrfStateChange( self, vrfName ):
      if vrfName != self.config_.switchHost.vrfName:
         return

      vrfStatus = self.vrfStatusLocal_.vrf.get( vrfName )
      netNs = None
      if vrfStatus and vrfStatus.state == VrfState.active:
         netNs = vrfStatus.networkNamespace

      bt0( "netNs: ", bv( netNs ) )
      if self.netNsInUse_:
         if self.netNsInUse_ == netNs:
            pass
         else:
            self._handleConfigChange()
      elif netNs is not None:
         self.netNsInUse_ = netNs
         self._maybeHandleControllers()
      else:
         pass
      
   def _sslEnabled( self ):
      if self.config_.sslProfileConfig.sslProfileName:
         return True
      return False

   def _handleConfigChange( self, notifiee=None ):
      bt0( "sourceIntf: ", bv( self.config_.switchHost.sourceIntf ),
           "sslProfileName: ", bv( self.config_.sslProfileConfig.sslProfileName ),
           "vrfName: ", bv( self.config_.switchHost.vrfName ) )
      # If anything changes, we tear things down and start over.
      self._doConnectionCleanup( force=True )
      self.status_.uuid = str( uuid.uuid1() )
      if self._sslEnabled():
         self._handleSsl()
      self._handleSourceIntf()
      self._handleVrf()

   def _handleController( self, _, controllerName ):
      add = controllerName in self.config_.controllerConfig
      bt0( bv( controllerName ), "add=", bv( str( add ) ) )
      if add:
         if controllerName in self.controllerConfigMan_:
            self.controllerConfigMan_[ controllerName ].doCleanup( force=True )
            del self.controllerConfigMan_[ controllerName ]
         sourceIntfIpAddr = self.srcIntfIpInUse_ or ipAddrZero
         self.controllerConfigMan_[ controllerName ] = \
             ControllerConfigManager( self.config_, self.status_,
                                      self.sslProfileConfig_,
                                      controllerName, self.netConfig_,
                                      self.overrideServiceConfigDir_,
                                      self.serviceConfigDir_, self.agentRoot_,
                                      sourceIntfIpAddr, self.ipConfig_,
                                      self.mlagConfig_, self.netNsInUse_ )

      elif controllerName in self.controllerConfigMan_:
         self.controllerConfigMan_[ controllerName ].doCleanup( force=True )
         del self.controllerConfigMan_[ controllerName ]
         # A controller is deconfigured. Try bringing up duplicated connections.
         for controllerConfigMan in self.controllerConfigMan_.itervalues():
            controllerConfigMan.reconnect()

   def _maybeHandleControllers( self ):
      bt8()
      for controller in self.config_.controllerConfig:
         self._maybeHandleController( None, controller )

   def _maybeHandleController( self, _, controllerName ):
      bt8()
      if controllerName in self.config_.controllerConfig:
         if ( ( self.srcIntf_ and not self.srcIntfIpInUse_ ) or
              ( not self.netNsInUse_ ) or
              ( self._sslEnabled() and
                not self.status_.sslProfileStatus.enabled ) ):
            return
      self._handleController( None, controllerName )

   def _copySslConfig( self, dest, sslReactor ):
      dest.tlsVersion = sslReactor.tlsVersion()
      dest.fipsMode = sslReactor.fipsMode()
      dest.certKeyPath = sslReactor.certKeyPath()
      dest.trustedCertsPath = sslReactor.trustedCertsPath()
      dest.dhParamPath = sslReactor.dhParamPath()
      dest.crlsPath = sslReactor.crlsPath()
      dest.cipherSuite = sslReactor.cipherSuite()

   def _handleSsl( self ):
      bt8()
      self.sslProfileStateReactor_ = ControllerSslProfileStateReactor(
                                    self.mgmtSecSslStatus_,
                                    self.config_.sslProfileConfig.sslProfileName,
                                    self._handleConfigChange,
                                    self._handleConfigChange,
                                    self._handleConfigChange,
                                    callBackNow=False )

      state = self.sslProfileStateReactor_.profileState()
      errCode = None
      if not state:
         bt8( "SSL profile does not exist" )
         errCode = SslProfileDisableReason.profileNotExist
      elif state == ProfileState.valid:
         if not self.sslProfileStateReactor_.certKeyPath():
            bt8( "Certificate not configured in SSL profile" )
            errCode = SslProfileDisableReason.certNotConfigured
         elif not self.sslProfileStateReactor_.trustedCertsPath():
            bt8( "Trusted certificates not configured in SSL profile" )
            errCode = SslProfileDisableReason.trustedCertNotConfigured
         else:
            bt8( "SSL profile is valid and ready to use" )
            errCode = SslProfileDisableReason.noError
      else:
         bt0( "SSL profile is invalid" )
         errCode = SslProfileDisableReason.profileInvalid
      if not self.status_.sslProfileStatus:
         self.status_.sslProfileStatus = (
               self.config_.sslProfileConfig.sslProfileName, )
      sslEnabled = errCode == SslProfileDisableReason.noError
      
      if sslEnabled:
         self.sslProfileConfig_ = Tac.newInstance( "Controller::SslProfileConfig",
               self.config_.sslProfileConfig.sslProfileName )
         self._copySslConfig( self.sslProfileConfig_, self.sslProfileStateReactor_ )
         
         # Copy the SSL config to sslProfileStatus
         if not self.status_.sslProfileStatus.sslProfileConfig:
            self.status_.sslProfileStatus.sslProfileConfig = (
                  self.config_.sslProfileConfig.sslProfileName, )
         self._copySslConfig( self.status_.sslProfileStatus.sslProfileConfig,
                              self.sslProfileStateReactor_ )

      # Set SSL status after populating sslConfig everywhere
      self.status_.sslProfileStatus.enabled = sslEnabled
      self.status_.sslProfileStatus.disableReason = errCode

   def _doConnectionCleanup( self, force=False ):
      bt0()
      self.status_.sslProfileStatus = None
      for controllerConfigMan in self.controllerConfigMan_.itervalues():
         controllerConfigMan.doCleanup( force=force )
      if force:
         self.controllerConfigMan_.clear()
      if self.sslProfileStateReactor_:
         self.sslProfileStateReactor_.close()
      self.sslProfileStateReactor_ = None
      if self.srcIntf_:
         self.srcIntfIpReactor_ = None
         self.srcIntfIpInUse_ = None
         self.agentRoot_.sourceIntfSm.unregisterSourceIntf( self.srcIntf_ )
         self.srcIntf_ = None
      self.netNsInUse_ = None
      self.status_.uuid = ""
      self.sslProfileConfig_ = None

   def doCleanup( self, force=False ):
      bt8()
      self.bridgeMacAddrReactor_ = None
      self.vrfStatusLocalReactor_ = None
      self.switchConfigReactor_ = None
      self.sslProfileConfigReactor_ = None
      self.controllerConfigReactor_ = None

      self._doConnectionCleanup( force=force )

class ControllerConfigManager( object ):
   """
   The ControllerConfigManager monitors the connectionConfig related to one
   controllerConfig (i.e., server host configuration). It resolves hostname if
   needed, and creates a controllerStatus and a ConectionMan for each resolved
   controller IP.
   """
   def __init__( self, config, status, sslProfileConfig, controllerName,
                 netConfig, overrideServiceConfigDir, serviceConfigDir,
                 agentRoot, srcIntfIpAddr, ipConfig, mlagConfig, netNs ):
      bt0( bv( controllerName ) )
      self.config_ = config
      self.status_ = status
      self.sslProfileConfig_ = sslProfileConfig
      self.overrideServiceConfigDir_ = overrideServiceConfigDir
      self.serviceConfigDir_ = serviceConfigDir
      self.agentRoot_ = agentRoot
      self.srcIntfIpAddr_ = srcIntfIpAddr
      self.ipConfig_ = ipConfig
      self.mlagConfig_ = mlagConfig
      self.netNs_ = netNs
      self.controllerName_ = controllerName
      self.netConfig_ = netConfig
      self.controllerConfig_ = self.config_.controllerConfig[ controllerName ]
      self.connectionConfig_ = self.controllerConfig_.connectionConfig

      self.controllerHostname_ = None
      self.dnsQuerier_ = None

      self.controllerConfigReactor_ = GenericReactor(
                                          self.controllerConfig_,
                                          [ "connectionConfig" ],
                                          self.handleConnectionConfig )
      self.connectionConfigReactor_ = None
      self.resolvedIps = set()
      # One connectionManager for each controller IP address.
      self.connectionMan_ = {}
      self.cleanupReactor_ = {}

      # Get started
      self.handleConnectionConfig()

   def maybeCreateConnectionMan( self, controllerIp=None, sockName=None ):
      bt0( bv( controllerIp ), "sockname=", bv( sockName ) )
      controllerStatusName = controllerIp or sockName
      if controllerStatusName in self.status_.controllerStatus:
         connectionStatus = self.status_.controllerStatus[ controllerStatusName ].\
                            connectionStatus
         if connectionShuttingDown( connectionStatus ):
            self.cleanupConnectionMan( controllerStatusName, force=True )
         else:
            # The controllerStatus already exists. User must have configured
            # multiple host entries for the same controllerIp. Return now without
            # creating a connectionMan. We will try again later in reconnect()
            # when any host entry is deconfigured.
            bt0( "Duplicated controller IP detected. configKey=",
                 bv( self.controllerName_ ), "statusKey=",
                 bv( controllerStatusName ) )
            return

      controllerStatus = self.status_.newControllerStatus( controllerStatusName )
      controllerStatus.connectionStatus = ()
      controllerStatus.heartbeatStatus = ( "heartbeatStatus", )
      controllerStatus.serviceStatusDir = ( "serviceStatusDir", )
      connectionStatus = controllerStatus.connectionStatus
      connectionStatus.switchHost = ( "", )
      connectionStatus.controller = ( "", )
      connectionStatus.switchHost.hostname = self.netConfig_.hostname

      connectionStatus.controller.hostname = self.connectionConfig_.hostname
      if controllerIp:
         connectionStatus.controller.ip = controllerIp
      elif sockName:
         connectionStatus.controller.sockname = sockName

      self.connectionMan_[ controllerStatusName ] = ConnectionManager(
         controllerStatusName, self.connectionConfig_, self.config_,
         self.status_, self.sslProfileConfig_, self.overrideServiceConfigDir_,
         self.serviceConfigDir_, self.agentRoot_, self.srcIntfIpAddr_,
         self.ipConfig_, self.mlagConfig_, self.netNs_ )

      self.cleanupReactor_[ controllerStatusName ] = GenericReactor(
                                                     connectionStatus,
                                                     [ "state" ],
                                                     self.handleOobState )

   def handleOobState( self, notifiee=None ):
      connectionStatus = notifiee.notifier_
      controller = connectionStatus.controller
      controllerStatusName = controller.ip or controller.sockname

      bt0( bv( controllerStatusName ),
           "state=", bv( str(  connectionStatus.state ) ) )

      if connectionShutdown( connectionStatus ):
         self.cleanupConnectionMan( controllerStatusName, force=True )

   def handleConfigChange( self, notifiee=None ):
      bt0( bv( self.controllerName_ ),
           "ip=", bv( self.connectionConfig_.ip ),
           "hostname=", bv( self.connectionConfig_.hostname ),
           "sockname=", bv( self.connectionConfig_.sockname ),
           "sysname=", bv( self.connectionConfig_.sysname ) )
      self.handleConnectionConfig()

   def cleanupConnectionMan( self, ipAddr, force=False ):
      if force:
         if ipAddr in self.cleanupReactor_:
            del self.cleanupReactor_[ ipAddr ]
         if ipAddr in self.connectionMan_:
            self.connectionMan_[ ipAddr ].doCleanup( force=True )
            del self.connectionMan_[ ipAddr ]
            if ipAddr in self.status_.controllerStatus:
               del self.status_.controllerStatus[ ipAddr ]
      else:
         if ipAddr in self.connectionMan_:
            self.connectionMan_[ ipAddr ].doCleanup()

   def reconnect( self ):
      bt0( bv( self.controllerName_ ) )
      for ipAddr in self.resolvedIps:
         if ( ipAddr in self.status_.controllerStatus and
              self.status_.controllerStatus[ ipAddr ].duplicatedConnection ):
            # We have kept the connectionMan for duplicatedConnection.
            # Delete it before trying to reconnect
            self.cleanupConnectionMan( ipAddr, force=True )
         self.maybeCreateConnectionMan( controllerIp=ipAddr )

   def doCleanup( self, force=False ):
      bt0( bv( self.controllerName_ ), bv( force ) )
      for ipAddr in self.resolvedIps:
         self.cleanupConnectionMan( ipAddr, force=force )

      self.clearRunning()

   def handleConnectionConfig( self, notifiee=None ):
      if not self.controllerConfig_.connectionConfig:
         self.doCleanup()
         return

      self.connectionConfig_ = self.controllerConfig_.connectionConfig
      if not self.connectionConfigReactor_:
         self.connectionConfigReactor_ = GenericReactor(
                                          self.connectionConfig_,
                                          [ 'hostname', 'ip',
                                            'sockname', 'sysname' ],
                                          self.handleConfigChange )
      bt0( bv( self.controllerName_ ) )
      self.doCleanup( force=True )
      # The CLI code enforces that we can't have both a hostname
      # and IP for the controller.
      if self.connectionConfig_.ip != ipAddrZero:
         self._handleControllerIp( self.connectionConfig_.ip )
      elif self.connectionConfig_.hostname:
         self._handleControllerHostname()
      elif self.connectionConfig_.sockname:
         # Not configurable via CLI but used in testing
         self.maybeCreateConnectionMan( sockName=self.connectionConfig_.sockname )

   def _handleControllerIp( self, ipAddr=None ):
      if not ipAddr or ipAddr == ipAddrZero:
         return
      self.resolvedIps.add( ipAddr )
      self.maybeCreateConnectionMan( controllerIp=ipAddr )
      bt0( bv( ipAddr ), bv( self.controllerName_ ) )

   def _handleControllerHostname( self ):
      bt8( bv( self.controllerName_ ), "resolving hostname:",
           bv( self.connectionConfig_.hostname ) )
      self.controllerHostname_ = self.connectionConfig_.hostname
      if self.dnsQuerier_:
         self.dnsQuerier_.clear()
         self.dnsQuerier_ = None
      self.dnsQuerier_ = Aresolve.Querier( self.handleHostnameResolution,
                                           shortTime=2, longTime=10 )
      self.dnsQuerier_.host( self.controllerHostname_ )

   def handleHostnameResolution( self, data ):
      bt0( bv( self.controllerName_ ), "hostname=", bv( data.name ),
           "valid=", bv( str( data.valid ) ) )
      # Aresolve has completed resolution. If it was successful, handle 
      # the resolved Ip address. If not, log a message and give up.
      if data.valid:
         if self.dnsQuerier_:
            self.dnsQuerier_.finishHost( self.controllerHostname_ )
         for ipAddr in data.ipAddress:
            self._handleControllerIp( ipAddr )
      else:
         bt0( "CVX_HOSTNAME_LOOKUP_FAIL" )
         Logging.log( CVX_HOSTNAME_LOOKUP_FAIL, # pylint: disable-msg=E0602
                      data.name )

   def clearRunning( self ):
      bt8( bv( self.controllerName_ ) )
      if self.dnsQuerier_:
         self.dnsQuerier_.clear()

class ConnectionManager( object ):
   """
   ConnectionManager is created for each resolved controller IP address
   (or sockname). It gathers ip and srcIp information for the connection to
   instantiate an OobClientSm.
   """
   def __init__( self, controllerStatusName, connectionConfig, config, status,
                 sslProfileConfig, overrideServiceConfigDir, serviceConfigDir,
                 agentRoot, srcIntfIpAddr, ipConfig, mlagConfig, netNs ):
      self.controllerStatusName_ = controllerStatusName
      self.connectionConfig_ = connectionConfig
      self.config_ = config
      self.status_ = status
      self.sslProfileConfig_ = sslProfileConfig
      self.agentRoot_ = agentRoot
      self.srcIntfIpAddr_ = srcIntfIpAddr
      self.ipConfig_ = ipConfig
      self.netNs_ = netNs
      self.mlagConfig_ = mlagConfig
      self.overrideServiceConfigDir_ = overrideServiceConfigDir
      self.serviceConfigDir_ = serviceConfigDir
      controllerStatus = self.status_.controllerStatus[ controllerStatusName ]
      self.connectionStatus_ = controllerStatus.connectionStatus

      self.srcIpRetryTimer_ = Tac.ClockNotifiee( self.handleSrcIpRetryTimer,
                                                 timeMin=Tac.endOfTime )

      # With a controller and switch IP we're pretty much ready to go.
      self.resolvedSwitchIpReactor_ = GenericReactor(
                                          self.connectionStatus_.switchHost,
                                          [ 'ip' ],
                                          self.finishConnectionStatus )

      self.statusReadyReactor_ = GenericReactor( self.connectionStatus_,
                                                 [ "ready" ],
                                                 self.handleReady )

      self.handleSourceIntf()

   def handleReady( self, notifiee=None ):
      bt0( bv( self.controllerStatusName_ ),
           "ready=", bv( str(  self.connectionStatus_.ready ) ) )

      if self.connectionStatus_.ready:
         assert self.controllerStatusName_ not in self.agentRoot_.oobClientSm
         self.agentRoot_.oobClientSm.newMember(
                                         self.controllerStatusName_,
                                         self.config_, self.status_,
                                         self.overrideServiceConfigDir_, 
                                         self.serviceConfigDir_ )
      else:
         self.agentRoot_.oobClientSm[ self.controllerStatusName_ ].\
               doCleanup( False )

   def _isMlagPeerLinkIp( self, srcIp ):
      if not ( self.mlagConfig_.enabled and self.mlagConfig_.localIntfId ):
         return False
      intfConfig = self.ipConfig_.ipIntfConfig.get(
                     self.mlagConfig_.localIntfId )
      if intfConfig and srcIp == intfConfig.addrWithMask.address:
         return True
      return False

   def _srcIpFromDst( self ):
      # Ask the kernel what our source address will be for a
      # given remote endpoint.
      bt0( bv( self.controllerStatusName_ ) )
      controllerIp = self.connectionStatus_.controller.ip
      srcIp = ipAddrZero
      s = socketAt( socket.AF_INET, socket.SOCK_DGRAM, ns=self.netNs_ )
      s.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
      s.bind( ( ipAddrZero, 0 ) )
      try:
         s.connect( ( controllerIp, Constants.controllerOobPort ) )
         srcIp = s.getsockname()[ 0 ]
         if self._isMlagPeerLinkIp( srcIp ):
            bt0( "_srcIpFromDst: Got mlag peer-link's IP %s", srcIp )
            srcIp = ipAddrZero
      except socket.error:
         bt0( "_srcIpFromDst failed to connect to [%s:%d]" %
              ( controllerIp, Constants.controllerOobPort ) )
      finally:
         s.close()
         if srcIp == ipAddrZero:
            self.srcIpRetryTimer_.timeMin = Tac.now() + 5

      return srcIp

   def handleSrcIpRetryTimer( self ):
      bt0( bv( self.controllerStatusName_ ) )
      if self.connectionStatus_.switchHost.ip == ipAddrZero:
         self.connectionStatus_.switchHost.ip = self._srcIpFromDst()

   def handleSourceIntf( self ):
      # The controller.ip (or sockname) should not change in the lifetime of a
      # ConnectionManager.
      assert ( self.connectionStatus_.controller.ip != ipAddrZero or
               not self.connectionConfig_.sockname )
      bt8( bv( self.controllerStatusName_ ), 
           bv( self.config_.switchHost.sourceIntf ) )
      if self.srcIntfIpAddr_ != ipAddrZero:
         self.connectionStatus_.switchHost.ip = self.srcIntfIpAddr_
      else:
         self.connectionStatus_.switchHost.ip = self._srcIpFromDst()
      bt8( "srcIntfConfig=", self.config_.switchHost.sourceIntf,
           "switchHostIp=", self.connectionStatus_.switchHost.ip )

   def finishConnectionStatus( self, notifiee=None ):
      if self.connectionStatus_.switchHost.ip == ipAddrZero:
         return # false alarm
      if ( self.connectionStatus_.ready and
           ( self.connectionStatus_.controller.srcIp !=
             self.connectionStatus_.switchHost.ip ) ):
         bt8( "Source-interface's IP addr changed" )
         self.connectionStatus_.controller.srcIp = ipAddrZero
         self.connectionStatus_.ready = False
      config = self.connectionConfig_
      self.connectionStatus_.controller.port = Constants.controllerOobPort

      # If sysname isn't configured, we assume that Controllerdb
      # has the default sysname.
      self.connectionStatus_.controller.sysname = ( config.sysname or
                                                    self.agentRoot_.defaultSysname )
      # Switch's sysname is not configurable 
      self.connectionStatus_.switchHost.sysname = self.agentRoot_.mySysname
      self.connectionStatus_.controller.sockname = config.sockname
      self.connectionStatus_.controller.srcIp = \
            self.connectionStatus_.switchHost.ip
      self.connectionStatus_.switchHost.netNs = self.netNs_
      self.connectionStatus_.controller.netNs = self.netNs_
      if self.status_.sslProfileStatus: # SSL is enabled
         self.connectionStatus_.switchHost.sslProfileConfig = \
               self.status_.sslProfileStatus.sslProfileConfig
         self.connectionStatus_.controller.sslProfileConfig = \
               self.status_.sslProfileStatus.sslProfileConfig
      bt8( "finishConnectionStatus: ready=True " )
      self.connectionStatus_.ready = True

   def doOobClientSmCleanup( self ):
      if self.controllerStatusName_ in self.agentRoot_.oobClientSm:
         self.agentRoot_.oobClientSm[ self.controllerStatusName_ ].\
               doCleanup( True )
         del self.agentRoot_.oobClientSm[ self.controllerStatusName_ ]

   def doCleanup( self, force=False ):
      bt8( bv( self.controllerStatusName_ ) )

      if force:
         self.doOobClientSmCleanup()
         return

      self.connectionStatus_.ready = False
      self.connectionStatus_.switchHost.ip = ipAddrZero
      self.connectionStatus_.switchHost.sslProfileConfig = None
      self.connectionStatus_.switchHost.netNs = DEFAULT_NS
      self.connectionStatus_.controller.netNs = DEFAULT_NS
      if self.srcIpRetryTimer_:
         self.srcIpRetryTimer_.timeMin = Tac.endOfTime

