# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import abc
import collections
import urlparse

import ReversibleSecretCli
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'ContainerTracerLib' )
debug = Tracing.trace8

PodInfo = collections.namedtuple( 'PodInfo', 'name phase nodeName' )

class ClientError( Exception ):
   pass

class BaseClient( object ):
   __metaclass__ = abc.ABCMeta
   @abc.abstractmethod
   def getNodes( self, nodeName=None):
      """Return list of node names for a cluster.

         If nodeName is provided the results will be filtered for exact matches.
      """
      pass

   @abc.abstractmethod
   def getPods( self, podName=None, nodeName=None):
      """Return a list of PodInfo tuples.

      The optional parameters will cause the query to return only exact matches
      for the provided parameters.
      """
      pass

class K8sThinClient( BaseClient ):
   def __init__( self, url, token, timeout=None ):
      # avoids importing requests and deps until it's really needed
      import requests
      import urllib3
      urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
      self.session = requests.Session()
      self.session.verify = False
      self.session.headers[ 'Authorization' ] = 'Bearer %s' % token
      self.session.headers[ 'Accept' ] = 'application/json'
      self.timeout = timeout
      self.baseURL = url + ( '' if url[ -1 : ] == '/' else '/' )

   def _get( self, path, kind, fieldSelectors=None ):
      # avoids importing requests and deps until it's really needed
      import requests
      url = urlparse.urljoin( self.baseURL, path )
      params = {}
      if fieldSelectors:
         params[ 'fieldSelector' ] = ','.join(
            '%s=%s' % i for i in fieldSelectors.items()
         )
      try:
         resp = self.session.get( url, params=params, timeout=self.timeout )
         resp.raise_for_status()
         retval = resp.json()
         if retval[ u'apiVersion' ] == u'v1' and retval[ u'kind' ] == kind:
            return retval[ 'items' ]
      except requests.exceptions.ConnectionError:
         raise ClientError( 'Unable to connect to cluster at %s' % self.baseURL )
      except KeyError as e:
         debug( 'Response missing expected key: %s' % e )
      except Exception as e:
         debug( 'Exception during client call: %s' % e )
      raise ClientError( 'Error processing request to %s ' % self.baseURL )


   def getNodes( self, nodeName=None, ):
      path = '/api/v1/nodes'
      f = {}
      if nodeName:
         f[ 'metadata.name' ] = nodeName

      return [
         n[ 'metadata' ][ 'name' ].encode( 'utf8' )
         for n in self._get( path, 'NodeList', f )
      ]

   def getPods( self, podName=None, nodeName=None ):
      path = '/api/v1/pods'
      f = {}
      if podName:
         f[ 'metadata.name' ] = podName
      if nodeName:
         f[ 'spec.nodeName' ] = nodeName

      return [
         PodInfo(
            pi[ 'metadata' ][ 'name' ].encode( 'utf8' ),
            pi[ 'status' ][ 'phase' ].encode( 'utf8' ),
            pi.get( 'spec', {} ).get( 'nodeName', '' ).encode( 'utf8' )
         ) for pi in self._get( path, 'PodList', f )
      ]

def getClient( config ):
   return K8sThinClient(
      config.url,
      ReversibleSecretCli.decodeKey( config.authSecret ),
      timeout=config.requestTimeout
   )
