# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import base64
import collections
import yaml

import Tracing

debug = Tracing.trace8

Config = collections.namedtuple(
   'Config',
   'server caCertData tokenData clientCertData clientKeyData'
)

class ValidationError( Exception ):
   pass

def parseConfig( filename, alternateContext=None ):
   kwargs = {}
   with open( filename ) as f:
      data = yaml.load( f )

   if not isinstance( data, dict ):
      raise ValidationError( 'Config file does not appear to be valid YAML' )
   currentCtx = data.get( 'current-context' )
   matchCtx = alternateContext or currentCtx

   try:
      ctx = [ c for c in data[ 'contexts' ] if c[ 'name' ] == matchCtx ][ 0 ]
   except ( KeyError, IndexError ):
      raise ValidationError( 'Unable to find usable context (%s)' % matchCtx )

   # find data from context intfo
   try:
      name = ctx.get( 'context', {} ).get( 'cluster' )
      cluster = [
         c for c in data[ 'clusters' ] if c[ 'name' ] == name
      ][ 0 ][ 'cluster' ]
   except ( KeyError, IndexError ):
      raise ValidationError( 'Unable to find %s in clusters.' % name )

   try:
      name = ctx.get( 'context', {} ).get( 'user' )
      user = [ u for u in data[ 'users' ] if u[ 'name' ] == name ][ 0 ][ 'user' ]
   except ( KeyError, IndexError ):
      raise ValidationError( 'Unable to find %s in users' % name )

   # attempt to populate config tuple
   try:
      kwargs[ 'server' ] = cluster[ 'server' ]
   except KeyError:
      raise ValidationError( 'Unable to find server URL for context %s' % matchCtx )

   def decode( d, k ):
      return base64.b64decode( d.get( k, '' ) ).decode( 'utf8' )

   kwargs[ 'caCertData' ] = decode( cluster, 'certificate-authority-data' )
   kwargs[ 'tokenData' ] = user.get( 'token', '' )
   kwargs[ 'clientCertData' ] = decode( user, 'client-certificate-data' )
   kwargs[ 'clientKeyData' ] = decode( user, 'client-key-data' )

   config = Config( **kwargs )

   if config.tokenData or ( config.clientCertData and config.clientKeyData ):
      return config

   raise ValidationError(
      'Unable to find authentication details for %s' % user[ 'name ' ]
   )

def validate( filename ):
   try:
      parseConfig( filename )
   except ValidationError:
      raise
   except Exception as e:
      debug( 'Unhandled exception: %s' % e )
      raise ValidationError( 'File does not appear to be a valid kubeconfig' )
