# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import threading

import Syscall
import Tac
import UtmpDump

MAX_HISTORY_SIZE = 100

class ConfigLockException( Exception ):
   pass

class UnableToAcquireLockException( ConfigLockException ):
   pass

class UnableToReleaseLockException( ConfigLockException ):
   pass

class ConfigureLockHistory( object ):
   def __init__( self, user, tty, location, acquireReason ):
      self.user = user
      self.tty = tty
      self.location = location if location != 'UnknownIpAddr' else None
      self.tid = Syscall.gettid()
      self.grabTime = Tac.now()
      self.acquireReason = acquireReason
      self.releaseTime = None
      self.releaseReason = None

class ConfigureLock( object ):
   def __init__( self ):
      self.owner_ = None
      self.lock_ = threading.Lock()
      self.history_ = []
      self.currHistory_ = None

   def getCurrUserInfo( self ):
      return self.currHistory_

   def getHistory( self ):
      return self.history_

   def acquireLock( self, reason=None ):
      # grabs the lock for the current user
      info = UtmpDump.getUserInfo()
      with self.lock_:
         if self.owner_ is not None:
            raise UnableToAcquireLockException()
         self.owner_ = Syscall.gettid()
         self.currHistory_ = ConfigureLockHistory( info[ 'user' ],
               info[ 'tty' ], info[ 'ipAddr' ], reason )
         self.history_.append( self.currHistory_ )
         if len( self.history_ ) > MAX_HISTORY_SIZE:
            self.history_.pop( 0 )

   def releaseLock( self, reason=None ):
      with self.lock_:
         if self.owner_ is None:
            # if there is no owner we shouldn't do the rest of this
            return
         self.owner_ = None
         self.currHistory_.releaseTime = Tac.now()
         self.currHistory_.releaseReason = reason
         self.currHistory_ = None

   def canRunConfigureCmds( self ):
      # returns of this thread can run configure commands.
      if self.isLockOwnedByThread():
         return True
      with self.lock_:
         return self.owner_ is None

   def isLockOwnedByThread( self ):
      # Returns if the current thread has the lock grabbed.
      # NB: we don't need a lock here because even if the owner does change
      # it could never change to or away from the current thread.
      return self.owner_ == Syscall.gettid()

   def isLockedOwned( self ):
      with self.lock_:
         return self.owner_ is not None
