# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

# Common config session-related code that is used by multiple packages.

import Logging
import UtmpDump
import CliSession
import CliSessionDataStore
import CliSessionOnCommit
import CommonGuards
import FileCliUtil
import SessionUrlUtil
import CliCommon
import Tracing
import os

t0 = Tracing.t0

guardNoSession = "no active session"

def sessionGuard( mode, token ):
   sessionName = CliSession.currentSession( mode.session_.entityManager_ )
   if sessionName is None:
      return guardNoSession
   return None

def sessionSsoGuard( mode, token ):
   ssoGuardCode = CommonGuards.ssoStandbyGuard( mode, token )
   if ssoGuardCode:
      return ssoGuardCode
   else:
      return sessionGuard( mode, token )

SYS_CONFIG_REPLACE_SUCCESS = None
Logging.logD( id="SYS_CONFIG_REPLACE_SUCCESS",
              severity=Logging.logNotice,
              format="User %s replaced running configuration with %s successfully "
                     "on %s (%s)",
              explanation="A network administrator has replaced the running "
                          "configuration of the system successfully.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_REPLACE_FAILURE = None
Logging.logD( id="SYS_CONFIG_REPLACE_FAILURE",
              severity=Logging.logError,
              format="Configuration replace operation by %s with %s failed, "
                     "on %s (%s)",
              explanation="A network administrator has attempted to replace the "
                          "running configuration of the system, but the operation "
                          "failed. The running configuration of the system has not "
                          "changed.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_ENTERED = None
Logging.logD( id="SYS_CONFIG_SESSION_ENTERED",
              severity=Logging.logNotice,
              format="User %s entered configuration session %s on %s (%s)",
              explanation="A network administrator has entered a configuration "
                          "session.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_EXITED = None
Logging.logD( id="SYS_CONFIG_SESSION_EXITED",
              severity=Logging.logNotice,
              format="User %s exited configuration session %s on %s (%s)",
              explanation="A network administrator has exited a configuration "
                          "session without committing or aborting, thereby having "
                          "no effect on the running configuration fo the system.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_COMMIT_SUCCESS = None
Logging.logD( id="SYS_CONFIG_SESSION_COMMIT_SUCCESS",
              severity=Logging.logNotice,
              format="User %s committed configuration session %s successfully "
                     "on %s (%s)",
              explanation="A network administrator has commited a configuration "
                          "session, thereby modifying the running configuration of"
                          " the system.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_COMMIT_TIMER_STARTED = None
Logging.logD( id="SYS_CONFIG_SESSION_COMMIT_TIMER_STARTED",
              severity=Logging.logNotice,
              format="User %s committed session %s on %s (%s), "
                     "with timer %d:%d:%d(hr:min:sec).",
              explanation="A network administrator has committed a"
                          "configuration session with a timer, thereby"
                          " modifying the running configuration of the "
                          "system temporarily.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_COMMIT_TIMER_UPDATED = None
Logging.logD( id="SYS_CONFIG_SESSION_COMMIT_TIMER_UPDATED",
              severity=Logging.logNotice,
              format="User %s updated commit timer with %d:%d:%d"
              "(hr:min:sec) for session %s on %s (%s).",
              explanation="A network administrator has updated the commit"
                           " timer of a configuration session.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_COMMIT_TIMER_COMPLETED = None
Logging.logD( id="SYS_CONFIG_SESSION_COMMIT_TIMER_COMPLETED",
              severity=Logging.logNotice,
              format="User %s has confirmed the session %s on %s (%s).",
              explanation="A network administrator has confirmed the"
                          "configuration session.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_COMMIT_FAILURE = None
Logging.logD( id="SYS_CONFIG_SESSION_COMMIT_FAILURE",
              severity=Logging.logError,
              format="User %s committed configuration session %s with errors "
                     "on %s (%s)",
              explanation="A network administrator has committed a configuration "
                          "session, but the operation failed. The system may be "
                          " in an inconsistent state.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_ABORTED = None
Logging.logD( id="SYS_CONFIG_SESSION_ABORTED",
              severity=Logging.logNotice,
              format="User %s aborted configuration session %s on %s (%s)",
              explanation="A network administrator has aborted a configuration "
                          "session, thereby having no effect on the running "
                          "configuration of the system.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_DELETED = None
Logging.logD( id="SYS_CONFIG_SESSION_DELETED",
              severity=Logging.logNotice,
              format="User %s deleted configuration session %s on %s (%s)",
              explanation="A network administrator has deleted a configuration "
                          "session, thereby having no effect on the running "
                          "configuration of the system.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

def doLog( slog, target, time=None ):
   info = UtmpDump.getUserInfo()
   if slog == SYS_CONFIG_SESSION_COMMIT_TIMER_STARTED:
      logParams = ( slog, info[ 'user' ], target, info[ 'tty' ],
                    info[ 'ipAddr' ] ) + time
   elif slog == SYS_CONFIG_SESSION_COMMIT_TIMER_UPDATED:
      logParams = ( slog, info[ 'user' ] ) + time + \
                  ( target, info[ 'tty' ], info[ 'ipAddr' ] )
   else:
      logParams = ( slog, info[ 'user' ], target, info[ 'tty' ],
                    info[ 'ipAddr' ] )
   Logging.log( *logParams )

def getComponentName( url ):
   return os.path.splitext( os.path.basename( url ) )[ 0 ]

LOAD_CONFIG_ERROR = "Configuration replace aborted due to errors in loading the file"

# Sets up a temporary config session for config replace.
# Create a temporary config session, rollback to clean config, and apply @surl to
# the session (alternatively, rollback from a given session).
# Returns a tuple of the temporary config session name and any error message, so that
# the session can be committed later.
def configReplaceSession( mode, surl, ignoreErrors=False, sessionName="",
                          md5=None, replace=True, noCommit=False,
                          loadAsStartupConfig=False, ignoreENOENT=False ):
   assert not sessionName or replace
   em = mode.entityManager
   CliSession.exitSession( em )

   target = "cfg"
   if surl and not sessionName:
      target = getComponentName( str( surl ) )
   if not surl and sessionName:
      target = sessionName

   targetSessionName = CliSession.uniqueSessionName( em, prefix="cfg" )
   response = CliSession.enterSession( targetSessionName, em, noCommit=noCommit )
   if response:
      return targetSessionName, response

   try:
      configSet = CliSession.sessionConfig.pendingChangeConfigDir.config
      pcc = configSet.get( targetSessionName )
      # add a description to userString if Cli provides one. TODO
      pcc.userString = "config %s of %s" % ( "replace" if replace else "copy",
                                             target )
      if sessionName:
         response = CliSession.rollbackFromSession( em, sessionName )
         if response:
            CliSession.abortSession( em )
            return targetSessionName, response
      else:
         if replace:
            # rollback from clean-config
            response = CliSession.rollbackSession( em )
            if response:
               CliSession.abortSession( em )
               return targetSessionName, response
         # load file:
         #
         # Disable AAA for config replace (there is no benefit doing it since
         # we just did rollback)
         sessionConfig = SessionUrlUtil.sessionConfig(
            mode.entityManager, True, mode.session,
            sessionName=targetSessionName )
         sessionConfig.raiseOnException = not ignoreErrors
         # Since we are replacing with a new config, no need to validate
         # the new one.
         sessionConfig.skipConfigCheck = True
         sessionConfig.startupConfig = loadAsStartupConfig
         sessionConfig.md5 = md5
         error = ""
         try:
            t0( 'Loading config from', surl, 'into session' )
            error = FileCliUtil.copyFile( None, mode, surl, sessionConfig,
                                          commandSource="configure replace",
                                          ignoreENOENT=ignoreENOENT )
         except CliCommon.LoadConfigError, e:
            error = LOAD_CONFIG_ERROR
         except CliCommon.MD5Error, e: # pylint: disable-msg=E1101
            error = e.msg

         if error != "":
            t0( 'Loading config resulted in error:', error )
            CliSession.abortSession( em )
            CliSession.exitSession( em )
            return targetSessionName, error
   except KeyboardInterrupt:
      CliSession.abortSession( em )
      return targetSessionName, "Keyboard Interrupt"
   else:
      return targetSessionName, ""
   finally:
      CliSession.exitSession( em )

# Invoke on-commit handlers on session. If there is an error, rollback to checkpoint
# and invoke the on-commit handlers again on the rolled back config.
# Returns
#  "OK": if on-commit handlers did not complain
#  "some err msg" to display to user if an on-commit handler complained
# Any error display cannot be done now and has to be delayed to ensure it is not
# buried after a forest of warning during the rollback...
# There are 3 error messages/cases:
#   the commit failed and we rolled back
#   the commit failed and the rollback failed too, because either
#      the checkpoint application failed,
#      the checkpoint was applied but its commit failed
def invokeOnCommitHandlersOrRevert( mode, sessionName, checkpointUrl ):
   em = mode.entityManager

   # We just committed the session. Now go on to commit registered
   # data store(s) and invoke on commit handlers. If there is an
   # error, revert to saved checkpoints.
   success = True
   if checkpointUrl:
      # Since we have a checkpoint of the main session to revert to,
      # also save a checkpoint of any registered data stores
      success = CliSessionDataStore.backup( mode, sessionName )

   if success:
      success = ( CliSessionDataStore.commit( mode, sessionName ) and
                  CliSessionOnCommit.invokeSessionOnCommitHandlers( mode,
                                                                    sessionName ) )

   errorPrefix = "ERROR: session commit failed"
   if success:
      ret = None
   elif not checkpointUrl:
      ret = "%s, no checkpoint to rollback to" % errorPrefix
   else:
      mode.addError( "Error during session commit, starting rollback" )
      # revert data store, which cannot fail
      CliSessionDataStore.restore( mode, sessionName )
      # Revert back to checkpointUrl if there is an error from the onCommit handlers
      rollbackSessName, error = configReplaceSession( mode, checkpointUrl )
      if not error:
         rollbackResponse = CliSession.commitSession( em,
                                          sessionName=rollbackSessName )
         if rollbackResponse:
            ret = "%s, commit of rollback failed too" % errorPrefix
            doLog( SYS_CONFIG_REPLACE_FAILURE, checkpointUrl )
         else:
            # Invoke the commit handlers again for the rolled back config
            CliSessionOnCommit.invokeSessionOnCommitHandlers( mode,
               sessionName )
            doLog( SYS_CONFIG_REPLACE_SUCCESS, checkpointUrl )
            ret = "%s, config rolled back" % errorPrefix
      else:
         ret = "%s, rollback failed too" % errorPrefix
         doLog( SYS_CONFIG_REPLACE_FAILURE, checkpointUrl )
   CliSession.discardSessionOnCommitHandlers( sessionName )
   return ret
