# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
Creates a mechanism to allow Cli commands to use config entities
from Sysdb (serialized through the Cli Session Manager) if there is no
pending session, but to use config entities from the pending session
if a session is in progress.

The basic idea is that the plugin obtains a proxy entity by calling
ConfigMount.mount, similar to LazyMount.mount.  The proxy entity checks on 
every access whether a session is active in this process, and if so, it 
returns the entity with the given pathname in the pending-config, rather than
in Sysdb.  It mounts, of course, as necessary.  If the entity is not yet
mounted, then it adds the mountPath to the root collection of the 
pending-config.

Limitations: the current implementation requires that config mounted
entities not be accessed from the activity thread (they may go blocked
on the mount).  This restriction precludes using then within
single-threaded agents (but, then, LazyMount already has the same restriction).
"""

import threading

import Tac
import functools
import Tracing
import EntityManager
import LazyMount
import CliSession as CS

_Proxy = LazyMount._Proxy  # pylint: disable-msg=W0212
_proxies = LazyMount._proxies # pylint: disable-msg=W0212

th = Tracing.defaultTraceHandle()
t0 = th.trace0
t1 = th.trace1
t8 = th.trace8
t9 = th.trace9

class ConfigMountProhibitedError( Exception ):
   # common reasons
   CONFIG_LOCKED = "configuration is locked by another session"

   def __init__( self, reason ):
      self.reason = reason
      # NOTE: this message is displayed by the CLI.
      msg = 'Unable to run this command (' + reason + ')'
      Exception.__init__( self, msg )

# threadLocalData object needs to be created just once and not per thread. This
# object grabs the locks when attributes are accessed. The attributes are local
# to the thread.
threadLocalData = threading.local()

def disableConfigMountsIs( disable ):
   threadLocalData.disableConfigMounts = disable
   return disable

def disableConfigMounts():
   return getattr( threadLocalData, 'disableConfigMounts', False )

def prohibitConfigMountsIs( prohibited ):
   threadLocalData.prohibitConfigMounts = prohibited

def prohibitConfigMounts():
   return getattr( threadLocalData, 'prohibitConfigMounts', False )

class ConfigMountDisabler( object ):
   """ Context manager that automatically disables configMounts"""
   def __init__( self, disable=True ):
      self.active_ = False
      self.disable_ = disable
      self.old_ = disableConfigMounts()

   def __enter__( self ):
      self.old_ = disableConfigMounts()
      self.active_ = True
      disableConfigMountsIs( self.disable_ )

   def __exit__( self, _exceptionType, _value, _traceback ):
      # should we check that self.disabled_ == True?  
      if self.old_ != disableConfigMounts():
         disableConfigMountsIs( self.old_ )
      self.active_ = False

class GlobalConfigDisabler( object ):
   """ Context manager that prohibits configMounts"""
   def __init__( self, prohibited ):
      self.prohibited_ = prohibited
      self.old_ = None

   def __enter__( self ):
      self.old_ = prohibitConfigMounts()
      prohibitConfigMountsIs( self.prohibited_ )

   def __exit__( self, _exceptionType, _value, _traceback ):
      prohibitConfigMountsIs( self.old_ )
      self.old_ = None

# Decorator for holding the configMount disabler for the duration of the
# function.
def withNoConfigMounts( func ):
   @functools.wraps( func )
   def _withNoConfigMounts( *args, **kwargs ):
      with ConfigMountDisabler():
         return func( *args, **kwargs )
   return _withNoConfigMounts

def withConfigMountsEnabled( func ):
   @functools.wraps( func )
   def _withConfigMounts( *args, **kwargs ):
      with ConfigMountDisabler( disable=False ):
         return func( *args, **kwargs )
   return _withConfigMounts

sessionCleanupHandlerRegistered_ = False

# all ConfigMount proxies to be notified of config session deletions
sessionCleanupNotifiees_ = {} # sessionName -> proxies

def configMountSessionCleanupHandler( sessionName ):
   # We use activity lock as deleting and freeing GenericIf objects
   # require activity lock. If we use another lock we'd still have
   # to grab activity lock first to avoid deadlock.
   with Tac.ActivityLockHolder():
      notifiees = sessionCleanupNotifiees_.get( sessionName, set() )
      for p in notifiees:
         t1( "Remove session", sessionName, "for", p )
         del p.sessionObj_[ sessionName ]
      if notifiees:
         del sessionCleanupNotifiees_[ sessionName ]

def mount( entityManager, path, typename, flags ):
   """Register a config mount.  A proxy for the mounted entity is returned.  
   Any access to the proxy will check whether we are currently in a session,
   or not, and check whether the mounted object has been mounted yet for
   this session.  This will cause the mount to be performed if it has not yet
   been done.

   Caution: Like LazyMount proxies, ConfigMount proxies should not be 
   accessed from the activity thread!
   """
   global sessionCleanupHandlerRegistered_
   if not sessionCleanupHandlerRegistered_:
      sessionCleanupHandlerRegistered_ = True
      CS.registerSessionCleanupHandler( configMountSessionCleanupHandler )

   if path[ -1 ] == "/":
      path = path[ 0: -1 ]
   if LazyMount.directMode_:
      return entityManager.root().entity[ EntityManager.cleanPath( path ) ]

   fullPath = entityManager.mountKey( path )
   p = _proxies.get( fullPath )
   newFlags = CS.prefixIs( entityManager, path, flags, typename )
   # with preinit profiles all config mount points have "wi" flags
   if 'r' in flags and 'w' not in flags and 'w' in newFlags:
      # This is to prevent destructive resync because of flags going from "w" to "r"
      print "path: " + path + ", type: " + typename + ", flags: " + flags +\
            ", newFlags: " + newFlags
      assert False, "Read-only ConfigMount not allowed"
   if p is not None:
      assert p.typename_ == typename, \
          "Conflicting types for the same mount path " + path

      if p.flags_ != newFlags:
         t0( "Upgrading flags for ConfigMount proxy type", typename, "at", path, 
             "from", p.flags_, "to", newFlags )
         p._upgradeFlags( newFlags ) # pylint: disable-msg=W0212
      if not isinstance( p.impl_, ProxyImpl ):
         t0( "Found LazyMount proxy. Converting it to ConfigMount proxy" )
         p.convertTo( ProxyImpl )
      return p
   else:
      t0( "Creating ConfigMount proxy for", typename, "at", path, 
          "w/ flags", flags, "(new: %s)" % newFlags )
      return LazyMount.mount( entityManager, path, typename, newFlags,
                              implClass=ProxyImpl )

force = LazyMount.force

def isConfigMount( proxy ):
   return isinstance( proxy.impl_, ProxyImpl )

def doDeferredMounts( block=True, callback=None ):
   """Perform all unmounted config mounts immediately, outside of
   a session. returns True if the callback will be called."""
   # Essentially the same implementation as LazyMount, but without the filter
   t0( "doDeferredMounts block:", block, "callback", callback )
   with ConfigMountDisabler():
      return LazyMount.doDeferredMounts( filterFunc=isConfigMount,
                                         block=block,
                                         callback=callback )

class MountGroupMgr( object ):
   " Context manager that automatically manages a mountGroup"
   def __init__( self, em, threadSafe=True, block=True ):
      self.em_ = em
      self.threadSafe_ = threadSafe
      self.block_ = block
      self.mg_ = None

   def __enter__( self ):
      self.mg_ = self.em_.mountGroup( threadSafe=self.threadSafe_ )
      return self.mg_

   def __exit__( self, _exceptionType, _value, _traceback ):
      if((( _exceptionType is not None ) or 
          ( _value is not None )) and
         ( self.mg_ is not None ) and 
         ( self.mg_.cMg_.mountStatus == "mountPending" )):
         self.mg_.close( callback=None, blocking=False )

# Can't use EntityManager.entity() because if we are called with Local
# em, then path is relative, if called with Remote em, then path is absolute
# We want to be oblivious of whether we are cohabiting (for tests), or not
# for the real product (and non-cohabiting tests).
def getEntity( em, relativePath, typeName ):
   """Return the already-mounted entity, at path relative to em.root(), with 
      type typeName.
      Raises TypeError if it is not the right type.  Raises KeyError
      if the entity is not found."""
   ent = em.root().entity[ relativePath ]
   if ent.tacType.fullTypeName != typeName:
      raise TypeError( "Entity %s is present but has a different type (%s)"
                       " than requested (%s)" %
                       ( relativePath, ent.tacType.fullTypeName, typeName ) )
   return ent

class ProxyImpl( LazyMount.ProxyImpl ): # pylint: disable-msg=W0212
   baseAttributes = ( LazyMount.ProxyImpl.baseAttributes |
                      set( [ 'sessionObj_' ] ) )

   def __init__( self, entityManager, path, typename, flags ):
      LazyMount.ProxyImpl.__init__( self, entityManager,
                                    path, typename, flags )
      # new members for session mount
      self.sessionObj_ = {} # name -> obj

   def __repr__( self ):
      return "<ConfigMount.ProxyImpl for %s at %s>" % ( self.typename_,
                                                        self.path_ )

   # Do we need to override from LazyMount because a different set of 
   # class baseAttributes?  Or is it sufficient to just change baseAttributes, above?
   def __setattr__( self, attr, value ):
      if attr in self.__class__.baseAttributes:
         object.__setattr__( self, attr, value )
      else:
         self._checkWritePermission()
         self._checkConfigRoot()
         obj = self._mount()
         setattr( obj, attr, value )

   def _checkConfigRoot( self ):
      # Check if the root is disabled
      configRootFilter = CS.activeConfigRootFilter( self.entityManager_ )
      if configRootFilter and self.path_ in configRootFilter.root:
         raise ConfigMountProhibitedError( configRootFilter.name.upper() +
                                           " is active" )

   def _tryMountSession( self, sessionName, activityLockTaken,
                         block=True, threadSafe=None, callback=None ):
      assert not ( callback and threadSafe )
      t8( "ConfigMount._tryMountSession() for session", sessionName,
          self.path_, "flags", self.flags_ )
      sessionObj = self.sessionObj_.get( sessionName )
      if sessionObj:
         t9( "ConfigMount session object present" )
         if callback:
            callback()
         return sessionObj

      t0( "ConfigMount", self.path_, "session", sessionName, "not mounted" )

      with Tac.ActivityUnlockHolder( activityLockTaken ):
         CS.addSessionRoot( self.entityManager_, self.path_ )

      # addSessionRoot should force the copy from Sysdb into the session.
      # enterSession should have mounted the session "wi".
      sessionPath = ( "session/sessionDir/%s/" % sessionName +
                      EntityManager.cleanPath( self.path_ ) )
      try:
         sessionObj = getEntity( self.entityManager_, sessionPath, self.typename_ )
      except KeyError:
         # The session might have been deleted. Return None and let the caller
         # retry.
         t0( "Cannot find session path", sessionPath )
         return None

      # register session notification (we have activity lock)
      self.sessionObj_[ sessionName ] = sessionObj
      sessionCleanupNotifiees_.setdefault( sessionName, set() ).add( self )
      if callback:
         callback()
      return sessionObj

   # overriding LazyMount.ProxyImpl version
   def _mount( self, block=True, threadSafe=None, callback=None ):
      sessionName = ( "" if disableConfigMounts()
                      else ( CS.currentSession( self.entityManager_ ) or "" ) )
      if not sessionName:
         if prohibitConfigMounts():
            raise ConfigMountProhibitedError(
               ConfigMountProhibitedError.CONFIG_LOCKED )
         return LazyMount.ProxyImpl._mount( self, # pylint: disable-msg=W0212
                                            block = block,
                                            threadSafe=threadSafe,
                                            callback=callback )

      with Tac.ActivityLockHolder() as lock:
         while True:
            obj = self._tryMountSession( sessionName,
                                         lock.taken(),
                                         block=block,
                                         threadSafe=threadSafe,
                                         callback=callback )
            if obj is not None:
               return obj

   def sysdbObj( self ):
      if self.obj_:
         return self.obj_
      with ConfigMountDisabler():
         return self._mount()
