# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Toggles.ProcMgrToggleLib
import Logging
import Tracing
import Tac
import os
import time

t0 = Tracing.trace0

SYS_MEMORY_EXHAUSTION_CLI_RESTART = None
restartFormat = "Restarting CLI due to memory exhaustion on the system."
restartRecommendedAction = "CLI will be operational again soon. "\
      "Please try to disable features, then run 'reset system memory exhaustion'."
Logging.logD( id="SYS_MEMORY_EXHAUSTION_CLI_RESTART",
   severity=Logging.logNotice,
   format=restartFormat,
   explanation=( "CLI server is restarted due to memory exhaustion on the system." ),
   recommendedAction=restartRecommendedAction )

class LowMemModeSm( Tac.Notifiee ):
   notifierTypeName = 'ProcMgr::LowMemoryModeStatus'
   def __init__( self, notifier ):
      self.notifier = notifier
      Tac.Notifiee.__init__( self, notifier )
      # Do not run handleStatus() here. Otherwise everytime ConfigAgent restarts
      # under low-memory mode, it will die again, leading to a crashing loop.

   @Tac.handler( 'status' )
   def handleStatus( self ):
      # If the lowMemMode status changed to True, ConfigAgent exits gracefully and 
      # will be restarted by ProcMgr
      if 'A4_CHROOT' in os.environ:
         # Ignore this in autobuild since we don't want CLI to die during tests.
         return
      if self.notifier.status:
         try:
            # Best-effort logging and broadcasting the restart message
            Logging.log( SYS_MEMORY_EXHAUSTION_CLI_RESTART )
            argv = [ 'wall', restartFormat + " " + restartRecommendedAction ]
            Tac.run( argv, asRoot=True )
            time.sleep( 1 )
         # pylint: disable-msg=bare-except
         except:
            pass
         # pylint: disable-msg=protected-access
         os._exit( 0 )

class LowMemModeSmStarter( object ):
   def __init__( self, em, mg, agent ):
      self.em_ = em
      self.agent_ = agent
      self.lowMemModeStatus_ = None
      self.lowMemModeSm = None
      self.lowMemModeToggled = Toggles.ProcMgrToggleLib.toggleLowMemModeEnabled()
      self.doMounts( mg )

   def doMounts( self, mg ):
      if self.lowMemModeToggled:
         self.lowMemModeStatus_ = mg.mount( 'sys/status/lowMemStatus',
                                       'ProcMgr::LowMemoryModeStatus', 'r' )
   
   def run( self ):
      if self.lowMemModeToggled:
         t0( "Creating LowMemModeSm" )
         self.lowMemModeSm = LowMemModeSm( self.lowMemModeStatus_ )

def Plugin( context ):
   context.registerStateMachine( LowMemModeSmStarter )
