#!/usr/bin/python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import Tracing
import Tac

t0 = Tracing.trace0

# from sys/logging/config to logging/config
class SysLogConfigReactor( Tac.Notifiee ):
   notifierTypeName = "LogMgr::LogConfig"

   def __init__( self, sysLogConfig, tacLogConfig ):
      Tac.Notifiee.__init__( self, sysLogConfig )
      self.tacLogConfig_ = tacLogConfig
      self.handleLoggingOn()
      self.handleRepeatInterval()
      self.handleSequenceNumbersEnabled()
      for name in sysLogConfig.facilityLogLevel:
         self.handleFacilityLogLevel( name )

   @Tac.handler( 'loggingOn' )
   def handleLoggingOn( self ):
      self.tacLogConfig_.loggingOn = self.notifier_.loggingOn

   @Tac.handler( 'repeatInterval' )
   def handleRepeatInterval( self ):
      self.tacLogConfig_.repeatInterval = self.notifier_.repeatInterval

   @Tac.handler( 'sequenceNumbersEnabled' )
   def handleSequenceNumbersEnabled( self ):
      self.tacLogConfig_.sequenceNumbersEnabled = \
           self.notifier_.sequenceNumbersEnabled

   @Tac.handler( 'facilityLogLevel' )
   def handleFacilityLogLevel( self, name ):
      # this name should always exist in tacLogConfig_
      level = self.notifier_.facilityLogLevel.get(
         name, self.notifier_.facilityLogLevelDefault )
      self.tacLogConfig_.facilityConfig[ name ].logLevel = level

class SysLogStatusReactor( Tac.Notifiee ):
   notifierTypeName = "LogMgr::LogStatus"

   def __init__( self, sysLogStatus, tacLogConfig ):
      Tac.Notifiee.__init__( self, sysLogStatus )
      self.tacLogConfig_ = tacLogConfig
      self.handleSeverityThreshold()

   @Tac.handler( 'severityThreshold' )
   def handleSeverityThreshold( self ):
      # We update the severity threshold of the Tac::LogConfig
      # object with this value as well, so that the logger can do
      # filtering right at the source instead of wasting time generating
      # log messages that are going to get dropped anyway.
      # This is done only on the active supervisor.
      self.tacLogConfig_.severityThreshold = self.notifier_.severityThreshold

class RedundancyStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Redundancy::RedundancyStatus'

   def __init__( self, redundancyStatus, sysLogConfig, sysLogStatus, tacLogConfig ):
      Tac.Notifiee.__init__( self, redundancyStatus )
      self.redundancyStatus_ = redundancyStatus
      self.sysLogConfig_ = sysLogConfig
      self.sysLogStatus_ = sysLogStatus
      self.tacLogConfig_ = tacLogConfig
      self.sysLogConfigReactor_ = None
      self.sysLogStatusReactor_ = None
      self.handleMode()

   @Tac.handler( 'mode' )
   def handleMode( self ):
      t0( "handleMode: protocol=%s, mode=%s" % ( self.notifier_.protocol,
                                                 self.notifier_.mode ) )
      if self.notifier_.mode == "active" or self.notifier_.protocol != "sso":
         self.sysLogConfigReactor_ = SysLogConfigReactor( self.sysLogConfig_,
                                                          self.tacLogConfig_ )
         self.sysLogStatusReactor_ = SysLogStatusReactor( self.sysLogStatus_,
                                                          self.tacLogConfig_ )

class LogMgrConfigAgentPluginStarter( object ):
   def __init__( self, em, mg, agent ):
      self.em_ = em
      self.agent_ = agent
      self.sysLogConfig_ = None
      self.sysLogStatus_ = None
      self.tacLogConfig_ = None
      self.redundancyStatusReactor_ = None

      self.doMounts( mg )

   def doMounts( self, mg ):
      self.sysLogConfig_ = mg.mount( 'sys/logging/config', 'LogMgr::LogConfig',
                                     'ri' )
      self.sysLogStatus_ = mg.mount( Cell.path( 'sys/logging/status' ),
                                     'LogMgr::LogStatus','r' )
      self.tacLogConfig_ = mg.mount( 'logging/config', 'Tac::LogConfig', 'w' )

   def run( self ):
      self.redundancyStatusReactor_ = RedundancyStatusReactor(
            self.agent_.redundancyStatus(), self.sysLogConfig_, self.sysLogStatus_,
            self.tacLogConfig_ )

def Plugin( context ):
   context.registerStateMachine( LogMgrConfigAgentPluginStarter )
