# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing
import Tac
from CloudHelper import CloudHelper
import CloudHa
import Logging
from CloudException import MissingConfigError, \
   InvalidJson, ConfigInvalid
from CloudUtil import defaultRecoveryWaitTime

t0 = Tracing.t0
t1 = Tracing.t1
t2 = Tracing.t2
t3 = Tracing.t3
t8 = Tracing.t8

Logging.logD( id='CLOUDHA_UPDATED_LOCAL_SUBNET_ROUTES',
              severity=Logging.logInfo,
              format="Routes for locally connected subnets now point "
                  "to my interfaces",
              explanation="Local subnets are now default routed via my local "
                  "interfaces",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

Logging.logD( id='CLOUDHA_UPDATED_PEER_SUBNET_ROUTES',
              severity=Logging.logInfo,
              format='Routes for peer connected subnets now point '
                  'to my interfaces',
              explanation='Peer subnets are now routed via my local '
                  'interfaces',
              recommendedAction=Logging.NO_ACTION_REQUIRED )

Logging.logD( id='CLOUDHA_ERROR_UPDATE_LOCAL_SUBNET_ROUTES',
              severity=Logging.logError,
              format="Error setting routes for locally connected subnets to "
                     "point to my interfaces( %s )",
              explanation="Error occurred when trying to set local subnet "
                     "routes to point to local interface.",
              recommendedAction="Check that the configured local subnets, interface "
                     "IDs and access credentials are valid." )

Logging.logD( id='CLOUDHA_ERROR_UPDATE_PEER_SUBNET_ROUTES',
              severity=Logging.logError,
              format="Error setting routes for peer subnets to "
                     "point to my interfaces( %s )",
              explanation="Error occurred when trying to set peer subnet "
                     "routes to point to local interface.",
              recommendedAction="Check that the configured peer subnets, local "
                     "interface IDs and access credentials are valid." )

Logging.logD( id='CLOUDHA_FAILOVER',
              severity=Logging.logError,
              format='Failed to hear from peer VEOS instance %s over %s.',
              explanation='Subnets connected to peer are now being routed via local '
                     'interfaces',
              recommendedAction="Check if the peer VEOS instance is still up and "
                     "has BFD connectivity to me." )

Logging.logD( id='CLOUDHA_ERROR_RUNTIME_CONFIG',
              severity=Logging.logError,
              format='Error validating Cloud-HA config( %s )',
              explanation='The runtime Cloud-HA config is not valid',
              recommendedAction="Check that the CLOUD-HA config is valid" )

Logging.logD( id='CLOUDHA_INVALID_JSON',
              severity=Logging.logError,
              format='CLOUD-HA config %s is invalid json( %s )',
              explanation='The Cloud-HA config file is not validi json',
              recommendedAction="Make sure that the CLOUD-HA config is "
                  "valid json file" )

Logging.logD( id='CLOUDHA_CONFIG_MISSING',
              severity=Logging.logError,
              format='CLOUD-HA config %s is missing or unreadable',
              explanation='The Cloud-HA config file is not present',
              recommendedAction="Make sure that the CLOUD-HA config is present" )

Logging.logD( id='CLOUDHA_INVALID_CONFIG',
              severity=Logging.logError,
              format='CLOUD-HA config is invalid( %s )',
              explanation='The Cloud-HA config file is invalid',
              recommendedAction="Make sure that the CLOUD-HA config is valid" )

Logging.logD( id='CLOUDHA_VALID_CONFIG',
              severity=Logging.logInfo,
              format='Successfully validated CLOUD-HA config',
              explanation='The Cloud-HA config is runtime checked to be correct',
              recommendedAction=Logging.NO_ACTION_REQUIRED )


######################################################################
#  
# Events for the State Machines 
#
#
######################################################################

peerDownEventName = "PEER_DOWN_EVENT"
peerAdminDownEventName = "PEER_ADMIN_DOWN_EVENT"
peerUpEventName = "PEER_UP_EVENT"
configChangedEventName = "CONFIG_CHANGE_EVENT"
validConfigEventName = "VALID_CONFIG_EVENT"
hysteresisTimerExpiredEventName = "HYSTERESIS_TIMER_EXPIRED"

class Input( object ):
   def __init__( self, name ):
      self.name = name
   def __str__( self ):
      return 'HA-EVENT-' + self.name

# The BFD has detected peer and/or self state is not down
# and grounds for failover.
class PeerDown( Input ):
   def __init__( self ):
      Input.__init__( self, peerDownEventName )
 
# BFD will notify 'admin down' if the peer or local BFD 
# has been shut down and/or peer bfd config is changed/
# removed. We don't want to go to failover for such cases.
# This event should normally trigger SM to go back to
# init state and wait for BFD state to go "UP" again.
class PeerAdminDown( Input ):
   def __init__( self ):
      Input.__init__( self, peerAdminDownEventName )

class PeerUp( Input ):
   def __init__( self ):
      Input.__init__( self, peerUpEventName )

class ValidConfig( Input ):
   def __init__( self ):
      Input.__init__( self, validConfigEventName )
      #self.config = config

class ConfigChanged( Input ):
   def __init__( self ):
      Input.__init__( self, configChangedEventName )

class HysteresisTimerExpired( Input ):
   def __init__( self ):
      Input.__init__( self, hysteresisTimerExpiredEventName )


######################################################################
#  
# States
#
#
######################################################################

haInitStateName = 'HA_INIT_STATE'
haDisabledStateName = 'HA_DISABLED_STATE'
haConnectedStateName = 'HA_CONNECTED_STATE'
haReadyStateName = 'HA_READY_STATE'
haFailoverStateName = 'HA_FAILOVER_STATE'

class State( object ):
   def __init__( self, name, stateMachine ):
      self.name = name
      
      # I doubt that we need to use weakref.proxy here
      # as the StateMachine class is holding the reference and
      # not the stateMachine instance object...
      self.stateMachine = stateMachine
      assert stateMachine

      # Derived class should fill all the transition rules.
      # The format is dictionary of events and each row is 
      # tuple of condition check function and nextState.
      # If this is unconditional transition, then condition function 
      # can be None.
      self.rule = {}
      # Add common rule
      self.rule = { configChangedEventName : ( None, haDisabledStateName ) }
      self.clockHandler = None
      # We may have to queue events if we are waiting for backend response.
      # I think we can only have config change events only as user can change
      # config any time under use while we are waiting for backend response.
      self.callback = None

   def run( self, inputEv ) :
      t8('NOP: %s should implement event handler %s if needed' % \
         ( self.name, inputEv ) )

   def __str__( self ):
      return self.name

   # This checks whether it is OK to go to next state. If not OK,
   # the main SM will queue the event and retrigger when appropriate.
   # We expect only multiple configChange events only though.
   # Revisit this assumption
   def stateChangePrecondition( self, event ):
      if self.stateMachine.cloudHelper and \
         self.stateMachine.cloudHelper.responsePending():
         # Start background response handling
         return False
      else:
         return True

   # This is called while exiting the current state.
   # Derive class can enhance it if needed.
   # This handles the common case of stopping any timer handlers
   def exitHandler( self, inputEvt ):
      t2( 'Exit Handler invoked for state %s ' % str( self) )
      if self.clockHandler and self.clockHandler.timeMin != Tac.endOfTime:
         t0( 'Canceling Clock Handler' )
         self.clockHandler.timeMin = Tac.endOfTime
         self.callback = None

   def asyncClockCallback( self ):
      t8( 'asynClockCallback' )
      if not self.stateMachine.cloudHelper.asyncResponseDone():
         # We need to aynchronously check backend response
         assert self.clockHandler
         t8( 'Restarting clock handler' )
         self.clockHandler.timeMin = Tac.now() + 0.5 
      else:
         t0( 'async clock callback invoked for state: %s callback: %s ' % ( \
            str( self ), self.callback ) )
         self.callback()
         self.callback = None
         # poke main SM to handle any pending events too
         self.stateMachine.handlePendingEvents()

   # Common api to async check backend notification
   def asyncResponseHandler( self, callback ):
      t2( 'asyncResponseHandler invoked for callback:', callback )
      assert not self.callback, 'Callback already set'
      assert callback, 'No callback given'
      self.callback = callback
      if not self.stateMachine.cloudHelper.asyncResponseDone():
         t0( 'Starting clockHandler' )
         # We need to aynchronously check backend response
         self.clockHandler = Tac.ClockNotifiee( self.asyncClockCallback, 
                  timeMin=Tac.now() + 0.5 )
      else:
         # We got response synchronously. Not sure if it can happen 
         # in real world. but handle for mock test case anyway...
         t0( 'Got immediate response' )
         self.callback()
         self.callback = None

class HaDisabledState( State ):
   def __init__( self, stateMachine ):
      State.__init__( self,  haDisabledStateName, stateMachine )
      self.rule.update( { validConfigEventName : ( None, haInitStateName ) } )
      self.validator = None

   def staticConfigValidation( self ):
      t2( 'Initiating static config validation' )
      assert self.stateMachine.agent
      agent = self.stateMachine.agent
      configHandler = agent.cloudConfigHandler
      assert configHandler
      try:
         configHandler.handleSysdbConfiguration( self.stateMachine )
      except InvalidJson, e:
         # pylint: disable-msg=E0602
         Logging.log( CLOUDHA_INVALID_JSON, e.fileName, e.error )
         t0( 'Error in %s: %s' % ( e.fileName, e.error ) )
         # We will rely on inotify to retriger us
         # after config change.
         return False
      except MissingConfigError, e:
         t0( e )
         # pylint: disable-msg=E0602
         Logging.log( CLOUDHA_CONFIG_MISSING, e.loc )  
         return  False
      except ConfigInvalid, e:
         t0( e )
         # pylint: disable-msg=E0602
         Logging.log( CLOUDHA_INVALID_CONFIG, e.error  )  
         return  False

      config = agent.cloudConfigHandler.config
      if not config:
         t0( 'HA Config is not found' )
         return False
      # Also start the statemachine to trigger 
      # creation of backend driver.
      self.stateMachine.config = config
      t0( ' Creating backend Helper' )
      self.stateMachine.createBackendHelper()
      return True

   def runTimeConfigValidation( self ):
      t2( 'Initiating runtime config validation' )
      # Update status to reflect our state accurately.
      #self.stateMachine.agent.cloudStatusHandler.initializeSysdbStatus()
      self.stateMachine.cloudHelper.runTimeConfigValidation()

   def asyncResponseCallback( self ):
      response = self.stateMachine.cloudHelper.getResponse()
      result = response.ret
      # pylint: disable-msg=E0602
      if result:
         t0( 'Backend successfully validated user provided config' )
         Logging.log( CLOUDHA_VALID_CONFIG )
         self.stateMachine.agent.cloudStatusHandler.updateSysdbStatus( 'valid', 
               'init' )
         self.stateMachine.sendEvent( ValidConfig() ) 
      else:
         t0( 'Backend Failed to validate provided config: %s exception: %s ' % \
               ( response.message, response.exception ) )
         self.stateMachine.agent.cloudStatusHandler.updateSysdbStatus( 'invalid', 
               'init' )
         Logging.log( CLOUDHA_ERROR_RUNTIME_CONFIG, response.message )

   def run( self, inputEv ):
      t0( 'Running in state: ', self ) 
      # Update status to reflect our state accurately.
      self.stateMachine.agent.cloudStatusHandler.initializeSysdbStatus()
      agent = self.stateMachine.agent
      configHandler = agent.cloudConfigHandler
      if self.staticConfigValidation():
         # Make sure the transition is allowed only if config is enabled if present
         genConfig = self.stateMachine.config[ 'generalConfig' ]   
         enable = genConfig.get( 'enable_optional', 'true' )
         if enable.lower() == 'true' :
            self.runTimeConfigValidation() 
            # Initiate background runtime config check
            self.asyncResponseHandler( self.asyncResponseCallback )
         else:
            t0( 'CloudHa exiting as disabled in config' )
            self.stateMachine.agent.cloudStatusHandler.updateSysdbStatus( \
                  state='disabled' )
            # Cleanup BFD etc..   
            configHandler.cleanup()
            return
      else:
         #Invalid config. Clear out our status 
         t0( 'static config check failed.' )
         self.stateMachine.agent.cloudStatusHandler.updateSysdbStatus( \
                  'invalid', state='init' )
         #Also clear old BFD entries.
         configHandler.cleanup()

class HaInitState( State ):
   def __init__( self, stateMachine ):
      State.__init__( self,  haInitStateName, stateMachine )
      self.rule.update( { 
            configChangedEventName : ( None , haDisabledStateName ),
            peerUpEventName : ( None , haConnectedStateName )   
         } )

   # Start the BFD session as our config is good to go.
   def run( self, inputEv ):
      t0( 'Running in state: ', self ) 
      self.stateMachine.startOrFlapBfdSession()
      self.stateMachine.agent.cloudStatusHandler.updateSysdbStatus( \
            state='waiting' ) 

class HaConnectedState( State ):
   def __init__( self, stateMachine ):
      State.__init__( self,  haConnectedStateName, stateMachine )
      self.rule.update( { 
            configChangedEventName : ( None, haDisabledStateName ),
            peerAdminDownEventName: ( None , haInitStateName ),  
            peerDownEventName: ( None , haInitStateName ),  
            peerUpEventName : ( None, haConnectedStateName ), 
            hysteresisTimerExpiredEventName : ( None, haReadyStateName ),
         } )
      self.timer_handler = None

   def timerHandler( self ):
      t0( " clockHandler called " )
      self.clockHandler.timeMin = Tac.endOfTime
      self.clockHandler =  None
      self.stateMachine.sendEvent( HysteresisTimerExpired() )

   # Start the hysteresis timer to go to HA_READY
   def run( self, inputEv ):
      t0( 'Running in state: ', self ) 
      config = self.stateMachine.config[ 'generalConfig' ]
      # Use default 30 sec
      hT = config.get( 'hysteresis_time_optional', None )
      hTime = int( hT ) if hT else defaultRecoveryWaitTime
      hysteresis_time = hTime
      t0( 'Starting hysteresis timer for %s ' % hysteresis_time )
      self.clockHandler = Tac.ClockNotifiee ( self.timerHandler, \
         timeMin=hysteresis_time + Tac.now() )
      self.stateMachine.agent.cloudStatusHandler.updateSysdbStatus( \
                  state='connected' )
class HaReadyState( State ):
   def __init__( self, stateMachine ):
      State.__init__( self,  haReadyStateName, stateMachine )
      self.rule.update(  { 
            configChangedEventName : ( None, haDisabledStateName ),
            peerAdminDownEventName: ( None , haInitStateName ),  
            peerDownEventName: ( None , haFailoverStateName ),  
         } )

   def asyncResponseCallback( self ):
      t8( 'asyncResponseCallback in HaReadyState' )
      assert self.stateMachine.cloudHelper.asyncResponseDone()
      response = self.stateMachine.cloudHelper.getResponse()
      # pylint: disable-msg=E0602
      if response.ret:
         Logging.log( CLOUDHA_UPDATED_LOCAL_SUBNET_ROUTES )
         t0( 'Updated local route tables to self')
      else:
         Logging.log( CLOUDHA_ERROR_UPDATE_LOCAL_SUBNET_ROUTES, \
            response.message )
         t0( 'Failed to Update local route tables to self ', \
            response.message )

   def run( self, inputEv ):
      t0( 'Running in state: ', self ) 
      # Get our traffic back to us from peer.
      self.stateMachine.updateLocalRoutes()
      self.stateMachine.agent.cloudStatusHandler.updateSysdbStatus( \
                  state='ready' )
      self.asyncResponseHandler( self.asyncResponseCallback )  
      
class HaFailoverState( State ):
   def __init__( self, stateMachine ):
      State.__init__( self,  haFailoverStateName, stateMachine )
      self.rule.update( { 
         configChangedEventName : ( None, haDisabledStateName ),
         peerUpEventName: ( None , haConnectedStateName ),  
      } )

   def asyncResponseCallback( self ):
      t8( 'asyncResponseCallback in HaFailoverState' )
      assert self.stateMachine.cloudHelper.asyncResponseDone()
      response = self.stateMachine.cloudHelper.getResponse()
      # pylint: disable-msg=E0602
      if response.ret:
         Logging.log( CLOUDHA_UPDATED_PEER_SUBNET_ROUTES )
         t0( 'Updated peer route tables to self')
      else:
         Logging.log( CLOUDHA_ERROR_UPDATE_PEER_SUBNET_ROUTES, \
            response.message )
         t0( 'Failed to Update peer route tables to self: ', \
            response.message )

   def run( self, inputEv ):
      t0( 'Running in state: ', self ) 
      # take over traffic from peer
      self.stateMachine.handleFailover()
      self.stateMachine.agent.cloudStatusHandler.updateSysdbStatus( \
                  state='failover' )
      self.asyncResponseHandler( self.asyncResponseCallback )  

   def exitHandler( self, inputEvt ):
      t0(' Exiting failover state' )
      if inputEvt.name == peerUpEventName:
         t0( 'Recovering from Failover' )
         self.stateMachine.agent.cloudStatusHandler.updateRecoveryTime()
      super( HaFailoverState, self).exitHandler( inputEvt ) 

######################################################################
#  
# State Machines
#
#
######################################################################

# Base handler for various cloud type such as AWS, Azure etc
# This basically drives the cloud HA state machine based on inputs from
# BFD status, config etc.
class CloudHaStateMachine( object ):

   def __init__( self, agent, cloudType ):
      #
      # Instantiate all States apriory. This has to be done here instead of 
      # usual in class method as we need to pass the 'self'.
      #
      CloudHaStateMachine.haDisabledState = HaDisabledState( self )
      CloudHaStateMachine.haInitState = HaInitState( self )
      CloudHaStateMachine.haConnectedState = HaConnectedState( self )
      CloudHaStateMachine.haReadyState = HaReadyState( self )
      CloudHaStateMachine.haFailoverState = HaFailoverState( self )
   
      CloudHaStateMachine.stateMachines = {
         str( CloudHaStateMachine.haDisabledState ) : \
            CloudHaStateMachine.haDisabledState,
         str( CloudHaStateMachine.haInitState ) :  \
            CloudHaStateMachine.haInitState,
         str( CloudHaStateMachine.haConnectedState ) : \
            CloudHaStateMachine.haConnectedState,
         str( CloudHaStateMachine.haReadyState ) :  \
            CloudHaStateMachine.haReadyState,
         str( CloudHaStateMachine.haFailoverState ) : \
            CloudHaStateMachine.haFailoverState,
      }

      self.cloudHelper = None
      self.agent = agent
      self.cloudType = cloudType
      self.ha_enabed = True
      self.config = None
      # Instantiate states
      self.cState = CloudHaStateMachine.haDisabledState
      self.cState.run( None )
      # pending events queue. Not sure whether it can be a set.
      self.eventQueue = []

   # send Event to SM
   def sendEvent( self, inputEv ):
      t1( ' Received event %s in state %s' % ( str( inputEv ), \
         str( self.cState ) ) )
      rule = self.cState.rule.get( inputEv.name )
      if rule:
         # Make sure transition condition is true if listed.
         if not rule[ 0 ] or rule[ 0 ]( inputEv.name ):
            # Make sure the precondition is met else we queue and
            # retry later.
            oldState = self.cState   
            if not oldState.stateChangePrecondition( inputEv ):
               t0(' Unable to transition out of %s event %s' % \
                     ( str( oldState ), str( inputEv ) ) )
               # No need to queue config change event more than once
               if not isinstance( inputEv, ConfigChanged ) or \
                     not inputEv.__class__ in \
                        [ i.__class__ for i in self.eventQueue ]:
                  t2( 'Queued event: %s' % inputEv )
                  self.eventQueue.append( inputEv )
               else:
                  t0( 'Ignoring/not queuing event %s in state %s' % \
                     (inputEv, self.cState ) )
               return
               
            self.cState = CloudHaStateMachine.stateMachines[ rule[ 1 ] ]
            t0( 'State Transition: %s ===(%s)===> %s' % ( oldState, inputEv, \
                  self.cState ) )
            # Run an exit handler for the state.
            oldState.exitHandler( inputEv )
            self.cState.run( inputEv )
      else:
         t0( 'Ignoring: Bad input event  %s in state %s' % \
            ( inputEv, self.cState ) ) 

   # This can be called by states to retrigger pending events
   def handlePendingEvents( self ):
      t0( 'Poked SM. Retriggering pending events if any, queue length: %s' % \
         len( self.eventQueue ) )
      while len( self.eventQueue ):
         event = self.eventQueue.pop()
         t0( 'Retriggering event %s' % event )
         self.sendEvent( event )
         t0( 'End of retriggering event %s' % event )

   def sendConfigChangedEvent( self ):
      self.sendEvent( ConfigChanged() )

   def startOrFlapBfdSession( self ):
      t0( 'Apply BFD config in SM' )
      self.agent.cloudConfigHandler.applyBfdConfig()
      t0( 'Creating Bfd status handler SM' )
      CloudHa.createBfdStatusSm( self.agent, self )

   def handleFailover( self ):
      # pylint: disable-msg=E0602
      self.agent.cloudStatusHandler.updateFailoverTime()
      peerIp = self.config[ 'bfdConfig' ][ 'peerVeosIp' ]
      peer_intf = self.config[ 'bfdConfig' ][ 'bfdSourceInterface' ]
      Logging.log( CLOUDHA_FAILOVER, peerIp, peer_intf )
      self.cloudHelper.updatePeerRoutes()

   def updateLocalRoutes( self  ):
      self.cloudHelper.updateLocalRoutes()

   def createBackendHelper( self ):
      assert self.config
      self.cloudHelper = CloudHelper( self.config, self.cloudType )
      assert self.cloudHelper
